# fmt: off
"""Stores the Location class and a list of each location in the game."""

from randomizer.Enums.Items import Items
from randomizer.Enums.Kongs import Kongs
from randomizer.Enums.Levels import Levels
from randomizer.Enums.Locations import Locations
from randomizer.Enums.MoveTypes import MoveTypes
from randomizer.Enums.Types import Types
from randomizer.Enums.VendorType import VendorType
from randomizer.Enums.Maps import Maps


class MapIDCombo:
    """A combination of a map and an associated item ID. If id == -1 and map == 0, has no model 2 item, ignore those."""

    def __init__(self, map=None, id=None, flag=None, kong=Kongs.any):
        """Initialize with given parameters."""
        self.map = map
        self.id = id
        self.flag = flag
        self.kong = kong


class Location:
    """A shufflable location at which a random item can be placed."""

    def __init__(self, level, name, default, location_type, kong=Kongs.any, data=None, logically_relevant=False):
        """Initialize with given parameters."""
        if data is None:
            data = []
        self.name = name
        self.default = default
        self.type = location_type
        self.item = None
        self.delayedItem = None
        self.constant = False
        self.is_reward = False
        self.map_id_list = None
        self.level = level
        self.kong = kong
        self.logically_relevant = logically_relevant  # This is True if this location is needed to derive the logic for another location
        self.placement_index = None
        self.inaccessible = False
        self.smallerShopsInaccessible = False
        self.tooExpensiveInaccessible = False
        self.is_shop = False
        if self.type == Types.Shop:
            self.movetype = data[0]
            self.index = data[1]
            self.vendor = data[2]
            lvl_index = self.level
            if lvl_index == Levels.DKIsles:
                lvl_index = 7
            lst = []
            if self.kong < 5:
                lst.append((self.vendor * 40) + (self.kong * 8) + lvl_index)
            else:
                for kong_index in range(5):
                    lst.append((self.vendor * 40) + (kong_index * 8) + lvl_index)
            self.placement_index = lst
            self.is_shop = True
        elif self.type in (Types.TrainingBarrel, Types.Shockwave):
            self.placement_index = [data[0]]
        elif self.type == Types.Blueprint:
            self.map = data[0]
            level_index = int(self.level)
            if self.level in (Levels.DKIsles, Levels.HideoutHelm):
                level_index = 7
            self.map_id_list = [MapIDCombo(0, -1, 469 + self.kong + (5 * level_index), self.kong)]
        elif self.type in [Types.Medal, Types.IslesMedal, Types.HalfMedal] and self.level != Levels.HideoutHelm:
            level_index = int(self.level)
            if self.type == Types.HalfMedal:
                if self.level == Levels.DKIsles:
                    level_index = 7
                self.map_id_list = [MapIDCombo(0, -1, 0x3D6 + (5 * level_index) + self.kong)]
            else:
                if self.level == Levels.DKIsles:
                    self.map_id_list = [MapIDCombo(0, -1, 0x3C6 + self.kong, self.kong)]
                else:
                    self.map_id_list = [MapIDCombo(0, -1, 549 + self.kong + (5 * level_index), self.kong)]
        elif self.type == Types.Hint:
            level_index = int(self.level)
            self.map_id_list = [MapIDCombo(0, -1, 0x384 + self.kong + (5 * level_index), self.kong)]
        elif self.type in (Types.Banana, Types.Key, Types.NintendoCoin, Types.RarewareCoin, Types.Crown, Types.Medal, Types.Bean, Types.Pearl, Types.Kong, Types.Fairy, Types.RainbowCoin, Types.CrateItem, Types.BoulderItem, Types.Enemies, Types.Cranky, Types.Candy, Types.Funky, Types.Snide):
            if data is None:
                self.map_id_list = []
            else:
                self.map_id_list = data
        self.default_mapid_data = self.map_id_list
        # "Reward" locations are locations that require an actor to exist for the location's item - something not all items have
        if self.default_mapid_data is not None and len(self.default_mapid_data) > 0 and type(self.default_mapid_data[0]) is MapIDCombo and self.default_mapid_data[0].id == -1:
            self.is_reward = True

    def PlaceItem(self, spoiler, item):
        """Place item at this location."""
        self.item = item
        # If we're placing a real move here, lock out mutually exclusive shop locations
        if item != Items.NoItem and self.type == Types.Shop:
            for location in ShopLocationReference[self.level][self.vendor]:
                if spoiler.LocationList[location].smallerShopsInaccessible:
                    continue
                # If this is a shared spot, lock out kong-specific locations in this shop
                if self.kong == Kongs.any and spoiler.LocationList[location].kong != Kongs.any:
                    spoiler.LocationList[location].inaccessible = True
                # If this is a kong-specific spot, lock out the shared location in this shop
                if self.kong != Kongs.any and spoiler.LocationList[location].kong == Kongs.any:
                    spoiler.LocationList[location].inaccessible = True
                    break  # There's only one shared spot to block

    def PlaceConstantItem(self, spoiler, item):
        """Place item at this location, and set constant so it's ignored in the spoiler."""
        self.PlaceItem(spoiler, item)
        self.constant = True

    def SetDelayedItem(self, item):
        """Set an item to be added back later."""
        self.delayedItem = item

    def PlaceDelayedItem(self, spoiler):
        """Place the delayed item at this location."""
        self.PlaceItem(spoiler, self.delayedItem)
        self.delayedItem = None

    def PlaceDefaultItem(self, spoiler):
        """Place whatever this location's default (vanilla) item is at it."""
        self.PlaceItem(spoiler, self.default)
        self.constant = True

    def UnplaceItem(self, spoiler):
        """Unplace an item here, which may affect the placement of other items."""
        self.item = None
        # If this is a shop location, we may have locked out a location we now need to undo
        if self.type == Types.Shop:
            # Check other locations in this shop
            for location_id in ShopLocationReference[self.level][self.vendor]:
                location_obj = spoiler.LocationList[location_id]
                # We always leave locations culled by smaller shops and locations deemed too expensive inaccessible
                if location_obj.smallerShopsInaccessible or location_obj.tooExpensiveInaccessible:
                    continue
                if location_obj.kong == Kongs.any and location_obj.inaccessible:
                    # If there are no other items remaining in this shop, then we can unlock the shared location
                    itemsInThisShop = len([location for location in ShopLocationReference[self.level][self.vendor] if spoiler.LocationList[location].item not in (None, Items.NoItem)])
                    if itemsInThisShop == 0:
                        location_obj.inaccessible = False
                # Locations are only inaccessible due to lockouts. If any exist, they're because this location caused them to be locked out.
                elif location_obj.inaccessible:
                    location_obj.inaccessible = False


DROPSANITY_FLAG_START = 0x426


LocationListOriginal = {
    # Training Barrel locations
    Locations.IslesVinesTrainingBarrel: Location(Levels.DKIsles, "Isles Vines Training Barrel", Items.Vines, Types.TrainingBarrel, Kongs.any, [123]),
    Locations.IslesSwimTrainingBarrel: Location(Levels.DKIsles, "Isles Dive Training Barrel", Items.Swim, Types.TrainingBarrel, Kongs.any, [120]),
    Locations.IslesOrangesTrainingBarrel: Location(Levels.DKIsles, "Isles Oranges Training Barrel", Items.Oranges, Types.TrainingBarrel, Kongs.any, [121]),
    Locations.IslesBarrelsTrainingBarrel: Location(Levels.DKIsles, "Isles Barrels Training Barrel", Items.Barrels, Types.TrainingBarrel, Kongs.any, [122]),
    # Basic moves that didn't require obtaining in vanilla
    Locations.IslesClimbing: Location(Levels.DKIsles, "Climbing Default Location", Items.Climbing, Types.Climbing, Kongs.any),
    # Pre-Given Moves
    Locations.IslesFirstMove: Location(Levels.DKIsles, "Isles Cranky's First Move", Items.ProgressiveSlam, Types.PreGivenMove),
    Locations.PreGiven_Location00: Location(Levels.DKIsles, "Pre-Given Move (00)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location01: Location(Levels.DKIsles, "Pre-Given Move (01)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location02: Location(Levels.DKIsles, "Pre-Given Move (02)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location03: Location(Levels.DKIsles, "Pre-Given Move (03)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location04: Location(Levels.DKIsles, "Pre-Given Move (04)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location05: Location(Levels.DKIsles, "Pre-Given Move (05)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location06: Location(Levels.DKIsles, "Pre-Given Move (06)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location07: Location(Levels.DKIsles, "Pre-Given Move (07)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location08: Location(Levels.DKIsles, "Pre-Given Move (08)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location09: Location(Levels.DKIsles, "Pre-Given Move (09)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location10: Location(Levels.DKIsles, "Pre-Given Move (10)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location11: Location(Levels.DKIsles, "Pre-Given Move (11)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location12: Location(Levels.DKIsles, "Pre-Given Move (12)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location13: Location(Levels.DKIsles, "Pre-Given Move (13)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location14: Location(Levels.DKIsles, "Pre-Given Move (14)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location15: Location(Levels.DKIsles, "Pre-Given Move (15)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location16: Location(Levels.DKIsles, "Pre-Given Move (16)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location17: Location(Levels.DKIsles, "Pre-Given Move (17)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location18: Location(Levels.DKIsles, "Pre-Given Move (18)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location19: Location(Levels.DKIsles, "Pre-Given Move (19)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location20: Location(Levels.DKIsles, "Pre-Given Move (20)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location21: Location(Levels.DKIsles, "Pre-Given Move (21)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location22: Location(Levels.DKIsles, "Pre-Given Move (22)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location23: Location(Levels.DKIsles, "Pre-Given Move (23)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location24: Location(Levels.DKIsles, "Pre-Given Move (24)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location25: Location(Levels.DKIsles, "Pre-Given Move (25)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location26: Location(Levels.DKIsles, "Pre-Given Move (26)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location27: Location(Levels.DKIsles, "Pre-Given Move (27)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location28: Location(Levels.DKIsles, "Pre-Given Move (28)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location29: Location(Levels.DKIsles, "Pre-Given Move (29)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location30: Location(Levels.DKIsles, "Pre-Given Move (30)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location31: Location(Levels.DKIsles, "Pre-Given Move (31)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location32: Location(Levels.DKIsles, "Pre-Given Move (32)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location33: Location(Levels.DKIsles, "Pre-Given Move (33)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location34: Location(Levels.DKIsles, "Pre-Given Move (34)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location35: Location(Levels.DKIsles, "Pre-Given Move (35)", Items.NoItem, Types.PreGivenMove),
    Locations.PreGiven_Location36: Location(Levels.DKIsles, "Pre-Given Move (36)", Items.NoItem, Types.PreGivenMove),
    # Shop Owners (Dummy Locations)
    Locations.ShopOwner_Location00: Location(Levels.DKIsles, "Pre-Given Shop (0)", Items.Cranky, Types.Cranky, Kongs.any, [MapIDCombo(0, -1, 0x3C2, Kongs.any)]),
    Locations.ShopOwner_Location01: Location(Levels.DKIsles, "Pre-Given Shop (1)", Items.Funky, Types.Funky, Kongs.any, [MapIDCombo(0, -1, 0x3C3, Kongs.any)]),
    Locations.ShopOwner_Location02: Location(Levels.DKIsles, "Pre-Given Shop (2)", Items.Candy, Types.Candy, Kongs.any, [MapIDCombo(0, -1, 0x3C4, Kongs.any)]),
    Locations.ShopOwner_Location03: Location(Levels.DKIsles, "Pre-Given Shop (3)", Items.Snide, Types.Snide, Kongs.any, [MapIDCombo(0, -1, 0x3C5, Kongs.any)]),
    # DK Isles locations
    Locations.IslesDonkeyMedal: Location(Levels.DKIsles, "Isles Donkey Medal", Items.BananaMedal, Types.IslesMedal, Kongs.donkey),
    Locations.IslesDiddyMedal: Location(Levels.DKIsles, "Isles Diddy Medal", Items.BananaMedal, Types.IslesMedal, Kongs.diddy),
    Locations.IslesLankyMedal: Location(Levels.DKIsles, "Isles Lanky Medal", Items.BananaMedal, Types.IslesMedal, Kongs.lanky),
    Locations.IslesTinyMedal: Location(Levels.DKIsles, "Isles Tiny Medal", Items.BananaMedal, Types.IslesMedal, Kongs.tiny),
    Locations.IslesChunkyMedal: Location(Levels.DKIsles, "Isles Chunky Medal", Items.BananaMedal, Types.IslesMedal, Kongs.chunky),
    Locations.IslesDonkeyJapesRock: Location(Levels.DKIsles, "Isles Japes Lobby Entrance Item", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(Maps.Isles, 0x4, 381, Kongs.donkey)]),  # Can be assigned to other kongs
    Locations.IslesTinyCagedBanana: Location(Levels.DKIsles, "Isles Tiny Feather Cage", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.Isles, 0x2B, 420, Kongs.tiny)]),
    Locations.IslesTinyInstrumentPad: Location(Levels.DKIsles, "Isles Tiny Saxophone Pad", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(0, -1, 425, Kongs.tiny)]),
    Locations.IslesLankyCagedBanana: Location(Levels.DKIsles, "Isles Lanky Grape Cage", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.Isles, 0x2F, 421, Kongs.lanky)]),
    Locations.IslesChunkyCagedBanana: Location(Levels.DKIsles, "Isles Chunky Pineapple Cage", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.Isles, 0x2D, 422, Kongs.chunky)]),
    Locations.IslesChunkyInstrumentPad: Location(Levels.DKIsles, "Isles Chunky Triangle Pad", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 424, Kongs.chunky)]),
    Locations.IslesChunkyPoundtheX: Location(Levels.DKIsles, "Isles Chunky Pound the X", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.Isles, 0x56, 431, Kongs.chunky)]),
    Locations.IslesBananaFairyIsland: Location(Levels.DKIsles, "Isles Fairy (Small Island)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.Isles, -1, 606)]),
    Locations.IslesBananaFairyCrocodisleIsle: Location(Levels.DKIsles, "Isles Fairy (Upper Krem Isles)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.Isles, -1, 607)]),
    Locations.IslesLankyPrisonOrangsprint: Location(Levels.DKIsles, "Isles Lanky Sprint Cage", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.KLumsy, 0x3, 429, Kongs.lanky)]),
    Locations.CameraAndShockwave: Location(Levels.DKIsles, "The Banana Fairy's Gift", Items.CameraAndShockwave, Types.Shockwave, Kongs.tiny, [124]),
    Locations.RarewareBanana: Location(Levels.DKIsles, "Returning the Banana Fairies", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.BananaFairyRoom, 0x1E, 301, Kongs.tiny)]),
    Locations.IslesLankyInstrumentPad: Location(Levels.DKIsles, "Isles Lanky Japes Instrument", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 398, Kongs.lanky)]),
    Locations.IslesTinyAztecLobby: Location(Levels.DKIsles, "Isles Tiny Aztec Lobby Barrel", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(0, -1, 402, Kongs.tiny)]),
    Locations.IslesDonkeyCagedBanana: Location(Levels.DKIsles, "Isles Donkey Coconut Cage", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.Isles, 0x4D, 419, Kongs.donkey)]),
    Locations.IslesDiddySnidesLobby: Location(Levels.DKIsles, "Isles Diddy Snides Spring Barrel", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 416, Kongs.diddy)]),
    Locations.IslesBattleArena1: Location(Levels.DKIsles, "Isles Battle Arena 1 (Snide's Room: Under Rock)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.SnidesCrown, -1, 615)]),
    Locations.IslesDonkeyInstrumentPad: Location(Levels.DKIsles, "Isles Donkey Bongos Pad", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(0, -1, 404, Kongs.donkey)]),
    Locations.IslesKasplatFactoryLobby: Location(Levels.DKIsles, "Isles Kasplat: Factory Lobby Box", Items.TinyBlueprint, Types.Blueprint, Kongs.tiny, [Maps.FranticFactoryLobby]),
    Locations.IslesBananaFairyFactoryLobby: Location(Levels.DKIsles, "Isles Fairy (Factory Lobby)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.FranticFactoryLobby, -1, 593)]),
    Locations.IslesTinyGalleonLobby: Location(Levels.DKIsles, "Isles Tiny Galleon Lobby Swim", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.GloomyGalleonLobby, 0x9, 403)]),
    Locations.IslesKasplatGalleonLobby: Location(Levels.DKIsles, "Isles Kasplat: Galleon Lobby", Items.ChunkyBlueprint, Types.Blueprint, Kongs.chunky, [Maps.GloomyGalleonLobby]),
    Locations.IslesDiddyCagedBanana: Location(Levels.DKIsles, "Isles Diddy Peanut Cage", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.Isles, 0x2E, 423, Kongs.diddy)]),
    Locations.IslesDiddySummit: Location(Levels.DKIsles, "Isles Diddy Summit Barrel", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 428, Kongs.diddy)]),
    Locations.IslesBattleArena2: Location(Levels.DKIsles, "Isles Battle Arena 2 (Fungi Lobby: Gorilla Gone Box)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.LobbyCrown, -1, 614)]),
    Locations.IslesBananaFairyForestLobby: Location(Levels.DKIsles, "Isles Fairy (Fungi Lobby)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.FungiForestLobby, -1, 594)]),
    Locations.IslesDonkeyLavaBanana: Location(Levels.DKIsles, "Isles Donkey Caves Lava", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.CrystalCavesLobby, 0x5, 411, Kongs.donkey)]),
    Locations.IslesDiddyInstrumentPad: Location(Levels.DKIsles, "Isles Diddy Guitar Pad", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 410, Kongs.diddy)]),
    Locations.IslesKasplatCavesLobby: Location(Levels.DKIsles, "Isles Kasplat: Caves Lobby Punch", Items.LankyBlueprint, Types.Blueprint, Kongs.lanky, [Maps.CrystalCavesLobby]),
    Locations.IslesLankyCastleLobby: Location(Levels.DKIsles, "Isles Lanky Castle Lobby Barrel", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 415, Kongs.lanky)]),
    Locations.IslesKasplatCastleLobby: Location(Levels.DKIsles, "Isles Kasplat: Castle Lobby", Items.DiddyBlueprint, Types.Blueprint, Kongs.diddy, [Maps.CreepyCastleLobby]),
    Locations.IslesChunkyHelmLobby: Location(Levels.DKIsles, "Isles Chunky Helm Lobby Barrel", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 406, Kongs.chunky)]),
    Locations.IslesKasplatHelmLobby: Location(Levels.DKIsles, "Isles Kasplat: Helm Lobby", Items.DonkeyBlueprint, Types.Blueprint, Kongs.donkey, [Maps.HideoutHelmLobby]),
    Locations.BananaHoard: Location(Levels.DKIsles, "Banana Hoard", Items.BananaHoard, Types.Constant),
    # Jungle Japes location
    Locations.JapesDonkeyMedal: Location(Levels.JungleJapes, "Japes Donkey Medal", Items.BananaMedal, Types.Medal, Kongs.donkey),
    Locations.JapesDiddyMedal: Location(Levels.JungleJapes, "Japes Diddy Medal", Items.BananaMedal, Types.Medal, Kongs.diddy),
    Locations.JapesLankyMedal: Location(Levels.JungleJapes, "Japes Lanky Medal", Items.BananaMedal, Types.Medal, Kongs.lanky),
    Locations.JapesTinyMedal: Location(Levels.JungleJapes, "Japes Tiny Medal", Items.BananaMedal, Types.Medal, Kongs.tiny),
    Locations.JapesChunkyMedal: Location(Levels.JungleJapes, "Japes Chunky Medal", Items.BananaMedal, Types.Medal, Kongs.chunky),
    Locations.DiddyKong: Location(Levels.JungleJapes, "Japes Cage: Diddy Kong", Items.Diddy, Types.Kong, Kongs.donkey, [MapIDCombo(0, -1, 6)], logically_relevant=True),
    Locations.JapesDonkeyFrontofCage: Location(Levels.JungleJapes, "Japes in Front of Diddy Cage", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(Maps.JungleJapes, 0x69, 4, Kongs.donkey)], logically_relevant=True),  # Can be assigned to other kongs
    Locations.JapesDonkeyFreeDiddy: Location(Levels.JungleJapes, "Japes Free Diddy Item", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(Maps.JungleJapes, 0x48, 5, Kongs.donkey)]),  # Can be assigned to other kongs
    Locations.JapesDonkeyCagedBanana: Location(Levels.JungleJapes, "Japes Donkey Floor Cage Banana", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.JungleJapes, 0x44, 20, Kongs.donkey)]),
    Locations.JapesDonkeyBaboonBlast: Location(Levels.JungleJapes, "Japes Donkey Baboon Blast", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.JapesBaboonBlast, 1, 3, Kongs.donkey)]),
    Locations.JapesDiddyCagedBanana: Location(Levels.JungleJapes, "Japes Diddy Timed Cage Banana", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.JungleJapes, 0x4D, 18, Kongs.diddy)]),
    Locations.JapesDiddyMountain: Location(Levels.JungleJapes, "Japes Diddy Top of Mountain", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.JungleJapes, 0x52, 23, Kongs.diddy)], logically_relevant=True),
    Locations.JapesLankyCagedBanana: Location(Levels.JungleJapes, "Japes Lanky Timed Cage Banana", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.JungleJapes, 0x4F, 19, Kongs.lanky)]),
    Locations.JapesTinyCagedBanana: Location(Levels.JungleJapes, "Japes Tiny Timed Cage Banana", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.JungleJapes, 0x4C, 21, Kongs.tiny)]),
    Locations.JapesChunkyBoulder: Location(Levels.JungleJapes, "Japes Chunky Boulder", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 25, Kongs.chunky)]),
    Locations.JapesChunkyCagedBanana: Location(Levels.JungleJapes, "Japes Chunky Timed Cage Banana", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.JungleJapes, 0x50, 22, Kongs.chunky)]),
    Locations.JapesBattleArena: Location(Levels.JungleJapes, "Japes Battle Arena (Near Funky)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.JapesCrown, -1, 609)]),
    Locations.JapesDiddyTunnel: Location(Levels.JungleJapes, "Japes Diddy Peanut Tunnel", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.JungleJapes, 0x1E, 31, Kongs.diddy)]),
    Locations.JapesLankyGrapeGate: Location(Levels.JungleJapes, "Japes Lanky Grape Gate Barrel", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 1, Kongs.lanky)]),
    Locations.JapesTinyFeatherGateBarrel: Location(Levels.JungleJapes, "Japes Tiny Feather Gate Barrel", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(0, -1, 2, Kongs.tiny)]),
    Locations.JapesKasplatLeftTunnelNear: Location(Levels.JungleJapes, "Japes Kasplat: Hive Tunnel Lower", Items.DonkeyBlueprint, Types.Blueprint, Kongs.donkey, [Maps.JungleJapes]),
    Locations.JapesKasplatLeftTunnelFar: Location(Levels.JungleJapes, "Japes Kasplat: Hive Tunnel Upper", Items.TinyBlueprint, Types.Blueprint, Kongs.tiny, [Maps.JungleJapes]),
    Locations.JapesTinyStump: Location(Levels.JungleJapes, "Japes Tiny Stump", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.JungleJapes, 0x68, 8, Kongs.tiny)]),
    Locations.JapesChunkyGiantBonusBarrel: Location(Levels.JungleJapes, "Japes Chunky Giant Bonus Barrel", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 28, Kongs.chunky)]),
    Locations.JapesTinyBeehive: Location(Levels.JungleJapes, "Japes Tiny Beehive", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.JapesTinyHive, 0x3F, 9, Kongs.tiny)]),
    Locations.JapesLankySlope: Location(Levels.JungleJapes, "Japes Lanky Slope Barrel", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 11, Kongs.lanky)]),
    Locations.JapesKasplatNearPaintingRoom: Location(Levels.JungleJapes, "Japes Kasplat: Near Painting Room", Items.DiddyBlueprint, Types.Blueprint, Kongs.diddy, [Maps.JungleJapes]),
    Locations.JapesKasplatNearLab: Location(Levels.JungleJapes, "Japes Kasplat: By Lanky Slope Bonus", Items.LankyBlueprint, Types.Blueprint, Kongs.lanky, [Maps.JungleJapes]),
    Locations.JapesBananaFairyRambiCave: Location(Levels.JungleJapes, "Japes Fairy (Rambi Door Pool)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, 589)]),
    Locations.JapesLankyFairyCave: Location(Levels.JungleJapes, "Japes Lanky Painting Room Zingers", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.JapesLankyCave, 0x4, 10, Kongs.lanky)]),
    Locations.JapesBananaFairyLankyCave: Location(Levels.JungleJapes, "Japes Fairy (Painting Room)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.JapesLankyCave, -1, 590)]),
    Locations.JapesDiddyMinecarts: Location(Levels.JungleJapes, "Japes Diddy Minecart", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 24, Kongs.diddy)]),
    Locations.JapesChunkyUnderground: Location(Levels.JungleJapes, "Japes Chunky Underground", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.JapesUnderGround, 0x3, 12, Kongs.chunky)]),
    Locations.JapesKasplatUnderground: Location(Levels.JungleJapes, "Japes Kasplat: Underground", Items.ChunkyBlueprint, Types.Blueprint, Kongs.chunky, [Maps.JapesUnderGround]),
    Locations.JapesKey: Location(Levels.JungleJapes, "Japes Boss Defeated", Items.JungleJapesKey, Types.Key, Kongs.any, [MapIDCombo(0, -1, 26)]),  # Can be assigned to any kong
    # Angry Aztec
    Locations.AztecDonkeyMedal: Location(Levels.AngryAztec, "Aztec Donkey Medal", Items.BananaMedal, Types.Medal, Kongs.donkey),
    Locations.AztecDiddyMedal: Location(Levels.AngryAztec, "Aztec Diddy Medal", Items.BananaMedal, Types.Medal, Kongs.diddy),
    Locations.AztecLankyMedal: Location(Levels.AngryAztec, "Aztec Lanky Medal", Items.BananaMedal, Types.Medal, Kongs.lanky),
    Locations.AztecTinyMedal: Location(Levels.AngryAztec, "Aztec Tiny Medal", Items.BananaMedal, Types.Medal, Kongs.tiny),
    Locations.AztecChunkyMedal: Location(Levels.AngryAztec, "Aztec Chunky Medal", Items.BananaMedal, Types.Medal, Kongs.chunky),
    Locations.AztecDonkeyFreeLlama: Location(Levels.AngryAztec, "Aztec Donkey Free Llama Blast", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.AngryAztec, 0x26, 51, Kongs.donkey)]),
    Locations.AztecChunkyVases: Location(Levels.AngryAztec, "Aztec Chunky Vases", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.AngryAztec, 0x23, 49, Kongs.chunky)]),
    Locations.AztecKasplatSandyBridge: Location(Levels.AngryAztec, "Aztec Kasplat: Behind DK Stone Door", Items.DonkeyBlueprint, Types.Blueprint, Kongs.donkey, [Maps.AngryAztec]),
    Locations.AztecKasplatOnTinyTemple: Location(Levels.AngryAztec, "Aztec Kasplat: On Tiny Temple", Items.DiddyBlueprint, Types.Blueprint, Kongs.diddy, [Maps.AngryAztec]),
    Locations.AztecTinyKlaptrapRoom: Location(Levels.AngryAztec, "Aztec Tiny Klaptrap Room", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.AztecTinyTemple, 0x7E, 65, Kongs.tiny)]),
    Locations.AztecChunkyKlaptrapRoom: Location(Levels.AngryAztec, "Aztec Chunky Klaptrap Room", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.AztecTinyTemple, 0x9, 64, Kongs.chunky)]),
    Locations.TinyKong: Location(Levels.AngryAztec, "Aztec Cage: Tiny Kong", Items.Tiny, Types.Kong, Kongs.diddy, [MapIDCombo(0, -1, 66)], logically_relevant=True),
    Locations.AztecDiddyFreeTiny: Location(Levels.AngryAztec, "Aztec Free Tiny Item", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, 0x5B, 67, Kongs.diddy)]),  # Can be assigned to other kongs
    Locations.AztecLankyVulture: Location(Levels.AngryAztec, "Aztec Lanky Vulture Shooting", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 68, Kongs.lanky)]),
    Locations.AztecBattleArena: Location(Levels.AngryAztec, "Aztec Battle Arena (Tiny Temple: Vulture Room)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.AztecCrown, -1, 610)]),
    Locations.AztecDonkeyQuicksandCave: Location(Levels.AngryAztec, "Aztec Donkey Sealed Quicksand Tunnel Barrel", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(0, -1, 62, Kongs.donkey)], logically_relevant=True),
    Locations.AztecDiddyRamGongs: Location(Levels.AngryAztec, "Aztec Diddy Ram Gongs", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.AngryAztec, 0xA3, 54, Kongs.diddy)]),
    Locations.AztecDiddyVultureRace: Location(Levels.AngryAztec, "Aztec Diddy Vulture Race", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.AngryAztec, 0xEB, 63, Kongs.diddy)]),
    Locations.AztecChunkyCagedBarrel: Location(Levels.AngryAztec, "Aztec Chunky Giant Caged Barrel", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 52, Kongs.chunky)]),
    Locations.AztecKasplatNearLab: Location(Levels.AngryAztec, "Aztec Kasplat: Hunky Chunky Barrel", Items.TinyBlueprint, Types.Blueprint, Kongs.tiny, [Maps.AngryAztec]),
    Locations.AztecDonkey5DoorTemple: Location(Levels.AngryAztec, "Aztec Donkey 5 Door Temple", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.AztecDonkey5DTemple, 0x6, 57, Kongs.donkey)]),
    Locations.AztecDiddy5DoorTemple: Location(Levels.AngryAztec, "Aztec Diddy 5 Door Temple", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.AztecDiddy5DTemple, 0x6, 56, Kongs.diddy)]),
    Locations.AztecLanky5DoorTemple: Location(Levels.AngryAztec, "Aztec Lanky 5 Door Temple", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 60, Kongs.lanky)]),
    Locations.AztecTiny5DoorTemple: Location(Levels.AngryAztec, "Aztec Tiny 5 Door Temple", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.AztecTiny5DTemple, 0x6, 58, Kongs.tiny)]),
    Locations.AztecBananaFairyTinyTemple: Location(Levels.AngryAztec, "Aztec Fairy (Tiny 5-Door Temple)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.AztecTiny5DTemple, -1, 601)]),
    Locations.AztecChunky5DoorTemple: Location(Levels.AngryAztec, "Aztec Chunky 5 Door Temple", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 59, Kongs.chunky)]),
    Locations.AztecKasplatChunky5DT: Location(Levels.AngryAztec, "Aztec Kasplat: Chunky 5-Door Temple", Items.ChunkyBlueprint, Types.Blueprint, Kongs.chunky, [Maps.AztecChunky5DTemple]),
    Locations.AztecTinyBeetleRace: Location(Levels.AngryAztec, "Aztec Tiny Beetle Race", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.AztecTinyRace, 0x48, 75, Kongs.tiny)]),
    Locations.LankyKong: Location(Levels.AngryAztec, "Aztec Cage: Lanky Kong", Items.Lanky, Types.Kong, Kongs.donkey, [MapIDCombo(0, -1, 70)], logically_relevant=True),
    Locations.AztecDonkeyFreeLanky: Location(Levels.AngryAztec, "Aztec Free Lanky Item", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, 0x6C, 77, Kongs.donkey)]),  # Can be assigned to other kongs
    Locations.AztecLankyLlamaTempleBarrel: Location(Levels.AngryAztec, "Aztec Lanky Llama Temple Barrel", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 73, Kongs.lanky)]),
    Locations.AztecLankyMatchingGame: Location(Levels.AngryAztec, "Aztec Lanky Matching Game", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.AztecLlamaTemple, 0x2B, 72, Kongs.lanky)]),
    Locations.AztecBananaFairyLlamaTemple: Location(Levels.AngryAztec, "Aztec Fairy (Llama Temple)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, 600)]),
    Locations.AztecTinyLlamaTemple: Location(Levels.AngryAztec, "Aztec Tiny Llama Temple Lava Pedestals", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.AztecLlamaTemple, 0xAA, 71, Kongs.tiny)]),
    Locations.AztecKasplatLlamaTemple: Location(Levels.AngryAztec, "Aztec Kasplat: Llama Temple Lava", Items.LankyBlueprint, Types.Blueprint, Kongs.lanky, [Maps.AztecLlamaTemple]),
    Locations.AztecKey: Location(Levels.AngryAztec, "Aztec Boss Defeated", Items.AngryAztecKey, Types.Key, Kongs.any, [MapIDCombo(0, -1, 74)]),  # Can be assigned to any kong
    # Frantic Factory locations
    Locations.FactoryDonkeyMedal: Location(Levels.FranticFactory, "Factory Donkey Medal", Items.BananaMedal, Types.Medal, Kongs.donkey),
    Locations.FactoryDiddyMedal: Location(Levels.FranticFactory, "Factory Diddy Medal", Items.BananaMedal, Types.Medal, Kongs.diddy),
    Locations.FactoryLankyMedal: Location(Levels.FranticFactory, "Factory Lanky Medal", Items.BananaMedal, Types.Medal, Kongs.lanky),
    Locations.FactoryTinyMedal: Location(Levels.FranticFactory, "Factory Tiny Medal", Items.BananaMedal, Types.Medal, Kongs.tiny),
    Locations.FactoryChunkyMedal: Location(Levels.FranticFactory, "Factory Chunky Medal", Items.BananaMedal, Types.Medal, Kongs.chunky),
    Locations.FactoryDonkeyNumberGame: Location(Levels.FranticFactory, "Factory Donkey Number Game", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.FranticFactory, 0x7E, 122, Kongs.donkey)]),
    Locations.FactoryDiddyBlockTower: Location(Levels.FranticFactory, "Factory Diddy Block Tower", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 135, Kongs.diddy)]),
    Locations.FactoryLankyTestingRoomBarrel: Location(Levels.FranticFactory, "Factory Lanky Testing Room Barrel", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 137, Kongs.lanky)]),
    Locations.FactoryTinyDartboard: Location(Levels.FranticFactory, "Factory Tiny Dartboard", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.FranticFactory, 0x82, 124, Kongs.tiny)]),
    Locations.FactoryKasplatBlocks: Location(Levels.FranticFactory, "Factory Kasplat: Block Tower", Items.ChunkyBlueprint, Types.Blueprint, Kongs.chunky, [Maps.FranticFactory]),
    Locations.FactoryBananaFairybyCounting: Location(Levels.FranticFactory, "Factory Fairy (Number Game)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, 602)]),
    Locations.FactoryBananaFairybyFunky: Location(Levels.FranticFactory, "Factory Fairy (Near Funky's)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, 591)]),
    Locations.FactoryDiddyRandD: Location(Levels.FranticFactory, "Factory Diddy Charge Enemies", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.FranticFactory, 0x60, 126, Kongs.diddy)]),
    Locations.FactoryLankyRandD: Location(Levels.FranticFactory, "Factory Lanky Piano Game", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.FranticFactory, 0x3E, 125, Kongs.lanky)]),
    Locations.FactoryChunkyRandD: Location(Levels.FranticFactory, "Factory Chunky Toy Monster", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.FranticFactory, 0x7C, 127, Kongs.chunky)]),
    Locations.FactoryKasplatRandD: Location(Levels.FranticFactory, "Factory Kasplat: R&D", Items.LankyBlueprint, Types.Blueprint, Kongs.lanky, [Maps.FranticFactory]),
    Locations.FactoryBattleArena: Location(Levels.FranticFactory, "Factory Battle Arena (Under R&D Grate)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.FactoryCrown, -1, 611)]),
    Locations.FactoryTinyCarRace: Location(Levels.FranticFactory, "Factory Tiny Car Race", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.FactoryTinyRace, 0x62, 139, Kongs.tiny)]),
    Locations.FactoryDiddyChunkyRoomBarrel: Location(Levels.FranticFactory, "Factory Diddy Storage Room Barrel", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 134, Kongs.diddy)]),
    Locations.FactoryDonkeyPowerHut: Location(Levels.FranticFactory, "Factory Donkey Power Hut", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.FactoryPowerHut, 0x2, 112, Kongs.donkey)]),
    Locations.ChunkyKong: Location(Levels.FranticFactory, "Factory Cage: Chunky Kong", Items.Chunky, Types.Kong, Kongs.lanky, [MapIDCombo(0, -1, 117)], logically_relevant=True),
    Locations.NintendoCoin: Location(Levels.FranticFactory, "DK Arcade Round 2", Items.NintendoCoin, Types.NintendoCoin, Kongs.donkey, [MapIDCombo(Maps.FranticFactory, 0x13E, 132)]),
    Locations.FactoryDonkeyDKArcade: Location(Levels.FranticFactory, "Factory Donkey DK Arcade Round 1", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.FranticFactory, 0x108, 130, Kongs.donkey), MapIDCombo(Maps.FactoryBaboonBlast, 0, 130, Kongs.donkey)]),
    Locations.FactoryLankyFreeChunky: Location(Levels.FranticFactory, "Factory Free Chunky Item", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(Maps.FranticFactory, 0x78, 118, Kongs.lanky)]),  # Can be assigned to other kongs
    Locations.FactoryTinybyArcade: Location(Levels.FranticFactory, "Factory Tiny Mini by Arcade", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.FranticFactory, 0x23, 123, Kongs.tiny)]),
    Locations.FactoryChunkyDarkRoom: Location(Levels.FranticFactory, "Factory Chunky Dark Room", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.FranticFactory, 0x63, 121, Kongs.chunky)]),
    Locations.FactoryChunkybyArcade: Location(Levels.FranticFactory, "Factory Chunky Barrel by Arcade", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 136, Kongs.chunky)]),
    Locations.FactoryKasplatProductionBottom: Location(Levels.FranticFactory, "Factory Kasplat: Base of Production", Items.DiddyBlueprint, Types.Blueprint, Kongs.diddy, [Maps.FranticFactory]),
    Locations.FactoryKasplatStorage: Location(Levels.FranticFactory, "Factory Kasplat: Pole to Arcade", Items.TinyBlueprint, Types.Blueprint, Kongs.tiny, [Maps.FranticFactory, Kongs.tiny]),
    Locations.FactoryDonkeyCrusherRoom: Location(Levels.FranticFactory, "Factory Donkey Crusher Room", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.FactoryCrusher, 0x7, 128, Kongs.donkey)]),
    Locations.FactoryDiddyProductionRoom: Location(Levels.FranticFactory, "Factory Diddy Production Spring", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.FranticFactory, 0x2C, 113, Kongs.diddy)]),
    Locations.FactoryLankyProductionRoom: Location(Levels.FranticFactory, "Factory Lanky Production Handstand", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.FranticFactory, 0x2A, 115, Kongs.lanky)]),
    Locations.FactoryTinyProductionRoom: Location(Levels.FranticFactory, "Factory Tiny Production Twirl", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(0, -1, 116, Kongs.tiny)]),
    Locations.FactoryChunkyProductionRoom: Location(Levels.FranticFactory, "Factory Chunky Production Timer", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.FranticFactory, 0x29, 114, Kongs.chunky)]),
    Locations.FactoryKasplatProductionTop: Location(Levels.FranticFactory, "Factory Kasplat: Upper Production Pipe", Items.DonkeyBlueprint, Types.Blueprint, Kongs.donkey, [Maps.FranticFactory]),
    Locations.FactoryKey: Location(Levels.FranticFactory, "Factory Boss Defeated", Items.FranticFactoryKey, Types.Key, Kongs.any, [MapIDCombo(0, -1, 138)]),  # Can be assigned to any kong
    # Gloomy Galleon locations
    Locations.GalleonDonkeyMedal: Location(Levels.GloomyGalleon, "Galleon Donkey Medal", Items.BananaMedal, Types.Medal, Kongs.donkey),
    Locations.GalleonDiddyMedal: Location(Levels.GloomyGalleon, "Galleon Diddy Medal", Items.BananaMedal, Types.Medal, Kongs.diddy),
    Locations.GalleonLankyMedal: Location(Levels.GloomyGalleon, "Galleon Lanky Medal", Items.BananaMedal, Types.Medal, Kongs.lanky),
    Locations.GalleonTinyMedal: Location(Levels.GloomyGalleon, "Galleon Tiny Medal", Items.BananaMedal, Types.Medal, Kongs.tiny),
    Locations.GalleonChunkyMedal: Location(Levels.GloomyGalleon, "Galleon Chunky Medal", Items.BananaMedal, Types.Medal, Kongs.chunky),
    Locations.GalleonChunkyChest: Location(Levels.GloomyGalleon, "Galleon Chunky Chest", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.GloomyGalleon, 0xE, 182, Kongs.chunky)]),
    Locations.GalleonKasplatNearLab: Location(Levels.GloomyGalleon, "Galleon Kasplat: Past Vines", Items.TinyBlueprint, Types.Blueprint, Kongs.tiny, [Maps.GloomyGalleon]),
    Locations.GalleonBattleArena: Location(Levels.GloomyGalleon, "Galleon Battle Arena (Under Cranky)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.GalleonCrown, -1, 612)]),
    Locations.GalleonBananaFairybyCranky: Location(Levels.GloomyGalleon, "Galleon Fairy (In Punch Chest)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, 592)]),
    Locations.GalleonChunkyCannonGame: Location(Levels.GloomyGalleon, "Galleon Chunky Cannon Game", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.GloomyGalleon, 0x32, 154, Kongs.chunky)]),
    Locations.GalleonKasplatCannons: Location(Levels.GloomyGalleon, "Galleon Kasplat: Cannon Game Room", Items.LankyBlueprint, Types.Blueprint, Kongs.lanky, [Maps.GloomyGalleon]),
    Locations.GalleonDiddyShipSwitch: Location(Levels.GloomyGalleon, "Galleon Diddy Top of Lighthouse", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.GloomyGalleon, 0x2D, 204, Kongs.diddy)]),
    Locations.GalleonLankyEnguardeChest: Location(Levels.GloomyGalleon, "Galleon Lanky Enguarde Chest", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.GloomyGalleon, 0x6B, 192, Kongs.lanky)]),
    Locations.GalleonKasplatLighthouseArea: Location(Levels.GloomyGalleon, "Galleon Kasplat: Lighthouse Alcove", Items.DiddyBlueprint, Types.Blueprint, Kongs.diddy, [Maps.GloomyGalleon]),
    Locations.GalleonDonkeyLighthouse: Location(Levels.GloomyGalleon, "Galleon Donkey Lighthouse", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.GalleonLighthouse, 0x2F, 157, Kongs.donkey)]),
    Locations.GalleonTinyPearls: Location(Levels.GloomyGalleon, "Galleon Tiny Mermaid Reward", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.GalleonMermaidRoom, 0xE, 191, Kongs.tiny)]),
    Locations.GalleonChunkySeasick: Location(Levels.GloomyGalleon, "Galleon Chunky Seasick", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.GalleonSickBay, 0x6, 166, Kongs.chunky)]),
    Locations.GalleonDonkeyFreetheSeal: Location(Levels.GloomyGalleon, "Galleon Donkey Free the Seal", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.GloomyGalleon, 0x2E, 193, Kongs.donkey)]),
    Locations.GalleonKasplatNearSub: Location(Levels.GloomyGalleon, "Galleon Kasplat: Musical Cactus", Items.ChunkyBlueprint, Types.Blueprint, Kongs.chunky, [Maps.GloomyGalleon]),
    Locations.GalleonDonkeySealRace: Location(Levels.GloomyGalleon, "Galleon Donkey Seal Race", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.GalleonSealRace, 0x3B, 165, Kongs.donkey)]),
    Locations.GalleonDiddyGoldTower: Location(Levels.GloomyGalleon, "Galleon Diddy Gold Tower Barrel", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 163, Kongs.diddy)], logically_relevant=True),
    Locations.GalleonLankyGoldTower: Location(Levels.GloomyGalleon, "Galleon Lanky Gold Tower Barrel", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 164, Kongs.lanky)]),
    Locations.GalleonKasplatGoldTower: Location(Levels.GloomyGalleon, "Galleon Kasplat: Diddy Gold Tower", Items.DonkeyBlueprint, Types.Blueprint, Kongs.donkey, [Maps.GloomyGalleon]),
    Locations.GalleonTinySubmarine: Location(Levels.GloomyGalleon, "Galleon Tiny Submarine Barrel", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(0, -1, 202, Kongs.tiny)]),
    Locations.GalleonDiddyMechafish: Location(Levels.GloomyGalleon, "Galleon Diddy Mechfish", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.GalleonMechafish, 0xF, 167, Kongs.diddy)]),
    Locations.GalleonLanky2DoorShip: Location(Levels.GloomyGalleon, "Galleon Lanky 2 Door Ship", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.Galleon2DShip, 0x0, 183, Kongs.lanky)]),
    Locations.GalleonTiny2DoorShip: Location(Levels.GloomyGalleon, "Galleon Tiny 2 Door Ship", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(0, -1, 184, Kongs.tiny)]),
    Locations.GalleonDonkey5DoorShip: Location(Levels.GloomyGalleon, "Galleon Donkey 5 Door Ship", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(0, -1, 200, Kongs.donkey)]),
    Locations.GalleonDiddy5DoorShip: Location(Levels.GloomyGalleon, "Galleon Diddy 5 Door Ship", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 198, Kongs.diddy)]),
    Locations.GalleonLanky5DoorShip: Location(Levels.GloomyGalleon, "Galleon Lanky 5 Door Ship", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.Galleon5DShipDiddyLankyChunky, 0xE, 199, Kongs.lanky)]),
    Locations.GalleonTiny5DoorShip: Location(Levels.GloomyGalleon, "Galleon Tiny 5 Door Ship", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.Galleon5DShipDKTiny, 0x21, 201, Kongs.tiny)]),
    Locations.GalleonBananaFairy5DoorShip: Location(Levels.GloomyGalleon, "Galleon Fairy (In Tiny's 5-Door Ship)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.Galleon5DShipDKTiny, -1, 603)]),
    Locations.GalleonChunky5DoorShip: Location(Levels.GloomyGalleon, "Galleon Chunky 5 Door Ship", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 197, Kongs.chunky)]),
    Locations.GalleonPearl0: Location(Levels.GloomyGalleon, "Treasure Chest Far Left Clam", Items.Pearl, Types.Pearl, Kongs.tiny, [MapIDCombo(Maps.GalleonTreasureChest, 0, 186, Kongs.tiny)]),
    Locations.GalleonPearl1: Location(Levels.GloomyGalleon, "Treasure Chest Center Clam", Items.Pearl, Types.Pearl, Kongs.tiny, [MapIDCombo(Maps.GalleonTreasureChest, 1, 187, Kongs.tiny)]),
    Locations.GalleonPearl2: Location(Levels.GloomyGalleon, "Treasure Chest Far Right Clam", Items.Pearl, Types.Pearl, Kongs.tiny, [MapIDCombo(Maps.GalleonTreasureChest, 2, 188, Kongs.tiny)]),
    Locations.GalleonPearl3: Location(Levels.GloomyGalleon, "Treasure Chest Close Right Clam", Items.Pearl, Types.Pearl, Kongs.tiny, [MapIDCombo(Maps.GalleonTreasureChest, 3, 189, Kongs.tiny)]),
    Locations.GalleonPearl4: Location(Levels.GloomyGalleon, "Treasure Chest Close Left Clam", Items.Pearl, Types.Pearl, Kongs.tiny, [MapIDCombo(Maps.GalleonTreasureChest, 4, 190, Kongs.tiny)]),
    Locations.GalleonKey: Location(Levels.GloomyGalleon, "Galleon Boss Defeated", Items.GloomyGalleonKey, Types.Key, Kongs.any, [MapIDCombo(0, -1, 168)]),  # Can be assigned to any kong
    # Fungi Forest locations
    Locations.ForestDonkeyMedal: Location(Levels.FungiForest, "Forest Donkey Medal", Items.BananaMedal, Types.Medal, Kongs.donkey),
    Locations.ForestDiddyMedal: Location(Levels.FungiForest, "Forest Diddy Medal", Items.BananaMedal, Types.Medal, Kongs.diddy),
    Locations.ForestLankyMedal: Location(Levels.FungiForest, "Forest Lanky Medal", Items.BananaMedal, Types.Medal, Kongs.lanky),
    Locations.ForestTinyMedal: Location(Levels.FungiForest, "Forest Tiny Medal", Items.BananaMedal, Types.Medal, Kongs.tiny),
    Locations.ForestChunkyMedal: Location(Levels.FungiForest, "Forest Chunky Medal", Items.BananaMedal, Types.Medal, Kongs.chunky),
    Locations.ForestChunkyMinecarts: Location(Levels.FungiForest, "Forest Chunky Minecart", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 215, Kongs.chunky)]),
    Locations.ForestDiddyTopofMushroom: Location(Levels.FungiForest, "Forest Diddy Top of Mushroom Barrel", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 211, Kongs.diddy)]),
    Locations.ForestTinyMushroomBarrel: Location(Levels.FungiForest, "Forest Tiny Mushroom Barrel", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(0, -1, 227, Kongs.tiny)]),
    Locations.ForestDonkeyBaboonBlast: Location(Levels.FungiForest, "Forest Donkey Baboon Blast", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.FungiForest, 0x39, 254, Kongs.donkey)]),
    Locations.ForestKasplatLowerMushroomExterior: Location(Levels.FungiForest, "Forest Kasplat: Low Mushroom Exterior", Items.TinyBlueprint, Types.Blueprint, Kongs.tiny, [Maps.FungiForest]),
    Locations.ForestDonkeyMushroomCannons: Location(Levels.FungiForest, "Forest Donkey Mushroom Cannons", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.ForestGiantMushroom, 0x3, 228, Kongs.donkey)]),
    Locations.ForestKasplatInsideMushroom: Location(Levels.FungiForest, "Forest Kasplat: Inside Giant Mushroom", Items.DiddyBlueprint, Types.Blueprint, Kongs.diddy, [Maps.ForestGiantMushroom]),
    Locations.ForestKasplatUpperMushroomExterior: Location(Levels.FungiForest, "Forest Kasplat: Mushroom Night Door", Items.ChunkyBlueprint, Types.Blueprint, Kongs.chunky, [Maps.FungiForest]),
    Locations.ForestBattleArena: Location(Levels.FungiForest, "Forest Battle Arena (Giant Mushroom High Ladder Platform)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.ForestCrown, -1, 613)]),
    Locations.ForestChunkyFacePuzzle: Location(Levels.FungiForest, "Forest Chunky Face Puzzle", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.ForestChunkyFaceRoom, 0x2, 225, Kongs.chunky)]),
    Locations.ForestLankyZingers: Location(Levels.FungiForest, "Forest Lanky Zinger Bounce", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.ForestLankyZingersRoom, 0x0, 226, Kongs.lanky)]),
    Locations.ForestLankyColoredMushrooms: Location(Levels.FungiForest, "Forest Lanky Colored Mushroom Slam", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 224, Kongs.lanky)]),
    Locations.ForestDiddyOwlRace: Location(Levels.FungiForest, "Forest Diddy Owl Race", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 250, Kongs.diddy)]),
    Locations.ForestLankyRabbitRace: Location(Levels.FungiForest, "Forest Lanky Rabbit Race", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.FungiForest, 0x57, 249)]),
    Locations.ForestKasplatOwlTree: Location(Levels.FungiForest, "Forest Kasplat: Under Owl Tree", Items.LankyBlueprint, Types.Blueprint, Kongs.lanky, [Maps.FungiForest]),
    Locations.ForestTinyAnthill: Location(Levels.FungiForest, "Forest Tiny Anthill Banana", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.ForestAnthill, 0x0, 205, Kongs.tiny)]),
    Locations.ForestDonkeyMill: Location(Levels.FungiForest, "Forest Donkey Mill Levers", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.FungiForest, 0x2B, 219, Kongs.donkey), MapIDCombo(Maps.ForestMillFront, 0xA, 219, Kongs.donkey)]),
    Locations.ForestDiddyCagedBanana: Location(Levels.FungiForest, "Forest Diddy Winch Cage", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.FungiForest, 0x28, 214, Kongs.diddy)]),
    Locations.ForestTinySpiderBoss: Location(Levels.FungiForest, "Forest Tiny Spider Boss", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.ForestSpider, 0x1, 247, Kongs.tiny)]),
    Locations.ForestChunkyKegs: Location(Levels.FungiForest, "Forest Chunky Keg Crushing", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.ForestMillFront, 0xD, 221, Kongs.chunky)]),
    Locations.ForestDiddyRafters: Location(Levels.FungiForest, "Forest Diddy Dark Rafters", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.ForestRafters, 0x3, 216, Kongs.diddy)]),
    Locations.ForestBananaFairyRafters: Location(Levels.FungiForest, "Forest Fairy (Dark Rafters)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.ForestRafters, -1, 595)]),
    Locations.ForestLankyAttic: Location(Levels.FungiForest, "Forest Lanky Attic Shooting", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.ForestMillAttic, 0x2, 217, Kongs.lanky)]),
    Locations.ForestKasplatNearBarn: Location(Levels.FungiForest, "Forest Kasplat: Behind DK's Barn", Items.DonkeyBlueprint, Types.Blueprint, Kongs.donkey, [Maps.FungiForest]),
    Locations.ForestDonkeyBarn: Location(Levels.FungiForest, "Forest Donkey Thornvine Barn Barrel", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(0, -1, 235, Kongs.donkey)]),
    Locations.ForestBananaFairyThornvines: Location(Levels.FungiForest, "Forest Fairy (Thornvine Barn)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.ForestThornvineBarn, -1, 596)]),
    Locations.ForestTinyBeanstalk: Location(Levels.FungiForest, "Forest Tiny Top of the Beanstalk", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.FungiForest, 0x50, 209, Kongs.tiny)]),
    Locations.ForestChunkyApple: Location(Levels.FungiForest, "Forest Chunky Apple Rescue", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.FungiForest, 0x3E, 253, Kongs.chunky)]),
    Locations.ForestBean: Location(Levels.FungiForest, "Forest Second Anthill Reward", Items.Bean, Types.Bean, Kongs.tiny, [MapIDCombo(Maps.ForestAnthill, 0x5, 0x300, Kongs.tiny)]),
    Locations.ForestKey: Location(Levels.FungiForest, "Forest Boss Defeated", Items.FungiForestKey, Types.Key, Kongs.any, [MapIDCombo(0, -1, 236)]),  # Can be assigned to any kong
    # Crystal Caves locations
    Locations.CavesDonkeyMedal: Location(Levels.CrystalCaves, "Caves Donkey Medal", Items.BananaMedal, Types.Medal, Kongs.donkey),
    Locations.CavesDiddyMedal: Location(Levels.CrystalCaves, "Caves Diddy Medal", Items.BananaMedal, Types.Medal, Kongs.diddy),
    Locations.CavesLankyMedal: Location(Levels.CrystalCaves, "Caves Lanky Medal", Items.BananaMedal, Types.Medal, Kongs.lanky),
    Locations.CavesTinyMedal: Location(Levels.CrystalCaves, "Caves Tiny Medal", Items.BananaMedal, Types.Medal, Kongs.tiny),
    Locations.CavesChunkyMedal: Location(Levels.CrystalCaves, "Caves Chunky Medal", Items.BananaMedal, Types.Medal, Kongs.chunky),
    Locations.CavesDonkeyBaboonBlast: Location(Levels.CrystalCaves, "Caves Donkey Baboon Blast", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.CrystalCaves, 0x32, 298, Kongs.donkey)]),
    Locations.CavesDiddyJetpackBarrel: Location(Levels.CrystalCaves, "Caves Diddy Jetpack Barrel", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 294, Kongs.diddy)]),
    Locations.CavesTinyCaveBarrel: Location(Levels.CrystalCaves, "Caves Tiny Mini Cave Barrel", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(0, -1, 295, Kongs.tiny)], logically_relevant=True),
    Locations.CavesTinyMonkeyportIgloo: Location(Levels.CrystalCaves, "Caves Tiny Monkeyport Igloo", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.CrystalCaves, 0x29, 297, Kongs.tiny)]),
    Locations.CavesChunkyGorillaGone: Location(Levels.CrystalCaves, "Caves Chunky Gorilla Gone", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.CrystalCaves, 0x3E, 268, Kongs.chunky)]),
    Locations.CavesKasplatNearLab: Location(Levels.CrystalCaves, "Caves Kasplat: Near Ice Castle", Items.DonkeyBlueprint, Types.Blueprint, Kongs.donkey, [Maps.CrystalCaves]),
    Locations.CavesKasplatNearFunky: Location(Levels.CrystalCaves, "Caves Kasplat: Mini Room by Funky", Items.DiddyBlueprint, Types.Blueprint, Kongs.diddy, [Maps.CrystalCaves]),
    Locations.CavesKasplatPillar: Location(Levels.CrystalCaves, "Caves Kasplat: On the Pillar", Items.LankyBlueprint, Types.Blueprint, Kongs.lanky, [Maps.CrystalCaves]),
    Locations.CavesKasplatNearCandy: Location(Levels.CrystalCaves, "Caves Kasplat: By the Far Warp 2", Items.TinyBlueprint, Types.Blueprint, Kongs.tiny, [Maps.CrystalCaves]),
    Locations.CavesLankyBeetleRace: Location(Levels.CrystalCaves, "Caves Lanky Beetle Race", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.CavesLankyRace, 0x1, 259, Kongs.lanky)]),
    Locations.CavesLankyCastle: Location(Levels.CrystalCaves, "Caves Lanky Ice Castle Slam Challenge", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.CavesFrozenCastle, 0x10, 271, Kongs.lanky)]),
    Locations.CavesChunkyTransparentIgloo: Location(Levels.CrystalCaves, "Caves Chunky Transparent Igloo", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.CrystalCaves, 0x28, 270, Kongs.chunky)]),
    Locations.CavesKasplatOn5DI: Location(Levels.CrystalCaves, "Caves Kasplat: On 5-Door Igloo", Items.ChunkyBlueprint, Types.Blueprint, Kongs.chunky, [Maps.CrystalCaves]),
    Locations.CavesDonkey5DoorIgloo: Location(Levels.CrystalCaves, "Caves Donkey 5 Door Igloo", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.CavesDonkeyIgloo, 0x1, 275, Kongs.donkey)]),
    Locations.CavesDiddy5DoorIgloo: Location(Levels.CrystalCaves, "Caves Diddy 5 Door Igloo", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.CavesDiddyIgloo, 0x1, 274, Kongs.diddy)]),
    Locations.CavesLanky5DoorIgloo: Location(Levels.CrystalCaves, "Caves Lanky 5 Door Igloo", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.CavesLankyIgloo, 0x3, 281, Kongs.lanky)]),
    Locations.CavesTiny5DoorIgloo: Location(Levels.CrystalCaves, "Caves Tiny 5 Door Igloo", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.CavesTinyIgloo, 0x2, 279, Kongs.tiny)]),
    Locations.CavesBananaFairyIgloo: Location(Levels.CrystalCaves, "Caves Fairy (Tiny Igloo)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.CavesTinyIgloo, -1, 597)]),
    Locations.CavesChunky5DoorIgloo: Location(Levels.CrystalCaves, "Caves Chunky 5 Door Igloo", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.CavesChunkyIgloo, 0x0, 278, Kongs.chunky)]),
    Locations.CavesDonkeyRotatingCabin: Location(Levels.CrystalCaves, "Caves Donkey Rotating Cabin", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.CavesRotatingCabin, 0x1, 276, Kongs.donkey)]),
    Locations.CavesBattleArena: Location(Levels.CrystalCaves, "Caves Battle Arena (Rotating Room: Left Portion)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.CavesCrown, -1, 616)]),
    Locations.CavesDonkey5DoorCabin: Location(Levels.CrystalCaves, "Caves Donkey 5 Door Cabin", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.CavesDonkeyCabin, 0x8, 261, Kongs.donkey)]),
    Locations.CavesDiddy5DoorCabinLower: Location(Levels.CrystalCaves, "Caves Diddy 5 Door Cabin Lower", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.CavesDiddyLowerCabin, 0x1, 262, Kongs.diddy)]),
    Locations.CavesDiddy5DoorCabinUpper: Location(Levels.CrystalCaves, "Caves Diddy 5 Door Cabin Upper", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.CavesDiddyUpperCabin, 0x4, 293, Kongs.diddy)]),
    Locations.CavesBananaFairyCabin: Location(Levels.CrystalCaves, "Caves Fairy (Diddy Candles Cabin)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.CavesDiddyUpperCabin, -1, 608)]),
    Locations.CavesLanky1DoorCabin: Location(Levels.CrystalCaves, "Caves Lanky Sprint Cabin", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.CavesLankyCabin, 0x1, 264, Kongs.lanky)]),
    Locations.CavesTiny5DoorCabin: Location(Levels.CrystalCaves, "Caves Tiny 5 Door Cabin", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.CavesTinyCabin, 0x0, 260, Kongs.tiny)]),
    Locations.CavesChunky5DoorCabin: Location(Levels.CrystalCaves, "Caves Chunky 5 Door Cabin", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 263, Kongs.chunky)]),
    Locations.CavesKey: Location(Levels.CrystalCaves, "Caves Boss Defeated", Items.CrystalCavesKey, Types.Key, Kongs.any, [MapIDCombo(0, -1, 292)]),  # Can be assigned to any kong
    # Creepy Castle locations
    Locations.CastleDonkeyMedal: Location(Levels.CreepyCastle, "Castle Donkey Medal", Items.BananaMedal, Types.Medal, Kongs.donkey),
    Locations.CastleDiddyMedal: Location(Levels.CreepyCastle, "Castle Diddy Medal", Items.BananaMedal, Types.Medal, Kongs.diddy),
    Locations.CastleLankyMedal: Location(Levels.CreepyCastle, "Castle Lanky Medal", Items.BananaMedal, Types.Medal, Kongs.lanky),
    Locations.CastleTinyMedal: Location(Levels.CreepyCastle, "Castle Tiny Medal", Items.BananaMedal, Types.Medal, Kongs.tiny),
    Locations.CastleChunkyMedal: Location(Levels.CreepyCastle, "Castle Chunky Medal", Items.BananaMedal, Types.Medal, Kongs.chunky),
    Locations.CastleDiddyAboveCastle: Location(Levels.CreepyCastle, "Castle Diddy Above Castle", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 350, Kongs.diddy)]),
    Locations.CastleKasplatHalfway: Location(Levels.CreepyCastle, "Castle Kasplat: Near Upper Warp 2", Items.LankyBlueprint, Types.Blueprint, Kongs.lanky, [Maps.CreepyCastle]),
    Locations.CastleKasplatLowerLedge: Location(Levels.CreepyCastle, "Castle Kasplat: On a lone platform", Items.TinyBlueprint, Types.Blueprint, Kongs.tiny, [Maps.CreepyCastle]),
    Locations.CastleDonkeyTree: Location(Levels.CreepyCastle, "Castle Donkey Tree Sniping", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.CastleTree, 0x8, 320, Kongs.donkey)]),
    Locations.CastleChunkyTree: Location(Levels.CreepyCastle, "Castle Chunky Tree Sniping Barrel", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 319, Kongs.chunky)]),
    Locations.CastleKasplatTree: Location(Levels.CreepyCastle, "Castle Kasplat: Inside the Tree", Items.DonkeyBlueprint, Types.Blueprint, Kongs.donkey, [Maps.CastleTree]),
    Locations.CastleBananaFairyTree: Location(Levels.CreepyCastle, "Castle Fairy (Tree Sniper Room)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.CastleTree, -1, 605)]),
    Locations.CastleDonkeyLibrary: Location(Levels.CreepyCastle, "Castle Donkey Library", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.CastleLibrary, 0x3, 313, Kongs.donkey)]),
    Locations.CastleDiddyBallroom: Location(Levels.CreepyCastle, "Castle Diddy Ballroom", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(0, -1, 305, Kongs.diddy)]),
    Locations.CastleBananaFairyBallroom: Location(Levels.CreepyCastle, "Castle Fairy (Near Car Race)", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.CastleMuseum, -1, 604)]),
    Locations.CastleTinyCarRace: Location(Levels.CreepyCastle, "Castle Tiny Car Race", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.CastleTinyRace, 0x1, 325, Kongs.tiny)]),
    Locations.CastleLankyTower: Location(Levels.CreepyCastle, "Castle Lanky Tower", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 306, Kongs.lanky)]),
    Locations.CastleLankyGreenhouse: Location(Levels.CreepyCastle, "Castle Lanky Greenhouse", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.CastleGreenhouse, 0x1, 323, Kongs.lanky)]),
    Locations.CastleBattleArena: Location(Levels.CreepyCastle, "Castle Battle Arena (Greenhouse: Center)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.CastleCrown, -1, 617)]),
    Locations.CastleTinyTrashCan: Location(Levels.CreepyCastle, "Castle Tiny Trash Can", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.CastleTrashCan, 0x4, 351, Kongs.tiny)]),
    Locations.CastleChunkyShed: Location(Levels.CreepyCastle, "Castle Chunky Shed", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.CastleShed, 0x6, 322, Kongs.chunky)]),
    Locations.CastleChunkyMuseum: Location(Levels.CreepyCastle, "Castle Chunky Museum", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(Maps.CastleMuseum, 0x7, 314, Kongs.chunky)]),
    Locations.CastleKasplatCrypt: Location(Levels.CreepyCastle, "Castle Kasplat: Lower Cave Center", Items.DiddyBlueprint, Types.Blueprint, Kongs.diddy, [Maps.CastleLowerCave]),
    Locations.CastleDiddyCrypt: Location(Levels.CreepyCastle, "Castle Diddy Crypt", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.CastleCrypt, 0x8, 310, Kongs.diddy)]),
    Locations.CastleChunkyCrypt: Location(Levels.CreepyCastle, "Castle Chunky Crypt", Items.GoldenBanana, Types.Banana, Kongs.chunky, [MapIDCombo(0, -1, 311, Kongs.chunky)]),
    Locations.CastleDonkeyMinecarts: Location(Levels.CreepyCastle, "Castle Donkey Minecart", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(0, -1, 318, Kongs.donkey)]),
    Locations.CastleLankyMausoleum: Location(Levels.CreepyCastle, "Castle Lanky Mausoleum", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(Maps.CastleMausoleum, 0x3, 308, Kongs.lanky)]),
    Locations.CastleTinyMausoleum: Location(Levels.CreepyCastle, "Castle Tiny Mausoleum", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(Maps.CastleMausoleum, 0xD, 309, Kongs.tiny)]),
    Locations.CastleTinyOverChasm: Location(Levels.CreepyCastle, "Castle Tiny Over Chasm", Items.GoldenBanana, Types.Banana, Kongs.tiny, [MapIDCombo(0, -1, 315, Kongs.tiny)]),
    Locations.CastleKasplatNearCandy: Location(Levels.CreepyCastle, "Castle Kasplat: Near Candy's", Items.ChunkyBlueprint, Types.Blueprint, Kongs.chunky, [Maps.CastleUpperCave]),
    Locations.CastleDonkeyDungeon: Location(Levels.CreepyCastle, "Castle Donkey Dungeon", Items.GoldenBanana, Types.Banana, Kongs.donkey, [MapIDCombo(Maps.CastleDungeon, 0xF, 326, Kongs.donkey)]),
    Locations.CastleDiddyDungeon: Location(Levels.CreepyCastle, "Castle Diddy Dungeon", Items.GoldenBanana, Types.Banana, Kongs.diddy, [MapIDCombo(Maps.CastleDungeon, 0xD, 353, Kongs.diddy)]),
    Locations.CastleLankyDungeon: Location(Levels.CreepyCastle, "Castle Lanky Dungeon", Items.GoldenBanana, Types.Banana, Kongs.lanky, [MapIDCombo(0, -1, 316, Kongs.lanky)]),
    Locations.CastleKey: Location(Levels.CreepyCastle, "Castle Boss Defeated", Items.CreepyCastleKey, Types.Key, Kongs.any, [MapIDCombo(0, -1, 317)]),  # Can be assigned to any kong
    # Hideout Helm locations
    Locations.HelmDonkey1: Location(Levels.HideoutHelm, "Helm Donkey Barrel 1", Items.HelmDonkey1, Types.Constant, Kongs.donkey),
    Locations.HelmDonkey2: Location(Levels.HideoutHelm, "Helm Donkey Barrel 2", Items.HelmDonkey2, Types.Constant, Kongs.donkey),
    Locations.HelmDiddy1: Location(Levels.HideoutHelm, "Helm Diddy Barrel 1", Items.HelmDiddy1, Types.Constant, Kongs.diddy),
    Locations.HelmDiddy2: Location(Levels.HideoutHelm, "Helm Diddy Barrel 2", Items.HelmDiddy2, Types.Constant, Kongs.diddy),
    Locations.HelmLanky1: Location(Levels.HideoutHelm, "Helm Lanky Barrel 1", Items.HelmLanky1, Types.Constant, Kongs.lanky),
    Locations.HelmLanky2: Location(Levels.HideoutHelm, "Helm Lanky Barrel 2", Items.HelmLanky2, Types.Constant, Kongs.lanky),
    Locations.HelmTiny1: Location(Levels.HideoutHelm, "Helm Tiny Barrel 1", Items.HelmTiny1, Types.Constant, Kongs.tiny),
    Locations.HelmTiny2: Location(Levels.HideoutHelm, "Helm Tiny Barrel 2", Items.HelmTiny2, Types.Constant, Kongs.tiny),
    Locations.HelmChunky1: Location(Levels.HideoutHelm, "Helm Chunky Barrel 1", Items.HelmChunky1, Types.Constant, Kongs.chunky),
    Locations.HelmChunky2: Location(Levels.HideoutHelm, "Helm Chunky Barrel 2", Items.HelmChunky2, Types.Constant, Kongs.chunky),
    Locations.HelmBattleArena: Location(Levels.HideoutHelm, "Helm Battle Arena (Top of Blast-o-Matic)", Items.BattleCrown, Types.Crown, Kongs.any, [MapIDCombo(Maps.HelmCrown, -1, 618)]),
    Locations.HelmDonkeyMedal: Location(Levels.HideoutHelm, "Helm Donkey Medal", Items.BananaMedal, Types.Medal, Kongs.donkey, [MapIDCombo(Maps.HideoutHelm, 0x5D, 584, Kongs.donkey)]),
    Locations.HelmChunkyMedal: Location(Levels.HideoutHelm, "Helm Chunky Medal", Items.BananaMedal, Types.Medal, Kongs.chunky, [MapIDCombo(Maps.HideoutHelm, 0x5E, 588, Kongs.chunky)]),
    Locations.HelmTinyMedal: Location(Levels.HideoutHelm, "Helm Tiny Medal", Items.BananaMedal, Types.Medal, Kongs.tiny, [MapIDCombo(Maps.HideoutHelm, 0x60, 587, Kongs.tiny)]),
    Locations.HelmLankyMedal: Location(Levels.HideoutHelm, "Helm Lanky Medal", Items.BananaMedal, Types.Medal, Kongs.lanky, [MapIDCombo(Maps.HideoutHelm, 0x5F, 586, Kongs.lanky)]),
    Locations.HelmDiddyMedal: Location(Levels.HideoutHelm, "Helm Diddy Medal", Items.BananaMedal, Types.Medal, Kongs.diddy, [MapIDCombo(Maps.HideoutHelm, 0x61, 585, Kongs.diddy)]),
    Locations.HelmBananaFairy1: Location(Levels.HideoutHelm, "Helm Fairy (Key 8 Room (1))", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, 598)]),
    Locations.HelmBananaFairy2: Location(Levels.HideoutHelm, "Helm Fairy (Key 8 Room (2))", Items.BananaFairy, Types.Fairy, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, 599)]),
    Locations.HelmKey: Location(Levels.HideoutHelm, "The End of Helm", Items.HideoutHelmKey, Types.Key, Kongs.any, [MapIDCombo(Maps.HideoutHelm, 0x5A, 380)]),

    # Normal shop locations
    Locations.SimianSlam: Location(Levels.DKIsles, "Isles Cranky Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Slam, 1, VendorType.Cranky]),
    Locations.BaboonBlast: Location(Levels.JungleJapes, "Japes Cranky Donkey", Items.BaboonBlast, Types.Shop, Kongs.donkey, [MoveTypes.Moves, 1, VendorType.Cranky]),
    Locations.ChimpyCharge: Location(Levels.JungleJapes, "Japes Cranky Diddy", Items.ChimpyCharge, Types.Shop, Kongs.diddy, [MoveTypes.Moves, 1, VendorType.Cranky]),
    Locations.Orangstand: Location(Levels.JungleJapes, "Japes Cranky Lanky", Items.Orangstand, Types.Shop, Kongs.lanky, [MoveTypes.Moves, 1, VendorType.Cranky]),
    Locations.MiniMonkey: Location(Levels.JungleJapes, "Japes Cranky Tiny", Items.MiniMonkey, Types.Shop, Kongs.tiny, [MoveTypes.Moves, 1, VendorType.Cranky]),
    Locations.HunkyChunky: Location(Levels.JungleJapes, "Japes Cranky Chunky", Items.HunkyChunky, Types.Shop, Kongs.chunky, [MoveTypes.Moves, 1, VendorType.Cranky]),
    Locations.CoconutGun: Location(Levels.JungleJapes, "Japes Funky Donkey", Items.Coconut, Types.Shop, Kongs.donkey, [MoveTypes.Guns, 1, VendorType.Funky]),
    Locations.PeanutGun: Location(Levels.JungleJapes, "Japes Funky Diddy", Items.Peanut, Types.Shop, Kongs.diddy, [MoveTypes.Guns, 1, VendorType.Funky]),
    Locations.GrapeGun: Location(Levels.JungleJapes, "Japes Funky Lanky", Items.Grape, Types.Shop, Kongs.lanky, [MoveTypes.Guns, 1, VendorType.Funky]),
    Locations.FeatherGun: Location(Levels.JungleJapes, "Japes Funky Tiny", Items.Feather, Types.Shop, Kongs.tiny, [MoveTypes.Guns, 1, VendorType.Funky]),
    Locations.PineappleGun: Location(Levels.JungleJapes, "Japes Funky Chunky", Items.Pineapple, Types.Shop, Kongs.chunky, [MoveTypes.Guns, 1, VendorType.Funky]),
    Locations.StrongKong: Location(Levels.AngryAztec, "Aztec Cranky Donkey", Items.StrongKong, Types.Shop, Kongs.donkey, [MoveTypes.Moves, 2, VendorType.Cranky]),
    Locations.RocketbarrelBoost: Location(Levels.AngryAztec, "Aztec Cranky Diddy", Items.RocketbarrelBoost, Types.Shop, Kongs.diddy, [MoveTypes.Moves, 2, VendorType.Cranky]),
    Locations.Bongos: Location(Levels.AngryAztec, "Aztec Candy Donkey", Items.Bongos, Types.Shop, Kongs.donkey, [MoveTypes.Instruments, 1, VendorType.Candy]),
    Locations.Guitar: Location(Levels.AngryAztec, "Aztec Candy Diddy", Items.Guitar, Types.Shop, Kongs.diddy, [MoveTypes.Instruments, 1, VendorType.Candy]),
    Locations.Trombone: Location(Levels.AngryAztec, "Aztec Candy Lanky", Items.Trombone, Types.Shop, Kongs.lanky, [MoveTypes.Instruments, 1, VendorType.Candy]),
    Locations.Saxophone: Location(Levels.AngryAztec, "Aztec Candy Tiny", Items.Saxophone, Types.Shop, Kongs.tiny, [MoveTypes.Instruments, 1, VendorType.Candy]),
    Locations.Triangle: Location(Levels.AngryAztec, "Aztec Candy Chunky", Items.Triangle, Types.Shop, Kongs.chunky, [MoveTypes.Instruments, 1, VendorType.Candy]),
    Locations.GorillaGrab: Location(Levels.FranticFactory, "Factory Cranky Donkey", Items.GorillaGrab, Types.Shop, Kongs.donkey, [MoveTypes.Moves, 3, VendorType.Cranky]),
    Locations.SimianSpring: Location(Levels.FranticFactory, "Factory Cranky Diddy", Items.SimianSpring, Types.Shop, Kongs.diddy, [MoveTypes.Moves, 3, VendorType.Cranky]),
    Locations.BaboonBalloon: Location(Levels.FranticFactory, "Factory Cranky Lanky", Items.BaboonBalloon, Types.Shop, Kongs.lanky, [MoveTypes.Moves, 2, VendorType.Cranky]),
    Locations.PonyTailTwirl: Location(Levels.FranticFactory, "Factory Cranky Tiny", Items.PonyTailTwirl, Types.Shop, Kongs.tiny, [MoveTypes.Moves, 2, VendorType.Cranky]),
    Locations.PrimatePunch: Location(Levels.FranticFactory, "Factory Cranky Chunky", Items.PrimatePunch, Types.Shop, Kongs.chunky, [MoveTypes.Moves, 2, VendorType.Cranky]),
    Locations.AmmoBelt1: Location(Levels.FranticFactory, "Factory Funky Shared", Items.ProgressiveAmmoBelt, Types.Shop, Kongs.any, [MoveTypes.AmmoBelt, 1, VendorType.Funky]),
    Locations.MusicUpgrade1: Location(Levels.GloomyGalleon, "Galleon Candy Shared", Items.ProgressiveInstrumentUpgrade, Types.Shop, Kongs.any, [MoveTypes.Instruments, 2, VendorType.Candy]),
    Locations.SuperSimianSlam: Location(Levels.FungiForest, "Forest Cranky Shared", Items.ProgressiveSlam, Types.Shop, Kongs.any, [MoveTypes.Slam, 2, VendorType.Cranky]),
    Locations.HomingAmmo: Location(Levels.FungiForest, "Forest Funky Shared", Items.HomingAmmo, Types.Shop, Kongs.any, [MoveTypes.Guns, 2, VendorType.Funky]),
    Locations.OrangstandSprint: Location(Levels.CrystalCaves, "Caves Cranky Lanky", Items.OrangstandSprint, Types.Shop, Kongs.lanky, [MoveTypes.Moves, 3, VendorType.Cranky]),
    Locations.Monkeyport: Location(Levels.CrystalCaves, "Caves Cranky Tiny", Items.Monkeyport, Types.Shop, Kongs.tiny, [MoveTypes.Moves, 3, VendorType.Cranky]),
    Locations.GorillaGone: Location(Levels.CrystalCaves, "Caves Cranky Chunky", Items.GorillaGone, Types.Shop, Kongs.chunky, [MoveTypes.Moves, 3, VendorType.Cranky]),
    Locations.AmmoBelt2: Location(Levels.CrystalCaves, "Caves Funky Shared", Items.ProgressiveAmmoBelt, Types.Shop, Kongs.any, [MoveTypes.AmmoBelt, 2, VendorType.Funky]),
    Locations.ThirdMelon: Location(Levels.CrystalCaves, "Caves Candy Shared", Items.ProgressiveInstrumentUpgrade, Types.Shop, Kongs.any, [MoveTypes.Instruments, 3, VendorType.Candy]),
    Locations.SuperDuperSimianSlam: Location(Levels.CreepyCastle, "Castle Cranky Shared", Items.ProgressiveSlam, Types.Shop, Kongs.any, [MoveTypes.Slam, 3, VendorType.Cranky]),
    Locations.SniperSight: Location(Levels.CreepyCastle, "Castle Funky Shared", Items.SniperSight, Types.Shop, Kongs.any, [MoveTypes.Guns, 3, VendorType.Funky]),
    Locations.MusicUpgrade2: Location(Levels.CreepyCastle, "Castle Candy Shared", Items.ProgressiveInstrumentUpgrade, Types.Shop, Kongs.any, [MoveTypes.Instruments, 4, VendorType.Candy]),
    Locations.RarewareCoin: Location(Levels.Shops, "Jetpac", Items.RarewareCoin, Types.RarewareCoin, Kongs.any, [MapIDCombo(Maps.Cranky, 0x2, 379)]),
    # Additional shop locations for randomized moves- Index doesn't really matter, just set to 0
    # Japes
    Locations.SharedJapesPotion: Location(Levels.JungleJapes, "Japes Cranky Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.SharedJapesGun: Location(Levels.JungleJapes, "Japes Funky Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Guns, 0, VendorType.Funky]),
    # Aztec
    Locations.SharedAztecPotion: Location(Levels.AngryAztec, "Aztec Cranky Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.LankyAztecPotion: Location(Levels.AngryAztec, "Aztec Cranky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.TinyAztecPotion: Location(Levels.AngryAztec, "Aztec Cranky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.ChunkyAztecPotion: Location(Levels.AngryAztec, "Aztec Cranky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.SharedAztecGun: Location(Levels.AngryAztec, "Aztec Funky Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DonkeyAztecGun: Location(Levels.AngryAztec, "Aztec Funky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DiddyAztecGun: Location(Levels.AngryAztec, "Aztec Funky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.LankyAztecGun: Location(Levels.AngryAztec, "Aztec Funky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.TinyAztecGun: Location(Levels.AngryAztec, "Aztec Funky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.ChunkyAztecGun: Location(Levels.AngryAztec, "Aztec Funky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.SharedAztecInstrument: Location(Levels.AngryAztec, "Aztec Candy Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Instruments, 0, VendorType.Candy]),
    # Factory
    Locations.SharedFactoryPotion: Location(Levels.FranticFactory, "Factory Cranky Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DonkeyFactoryGun: Location(Levels.FranticFactory, "Factory Funky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DiddyFactoryGun: Location(Levels.FranticFactory, "Factory Funky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.LankyFactoryGun: Location(Levels.FranticFactory, "Factory Funky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.TinyFactoryGun: Location(Levels.FranticFactory, "Factory Funky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.ChunkyFactoryGun: Location(Levels.FranticFactory, "Factory Funky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.SharedFactoryInstrument: Location(Levels.FranticFactory, "Factory Candy Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.DonkeyFactoryInstrument: Location(Levels.FranticFactory, "Factory Candy Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.DiddyFactoryInstrument: Location(Levels.FranticFactory, "Factory Candy Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.LankyFactoryInstrument: Location(Levels.FranticFactory, "Factory Candy Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.TinyFactoryInstrument: Location(Levels.FranticFactory, "Factory Candy Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.ChunkyFactoryInstrument: Location(Levels.FranticFactory, "Factory Candy Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Instruments, 0, VendorType.Candy]),
    # Galleon
    Locations.SharedGalleonPotion: Location(Levels.GloomyGalleon, "Galleon Cranky Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DonkeyGalleonPotion: Location(Levels.GloomyGalleon, "Galleon Cranky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DiddyGalleonPotion: Location(Levels.GloomyGalleon, "Galleon Cranky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.LankyGalleonPotion: Location(Levels.GloomyGalleon, "Galleon Cranky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.TinyGalleonPotion: Location(Levels.GloomyGalleon, "Galleon Cranky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.ChunkyGalleonPotion: Location(Levels.GloomyGalleon, "Galleon Cranky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.SharedGalleonGun: Location(Levels.GloomyGalleon, "Galleon Funky Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DonkeyGalleonGun: Location(Levels.GloomyGalleon, "Galleon Funky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DiddyGalleonGun: Location(Levels.GloomyGalleon, "Galleon Funky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.LankyGalleonGun: Location(Levels.GloomyGalleon, "Galleon Funky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.TinyGalleonGun: Location(Levels.GloomyGalleon, "Galleon Funky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.ChunkyGalleonGun: Location(Levels.GloomyGalleon, "Galleon Funky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DonkeyGalleonInstrument: Location(Levels.GloomyGalleon, "Galleon Candy Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.DiddyGalleonInstrument: Location(Levels.GloomyGalleon, "Galleon Candy Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.LankyGalleonInstrument: Location(Levels.GloomyGalleon, "Galleon Candy Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.TinyGalleonInstrument: Location(Levels.GloomyGalleon, "Galleon Candy Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.ChunkyGalleonInstrument: Location(Levels.GloomyGalleon, "Galleon Candy Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Instruments, 0, VendorType.Candy]),
    # Forest
    Locations.DonkeyForestPotion: Location(Levels.FungiForest, "Forest Cranky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DiddyForestPotion: Location(Levels.FungiForest, "Forest Cranky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.LankyForestPotion: Location(Levels.FungiForest, "Forest Cranky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.TinyForestPotion: Location(Levels.FungiForest, "Forest Cranky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.ChunkyForestPotion: Location(Levels.FungiForest, "Forest Cranky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DonkeyForestGun: Location(Levels.FungiForest, "Forest Funky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DiddyForestGun: Location(Levels.FungiForest, "Forest Funky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.LankyForestGun: Location(Levels.FungiForest, "Forest Funky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.TinyForestGun: Location(Levels.FungiForest, "Forest Funky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.ChunkyForestGun: Location(Levels.FungiForest, "Forest Funky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Guns, 0, VendorType.Funky]),
    # Caves
    Locations.SharedCavesPotion: Location(Levels.CrystalCaves, "Caves Cranky Shared", Items.NoItem, Types.Shop, Kongs.any, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DonkeyCavesPotion: Location(Levels.CrystalCaves, "Caves Cranky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DiddyCavesPotion: Location(Levels.CrystalCaves, "Caves Cranky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DonkeyCavesGun: Location(Levels.CrystalCaves, "Caves Funky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DiddyCavesGun: Location(Levels.CrystalCaves, "Caves Funky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.LankyCavesGun: Location(Levels.CrystalCaves, "Caves Funky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.TinyCavesGun: Location(Levels.CrystalCaves, "Caves Funky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.ChunkyCavesGun: Location(Levels.CrystalCaves, "Caves Funky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DonkeyCavesInstrument: Location(Levels.CrystalCaves, "Caves Candy Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.DiddyCavesInstrument: Location(Levels.CrystalCaves, "Caves Candy Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.LankyCavesInstrument: Location(Levels.CrystalCaves, "Caves Candy Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.TinyCavesInstrument: Location(Levels.CrystalCaves, "Caves Candy Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.ChunkyCavesInstrument: Location(Levels.CrystalCaves, "Caves Candy Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Instruments, 0, VendorType.Candy]),
    # Castle
    Locations.DonkeyCastlePotion: Location(Levels.CreepyCastle, "Castle Cranky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DiddyCastlePotion: Location(Levels.CreepyCastle, "Castle Cranky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.LankyCastlePotion: Location(Levels.CreepyCastle, "Castle Cranky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.TinyCastlePotion: Location(Levels.CreepyCastle, "Castle Cranky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.ChunkyCastlePotion: Location(Levels.CreepyCastle, "Castle Cranky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DonkeyCastleGun: Location(Levels.CreepyCastle, "Castle Funky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DiddyCastleGun: Location(Levels.CreepyCastle, "Castle Funky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.LankyCastleGun: Location(Levels.CreepyCastle, "Castle Funky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.TinyCastleGun: Location(Levels.CreepyCastle, "Castle Funky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.ChunkyCastleGun: Location(Levels.CreepyCastle, "Castle Funky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Guns, 0, VendorType.Funky]),
    Locations.DonkeyCastleInstrument: Location(Levels.CreepyCastle, "Castle Candy Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.DiddyCastleInstrument: Location(Levels.CreepyCastle, "Castle Candy Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.LankyCastleInstrument: Location(Levels.CreepyCastle, "Castle Candy Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.TinyCastleInstrument: Location(Levels.CreepyCastle, "Castle Candy Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Instruments, 0, VendorType.Candy]),
    Locations.ChunkyCastleInstrument: Location(Levels.CreepyCastle, "Castle Candy Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Instruments, 0, VendorType.Candy]),
    # Isles
    Locations.DonkeyIslesPotion: Location(Levels.DKIsles, "Isles Cranky Donkey", Items.NoItem, Types.Shop, Kongs.donkey, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.DiddyIslesPotion: Location(Levels.DKIsles, "Isles Cranky Diddy", Items.NoItem, Types.Shop, Kongs.diddy, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.LankyIslesPotion: Location(Levels.DKIsles, "Isles Cranky Lanky", Items.NoItem, Types.Shop, Kongs.lanky, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.TinyIslesPotion: Location(Levels.DKIsles, "Isles Cranky Tiny", Items.NoItem, Types.Shop, Kongs.tiny, [MoveTypes.Moves, 0, VendorType.Cranky]),
    Locations.ChunkyIslesPotion: Location(Levels.DKIsles, "Isles Cranky Chunky", Items.NoItem, Types.Shop, Kongs.chunky, [MoveTypes.Moves, 0, VendorType.Cranky]),

    # Blueprints
    Locations.TurnInJungleJapesDonkeyBlueprint: Location(Levels.Snide, "Turning In 1 Blueprint", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x3fe)]),
    Locations.TurnInJungleJapesDiddyBlueprint: Location(Levels.Snide, "Turning In 2 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x3ff)]),
    Locations.TurnInJungleJapesLankyBlueprint: Location(Levels.Snide, "Turning In 3 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x400)]),
    Locations.TurnInJungleJapesTinyBlueprint: Location(Levels.Snide, "Turning In 4 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x401)]),
    Locations.TurnInJungleJapesChunkyBlueprint: Location(Levels.Snide, "Turning In 5 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x402)]),
    Locations.TurnInAngryAztecDonkeyBlueprint: Location(Levels.Snide, "Turning In 6 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x403)]),
    Locations.TurnInAngryAztecDiddyBlueprint: Location(Levels.Snide, "Turning In 7 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x404)]),
    Locations.TurnInAngryAztecLankyBlueprint: Location(Levels.Snide, "Turning In 8 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x405)]),
    Locations.TurnInAngryAztecTinyBlueprint: Location(Levels.Snide, "Turning In 9 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x406)]),
    Locations.TurnInAngryAztecChunkyBlueprint: Location(Levels.Snide, "Turning In 10 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x407)]),
    Locations.TurnInFranticFactoryDonkeyBlueprint: Location(Levels.Snide, "Turning In 11 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x408)]),
    Locations.TurnInFranticFactoryDiddyBlueprint: Location(Levels.Snide, "Turning In 12 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x409)]),
    Locations.TurnInFranticFactoryLankyBlueprint: Location(Levels.Snide, "Turning In 13 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x40a)]),
    Locations.TurnInFranticFactoryTinyBlueprint: Location(Levels.Snide, "Turning In 14 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x40b)]),
    Locations.TurnInFranticFactoryChunkyBlueprint: Location(Levels.Snide, "Turning In 15 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x40c)]),
    Locations.TurnInGloomyGalleonDonkeyBlueprint: Location(Levels.Snide, "Turning In 16 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x40d)]),
    Locations.TurnInGloomyGalleonDiddyBlueprint: Location(Levels.Snide, "Turning In 17 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x40e)]),
    Locations.TurnInGloomyGalleonLankyBlueprint: Location(Levels.Snide, "Turning In 18 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x40f)]),
    Locations.TurnInGloomyGalleonTinyBlueprint: Location(Levels.Snide, "Turning In 19 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x410)]),
    Locations.TurnInGloomyGalleonChunkyBlueprint: Location(Levels.Snide, "Turning In 20 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x411)]),
    Locations.TurnInFungiForestDonkeyBlueprint: Location(Levels.Snide, "Turning In 21 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x412)]),
    Locations.TurnInFungiForestDiddyBlueprint: Location(Levels.Snide, "Turning In 22 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x413)]),
    Locations.TurnInFungiForestLankyBlueprint: Location(Levels.Snide, "Turning In 23 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x414)]),
    Locations.TurnInFungiForestTinyBlueprint: Location(Levels.Snide, "Turning In 24 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x415)]),
    Locations.TurnInFungiForestChunkyBlueprint: Location(Levels.Snide, "Turning In 25 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x416)]),
    Locations.TurnInCrystalCavesDonkeyBlueprint: Location(Levels.Snide, "Turning In 26 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x417)]),
    Locations.TurnInCrystalCavesDiddyBlueprint: Location(Levels.Snide, "Turning In 27 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x418)]),
    Locations.TurnInCrystalCavesLankyBlueprint: Location(Levels.Snide, "Turning In 28 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x419)]),
    Locations.TurnInCrystalCavesTinyBlueprint: Location(Levels.Snide, "Turning In 29 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x41a)]),
    Locations.TurnInCrystalCavesChunkyBlueprint: Location(Levels.Snide, "Turning In 30 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x41b)]),
    Locations.TurnInCreepyCastleDonkeyBlueprint: Location(Levels.Snide, "Turning In 31 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x41c)]),
    Locations.TurnInCreepyCastleDiddyBlueprint: Location(Levels.Snide, "Turning In 32 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x41d)]),
    Locations.TurnInCreepyCastleLankyBlueprint: Location(Levels.Snide, "Turning In 33 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x41e)]),
    Locations.TurnInCreepyCastleTinyBlueprint: Location(Levels.Snide, "Turning In 34 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x41f)]),
    Locations.TurnInCreepyCastleChunkyBlueprint: Location(Levels.Snide, "Turning In 35 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x420)]),
    Locations.TurnInDKIslesDonkeyBlueprint: Location(Levels.Snide, "Turning In 36 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x421)]),
    Locations.TurnInDKIslesDiddyBlueprint: Location(Levels.Snide, "Turning In 37 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x422)]),
    Locations.TurnInDKIslesLankyBlueprint: Location(Levels.Snide, "Turning In 38 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x423)]),
    Locations.TurnInDKIslesTinyBlueprint: Location(Levels.Snide, "Turning In 39 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x424)]),
    Locations.TurnInDKIslesChunkyBlueprint: Location(Levels.Snide, "Turning In 40 Blueprints", Items.GoldenBanana, Types.Banana, Kongs.any, [MapIDCombo(0, -1, 0x425)]),

    # Hint Doors
    Locations.JapesDonkeyDoor: Location(Levels.JungleJapes, "Japes Donkey Hint Door", Items.JapesDonkeyHint, Types.Hint, Kongs.donkey),
    Locations.JapesDiddyDoor: Location(Levels.JungleJapes, "Japes Diddy Hint Door", Items.JapesDiddyHint, Types.Hint, Kongs.diddy),
    Locations.JapesLankyDoor: Location(Levels.JungleJapes, "Japes Lanky Hint Door", Items.JapesLankyHint, Types.Hint, Kongs.lanky),
    Locations.JapesTinyDoor: Location(Levels.JungleJapes, "Japes Tiny Hint Door", Items.JapesTinyHint, Types.Hint, Kongs.tiny),
    Locations.JapesChunkyDoor: Location(Levels.JungleJapes, "Japes Chunky Hint Door", Items.JapesChunkyHint, Types.Hint, Kongs.chunky),
    Locations.AztecDonkeyDoor: Location(Levels.AngryAztec, "Aztec Donkey Hint Door", Items.AztecDonkeyHint, Types.Hint, Kongs.donkey),
    Locations.AztecDiddyDoor: Location(Levels.AngryAztec, "Aztec Diddy Hint Door", Items.AztecDiddyHint, Types.Hint, Kongs.diddy),
    Locations.AztecLankyDoor: Location(Levels.AngryAztec, "Aztec Lanky Hint Door", Items.AztecLankyHint, Types.Hint, Kongs.lanky),
    Locations.AztecTinyDoor: Location(Levels.AngryAztec, "Aztec Tiny Hint Door", Items.AztecTinyHint, Types.Hint, Kongs.tiny),
    Locations.AztecChunkyDoor: Location(Levels.AngryAztec, "Aztec Chunky Hint Door", Items.AztecChunkyHint, Types.Hint, Kongs.chunky),
    Locations.FactoryDonkeyDoor: Location(Levels.FranticFactory, "Factory Donkey Hint Door", Items.FactoryDonkeyHint, Types.Hint, Kongs.donkey),
    Locations.FactoryDiddyDoor: Location(Levels.FranticFactory, "Factory Diddy Hint Door", Items.FactoryDiddyHint, Types.Hint, Kongs.diddy),
    Locations.FactoryLankyDoor: Location(Levels.FranticFactory, "Factory Lanky Hint Door", Items.FactoryLankyHint, Types.Hint, Kongs.lanky),
    Locations.FactoryTinyDoor: Location(Levels.FranticFactory, "Factory Tiny Hint Door", Items.FactoryTinyHint, Types.Hint, Kongs.tiny),
    Locations.FactoryChunkyDoor: Location(Levels.FranticFactory, "Factory Chunky Hint Door", Items.FactoryChunkyHint, Types.Hint, Kongs.chunky),
    Locations.GalleonDonkeyDoor: Location(Levels.GloomyGalleon, "Galleon Donkey Hint Door", Items.GalleonDonkeyHint, Types.Hint, Kongs.donkey),
    Locations.GalleonDiddyDoor: Location(Levels.GloomyGalleon, "Galleon Diddy Hint Door", Items.GalleonDiddyHint, Types.Hint, Kongs.diddy),
    Locations.GalleonLankyDoor: Location(Levels.GloomyGalleon, "Galleon Lanky Hint Door", Items.GalleonLankyHint, Types.Hint, Kongs.lanky),
    Locations.GalleonTinyDoor: Location(Levels.GloomyGalleon, "Galleon Tiny Hint Door", Items.GalleonTinyHint, Types.Hint, Kongs.tiny),
    Locations.GalleonChunkyDoor: Location(Levels.GloomyGalleon, "Galleon Chunky Hint Door", Items.GalleonChunkyHint, Types.Hint, Kongs.chunky),
    Locations.ForestDonkeyDoor: Location(Levels.FungiForest, "Forest Donkey Hint Door", Items.ForestDonkeyHint, Types.Hint, Kongs.donkey),
    Locations.ForestDiddyDoor: Location(Levels.FungiForest, "Forest Diddy Hint Door", Items.ForestDiddyHint, Types.Hint, Kongs.diddy),
    Locations.ForestLankyDoor: Location(Levels.FungiForest, "Forest Lanky Hint Door", Items.ForestLankyHint, Types.Hint, Kongs.lanky),
    Locations.ForestTinyDoor: Location(Levels.FungiForest, "Forest Tiny Hint Door", Items.ForestTinyHint, Types.Hint, Kongs.tiny),
    Locations.ForestChunkyDoor: Location(Levels.FungiForest, "Forest Chunky Hint Door", Items.ForestChunkyHint, Types.Hint, Kongs.chunky),
    Locations.CavesDonkeyDoor: Location(Levels.CrystalCaves, "Caves Donkey Hint Door", Items.CavesDonkeyHint, Types.Hint, Kongs.donkey),
    Locations.CavesDiddyDoor: Location(Levels.CrystalCaves, "Caves Diddy Hint Door", Items.CavesDiddyHint, Types.Hint, Kongs.diddy),
    Locations.CavesLankyDoor: Location(Levels.CrystalCaves, "Caves Lanky Hint Door", Items.CavesLankyHint, Types.Hint, Kongs.lanky),
    Locations.CavesTinyDoor: Location(Levels.CrystalCaves, "Caves Tiny Hint Door", Items.CavesTinyHint, Types.Hint, Kongs.tiny),
    Locations.CavesChunkyDoor: Location(Levels.CrystalCaves, "Caves Chunky Hint Door", Items.CavesChunkyHint, Types.Hint, Kongs.chunky),
    Locations.CastleDonkeyDoor: Location(Levels.CreepyCastle, "Castle Donkey Hint Door", Items.CastleDonkeyHint, Types.Hint, Kongs.donkey),
    Locations.CastleDiddyDoor: Location(Levels.CreepyCastle, "Castle Diddy Hint Door", Items.CastleDiddyHint, Types.Hint, Kongs.diddy),
    Locations.CastleLankyDoor: Location(Levels.CreepyCastle, "Castle Lanky Hint Door", Items.CastleLankyHint, Types.Hint, Kongs.lanky),
    Locations.CastleTinyDoor: Location(Levels.CreepyCastle, "Castle Tiny Hint Door", Items.CastleTinyHint, Types.Hint, Kongs.tiny),
    Locations.CastleChunkyDoor: Location(Levels.CreepyCastle, "Castle Chunky Hint Door", Items.CastleChunkyHint, Types.Hint, Kongs.chunky),

    # Progressive Hints
    Locations.ProgressiveHint_01: Location(Levels.JungleJapes, "Progressive Hint 1", Items.JapesDonkeyHint, Types.ProgressiveHint, Kongs.donkey),
    Locations.ProgressiveHint_02: Location(Levels.JungleJapes, "Progressive Hint 2", Items.JapesDiddyHint, Types.ProgressiveHint, Kongs.diddy),
    Locations.ProgressiveHint_03: Location(Levels.JungleJapes, "Progressive Hint 3", Items.JapesLankyHint, Types.ProgressiveHint, Kongs.lanky),
    Locations.ProgressiveHint_04: Location(Levels.JungleJapes, "Progressive Hint 4", Items.JapesTinyHint, Types.ProgressiveHint, Kongs.tiny),
    Locations.ProgressiveHint_05: Location(Levels.JungleJapes, "Progressive Hint 5", Items.JapesChunkyHint, Types.ProgressiveHint, Kongs.chunky),
    Locations.ProgressiveHint_06: Location(Levels.AngryAztec, "Progressive Hint 6", Items.AztecDonkeyHint, Types.ProgressiveHint, Kongs.donkey),
    Locations.ProgressiveHint_07: Location(Levels.AngryAztec, "Progressive Hint 7", Items.AztecDiddyHint, Types.ProgressiveHint, Kongs.diddy),
    Locations.ProgressiveHint_08: Location(Levels.AngryAztec, "Progressive Hint 8", Items.AztecLankyHint, Types.ProgressiveHint, Kongs.lanky),
    Locations.ProgressiveHint_09: Location(Levels.AngryAztec, "Progressive Hint 9", Items.AztecTinyHint, Types.ProgressiveHint, Kongs.tiny),
    Locations.ProgressiveHint_10: Location(Levels.AngryAztec, "Progressive Hint 10", Items.AztecChunkyHint, Types.ProgressiveHint, Kongs.chunky),
    Locations.ProgressiveHint_11: Location(Levels.FranticFactory, "Progressive Hint 11", Items.FactoryDonkeyHint, Types.ProgressiveHint, Kongs.donkey),
    Locations.ProgressiveHint_12: Location(Levels.FranticFactory, "Progressive Hint 12", Items.FactoryDiddyHint, Types.ProgressiveHint, Kongs.diddy),
    Locations.ProgressiveHint_13: Location(Levels.FranticFactory, "Progressive Hint 13", Items.FactoryLankyHint, Types.ProgressiveHint, Kongs.lanky),
    Locations.ProgressiveHint_14: Location(Levels.FranticFactory, "Progressive Hint 14", Items.FactoryTinyHint, Types.ProgressiveHint, Kongs.tiny),
    Locations.ProgressiveHint_15: Location(Levels.FranticFactory, "Progressive Hint 15", Items.FactoryChunkyHint, Types.ProgressiveHint, Kongs.chunky),
    Locations.ProgressiveHint_16: Location(Levels.GloomyGalleon, "Progressive Hint 16", Items.GalleonDonkeyHint, Types.ProgressiveHint, Kongs.donkey),
    Locations.ProgressiveHint_17: Location(Levels.GloomyGalleon, "Progressive Hint 17", Items.GalleonDiddyHint, Types.ProgressiveHint, Kongs.diddy),
    Locations.ProgressiveHint_18: Location(Levels.GloomyGalleon, "Progressive Hint 18", Items.GalleonLankyHint, Types.ProgressiveHint, Kongs.lanky),
    Locations.ProgressiveHint_19: Location(Levels.GloomyGalleon, "Progressive Hint 19", Items.GalleonTinyHint, Types.ProgressiveHint, Kongs.tiny),
    Locations.ProgressiveHint_20: Location(Levels.GloomyGalleon, "Progressive Hint 20", Items.GalleonChunkyHint, Types.ProgressiveHint, Kongs.chunky),
    Locations.ProgressiveHint_21: Location(Levels.FungiForest, "Progressive Hint 21", Items.ForestDonkeyHint, Types.ProgressiveHint, Kongs.donkey),
    Locations.ProgressiveHint_22: Location(Levels.FungiForest, "Progressive Hint 22", Items.ForestDiddyHint, Types.ProgressiveHint, Kongs.diddy),
    Locations.ProgressiveHint_23: Location(Levels.FungiForest, "Progressive Hint 23", Items.ForestLankyHint, Types.ProgressiveHint, Kongs.lanky),
    Locations.ProgressiveHint_24: Location(Levels.FungiForest, "Progressive Hint 24", Items.ForestTinyHint, Types.ProgressiveHint, Kongs.tiny),
    Locations.ProgressiveHint_25: Location(Levels.FungiForest, "Progressive Hint 25", Items.ForestChunkyHint, Types.ProgressiveHint, Kongs.chunky),
    Locations.ProgressiveHint_26: Location(Levels.CrystalCaves, "Progressive Hint 26", Items.CavesDonkeyHint, Types.ProgressiveHint, Kongs.donkey),
    Locations.ProgressiveHint_27: Location(Levels.CrystalCaves, "Progressive Hint 27", Items.CavesDiddyHint, Types.ProgressiveHint, Kongs.diddy),
    Locations.ProgressiveHint_28: Location(Levels.CrystalCaves, "Progressive Hint 28", Items.CavesLankyHint, Types.ProgressiveHint, Kongs.lanky),
    Locations.ProgressiveHint_29: Location(Levels.CrystalCaves, "Progressive Hint 29", Items.CavesTinyHint, Types.ProgressiveHint, Kongs.tiny),
    Locations.ProgressiveHint_30: Location(Levels.CrystalCaves, "Progressive Hint 30", Items.CavesChunkyHint, Types.ProgressiveHint, Kongs.chunky),
    Locations.ProgressiveHint_31: Location(Levels.CreepyCastle, "Progressive Hint 31", Items.CastleDonkeyHint, Types.ProgressiveHint, Kongs.donkey),
    Locations.ProgressiveHint_32: Location(Levels.CreepyCastle, "Progressive Hint 32", Items.CastleDiddyHint, Types.ProgressiveHint, Kongs.diddy),
    Locations.ProgressiveHint_33: Location(Levels.CreepyCastle, "Progressive Hint 33", Items.CastleLankyHint, Types.ProgressiveHint, Kongs.lanky),
    Locations.ProgressiveHint_34: Location(Levels.CreepyCastle, "Progressive Hint 34", Items.CastleTinyHint, Types.ProgressiveHint, Kongs.tiny),
    Locations.ProgressiveHint_35: Location(Levels.CreepyCastle, "Progressive Hint 35", Items.CastleChunkyHint, Types.ProgressiveHint, Kongs.chunky),

    # Rainbow Coins - Has to be in order of map index
    Locations.RainbowCoin_Location00: Location(Levels.JungleJapes, "Japes Dirt: Painting Hill", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, 0x29E)]),  # Painting Hill
    Locations.RainbowCoin_Location01: Location(Levels.AngryAztec, "Aztec Dirt: Chunky Temple", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.AztecChunky5DTemple, -1, 0x29F)]),  # Chunky 5DT
    Locations.RainbowCoin_Location02: Location(Levels.FranticFactory, "Factory Dirt: Dark Room", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, 0x2A0)]),  # Dark Room
    Locations.RainbowCoin_Location03: Location(Levels.DKIsles, "Isles Dirt: Cabin Isle", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.Isles, -1, 0x2A1)]),  # Fungi Entrance
    Locations.RainbowCoin_Location04: Location(Levels.DKIsles, "Isles Dirt: Under Caves Lobby", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.Isles, -1, 0x2A2)]),  # Caves Slope
    Locations.RainbowCoin_Location05: Location(Levels.DKIsles, "Isles Dirt: Aztec Roof", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.Isles, -1, 0x2A3)]),  # Aztec Roof
    Locations.RainbowCoin_Location06: Location(Levels.AngryAztec, "Aztec Dirt: Oasis", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, 0x2A4)]),  # Oasis
    Locations.RainbowCoin_Location07: Location(Levels.FungiForest, "Forest Dirt: Mills Grass", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, 0x2A5)]),  # Isotarge Coin
    Locations.RainbowCoin_Location08: Location(Levels.FungiForest, "Forest Dirt: Beanstalk", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, 0x2A6)]),  # Beanstalk
    Locations.RainbowCoin_Location09: Location(Levels.GloomyGalleon, "Galleon Dirt: Lighthouse", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.GalleonLighthouse, -1, 0x2A7)]),  # Lighthouse
    Locations.RainbowCoin_Location10: Location(Levels.CrystalCaves, "Caves Dirt: Giant Kosha", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, 0x2A8)]),  # Giant Kosha
    Locations.RainbowCoin_Location11: Location(Levels.CreepyCastle, "Castle Dirt: Top Floor", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, 0x2A9)]),  # Castle Top
    Locations.RainbowCoin_Location12: Location(Levels.DKIsles, "Isles Dirt: Back of Prison", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.KLumsy, -1, 0x2AA)]),  # K. Lumsy
    Locations.RainbowCoin_Location13: Location(Levels.DKIsles, "Isles Dirt: Training Grounds Rear Tunnel", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.TrainingGrounds, -1, 0x2AB)]),  # Back of TG
    Locations.RainbowCoin_Location14: Location(Levels.DKIsles, "Isles Dirt: Banana Hoard", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.TrainingGrounds, -1, 0x2AC)]),  # Banana Hoard
    Locations.RainbowCoin_Location15: Location(Levels.DKIsles, "Isles Dirt: Castle Lobby", Items.RainbowCoin, Types.RainbowCoin, Kongs.any, [MapIDCombo(Maps.CreepyCastleLobby, -1, 0x2AD)]),  # Castle Lobby

    Locations.MelonCrate_Location00: Location(Levels.JungleJapes, "Japes Crate: Behind the Mountain", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, 0x3AC)]),  # Japes behind mountain
    Locations.MelonCrate_Location01: Location(Levels.JungleJapes, "Japes Crate: In the Rambi Cave", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, 0x3AD)]),  # Japes near cb boulder
    Locations.MelonCrate_Location02: Location(Levels.AngryAztec, "Aztec Crate: Llama Temple Entrance", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, 0x3AE)]),  # Llama Temple
    Locations.MelonCrate_Location03: Location(Levels.FranticFactory, "Factory Crate: Near Funky", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, 0x3AF)]),  # Factory near funky
    Locations.MelonCrate_Location04: Location(Levels.FranticFactory, "Factory Crate: Near Candy", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, 0x3B0)]),  # Factory near candy/cranky
    Locations.MelonCrate_Location05: Location(Levels.GloomyGalleon, "Galleon Crate: Near Cactus", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, 0x3B1)]),  # Galleon near cactus
    Locations.MelonCrate_Location06: Location(Levels.AngryAztec, "Aztec Crate: On Llama Temple", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, 0x3B2)]),  # Aztec on Llama
    Locations.MelonCrate_Location07: Location(Levels.AngryAztec, "Aztec Crate: Near Gong Tower", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, 0x3B3)]),  # Aztec near gong tower
    Locations.MelonCrate_Location08: Location(Levels.FungiForest, "Forest Crate: Near Owl Tree", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, 0x3B4)]),  # Fungi near owl tree
    Locations.MelonCrate_Location09: Location(Levels.FungiForest, "Forest Crate: Near Thornvine Barn", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, 0x3B5)]),  # Fungi near DK barn
    Locations.MelonCrate_Location10: Location(Levels.FungiForest, "Forest Crate: Behind Dark Attic", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, 0x3B6)]),  # Fungi behind dark attic
    Locations.MelonCrate_Location11: Location(Levels.FungiForest, "Forest Crate: In Thornvine Barn", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.ForestThornvineBarn, -1, 0x3B7)]),  # Fungi in DK Barn
    Locations.MelonCrate_Location12: Location(Levels.CreepyCastle, "Castle Crate: Behind Mausoleum Entrance", Items.CrateMelon, Types.CrateItem, Kongs.any, [MapIDCombo(Maps.CastleLowerCave, -1, 0x3B8)]),  # Crypt behind Mausoleum entrance

    Locations.HoldableBoulderIslesNearAztec: Location(Levels.DKIsles, "Isles Boulder: Near Level 2", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.Isles, -1, 0x2AE)]),
    Locations.HoldableBoulderIslesNearCaves: Location(Levels.DKIsles, "Isles Boulder: Near Level 6", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.Isles, -1, 0x2AF)]),
    Locations.HoldableBoulderAztec: Location(Levels.AngryAztec, "Aztec Boulder: Tunnel", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, 0x2B0)]),
    Locations.HoldableBoulderCavesSmall: Location(Levels.CrystalCaves, "Caves Boulder: Small", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, 0x2B1)]),
    Locations.HoldableBoulderCavesLarge: Location(Levels.CrystalCaves, "Caves Boulder: Large", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, 0x2B2)]),
    Locations.HoldableBoulderMuseum: Location(Levels.CreepyCastle, "Castle Boulder: Museum", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.CastleMuseum, -1, 0x2B3)]),
    Locations.HoldableBoulderJapesLobby: Location(Levels.DKIsles, "Isles Boulder: Japes Lobby", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.JungleJapesLobby, -1, 0x2B4)]),
    Locations.HoldableBoulderCastleLobby: Location(Levels.DKIsles, "Isles Boulder: Castle Lobby", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.CreepyCastleLobby, -1, 0x2B5)]),
    Locations.HoldableBoulderCavesLobby: Location(Levels.DKIsles, "Isles Boulder: Caves Lobby", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.CrystalCavesLobby, -1, 0x2B6)]),
    Locations.HoldableKegMillFrontNear: Location(Levels.FungiForest, "Forest Keg: Mill Front Near", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.ForestMillFront, -1, 0x2B7)]),
    Locations.HoldableKegMillFrontFar: Location(Levels.FungiForest, "Forest Keg: Mill Front Far", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.ForestMillFront, -1, 0x2B8)]),
    Locations.HoldableKegMillRear: Location(Levels.FungiForest, "Forest Keg: Mill Back", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.ForestMillBack, -1, 0x2B9)]),
    Locations.HoldableVaseCircle: Location(Levels.AngryAztec, "Aztec Vase: Circle", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, 0x2BA)]),
    Locations.HoldableVaseColon: Location(Levels.AngryAztec, "Aztec Vase: Colon", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, 0x2BB)]),
    Locations.HoldableVaseTriangle: Location(Levels.AngryAztec, "Aztec Vase: Triangle", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, 0x2BC)]),
    Locations.HoldableVasePlus: Location(Levels.AngryAztec, "Aztec Vase: Plus", Items.BoulderItem, Types.BoulderItem, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, 0x2BD)]),

    Locations.JapesMainEnemy_Start: Location(Levels.JungleJapes, "Japes Enemy: Start", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x0)]),
    Locations.JapesMainEnemy_DiddyCavern: Location(Levels.JungleJapes, "Japes Enemy: Diddy Cavern", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x1)]),
    Locations.JapesMainEnemy_Tunnel0: Location(Levels.JungleJapes, "Japes Enemy: Tunnel (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x2)]),
    Locations.JapesMainEnemy_Tunnel1: Location(Levels.JungleJapes, "Japes Enemy: Tunnel (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x3)]),
    Locations.JapesMainEnemy_Storm0: Location(Levels.JungleJapes, "Japes Enemy: Storm (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x4)]),
    Locations.JapesMainEnemy_Storm1: Location(Levels.JungleJapes, "Japes Enemy: Storm (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x5)]),
    Locations.JapesMainEnemy_Storm2: Location(Levels.JungleJapes, "Japes Enemy: Storm (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x6)]),
    Locations.JapesMainEnemy_Hive0: Location(Levels.JungleJapes, "Japes Enemy: Hive (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x7)]),
    Locations.JapesMainEnemy_Hive1: Location(Levels.JungleJapes, "Japes Enemy: Hive (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x8)]),
    Locations.JapesMainEnemy_Hive2: Location(Levels.JungleJapes, "Japes Enemy: Hive (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x9)]),
    Locations.JapesMainEnemy_Hive3: Location(Levels.JungleJapes, "Japes Enemy: Hive (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0xa)]),
    Locations.JapesMainEnemy_Hive4: Location(Levels.JungleJapes, "Japes Enemy: Hive (4)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0xb)]),
    Locations.JapesMainEnemy_KilledInDemo: Location(Levels.JungleJapes, "Japes Enemy: Killed In Demo", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0xc)]),
    Locations.JapesMainEnemy_NearUnderground: Location(Levels.JungleJapes, "Japes Enemy: Near Underground", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0xd)]),
    Locations.JapesMainEnemy_NearPainting0: Location(Levels.JungleJapes, "Japes Enemy: Near Painting (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0xe)]),
    Locations.JapesMainEnemy_NearPainting1: Location(Levels.JungleJapes, "Japes Enemy: Near Painting (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0xf)]),
    Locations.JapesMainEnemy_NearPainting2: Location(Levels.JungleJapes, "Japes Enemy: Near Painting (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x10)]),
    Locations.JapesMainEnemy_Mountain: Location(Levels.JungleJapes, "Japes Enemy: Mountain", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x11)]),
    Locations.JapesMainEnemy_FeatherTunnel: Location(Levels.JungleJapes, "Japes Enemy: Feather Tunnel", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x12)]),
    Locations.JapesMainEnemy_MiddleTunnel: Location(Levels.JungleJapes, "Japes Enemy: Middle Tunnel", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapes, -1, DROPSANITY_FLAG_START + 0x13)]),
    Locations.JapesLobbyEnemy_Enemy0: Location(Levels.DKIsles, "Isles Japes Lobby Enemy: Enemy (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapesLobby, -1, DROPSANITY_FLAG_START + 0x14)]),
    Locations.JapesLobbyEnemy_Enemy1: Location(Levels.DKIsles, "Isles Japes Lobby Enemy: Enemy (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JungleJapesLobby, -1, DROPSANITY_FLAG_START + 0x15)]),
    # Locations.JapesPaintingEnemy_Gauntlet0: Location(Levels.JungleJapes, "Japes Lanky Cave Enemy: Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesLankyCave, -1, DROPSANITY_FLAG_START + 0x16)]),
    # Locations.JapesPaintingEnemy_Gauntlet1: Location(Levels.JungleJapes, "Japes Lanky Cave Enemy: Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesLankyCave, -1, DROPSANITY_FLAG_START + 0x17)]),
    # Locations.JapesPaintingEnemy_Gauntlet2: Location(Levels.JungleJapes, "Japes Lanky Cave Enemy: Gauntlet2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesLankyCave, -1, DROPSANITY_FLAG_START + 0x18)]),
    # Locations.JapesPaintingEnemy_Gauntlet3: Location(Levels.JungleJapes, "Japes Lanky Cave Enemy: Gauntlet3", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesLankyCave, -1, DROPSANITY_FLAG_START + 0x19)]),
    # Locations.JapesPaintingEnemy_Gauntlet4: Location(Levels.JungleJapes, "Japes Lanky Cave Enemy: Gauntlet4", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesLankyCave, -1, DROPSANITY_FLAG_START + 0x1a)]),
    Locations.JapesMountainEnemy_Start0: Location(Levels.JungleJapes, "Japes Mountain Enemy: Start (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x1b)]),
    Locations.JapesMountainEnemy_Start1: Location(Levels.JungleJapes, "Japes Mountain Enemy: Start (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x1c)]),
    Locations.JapesMountainEnemy_Start2: Location(Levels.JungleJapes, "Japes Mountain Enemy: Start (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x1d)]),
    Locations.JapesMountainEnemy_Start3: Location(Levels.JungleJapes, "Japes Mountain Enemy: Start (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x1e)]),
    Locations.JapesMountainEnemy_Start4: Location(Levels.JungleJapes, "Japes Mountain Enemy: Start (4)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x1f)]),
    Locations.JapesMountainEnemy_NearGateSwitch0: Location(Levels.JungleJapes, "Japes Mountain Enemy: Near Gate Switch (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x20)]),
    Locations.JapesMountainEnemy_NearGateSwitch1: Location(Levels.JungleJapes, "Japes Mountain Enemy: Near Gate Switch (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x21)]),
    Locations.JapesMountainEnemy_HiLo: Location(Levels.JungleJapes, "Japes Mountain Enemy: Hi Lo", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x22)]),
    Locations.JapesMountainEnemy_Conveyor0: Location(Levels.JungleJapes, "Japes Mountain Enemy: Conveyor (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x23)]),
    Locations.JapesMountainEnemy_Conveyor1: Location(Levels.JungleJapes, "Japes Mountain Enemy: Conveyor (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesMountain, -1, DROPSANITY_FLAG_START + 0x24)]),
    Locations.JapesShellhiveEnemy_FirstRoom: Location(Levels.JungleJapes, "Japes Tiny Hive Enemy: First Room", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesTinyHive, -1, DROPSANITY_FLAG_START + 0x25)]),
    Locations.JapesShellhiveEnemy_SecondRoom0: Location(Levels.JungleJapes, "Japes Tiny Hive Enemy: Second Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesTinyHive, -1, DROPSANITY_FLAG_START + 0x26)]),
    Locations.JapesShellhiveEnemy_SecondRoom1: Location(Levels.JungleJapes, "Japes Tiny Hive Enemy: Second Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesTinyHive, -1, DROPSANITY_FLAG_START + 0x27)]),
    Locations.JapesShellhiveEnemy_ThirdRoom0: Location(Levels.JungleJapes, "Japes Tiny Hive Enemy: Third Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesTinyHive, -1, DROPSANITY_FLAG_START + 0x28)]),
    Locations.JapesShellhiveEnemy_ThirdRoom1: Location(Levels.JungleJapes, "Japes Tiny Hive Enemy: Third Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesTinyHive, -1, DROPSANITY_FLAG_START + 0x29)]),
    Locations.JapesShellhiveEnemy_ThirdRoom2: Location(Levels.JungleJapes, "Japes Tiny Hive Enemy: Third Room (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesTinyHive, -1, DROPSANITY_FLAG_START + 0x2a)]),
    Locations.JapesShellhiveEnemy_ThirdRoom3: Location(Levels.JungleJapes, "Japes Tiny Hive Enemy: Third Room (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesTinyHive, -1, DROPSANITY_FLAG_START + 0x2b)]),
    Locations.JapesShellhiveEnemy_MainRoom: Location(Levels.JungleJapes, "Japes Tiny Hive Enemy: Main Room", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.JapesTinyHive, -1, DROPSANITY_FLAG_START + 0x2c)]),

    Locations.AztecMainEnemy_VaseRoom0: Location(Levels.AngryAztec, "Aztec Enemy: Vase Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x2d)]),
    Locations.AztecMainEnemy_VaseRoom1: Location(Levels.AngryAztec, "Aztec Enemy: Vase Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x2e)]),
    Locations.AztecMainEnemy_VaseRoom2: Location(Levels.AngryAztec, "Aztec Enemy: Vase Room (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x2f)]),
    Locations.AztecMainEnemy_TunnelPad0: Location(Levels.AngryAztec, "Aztec Enemy: Tunnel Pad (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x30)]),
    Locations.AztecMainEnemy_TunnelCage0: Location(Levels.AngryAztec, "Aztec Enemy: Tunnel Cage (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x31)]),
    Locations.AztecMainEnemy_TunnelCage1: Location(Levels.AngryAztec, "Aztec Enemy: Tunnel Cage (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x32)]),
    Locations.AztecMainEnemy_TunnelCage2: Location(Levels.AngryAztec, "Aztec Enemy: Tunnel Cage (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x33)]),
    Locations.AztecMainEnemy_StartingTunnel0: Location(Levels.AngryAztec, "Aztec Enemy: Starting Tunnel (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x34)]),
    Locations.AztecMainEnemy_StartingTunnel1: Location(Levels.AngryAztec, "Aztec Enemy: Starting Tunnel (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x35)]),
    Locations.AztecMainEnemy_OasisDoor: Location(Levels.AngryAztec, "Aztec Enemy: Oasis Door", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x36)]),
    Locations.AztecMainEnemy_TunnelCage3: Location(Levels.AngryAztec, "Aztec Enemy: Tunnel Cage (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x37)]),
    Locations.AztecMainEnemy_OutsideLlama: Location(Levels.AngryAztec, "Aztec Enemy: Outside Llama", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x38)]),
    Locations.AztecMainEnemy_OutsideTower: Location(Levels.AngryAztec, "Aztec Enemy: Outside Tower", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x39)]),
    Locations.AztecMainEnemy_TunnelPad1: Location(Levels.AngryAztec, "Aztec Enemy: Tunnel Pad (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x3a)]),
    Locations.AztecMainEnemy_NearCandy: Location(Levels.AngryAztec, "Aztec Enemy: Near Candy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x3b)]),
    Locations.AztecMainEnemy_AroundTotem: Location(Levels.AngryAztec, "Aztec Enemy: Around Totem", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x3c)]),
    Locations.AztecMainEnemy_StartingTunnel2: Location(Levels.AngryAztec, "Aztec Enemy: Starting Tunnel (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x3d)]),
    Locations.AztecMainEnemy_StartingTunnel3: Location(Levels.AngryAztec, "Aztec Enemy: Starting Tunnel (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x3e)]),
    Locations.AztecMainEnemy_OutsideSnide: Location(Levels.AngryAztec, "Aztec Enemy: Outside Snide", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x3f)]),
    Locations.AztecMainEnemy_Outside5DT: Location(Levels.AngryAztec, "Aztec Enemy: Outside 5DT", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x40)]),
    Locations.AztecMainEnemy_NearSnoopTunnel: Location(Levels.AngryAztec, "Aztec Enemy: Near Sealed Quicksand Tunnel", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztec, -1, DROPSANITY_FLAG_START + 0x41)]),
    # Locations.AztecLobbyEnemy_Pad0: Location(Levels.DKIsles, "Aztec Lobby Enemy: Pad0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztecLobby, -1, DROPSANITY_FLAG_START + 0x42)]),
    # Locations.AztecLobbyEnemy_Pad1: Location(Levels.DKIsles, "Aztec Lobby Enemy: Pad1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AngryAztecLobby, -1, DROPSANITY_FLAG_START + 0x43)]),
    Locations.AztecDK5DTEnemy_StartTrap0: Location(Levels.AngryAztec, "Aztec Donkey 5DTemple Enemy: Start Trap (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDonkey5DTemple, -1, DROPSANITY_FLAG_START + 0x44)]),
    Locations.AztecDK5DTEnemy_StartTrap1: Location(Levels.AngryAztec, "Aztec Donkey 5DTemple Enemy: Start Trap (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDonkey5DTemple, -1, DROPSANITY_FLAG_START + 0x45)]),
    Locations.AztecDK5DTEnemy_StartTrap2: Location(Levels.AngryAztec, "Aztec Donkey 5DTemple Enemy: Start Trap (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDonkey5DTemple, -1, DROPSANITY_FLAG_START + 0x46)]),
    Locations.AztecDK5DTEnemy_EndTrap0: Location(Levels.AngryAztec, "Aztec Donkey 5DTemple Enemy: End Trap (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDonkey5DTemple, -1, DROPSANITY_FLAG_START + 0x47)]),
    Locations.AztecDK5DTEnemy_EndTrap1: Location(Levels.AngryAztec, "Aztec Donkey 5DTemple Enemy: End Trap (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDonkey5DTemple, -1, DROPSANITY_FLAG_START + 0x48)]),
    Locations.AztecDK5DTEnemy_EndTrap2: Location(Levels.AngryAztec, "Aztec Donkey 5DTemple Enemy: End Trap (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDonkey5DTemple, -1, DROPSANITY_FLAG_START + 0x49)]),
    Locations.AztecDK5DTEnemy_EndPath0: Location(Levels.AngryAztec, "Aztec Donkey 5DTemple Enemy: End Path (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDonkey5DTemple, -1, DROPSANITY_FLAG_START + 0x4a)]),
    Locations.AztecDK5DTEnemy_EndPath1: Location(Levels.AngryAztec, "Aztec Donkey 5DTemple Enemy: End Path (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDonkey5DTemple, -1, DROPSANITY_FLAG_START + 0x4b)]),
    Locations.AztecDK5DTEnemy_StartPath: Location(Levels.AngryAztec, "Aztec Donkey 5DTemple Enemy: Start Path", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDonkey5DTemple, -1, DROPSANITY_FLAG_START + 0x4c)]),
    Locations.AztecDiddy5DTEnemy_EndTrap0: Location(Levels.AngryAztec, "Aztec Diddy 5DTemple Enemy: End Trap (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDiddy5DTemple, -1, DROPSANITY_FLAG_START + 0x4d)]),
    Locations.AztecDiddy5DTEnemy_EndTrap1: Location(Levels.AngryAztec, "Aztec Diddy 5DTemple Enemy: End Trap (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDiddy5DTemple, -1, DROPSANITY_FLAG_START + 0x4e)]),
    Locations.AztecDiddy5DTEnemy_EndTrap2: Location(Levels.AngryAztec, "Aztec Diddy 5DTemple Enemy: End Trap (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDiddy5DTemple, -1, DROPSANITY_FLAG_START + 0x4f)]),
    Locations.AztecDiddy5DTEnemy_StartLeft0: Location(Levels.AngryAztec, "Aztec Diddy 5DTemple Enemy: Start Left (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDiddy5DTemple, -1, DROPSANITY_FLAG_START + 0x50)]),
    Locations.AztecDiddy5DTEnemy_StartLeft1: Location(Levels.AngryAztec, "Aztec Diddy 5DTemple Enemy: Start Left (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDiddy5DTemple, -1, DROPSANITY_FLAG_START + 0x51)]),
    Locations.AztecDiddy5DTEnemy_Reward: Location(Levels.AngryAztec, "Aztec Diddy 5DTemple Enemy: Reward", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDiddy5DTemple, -1, DROPSANITY_FLAG_START + 0x52)]),
    Locations.AztecDiddy5DTEnemy_SecondSwitch: Location(Levels.AngryAztec, "Aztec Diddy 5DTemple Enemy: Second Switch", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecDiddy5DTemple, -1, DROPSANITY_FLAG_START + 0x53)]),
    Locations.AztecLanky5DTEnemy_JoiningPaths: Location(Levels.AngryAztec, "Aztec Lanky 5DTemple Enemy: Joining Paths", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLanky5DTemple, -1, DROPSANITY_FLAG_START + 0x54)]),
    Locations.AztecLanky5DTEnemy_EndTrap: Location(Levels.AngryAztec, "Aztec Lanky 5DTemple Enemy: End Trap", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLanky5DTemple, -1, DROPSANITY_FLAG_START + 0x55)]),
    Locations.AztecLanky5DTEnemy_Reward: Location(Levels.AngryAztec, "Aztec Lanky 5DTemple Enemy: Reward", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLanky5DTemple, -1, DROPSANITY_FLAG_START + 0x56)]),
    Locations.AztecTiny5DTEnemy_StartRightFront: Location(Levels.AngryAztec, "Aztec Tiny 5DTemple Enemy: Start Right Front", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTiny5DTemple, -1, DROPSANITY_FLAG_START + 0x57)]),
    Locations.AztecTiny5DTEnemy_StartLeftBack: Location(Levels.AngryAztec, "Aztec Tiny 5DTemple Enemy: Start Left Back", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTiny5DTemple, -1, DROPSANITY_FLAG_START + 0x58)]),
    Locations.AztecTiny5DTEnemy_StartRightBack: Location(Levels.AngryAztec, "Aztec Tiny 5DTemple Enemy: Start Right Back", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTiny5DTemple, -1, DROPSANITY_FLAG_START + 0x59)]),
    Locations.AztecTiny5DTEnemy_StartLeftFront: Location(Levels.AngryAztec, "Aztec Tiny 5DTemple Enemy: Start Left Front", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTiny5DTemple, -1, DROPSANITY_FLAG_START + 0x5a)]),
    Locations.AztecTiny5DTEnemy_Reward0: Location(Levels.AngryAztec, "Aztec Tiny 5DTemple Enemy: Reward (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTiny5DTemple, -1, DROPSANITY_FLAG_START + 0x5b)]),
    Locations.AztecTiny5DTEnemy_Reward1: Location(Levels.AngryAztec, "Aztec Tiny 5DTemple Enemy: Reward (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTiny5DTemple, -1, DROPSANITY_FLAG_START + 0x5c)]),
    Locations.AztecTiny5DTEnemy_DeadEnd0: Location(Levels.AngryAztec, "Aztec Tiny 5DTemple Enemy: Dead End (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTiny5DTemple, -1, DROPSANITY_FLAG_START + 0x5d)]),
    Locations.AztecTiny5DTEnemy_DeadEnd1: Location(Levels.AngryAztec, "Aztec Tiny 5DTemple Enemy: Dead End (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTiny5DTemple, -1, DROPSANITY_FLAG_START + 0x5e)]),
    Locations.AztecChunky5DTEnemy_StartRight: Location(Levels.AngryAztec, "Aztec Chunky 5DTemple Enemy: Start Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecChunky5DTemple, -1, DROPSANITY_FLAG_START + 0x5f)]),
    Locations.AztecChunky5DTEnemy_StartLeft: Location(Levels.AngryAztec, "Aztec Chunky 5DTemple Enemy: Start Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecChunky5DTemple, -1, DROPSANITY_FLAG_START + 0x60)]),
    Locations.AztecChunky5DTEnemy_SecondRight: Location(Levels.AngryAztec, "Aztec Chunky 5DTemple Enemy: Second Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecChunky5DTemple, -1, DROPSANITY_FLAG_START + 0x61)]),
    Locations.AztecChunky5DTEnemy_SecondLeft: Location(Levels.AngryAztec, "Aztec Chunky 5DTemple Enemy: Second Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecChunky5DTemple, -1, DROPSANITY_FLAG_START + 0x62)]),
    Locations.AztecChunky5DTEnemy_Reward: Location(Levels.AngryAztec, "Aztec Chunky 5DTemple Enemy: Reward", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecChunky5DTemple, -1, DROPSANITY_FLAG_START + 0x63)]),
    Locations.AztecLlamaEnemy_KongFreeInstrument: Location(Levels.AngryAztec, "Aztec Llama Temple Enemy: Kong Free Instrument", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, DROPSANITY_FLAG_START + 0x64)]),
    Locations.AztecLlamaEnemy_DinoInstrument: Location(Levels.AngryAztec, "Aztec Llama Temple Enemy: Dino Instrument", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, DROPSANITY_FLAG_START + 0x65)]),
    Locations.AztecLlamaEnemy_Matching0: Location(Levels.AngryAztec, "Aztec Llama Temple Enemy: Matching0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, DROPSANITY_FLAG_START + 0x66)]),
    Locations.AztecLlamaEnemy_Matching1: Location(Levels.AngryAztec, "Aztec Llama Temple Enemy: Matching1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, DROPSANITY_FLAG_START + 0x67)]),
    Locations.AztecLlamaEnemy_Right: Location(Levels.AngryAztec, "Aztec Llama Temple Enemy: Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, DROPSANITY_FLAG_START + 0x68)]),
    Locations.AztecLlamaEnemy_Left: Location(Levels.AngryAztec, "Aztec Llama Temple Enemy: Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, DROPSANITY_FLAG_START + 0x69)]),
    Locations.AztecLlamaEnemy_MelonCrate: Location(Levels.AngryAztec, "Aztec Llama Temple Enemy: Melon Crate", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, DROPSANITY_FLAG_START + 0x6a)]),
    Locations.AztecLlamaEnemy_SlamSwitch: Location(Levels.AngryAztec, "Aztec Llama Temple Enemy: Slam Switch", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecLlamaTemple, -1, DROPSANITY_FLAG_START + 0x6b)]),
    # Locations.AztecTempleEnemy_Rotating00: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating00", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x6c)]),
    # Locations.AztecTempleEnemy_Rotating01: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating01", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x6d)]),
    # Locations.AztecTempleEnemy_Rotating02: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating02", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x6e)]),
    # Locations.AztecTempleEnemy_Rotating03: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating03", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x6f)]),
    # Locations.AztecTempleEnemy_Rotating04: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating04", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x70)]),
    # Locations.AztecTempleEnemy_Rotating05: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating05", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x71)]),
    # Locations.AztecTempleEnemy_Rotating06: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating06", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x72)]),
    # Locations.AztecTempleEnemy_Rotating07: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating07", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x73)]),
    # Locations.AztecTempleEnemy_Rotating08: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating08", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x74)]),
    # Locations.AztecTempleEnemy_Rotating09: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating09", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x75)]),
    # Locations.AztecTempleEnemy_Rotating10: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating10", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x76)]),
    # Locations.AztecTempleEnemy_Rotating11: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating11", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x77)]),
    # Locations.AztecTempleEnemy_Rotating12: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating12", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x78)]),
    # Locations.AztecTempleEnemy_Rotating13: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating13", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x79)]),
    # Locations.AztecTempleEnemy_Rotating14: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating14", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x7a)]),
    # Locations.AztecTempleEnemy_Rotating15: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Rotating15", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x7b)]),
    # Locations.AztecTempleEnemy_MiniRoom00: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Mini Room00", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x7c)]),
    # Locations.AztecTempleEnemy_MiniRoom01: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Mini Room01", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x7d)]),
    # Locations.AztecTempleEnemy_MiniRoom02: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Mini Room02", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x7e)]),
    # Locations.AztecTempleEnemy_MiniRoom03: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Mini Room03", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x7f)]),
    Locations.AztecTempleEnemy_GuardRotating0: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Guard Rotating (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x80)]),
    Locations.AztecTempleEnemy_GuardRotating1: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Guard Rotating (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x81)]),
    Locations.AztecTempleEnemy_MainRoom0: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Main Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x82)]),
    Locations.AztecTempleEnemy_MainRoom1: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Main Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x83)]),
    Locations.AztecTempleEnemy_MainRoom2: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Main Room (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x84)]),
    Locations.AztecTempleEnemy_KongRoom0: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Kong Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x85)]),
    Locations.AztecTempleEnemy_KongRoom1: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Kong Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x86)]),
    Locations.AztecTempleEnemy_KongRoom2: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Kong Room (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x87)]),
    Locations.AztecTempleEnemy_KongRoom3: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Kong Room (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x88)]),
    Locations.AztecTempleEnemy_KongRoom4: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Kong Room (4)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x89)]),
    # Locations.AztecTempleEnemy_Underwater: Location(Levels.AngryAztec, "Aztec Tiny Temple Enemy: Underwater", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.AztecTinyTemple, -1, DROPSANITY_FLAG_START + 0x8a)]),

    Locations.FactoryMainEnemy_CandyCranky0: Location(Levels.FranticFactory, "Factory Enemy: Candy Cranky (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x8b)]),
    Locations.FactoryMainEnemy_CandyCranky1: Location(Levels.FranticFactory, "Factory Enemy: Candy Cranky (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x8c)]),
    Locations.FactoryMainEnemy_LobbyLeft: Location(Levels.FranticFactory, "Factory Enemy: Lobby Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x8d)]),
    Locations.FactoryMainEnemy_LobbyRight: Location(Levels.FranticFactory, "Factory Enemy: Lobby Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x8e)]),
    Locations.FactoryMainEnemy_StorageRoom: Location(Levels.FranticFactory, "Factory Enemy: Storage Room", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x8f)]),
    Locations.FactoryMainEnemy_BlockTower0: Location(Levels.FranticFactory, "Factory Enemy: Block Tower (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x90)]),
    Locations.FactoryMainEnemy_BlockTower1: Location(Levels.FranticFactory, "Factory Enemy: Block Tower (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x91)]),
    Locations.FactoryMainEnemy_BlockTower2: Location(Levels.FranticFactory, "Factory Enemy: Block Tower (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x92)]),
    Locations.FactoryMainEnemy_TunnelToHatch: Location(Levels.FranticFactory, "Factory Enemy: Tunnel To Hatch", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x93)]),
    Locations.FactoryMainEnemy_TunnelToProd0: Location(Levels.FranticFactory, "Factory Enemy: Tunnel To Prod (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x94)]),
    Locations.FactoryMainEnemy_TunnelToProd1: Location(Levels.FranticFactory, "Factory Enemy: Tunnel To Prod (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x95)]),
    Locations.FactoryMainEnemy_TunnelToBlockTower: Location(Levels.FranticFactory, "Factory Enemy: Tunnel To Block Tower", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x96)]),
    Locations.FactoryMainEnemy_TunnelToRace0: Location(Levels.FranticFactory, "Factory Enemy: Tunnel To Race (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x97)]),
    Locations.FactoryMainEnemy_TunnelToRace1: Location(Levels.FranticFactory, "Factory Enemy: Tunnel To Race (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x98)]),
    Locations.FactoryMainEnemy_LowWarp4: Location(Levels.FranticFactory, "Factory Enemy: Low Warp 4", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x99)]),
    Locations.FactoryMainEnemy_DiddySwitch: Location(Levels.FranticFactory, "Factory Enemy: Diddy Switch", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x9a)]),
    Locations.FactoryMainEnemy_ToBlockTowerTunnel: Location(Levels.FranticFactory, "Factory Enemy: To Block Tower Tunnel", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x9b)]),
    Locations.FactoryMainEnemy_DarkRoom0: Location(Levels.FranticFactory, "Factory Enemy: Dark Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x9c)]),
    Locations.FactoryMainEnemy_DarkRoom1: Location(Levels.FranticFactory, "Factory Enemy: Dark Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x9d)]),
    # Locations.FactoryMainEnemy_BHDM0: Location(Levels.FranticFactory, "Factory Enemy: BHDM0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x9e)]),
    # Locations.FactoryMainEnemy_BHDM1: Location(Levels.FranticFactory, "Factory Enemy: BHDM1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0x9f)]),
    # Locations.FactoryMainEnemy_BHDM2: Location(Levels.FranticFactory, "Factory Enemy: BHDM2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa0)]),
    # Locations.FactoryMainEnemy_BHDM3: Location(Levels.FranticFactory, "Factory Enemy: BHDM3", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa1)]),
    # Locations.FactoryMainEnemy_BHDM4: Location(Levels.FranticFactory, "Factory Enemy: BHDM4", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa2)]),
    # Locations.FactoryMainEnemy_BHDM5: Location(Levels.FranticFactory, "Factory Enemy: BHDM5", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa3)]),
    # Locations.FactoryMainEnemy_BHDM6: Location(Levels.FranticFactory, "Factory Enemy: BHDM6", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa4)]),
    # Locations.FactoryMainEnemy_BHDM7: Location(Levels.FranticFactory, "Factory Enemy: BHDM7", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa5)]),
    # Locations.FactoryMainEnemy_BHDM8: Location(Levels.FranticFactory, "Factory Enemy: BHDM8", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa6)]),
    # Locations.FactoryMainEnemy_BHDM9: Location(Levels.FranticFactory, "Factory Enemy: BHDM9", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa7)]),
    # Locations.FactoryMainEnemy_1342Gauntlet0: Location(Levels.FranticFactory, "Factory Enemy: 1342Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa8)]),
    # Locations.FactoryMainEnemy_1342Gauntlet1: Location(Levels.FranticFactory, "Factory Enemy: 1342Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xa9)]),
    # Locations.FactoryMainEnemy_1342Gauntlet2: Location(Levels.FranticFactory, "Factory Enemy: 1342Gauntlet2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xaa)]),
    # Locations.FactoryMainEnemy_3124Gauntlet0: Location(Levels.FranticFactory, "Factory Enemy: 3124Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xab)]),
    # Locations.FactoryMainEnemy_3124Gauntlet1: Location(Levels.FranticFactory, "Factory Enemy: 3124Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xac)]),
    # Locations.FactoryMainEnemy_3124Gauntlet2: Location(Levels.FranticFactory, "Factory Enemy: 3124Gauntlet2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xad)]),
    # Locations.FactoryMainEnemy_4231Gauntlet0: Location(Levels.FranticFactory, "Factory Enemy: 4231Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xae)]),
    # Locations.FactoryMainEnemy_4231Gauntlet1: Location(Levels.FranticFactory, "Factory Enemy: 4231Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactory, -1, DROPSANITY_FLAG_START + 0xaf)]),
    Locations.FactoryLobbyEnemy_Enemy0: Location(Levels.DKIsles, "Isles Factory Lobby Enemy: Enemy (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FranticFactoryLobby, -1, DROPSANITY_FLAG_START + 0xb0)]),
    Locations.GalleonMainEnemy_ChestRoom0: Location(Levels.GloomyGalleon, "Galleon Enemy: Chest Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, DROPSANITY_FLAG_START + 0xb1)]),
    Locations.GalleonMainEnemy_ChestRoom1: Location(Levels.GloomyGalleon, "Galleon Enemy: Chest Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, DROPSANITY_FLAG_START + 0xb2)]),
    Locations.GalleonMainEnemy_NearVineCannon: Location(Levels.GloomyGalleon, "Galleon Enemy: Near Vine Cannon", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, DROPSANITY_FLAG_START + 0xb3)]),
    Locations.GalleonMainEnemy_CrankyCannon: Location(Levels.GloomyGalleon, "Galleon Enemy: Cranky Cannon", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, DROPSANITY_FLAG_START + 0xb4)]),
    # Locations.GalleonMainEnemy_Submarine: Location(Levels.GloomyGalleon, "Galleon Enemy: Submarine", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, DROPSANITY_FLAG_START + 0xb5)]),
    # Locations.GalleonMainEnemy_5DS0: Location(Levels.GloomyGalleon, "Galleon Enemy: 5DS0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, DROPSANITY_FLAG_START + 0xb6)]),
    # Locations.GalleonMainEnemy_5DS1: Location(Levels.GloomyGalleon, "Galleon Enemy: 5DS1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, DROPSANITY_FLAG_START + 0xb7)]),
    Locations.GalleonMainEnemy_PeanutTunnel: Location(Levels.GloomyGalleon, "Galleon Enemy: Peanut Tunnel", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, DROPSANITY_FLAG_START + 0xb8)]),
    Locations.GalleonMainEnemy_CoconutTunnel: Location(Levels.GloomyGalleon, "Galleon Enemy: Coconut Tunnel", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GloomyGalleon, -1, DROPSANITY_FLAG_START + 0xb9)]),
    Locations.GalleonLighthouseEnemy_Enemy0: Location(Levels.GloomyGalleon, "Galleon Lighthouse Enemy: Enemy (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GalleonLighthouse, -1, DROPSANITY_FLAG_START + 0xba)]),
    Locations.GalleonLighthouseEnemy_Enemy1: Location(Levels.GloomyGalleon, "Galleon Lighthouse Enemy: Enemy (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GalleonLighthouse, -1, DROPSANITY_FLAG_START + 0xbb)]),
    # Locations.Galleon5DSDLCEnemy_Diddy: Location(Levels.GloomyGalleon, "Galleon5DShip Diddy Lanky Chunky Enemy: Diddy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon5DShipDiddyLankyChunky, -1, DROPSANITY_FLAG_START + 0xbc)]),
    # Locations.Galleon5DSDLCEnemy_Chunky: Location(Levels.GloomyGalleon, "Galleon5DShip Diddy Lanky Chunky Enemy: Chunky", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon5DShipDiddyLankyChunky, -1, DROPSANITY_FLAG_START + 0xbd)]),
    # Locations.Galleon5DSDLCEnemy_Lanky: Location(Levels.GloomyGalleon, "Galleon5DShip Diddy Lanky Chunky Enemy: Lanky", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon5DShipDiddyLankyChunky, -1, DROPSANITY_FLAG_START + 0xbe)]),
    # Locations.Galleon5DSDTEnemy_DK0: Location(Levels.GloomyGalleon, "Galleon5DShip DKTiny Enemy: DK0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon5DShipDKTiny, -1, DROPSANITY_FLAG_START + 0xbf)]),
    # Locations.Galleon5DSDTEnemy_DK1: Location(Levels.GloomyGalleon, "Galleon5DShip DKTiny Enemy: DK1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon5DShipDKTiny, -1, DROPSANITY_FLAG_START + 0xc0)]),
    # Locations.Galleon5DSDTEnemy_DK2: Location(Levels.GloomyGalleon, "Galleon5DShip DKTiny Enemy: DK2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon5DShipDKTiny, -1, DROPSANITY_FLAG_START + 0xc1)]),
    # Locations.Galleon5DSDTEnemy_TinyCage: Location(Levels.GloomyGalleon, "Galleon5DShip DKTiny Enemy: Tiny Cage", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon5DShipDKTiny, -1, DROPSANITY_FLAG_START + 0xc2)]),
    # Locations.Galleon5DSDTEnemy_TinyBed: Location(Levels.GloomyGalleon, "Galleon5DShip DKTiny Enemy: Tiny Bed", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon5DShipDKTiny, -1, DROPSANITY_FLAG_START + 0xc3)]),
    # Locations.Galleon2DSEnemy_Tiny0: Location(Levels.GloomyGalleon, "Galleon2DShip Enemy: Tiny0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon2DShip, -1, DROPSANITY_FLAG_START + 0xc4)]),
    # Locations.Galleon2DSEnemy_Tiny1: Location(Levels.GloomyGalleon, "Galleon2DShip Enemy: Tiny1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Galleon2DShip, -1, DROPSANITY_FLAG_START + 0xc5)]),
    # Locations.GalleonSubEnemy_Enemy0: Location(Levels.GloomyGalleon, "Galleon Submarine Enemy: Enemy0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GalleonSubmarine, -1, DROPSANITY_FLAG_START + 0xc6)]),
    # Locations.GalleonSubEnemy_Enemy1: Location(Levels.GloomyGalleon, "Galleon Submarine Enemy: Enemy1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GalleonSubmarine, -1, DROPSANITY_FLAG_START + 0xc7)]),
    # Locations.GalleonSubEnemy_Enemy2: Location(Levels.GloomyGalleon, "Galleon Submarine Enemy: Enemy2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GalleonSubmarine, -1, DROPSANITY_FLAG_START + 0xc8)]),
    # Locations.GalleonSubEnemy_Enemy3: Location(Levels.GloomyGalleon, "Galleon Submarine Enemy: Enemy3", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.GalleonSubmarine, -1, DROPSANITY_FLAG_START + 0xc9)]),
    Locations.ForestMainEnemy_HollowTree0: Location(Levels.FungiForest, "Forest Enemy: Hollow Tree (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xca)]),
    Locations.ForestMainEnemy_HollowTree1: Location(Levels.FungiForest, "Forest Enemy: Hollow Tree (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xcb)]),
    Locations.ForestMainEnemy_HollowTreeEntrance: Location(Levels.FungiForest, "Forest Enemy: Hollow Tree Entrance", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xcc)]),
    Locations.ForestMainEnemy_TreeMelonCrate0: Location(Levels.FungiForest, "Forest Enemy: Tree Melon Crate (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xcd)]),
    Locations.ForestMainEnemy_TreeMelonCrate1: Location(Levels.FungiForest, "Forest Enemy: Tree Melon Crate (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xce)]),
    Locations.ForestMainEnemy_TreeMelonCrate2: Location(Levels.FungiForest, "Forest Enemy: Tree Melon Crate (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xcf)]),
    Locations.ForestMainEnemy_AppleGauntlet0: Location(Levels.FungiForest, "Forest Enemy: Apple Gauntlet (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd0)]),
    Locations.ForestMainEnemy_AppleGauntlet1: Location(Levels.FungiForest, "Forest Enemy: Apple Gauntlet (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd1)]),
    Locations.ForestMainEnemy_AppleGauntlet2: Location(Levels.FungiForest, "Forest Enemy: Apple Gauntlet (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd2)]),
    Locations.ForestMainEnemy_AppleGauntlet3: Location(Levels.FungiForest, "Forest Enemy: Apple Gauntlet (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd3)]),
    Locations.ForestMainEnemy_NearBeanstalk0: Location(Levels.FungiForest, "Forest Enemy: Near Beanstalk (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd4)]),
    Locations.ForestMainEnemy_NearBeanstalk1: Location(Levels.FungiForest, "Forest Enemy: Near Beanstalk (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd5)]),
    Locations.ForestMainEnemy_GreenTunnel: Location(Levels.FungiForest, "Forest Enemy: Green Tunnel", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd6)]),
    Locations.ForestMainEnemy_NearLowWarp5: Location(Levels.FungiForest, "Forest Enemy: Near Low Warp 5", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd7)]),
    Locations.ForestMainEnemy_NearPinkTunnelBounceTag: Location(Levels.FungiForest, "Forest Enemy: Near Pink Tunnel Bounce Tag", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd8)]),
    Locations.ForestMainEnemy_NearGMRocketbarrel: Location(Levels.FungiForest, "Forest Enemy: Near Giant Mushroom Rocketbarrel", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xd9)]),
    Locations.ForestMainEnemy_BetweenYellowTunnelAndRB: Location(Levels.FungiForest, "Forest Enemy: Between Yellow Tunnel And RB", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xda)]),
    Locations.ForestMainEnemy_NearCranky: Location(Levels.FungiForest, "Forest Enemy: Near Cranky", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xdb)]),
    Locations.ForestMainEnemy_NearPinkTunnelGM: Location(Levels.FungiForest, "Forest Enemy: Near Pink Tunnel Giant Mushroom", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xdc)]),
    Locations.ForestMainEnemy_GMRearTag: Location(Levels.FungiForest, "Forest Enemy: Giant Mushroom Rear Tag", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xdd)]),
    Locations.ForestMainEnemy_NearFacePuzzle: Location(Levels.FungiForest, "Forest Enemy: Near Face Puzzle", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xde)]),
    Locations.ForestMainEnemy_NearCrown: Location(Levels.FungiForest, "Forest Enemy: Near Crown", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xdf)]),
    Locations.ForestMainEnemy_NearHighWarp5: Location(Levels.FungiForest, "Forest Enemy: Near High Warp 5", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe0)]),
    Locations.ForestMainEnemy_TopOfMushroom: Location(Levels.FungiForest, "Forest Enemy: Top Of Mushroom", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe1)]),
    Locations.ForestMainEnemy_NearAppleDropoff: Location(Levels.FungiForest, "Forest Enemy: Near Apple Dropoff", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe2)]),
    Locations.ForestMainEnemy_NearDKPortal: Location(Levels.FungiForest, "Forest Enemy: Near DKPortal", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe3)]),
    Locations.ForestMainEnemy_NearWellTag: Location(Levels.FungiForest, "Forest Enemy: Near Well Tag", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe4)]),
    Locations.ForestMainEnemy_YellowTunnel0: Location(Levels.FungiForest, "Forest Enemy: Yellow Tunnel (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe5)]),
    Locations.ForestMainEnemy_YellowTunnel1: Location(Levels.FungiForest, "Forest Enemy: Yellow Tunnel (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe6)]),
    Locations.ForestMainEnemy_YellowTunnel2: Location(Levels.FungiForest, "Forest Enemy: Yellow Tunnel (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe7)]),
    Locations.ForestMainEnemy_YellowTunnel3: Location(Levels.FungiForest, "Forest Enemy: Yellow Tunnel (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe8)]),
    Locations.ForestMainEnemy_NearSnide: Location(Levels.FungiForest, "Forest Enemy: Near Snide", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xe9)]),
    Locations.ForestMainEnemy_NearIsoCoin: Location(Levels.FungiForest, "Forest Enemy: Near the hidden Rainbow Coin", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xea)]),
    Locations.ForestMainEnemy_NearBBlast: Location(Levels.FungiForest, "Forest Enemy: Near BBlast", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xeb)]),
    Locations.ForestMainEnemy_NearDarkAttic: Location(Levels.FungiForest, "Forest Enemy: Near Dark Attic", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xec)]),
    Locations.ForestMainEnemy_NearWellExit: Location(Levels.FungiForest, "Forest Enemy: Near Well Exit", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xed)]),
    Locations.ForestMainEnemy_NearBlueTunnel: Location(Levels.FungiForest, "Forest Enemy: Near Blue Tunnel", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xee)]),
    Locations.ForestMainEnemy_Thornvine0: Location(Levels.FungiForest, "Forest Enemy: Thornvine (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xef)]),
    Locations.ForestMainEnemy_Thornvine1: Location(Levels.FungiForest, "Forest Enemy: Thornvine (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xf0)]),
    Locations.ForestMainEnemy_Thornvine2: Location(Levels.FungiForest, "Forest Enemy: Thornvine (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xf1)]),
    Locations.ForestMainEnemy_ThornvineEntrance: Location(Levels.FungiForest, "Forest Enemy: Thornvine Entrance", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.FungiForest, -1, DROPSANITY_FLAG_START + 0xf2)]),
    Locations.ForestAnthillEnemy_Gauntlet0: Location(Levels.FungiForest, "Forest Anthill Enemy: Gauntlet (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestAnthill, -1, DROPSANITY_FLAG_START + 0xf3)]),
    Locations.ForestAnthillEnemy_Gauntlet1: Location(Levels.FungiForest, "Forest Anthill Enemy: Gauntlet (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestAnthill, -1, DROPSANITY_FLAG_START + 0xf4)]),
    Locations.ForestAnthillEnemy_Gauntlet2: Location(Levels.FungiForest, "Forest Anthill Enemy: Gauntlet (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestAnthill, -1, DROPSANITY_FLAG_START + 0xf5)]),
    Locations.ForestAnthillEnemy_Gauntlet3: Location(Levels.FungiForest, "Forest Anthill Enemy: Gauntlet (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestAnthill, -1, DROPSANITY_FLAG_START + 0xf6)]),
    Locations.ForestWinchEnemy_Enemy: Location(Levels.FungiForest, "Forest Winch Room Enemy: Enemy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestWinchRoom, -1, DROPSANITY_FLAG_START + 0xf7)]),
    Locations.ForestThornBarnEnemy_Enemy: Location(Levels.FungiForest, "Forest Thornvine Barn Enemy: Enemy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestThornvineBarn, -1, DROPSANITY_FLAG_START + 0xf8)]),
    Locations.ForestMillFrontEnemy_Enemy: Location(Levels.FungiForest, "Forest Mill Front Enemy: Enemy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestMillFront, -1, DROPSANITY_FLAG_START + 0xf9)]),
    Locations.ForestMillRearEnemy_Enemy: Location(Levels.FungiForest, "Forest Mill Back Enemy: Enemy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestMillBack, -1, DROPSANITY_FLAG_START + 0xfa)]),
    Locations.ForestGMEnemy_AboveNightDoor: Location(Levels.FungiForest, "Forest Giant Mushroom Enemy: Above Night Door", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestGiantMushroom, -1, DROPSANITY_FLAG_START + 0xfb)]),
    Locations.ForestGMEnemy_Path0: Location(Levels.FungiForest, "Forest Giant Mushroom Enemy: Path (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestGiantMushroom, -1, DROPSANITY_FLAG_START + 0xfc)]),
    Locations.ForestGMEnemy_Path1: Location(Levels.FungiForest, "Forest Giant Mushroom Enemy: Path (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestGiantMushroom, -1, DROPSANITY_FLAG_START + 0xfd)]),
    # Locations.ForestLankyAtticEnemy_Gauntlet0: Location(Levels.FungiForest, "Forest Mill Attic Enemy: Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestMillAttic, -1, DROPSANITY_FLAG_START + 0xfe)]),
    # Locations.ForestLankyAtticEnemy_Gauntlet1: Location(Levels.FungiForest, "Forest Mill Attic Enemy: Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestMillAttic, -1, DROPSANITY_FLAG_START + 0xff)]),
    # Locations.ForestLankyAtticEnemy_Gauntlet2: Location(Levels.FungiForest, "Forest Mill Attic Enemy: Gauntlet2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestMillAttic, -1, DROPSANITY_FLAG_START + 0x100)]),
    Locations.ForestLeapEnemy_Enemy0: Location(Levels.FungiForest, "Forest Lanky Zingers Room Enemy: Enemy (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestLankyZingersRoom, -1, DROPSANITY_FLAG_START + 0x101)]),
    Locations.ForestLeapEnemy_Enemy1: Location(Levels.FungiForest, "Forest Lanky Zingers Room Enemy: Enemy (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestLankyZingersRoom, -1, DROPSANITY_FLAG_START + 0x102)]),
    Locations.ForestFacePuzzleEnemy_Enemy: Location(Levels.FungiForest, "Forest Chunky Face Room Enemy: Enemy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestChunkyFaceRoom, -1, DROPSANITY_FLAG_START + 0x103)]),
    # Locations.ForestSpiderEnemy_Gauntlet0: Location(Levels.FungiForest, "Forest Spider Enemy: Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestSpider, -1, DROPSANITY_FLAG_START + 0x104)]),
    # Locations.ForestSpiderEnemy_Gauntlet1: Location(Levels.FungiForest, "Forest Spider Enemy: Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestSpider, -1, DROPSANITY_FLAG_START + 0x105)]),
    # Locations.ForestSpiderEnemy_Gauntlet2: Location(Levels.FungiForest, "Forest Spider Enemy: Gauntlet2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.ForestSpider, -1, DROPSANITY_FLAG_START + 0x106)]),
    Locations.CavesMainEnemy_Start: Location(Levels.CrystalCaves, "Caves Enemy: Start", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, DROPSANITY_FLAG_START + 0x107)]),
    Locations.CavesMainEnemy_NearIceCastle: Location(Levels.CrystalCaves, "Caves Enemy: Near Ice Castle", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, DROPSANITY_FLAG_START + 0x108)]),
    Locations.CavesMainEnemy_Outside5DC: Location(Levels.CrystalCaves, "Caves Enemy: Outside 5DC", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, DROPSANITY_FLAG_START + 0x109)]),
    Locations.CavesMainEnemy_1DCWaterfall: Location(Levels.CrystalCaves, "Caves Enemy: 1DC Waterfall", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, DROPSANITY_FLAG_START + 0x10a)]),
    Locations.CavesMainEnemy_NearFunky: Location(Levels.CrystalCaves, "Caves Enemy: Near Funky", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, DROPSANITY_FLAG_START + 0x10b)]),
    Locations.CavesMainEnemy_NearSnide: Location(Levels.CrystalCaves, "Caves Enemy: Near Snide", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, DROPSANITY_FLAG_START + 0x10c)]),
    Locations.CavesMainEnemy_NearBonusRoom: Location(Levels.CrystalCaves, "Caves Enemy: Near Bonus Room", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, DROPSANITY_FLAG_START + 0x10d)]),
    Locations.CavesMainEnemy_1DCHeadphones: Location(Levels.CrystalCaves, "Caves Enemy: 1DC Headphones", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, DROPSANITY_FLAG_START + 0x10e)]),
    # Locations.CavesMainEnemy_GiantKosha: Location(Levels.CrystalCaves, "Caves Enemy: Giant Kosha", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CrystalCaves, -1, DROPSANITY_FLAG_START + 0x10f)]),
    Locations.Caves5DIDKEnemy_Right: Location(Levels.CrystalCaves, "Caves Donkey Igloo Enemy: Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDonkeyIgloo, -1, DROPSANITY_FLAG_START + 0x110)]),
    Locations.Caves5DIDKEnemy_Left: Location(Levels.CrystalCaves, "Caves Donkey Igloo Enemy: Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDonkeyIgloo, -1, DROPSANITY_FLAG_START + 0x111)]),
    # Locations.Caves5DILankyEnemy_First0: Location(Levels.CrystalCaves, "Caves Lanky Igloo Enemy: First (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesLankyIgloo, -1, DROPSANITY_FLAG_START + 0x112)]),
    # Locations.Caves5DILankyEnemy_First1: Location(Levels.CrystalCaves, "Caves Lanky Igloo Enemy: First (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesLankyIgloo, -1, DROPSANITY_FLAG_START + 0x113)]),
    # Locations.Caves5DILankyEnemy_Second0: Location(Levels.CrystalCaves, "Caves Lanky Igloo Enemy: Second (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesLankyIgloo, -1, DROPSANITY_FLAG_START + 0x114)]),
    # Locations.Caves5DILankyEnemy_Second1: Location(Levels.CrystalCaves, "Caves Lanky Igloo Enemy: Second (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesLankyIgloo, -1, DROPSANITY_FLAG_START + 0x115)]),
    # Locations.Caves5DILankyEnemy_Second2: Location(Levels.CrystalCaves, "Caves Lanky Igloo Enemy: Second (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesLankyIgloo, -1, DROPSANITY_FLAG_START + 0x116)]),
    Locations.Caves5DITinyEnemy_BigEnemy: Location(Levels.CrystalCaves, "Caves Tiny Igloo Enemy: Big Enemy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesTinyIgloo, -1, DROPSANITY_FLAG_START + 0x117)]),
    # Locations.Caves5DIChunkyEnemy_Gauntlet00: Location(Levels.CrystalCaves, "Caves Chunky Igloo Enemy: Gauntlet00", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesChunkyIgloo, -1, DROPSANITY_FLAG_START + 0x118)]),
    # Locations.Caves5DIChunkyEnemy_Gauntlet01: Location(Levels.CrystalCaves, "Caves Chunky Igloo Enemy: Gauntlet01", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesChunkyIgloo, -1, DROPSANITY_FLAG_START + 0x119)]),
    # Locations.Caves5DIChunkyEnemy_Gauntlet02: Location(Levels.CrystalCaves, "Caves Chunky Igloo Enemy: Gauntlet02", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesChunkyIgloo, -1, DROPSANITY_FLAG_START + 0x11a)]),
    # Locations.Caves5DIChunkyEnemy_Gauntlet03: Location(Levels.CrystalCaves, "Caves Chunky Igloo Enemy: Gauntlet03", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesChunkyIgloo, -1, DROPSANITY_FLAG_START + 0x11b)]),
    # Locations.Caves5DIChunkyEnemy_Gauntlet04: Location(Levels.CrystalCaves, "Caves Chunky Igloo Enemy: Gauntlet04", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesChunkyIgloo, -1, DROPSANITY_FLAG_START + 0x11c)]),
    Locations.Caves1DCEnemy_Near: Location(Levels.CrystalCaves, "Caves Lanky Cabin Enemy: Near", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesLankyCabin, -1, DROPSANITY_FLAG_START + 0x11d)]),
    # Locations.Caves1DCEnemy_Far: Location(Levels.CrystalCaves, "Caves Lanky Cabin Enemy: Far", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesLankyCabin, -1, DROPSANITY_FLAG_START + 0x11e)]),
    # Locations.Caves5DCDKEnemy_Gauntlet0: Location(Levels.CrystalCaves, "Caves Donkey Cabin Enemy: Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDonkeyCabin, -1, DROPSANITY_FLAG_START + 0x11f)]),
    # Locations.Caves5DCDKEnemy_Gauntlet1: Location(Levels.CrystalCaves, "Caves Donkey Cabin Enemy: Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDonkeyCabin, -1, DROPSANITY_FLAG_START + 0x120)]),
    # Locations.Caves5DCDKEnemy_Gauntlet2: Location(Levels.CrystalCaves, "Caves Donkey Cabin Enemy: Gauntlet2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDonkeyCabin, -1, DROPSANITY_FLAG_START + 0x121)]),
    # Locations.Caves5DCDKEnemy_Gauntlet3: Location(Levels.CrystalCaves, "Caves Donkey Cabin Enemy: Gauntlet3", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDonkeyCabin, -1, DROPSANITY_FLAG_START + 0x122)]),
    # Locations.Caves5DCDKEnemy_Gauntlet4: Location(Levels.CrystalCaves, "Caves Donkey Cabin Enemy: Gauntlet4", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDonkeyCabin, -1, DROPSANITY_FLAG_START + 0x123)]),
    # Locations.Caves5DCDKEnemy_Gauntlet5: Location(Levels.CrystalCaves, "Caves Donkey Cabin Enemy: Gauntlet5", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDonkeyCabin, -1, DROPSANITY_FLAG_START + 0x124)]),
    # Locations.Caves5DCDiddyLowEnemy_CloseRight: Location(Levels.CrystalCaves, "Caves Diddy Lower Cabin Enemy: Close Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyLowerCabin, -1, DROPSANITY_FLAG_START + 0x125)]),
    # Locations.Caves5DCDiddyLowEnemy_FarRight: Location(Levels.CrystalCaves, "Caves Diddy Lower Cabin Enemy: Far Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyLowerCabin, -1, DROPSANITY_FLAG_START + 0x126)]),
    # Locations.Caves5DCDiddyLowEnemy_CloseLeft: Location(Levels.CrystalCaves, "Caves Diddy Lower Cabin Enemy: Close Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyLowerCabin, -1, DROPSANITY_FLAG_START + 0x127)]),
    # Locations.Caves5DCDiddyLowEnemy_FarLeft: Location(Levels.CrystalCaves, "Caves Diddy Lower Cabin Enemy: Far Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyLowerCabin, -1, DROPSANITY_FLAG_START + 0x128)]),
    # Locations.Caves5DCDiddyLowEnemy_Center0: Location(Levels.CrystalCaves, "Caves Diddy Lower Cabin Enemy: Center0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyLowerCabin, -1, DROPSANITY_FLAG_START + 0x129)]),
    # Locations.Caves5DCDiddyLowEnemy_Center1: Location(Levels.CrystalCaves, "Caves Diddy Lower Cabin Enemy: Center1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyLowerCabin, -1, DROPSANITY_FLAG_START + 0x12a)]),
    # Locations.Caves5DCDiddyLowEnemy_Center2: Location(Levels.CrystalCaves, "Caves Diddy Lower Cabin Enemy: Center2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyLowerCabin, -1, DROPSANITY_FLAG_START + 0x12b)]),
    # Locations.Caves5DCDiddyLowEnemy_Center3: Location(Levels.CrystalCaves, "Caves Diddy Lower Cabin Enemy: Center3", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyLowerCabin, -1, DROPSANITY_FLAG_START + 0x12c)]),
    # Locations.Caves5DCDiddyUpperEnemy_Enemy0: Location(Levels.CrystalCaves, "Caves Diddy Upper Cabin Enemy: Enemy (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyUpperCabin, -1, DROPSANITY_FLAG_START + 0x12d)]),
    # Locations.Caves5DCDiddyUpperEnemy_Enemy1: Location(Levels.CrystalCaves, "Caves Diddy Upper Cabin Enemy: Enemy (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesDiddyUpperCabin, -1, DROPSANITY_FLAG_START + 0x12e)]),
    # Locations.Caves5DCTinyEnemy_Gauntlet0: Location(Levels.CrystalCaves, "Caves Tiny Cabin Enemy: Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesTinyCabin, -1, DROPSANITY_FLAG_START + 0x12f)]),
    # Locations.Caves5DCTinyEnemy_Gauntlet1: Location(Levels.CrystalCaves, "Caves Tiny Cabin Enemy: Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesTinyCabin, -1, DROPSANITY_FLAG_START + 0x130)]),
    # Locations.Caves5DCTinyEnemy_Gauntlet2: Location(Levels.CrystalCaves, "Caves Tiny Cabin Enemy: Gauntlet2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesTinyCabin, -1, DROPSANITY_FLAG_START + 0x131)]),
    # Locations.Caves5DCTinyEnemy_Gauntlet3: Location(Levels.CrystalCaves, "Caves Tiny Cabin Enemy: Gauntlet3", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesTinyCabin, -1, DROPSANITY_FLAG_START + 0x132)]),
    # Locations.Caves5DCTinyEnemy_Gauntlet4: Location(Levels.CrystalCaves, "Caves Tiny Cabin Enemy: Gauntlet4", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CavesTinyCabin, -1, DROPSANITY_FLAG_START + 0x133)]),
    Locations.CastleMainEnemy_NearBridge0: Location(Levels.CreepyCastle, "Castle Enemy: Near Bridge (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x134)]),
    Locations.CastleMainEnemy_NearBridge1: Location(Levels.CreepyCastle, "Castle Enemy: Near Bridge (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x135)]),
    Locations.CastleMainEnemy_WoodenExtrusion0: Location(Levels.CreepyCastle, "Castle Enemy: Wooden Extrusion (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x136)]),
    Locations.CastleMainEnemy_WoodenExtrusion1: Location(Levels.CreepyCastle, "Castle Enemy: Wooden Extrusion (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x137)]),
    Locations.CastleMainEnemy_NearShed: Location(Levels.CreepyCastle, "Castle Enemy: Near Shed", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x138)]),
    Locations.CastleMainEnemy_NearLibrary: Location(Levels.CreepyCastle, "Castle Enemy: Near Library", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x139)]),
    Locations.CastleMainEnemy_NearTower: Location(Levels.CreepyCastle, "Castle Enemy: Near Tower", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x13a)]),
    Locations.CastleMainEnemy_MuseumSteps: Location(Levels.CreepyCastle, "Castle Enemy: Museum Steps", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x13b)]),
    Locations.CastleMainEnemy_NearLowCave: Location(Levels.CreepyCastle, "Castle Enemy: Near Low Cave", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x13c)]),
    Locations.CastleMainEnemy_PathToLowKasplat: Location(Levels.CreepyCastle, "Castle Enemy: Path To Low Kasplat", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x13d)]),
    Locations.CastleMainEnemy_LowTnS: Location(Levels.CreepyCastle, "Castle Enemy: Low TnS", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x13e)]),
    Locations.CastleMainEnemy_PathToDungeon: Location(Levels.CreepyCastle, "Castle Enemy: Path To Dungeon", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x13f)]),
    Locations.CastleMainEnemy_NearHeadphones: Location(Levels.CreepyCastle, "Castle Enemy: Near Headphones", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastle, -1, DROPSANITY_FLAG_START + 0x140)]),
    Locations.CastleLobbyEnemy_Left: Location(Levels.DKIsles, "Isles Castle Lobby Enemy: Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastleLobby, -1, DROPSANITY_FLAG_START + 0x141)]),
    Locations.CastleLobbyEnemy_FarRight: Location(Levels.DKIsles, "Isles Castle Lobby Enemy: Far Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastleLobby, -1, DROPSANITY_FLAG_START + 0x142)]),
    Locations.CastleLobbyEnemy_NearRight: Location(Levels.DKIsles, "Isles Castle Lobby Enemy: Near Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CreepyCastleLobby, -1, DROPSANITY_FLAG_START + 0x143)]),
    # Locations.CastleBallroomEnemy_Board00: Location(Levels.CreepyCastle, "Castle Ballroom Enemy: Board00", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBallroom, -1, DROPSANITY_FLAG_START + 0x144)]),
    # Locations.CastleBallroomEnemy_Board01: Location(Levels.CreepyCastle, "Castle Ballroom Enemy: Board01", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBallroom, -1, DROPSANITY_FLAG_START + 0x145)]),
    # Locations.CastleBallroomEnemy_Board02: Location(Levels.CreepyCastle, "Castle Ballroom Enemy: Board02", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBallroom, -1, DROPSANITY_FLAG_START + 0x146)]),
    # Locations.CastleBallroomEnemy_Board03: Location(Levels.CreepyCastle, "Castle Ballroom Enemy: Board03", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBallroom, -1, DROPSANITY_FLAG_START + 0x147)]),
    # Locations.CastleBallroomEnemy_Board04: Location(Levels.CreepyCastle, "Castle Ballroom Enemy: Board04", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBallroom, -1, DROPSANITY_FLAG_START + 0x148)]),
    Locations.CastleBallroomEnemy_Start: Location(Levels.CreepyCastle, "Castle Ballroom Enemy: Start", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBallroom, -1, DROPSANITY_FLAG_START + 0x149)]),
    Locations.CastleDungeonEnemy_FaceRoom: Location(Levels.CreepyCastle, "Castle Dungeon Enemy: Face Room", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleDungeon, -1, DROPSANITY_FLAG_START + 0x14a)]),
    Locations.CastleDungeonEnemy_ChairRoom: Location(Levels.CreepyCastle, "Castle Dungeon Enemy: Chair Room", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleDungeon, -1, DROPSANITY_FLAG_START + 0x14b)]),
    Locations.CastleDungeonEnemy_OutsideLankyRoom: Location(Levels.CreepyCastle, "Castle Dungeon Enemy: Outside Lanky Room", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleDungeon, -1, DROPSANITY_FLAG_START + 0x14c)]),
    # Locations.CastleShedEnemy_Gauntlet00: Location(Levels.CreepyCastle, "Castle Shed Enemy: Gauntlet00", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleShed, -1, DROPSANITY_FLAG_START + 0x14d)]),
    # Locations.CastleShedEnemy_Gauntlet01: Location(Levels.CreepyCastle, "Castle Shed Enemy: Gauntlet01", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleShed, -1, DROPSANITY_FLAG_START + 0x14e)]),
    # Locations.CastleShedEnemy_Gauntlet02: Location(Levels.CreepyCastle, "Castle Shed Enemy: Gauntlet02", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleShed, -1, DROPSANITY_FLAG_START + 0x14f)]),
    # Locations.CastleShedEnemy_Gauntlet03: Location(Levels.CreepyCastle, "Castle Shed Enemy: Gauntlet03", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleShed, -1, DROPSANITY_FLAG_START + 0x150)]),
    # Locations.CastleShedEnemy_Gauntlet04: Location(Levels.CreepyCastle, "Castle Shed Enemy: Gauntlet04", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleShed, -1, DROPSANITY_FLAG_START + 0x151)]),
    Locations.CastleLowCaveEnemy_NearCrypt: Location(Levels.CreepyCastle, "Castle Lower Cave Enemy: Near Crypt", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLowerCave, -1, DROPSANITY_FLAG_START + 0x152)]),
    Locations.CastleLowCaveEnemy_StairRight: Location(Levels.CreepyCastle, "Castle Lower Cave Enemy: Stair Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLowerCave, -1, DROPSANITY_FLAG_START + 0x153)]),
    Locations.CastleLowCaveEnemy_StairLeft: Location(Levels.CreepyCastle, "Castle Lower Cave Enemy: Stair Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLowerCave, -1, DROPSANITY_FLAG_START + 0x154)]),
    Locations.CastleLowCaveEnemy_NearMausoleum: Location(Levels.CreepyCastle, "Castle Lower Cave Enemy: Near Mausoleum", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLowerCave, -1, DROPSANITY_FLAG_START + 0x155)]),
    Locations.CastleLowCaveEnemy_NearFunky: Location(Levels.CreepyCastle, "Castle Lower Cave Enemy: Near Funky", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLowerCave, -1, DROPSANITY_FLAG_START + 0x156)]),
    Locations.CastleLowCaveEnemy_NearTag: Location(Levels.CreepyCastle, "Castle Lower Cave Enemy: Near Tag", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLowerCave, -1, DROPSANITY_FLAG_START + 0x157)]),
    Locations.CastleCryptEnemy_DiddyCoffin0: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Diddy Coffin (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x158)]),
    Locations.CastleCryptEnemy_DiddyCoffin1: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Diddy Coffin (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x159)]),
    Locations.CastleCryptEnemy_DiddyCoffin2: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Diddy Coffin (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x15a)]),
    Locations.CastleCryptEnemy_DiddyCoffin3: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Diddy Coffin (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x15b)]),
    Locations.CastleCryptEnemy_ChunkyCoffin0: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Chunky Coffin (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x15c)]),
    Locations.CastleCryptEnemy_ChunkyCoffin1: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Chunky Coffin (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x15d)]),
    Locations.CastleCryptEnemy_ChunkyCoffin2: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Chunky Coffin (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x15e)]),
    Locations.CastleCryptEnemy_ChunkyCoffin3: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Chunky Coffin (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x15f)]),
    Locations.CastleCryptEnemy_MinecartEntry: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Minecart Entry", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x160)]),
    Locations.CastleCryptEnemy_Fork: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Fork", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x161)]),
    Locations.CastleCryptEnemy_NearDiddy: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Near Diddy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x162)]),
    Locations.CastleCryptEnemy_NearChunky: Location(Levels.CreepyCastle, "Castle Crypt Enemy: Near Chunky", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleCrypt, -1, DROPSANITY_FLAG_START + 0x163)]),
    Locations.CastleMausoleumEnemy_TinyPath: Location(Levels.CreepyCastle, "Castle Mausoleum Enemy: Tiny Path", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleMausoleum, -1, DROPSANITY_FLAG_START + 0x164)]),
    Locations.CastleMausoleumEnemy_LankyPath0: Location(Levels.CreepyCastle, "Castle Mausoleum Enemy: Lanky Path (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleMausoleum, -1, DROPSANITY_FLAG_START + 0x165)]),
    Locations.CastleMausoleumEnemy_LankyPath1: Location(Levels.CreepyCastle, "Castle Mausoleum Enemy: Lanky Path (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleMausoleum, -1, DROPSANITY_FLAG_START + 0x166)]),
    Locations.CastleUpperCaveEnemy_NearDungeon: Location(Levels.CreepyCastle, "Castle Upper Cave Enemy: Near Dungeon", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleUpperCave, -1, DROPSANITY_FLAG_START + 0x167)]),
    # Locations.CastleUpperCaveEnemy_Pit: Location(Levels.CreepyCastle, "Castle Upper Cave Enemy: Pit", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleUpperCave, -1, DROPSANITY_FLAG_START + 0x168)]),
    Locations.CastleUpperCaveEnemy_NearPit: Location(Levels.CreepyCastle, "Castle Upper Cave Enemy: Near Pit", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleUpperCave, -1, DROPSANITY_FLAG_START + 0x169)]),
    Locations.CastleUpperCaveEnemy_NearEntrance: Location(Levels.CreepyCastle, "Castle Upper Cave Enemy: Near Entrance", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleUpperCave, -1, DROPSANITY_FLAG_START + 0x16a)]),
    # Locations.CastleKKOEnemy_CenterEnemy: Location(Levels.CreepyCastle, "Castle Boss Enemy: Center Enemy", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBoss, -1, DROPSANITY_FLAG_START + 0x16b)]),
    # Locations.CastleKKOEnemy_WaterEnemy00: Location(Levels.CreepyCastle, "Castle Boss Enemy: Water Enemy00", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBoss, -1, DROPSANITY_FLAG_START + 0x16c)]),
    # Locations.CastleKKOEnemy_WaterEnemy01: Location(Levels.CreepyCastle, "Castle Boss Enemy: Water Enemy01", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBoss, -1, DROPSANITY_FLAG_START + 0x16d)]),
    # Locations.CastleKKOEnemy_WaterEnemy02: Location(Levels.CreepyCastle, "Castle Boss Enemy: Water Enemy02", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBoss, -1, DROPSANITY_FLAG_START + 0x16e)]),
    # Locations.CastleKKOEnemy_WaterEnemy03: Location(Levels.CreepyCastle, "Castle Boss Enemy: Water Enemy03", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleBoss, -1, DROPSANITY_FLAG_START + 0x16f)]),
    # Locations.CastleLibraryEnemy_Gauntlet00: Location(Levels.CreepyCastle, "Castle Library Enemy: Gauntlet00", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x170)]),
    # Locations.CastleLibraryEnemy_Gauntlet01: Location(Levels.CreepyCastle, "Castle Library Enemy: Gauntlet01", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x171)]),
    # Locations.CastleLibraryEnemy_Gauntlet02: Location(Levels.CreepyCastle, "Castle Library Enemy: Gauntlet02", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x172)]),
    # Locations.CastleLibraryEnemy_Gauntlet03: Location(Levels.CreepyCastle, "Castle Library Enemy: Gauntlet03", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x173)]),
    # Locations.CastleLibraryEnemy_Corridor00: Location(Levels.CreepyCastle, "Castle Library Enemy: Corridor00", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x174)]),
    # Locations.CastleLibraryEnemy_Corridor01: Location(Levels.CreepyCastle, "Castle Library Enemy: Corridor01", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x175)]),
    # Locations.CastleLibraryEnemy_Corridor02: Location(Levels.CreepyCastle, "Castle Library Enemy: Corridor02", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x176)]),
    # Locations.CastleLibraryEnemy_Corridor03: Location(Levels.CreepyCastle, "Castle Library Enemy: Corridor03", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x177)]),
    # Locations.CastleLibraryEnemy_Corridor04: Location(Levels.CreepyCastle, "Castle Library Enemy: Corridor04", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x178)]),
    # Locations.CastleLibraryEnemy_Corridor05: Location(Levels.CreepyCastle, "Castle Library Enemy: Corridor05", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x179)]),
    Locations.CastleLibraryEnemy_ForkLeft0: Location(Levels.CreepyCastle, "Castle Library Enemy: Fork Left (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x17a)]),
    Locations.CastleLibraryEnemy_ForkLeft1: Location(Levels.CreepyCastle, "Castle Library Enemy: Fork Left (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x17b)]),
    Locations.CastleLibraryEnemy_ForkCenter: Location(Levels.CreepyCastle, "Castle Library Enemy: Fork Center", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x17c)]),
    Locations.CastleLibraryEnemy_ForkRight: Location(Levels.CreepyCastle, "Castle Library Enemy: Fork Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleLibrary, -1, DROPSANITY_FLAG_START + 0x17d)]),
    Locations.CastleMuseumEnemy_MainFloor0: Location(Levels.CreepyCastle, "Castle Museum Enemy: Main Floor (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleMuseum, -1, DROPSANITY_FLAG_START + 0x17e)]),
    Locations.CastleMuseumEnemy_MainFloor1: Location(Levels.CreepyCastle, "Castle Museum Enemy: Main Floor (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleMuseum, -1, DROPSANITY_FLAG_START + 0x17f)]),
    Locations.CastleMuseumEnemy_MainFloor2: Location(Levels.CreepyCastle, "Castle Museum Enemy: Main Floor (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleMuseum, -1, DROPSANITY_FLAG_START + 0x180)]),
    Locations.CastleMuseumEnemy_MainFloor3: Location(Levels.CreepyCastle, "Castle Museum Enemy: Main Floor (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleMuseum, -1, DROPSANITY_FLAG_START + 0x181)]),
    Locations.CastleMuseumEnemy_Start: Location(Levels.CreepyCastle, "Castle Museum Enemy: Start", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleMuseum, -1, DROPSANITY_FLAG_START + 0x182)]),
    # Locations.CastleTowerEnemy_Gauntlet0: Location(Levels.CreepyCastle, "Castle Tower Enemy: Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTower, -1, DROPSANITY_FLAG_START + 0x183)]),
    # Locations.CastleTowerEnemy_Gauntlet1: Location(Levels.CreepyCastle, "Castle Tower Enemy: Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTower, -1, DROPSANITY_FLAG_START + 0x184)]),
    # Locations.CastleTowerEnemy_Gauntlet2: Location(Levels.CreepyCastle, "Castle Tower Enemy: Gauntlet2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTower, -1, DROPSANITY_FLAG_START + 0x185)]),
    # Locations.CastleTowerEnemy_Gauntlet3: Location(Levels.CreepyCastle, "Castle Tower Enemy: Gauntlet3", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTower, -1, DROPSANITY_FLAG_START + 0x186)]),
    # Locations.CastleTowerEnemy_Gauntlet4: Location(Levels.CreepyCastle, "Castle Tower Enemy: Gauntlet4", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTower, -1, DROPSANITY_FLAG_START + 0x187)]),
    # Locations.CastleTrashEnemy_Gauntlet0: Location(Levels.CreepyCastle, "Castle Trash Can Enemy: Gauntlet0", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTrashCan, -1, DROPSANITY_FLAG_START + 0x188)]),
    # Locations.CastleTrashEnemy_Gauntlet1: Location(Levels.CreepyCastle, "Castle Trash Can Enemy: Gauntlet1", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTrashCan, -1, DROPSANITY_FLAG_START + 0x189)]),
    # Locations.CastleTrashEnemy_Gauntlet2: Location(Levels.CreepyCastle, "Castle Trash Can Enemy: Gauntlet2", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTrashCan, -1, DROPSANITY_FLAG_START + 0x18a)]),
    # Locations.CastleTrashEnemy_Gauntlet3: Location(Levels.CreepyCastle, "Castle Trash Can Enemy: Gauntlet3", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTrashCan, -1, DROPSANITY_FLAG_START + 0x18b)]),
    # Locations.CastleTrashEnemy_Gauntlet4: Location(Levels.CreepyCastle, "Castle Trash Can Enemy: Gauntlet4", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTrashCan, -1, DROPSANITY_FLAG_START + 0x18c)]),
    Locations.CastleTreeEnemy_StartRoom0: Location(Levels.CreepyCastle, "Castle Tree Enemy: Start Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTree, -1, DROPSANITY_FLAG_START + 0x18d)]),
    Locations.CastleTreeEnemy_StartRoom1: Location(Levels.CreepyCastle, "Castle Tree Enemy: Start Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.CastleTree, -1, DROPSANITY_FLAG_START + 0x18e)]),
    Locations.HelmMainEnemy_Start0: Location(Levels.HideoutHelm, "Helm Enemy: Start (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x18f)]),
    Locations.HelmMainEnemy_Start1: Location(Levels.HideoutHelm, "Helm Enemy: Start (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x190)]),
    Locations.HelmMainEnemy_Hill: Location(Levels.HideoutHelm, "Helm Enemy: Hill", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x191)]),
    Locations.HelmMainEnemy_SwitchRoom0: Location(Levels.HideoutHelm, "Helm Enemy: Switch Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x192)]),
    Locations.HelmMainEnemy_SwitchRoom1: Location(Levels.HideoutHelm, "Helm Enemy: Switch Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x193)]),
    Locations.HelmMainEnemy_MiniRoom0: Location(Levels.HideoutHelm, "Helm Enemy: Mini Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x194)]),
    Locations.HelmMainEnemy_MiniRoom1: Location(Levels.HideoutHelm, "Helm Enemy: Mini Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x195)]),
    Locations.HelmMainEnemy_MiniRoom2: Location(Levels.HideoutHelm, "Helm Enemy: Mini Room (2)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x196)]),
    Locations.HelmMainEnemy_MiniRoom3: Location(Levels.HideoutHelm, "Helm Enemy: Mini Room (3)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x197)]),
    Locations.HelmMainEnemy_DKRoom: Location(Levels.HideoutHelm, "Helm Enemy: DKRoom", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x198)]),
    Locations.HelmMainEnemy_ChunkyRoom0: Location(Levels.HideoutHelm, "Helm Enemy: Chunky Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x199)]),
    Locations.HelmMainEnemy_ChunkyRoom1: Location(Levels.HideoutHelm, "Helm Enemy: Chunky Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x19a)]),
    Locations.HelmMainEnemy_TinyRoom: Location(Levels.HideoutHelm, "Helm Enemy: Tiny Room", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x19b)]),
    Locations.HelmMainEnemy_LankyRoom0: Location(Levels.HideoutHelm, "Helm Enemy: Lanky Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x19c)]),
    Locations.HelmMainEnemy_LankyRoom1: Location(Levels.HideoutHelm, "Helm Enemy: Lanky Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x19d)]),
    Locations.HelmMainEnemy_DiddyRoom0: Location(Levels.HideoutHelm, "Helm Enemy: Diddy Room (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x19e)]),
    Locations.HelmMainEnemy_DiddyRoom1: Location(Levels.HideoutHelm, "Helm Enemy: Diddy Room (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x19f)]),
    Locations.HelmMainEnemy_NavRight: Location(Levels.HideoutHelm, "Helm Enemy: Nav Right", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x1a0)]),
    Locations.HelmMainEnemy_NavLeft: Location(Levels.HideoutHelm, "Helm Enemy: Nav Left", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.HideoutHelm, -1, DROPSANITY_FLAG_START + 0x1a1)]),
    Locations.IslesMainEnemy_PineappleCage0: Location(Levels.DKIsles, "Isles Enemy: Pineapple Cage (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1a2)]),
    Locations.IslesMainEnemy_FungiCannon0: Location(Levels.DKIsles, "Isles Enemy: Fungi Cannon (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1a3)]),
    Locations.IslesMainEnemy_JapesEntrance: Location(Levels.DKIsles, "Isles Enemy: Japes Entrance", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1a4)]),
    Locations.IslesMainEnemy_MonkeyportPad: Location(Levels.DKIsles, "Isles Enemy: Monkeyport Pad", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1a5)]),
    Locations.IslesMainEnemy_UpperFactoryPath: Location(Levels.DKIsles, "Isles Enemy: Upper Factory Path", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1a6)]),
    Locations.IslesMainEnemy_NearAztec: Location(Levels.DKIsles, "Isles Enemy: Near Aztec", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1a7)]),
    Locations.IslesMainEnemy_FungiCannon1: Location(Levels.DKIsles, "Isles Enemy: Fungi Cannon (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1a8)]),
    Locations.IslesMainEnemy_PineappleCage1: Location(Levels.DKIsles, "Isles Enemy: Pineapple Cage (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1a9)]),
    Locations.IslesMainEnemy_LowerFactoryPath0: Location(Levels.DKIsles, "Isles Enemy: Lower Factory Path (0)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1aa)]),
    Locations.IslesMainEnemy_LowerFactoryPath1: Location(Levels.DKIsles, "Isles Enemy: Lower Factory Path (1)", Items.EnemyItem, Types.Enemies, Kongs.any, [MapIDCombo(Maps.Isles, -1, DROPSANITY_FLAG_START + 0x1ab)]),
    # Krem Kap
    Locations.KremKap_JapesMainEnemy_Start: Location(Levels.JungleJapes, "Photo of Japes Enemy: Start", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_DiddyCavern: Location(Levels.JungleJapes, "Photo of Japes Enemy: Diddy Cavern", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Tunnel0: Location(Levels.JungleJapes, "Photo of Japes Enemy: Tunnel (0)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Tunnel1: Location(Levels.JungleJapes, "Photo of Japes Enemy: Tunnel (1)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Storm0: Location(Levels.JungleJapes, "Photo of Japes Enemy: Storm (0)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Storm1: Location(Levels.JungleJapes, "Photo of Japes Enemy: Storm (1)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Storm2: Location(Levels.JungleJapes, "Photo of Japes Enemy: Storm (2)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Hive0: Location(Levels.JungleJapes, "Photo of Japes Enemy: Hive (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Hive1: Location(Levels.JungleJapes, "Photo of Japes Enemy: Hive (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Hive2: Location(Levels.JungleJapes, "Photo of Japes Enemy: Hive (2)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Hive3: Location(Levels.JungleJapes, "Photo of Japes Enemy: Hive (3)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Hive4: Location(Levels.JungleJapes, "Photo of Japes Enemy: Hive (4)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_KilledInDemo: Location(Levels.JungleJapes, "Photo of Japes Enemy: Killed In Demo", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_NearUnderground: Location(Levels.JungleJapes, "Photo of Japes Enemy: Near Underground", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_NearPainting0: Location(Levels.JungleJapes, "Photo of Japes Enemy: Near Painting (0)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_NearPainting1: Location(Levels.JungleJapes, "Photo of Japes Enemy: Near Painting (1)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_NearPainting2: Location(Levels.JungleJapes, "Photo of Japes Enemy: Near Painting (2)", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_Mountain: Location(Levels.JungleJapes, "Photo of Japes Enemy: Mountain", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_FeatherTunnel: Location(Levels.JungleJapes, "Photo of Japes Enemy: Feather Tunnel", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMainEnemy_MiddleTunnel: Location(Levels.JungleJapes, "Photo of Japes Enemy: Middle Tunnel", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesLobbyEnemy_Enemy0: Location(Levels.DKIsles, "Photo of Jungle Japes Lobby Enemy: Enemy (0)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesLobbyEnemy_Enemy1: Location(Levels.DKIsles, "Photo of Jungle Japes Lobby Enemy: Enemy (1)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_JapesPaintingEnemy_Gauntlet0: Location(Levels.JungleJapes, "Photo of Japes Lanky Cave Enemy: Gauntlet0", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_JapesPaintingEnemy_Gauntlet1: Location(Levels.JungleJapes, "Photo of Japes Lanky Cave Enemy: Gauntlet1", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_JapesPaintingEnemy_Gauntlet2: Location(Levels.JungleJapes, "Photo of Japes Lanky Cave Enemy: Gauntlet2", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_JapesPaintingEnemy_Gauntlet3: Location(Levels.JungleJapes, "Photo of Japes Lanky Cave Enemy: Gauntlet3", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_JapesPaintingEnemy_Gauntlet4: Location(Levels.JungleJapes, "Photo of Japes Lanky Cave Enemy: Gauntlet4", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_Start0: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Start (0)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_Start1: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Start (1)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_Start2: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Start (2)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_Start3: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Start (3)", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_Start4: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Start (4)", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_NearGateSwitch0: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Near Gate Switch (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_NearGateSwitch1: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Near Gate Switch (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_HiLo: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Hi Lo", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_Conveyor0: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Conveyor (0)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesMountainEnemy_Conveyor1: Location(Levels.JungleJapes, "Photo of Japes Mountain Enemy: Conveyor (1)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesShellhiveEnemy_FirstRoom: Location(Levels.JungleJapes, "Photo of Japes Tiny Hive Enemy: First Room", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesShellhiveEnemy_SecondRoom0: Location(Levels.JungleJapes, "Photo of Japes Tiny Hive Enemy: Second Room (0)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesShellhiveEnemy_SecondRoom1: Location(Levels.JungleJapes, "Photo of Japes Tiny Hive Enemy: Second Room (1)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesShellhiveEnemy_ThirdRoom0: Location(Levels.JungleJapes, "Photo of Japes Tiny Hive Enemy: Third Room (0)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesShellhiveEnemy_ThirdRoom1: Location(Levels.JungleJapes, "Photo of Japes Tiny Hive Enemy: Third Room (1)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesShellhiveEnemy_ThirdRoom2: Location(Levels.JungleJapes, "Photo of Japes Tiny Hive Enemy: Third Room (2)", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesShellhiveEnemy_ThirdRoom3: Location(Levels.JungleJapes, "Photo of Japes Tiny Hive Enemy: Third Room (3)", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_JapesShellhiveEnemy_MainRoom: Location(Levels.JungleJapes, "Photo of Japes Tiny Hive Enemy: Main Room", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),

    Locations.KremKap_AztecMainEnemy_VaseRoom0: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Vase Room (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_VaseRoom1: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Vase Room (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_VaseRoom2: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Vase Room (2)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_TunnelPad0: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Tunnel Pad (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_TunnelCage0: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Tunnel Cage (0)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_TunnelCage1: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Tunnel Cage (1)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_TunnelCage2: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Tunnel Cage (2)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_StartingTunnel0: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Starting Tunnel (0)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_StartingTunnel1: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Starting Tunnel (1)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_OasisDoor: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Oasis Door", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_TunnelCage3: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Tunnel Cage (3)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_OutsideLlama: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Outside Llama", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_OutsideTower: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Outside Tower", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_TunnelPad1: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Tunnel Pad (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_NearCandy: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Near Candy", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_AroundTotem: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Around Totem", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_StartingTunnel2: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Starting Tunnel (2)", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_StartingTunnel3: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Starting Tunnel (3)", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_OutsideSnide: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Outside Snide", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_Outside5DT: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Outside 5DT", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecMainEnemy_NearSnoopTunnel: Location(Levels.AngryAztec, "Photo of Aztec Enemy: Near Sealed Quicksand Tunnel", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecLobbyEnemy_Pad0: Location(Levels.DKIsles, "Photo of Angry Aztec Lobby Enemy: Pad0", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecLobbyEnemy_Pad1: Location(Levels.DKIsles, "Photo of Angry Aztec Lobby Enemy: Pad1", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDK5DTEnemy_StartTrap0: Location(Levels.AngryAztec, "Photo of Aztec Donkey 5DTemple Enemy: Start Trap (0)", Items.PhotoKaboom, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDK5DTEnemy_StartTrap1: Location(Levels.AngryAztec, "Photo of Aztec Donkey 5DTemple Enemy: Start Trap (1)", Items.PhotoKaboom, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDK5DTEnemy_StartTrap2: Location(Levels.AngryAztec, "Photo of Aztec Donkey 5DTemple Enemy: Start Trap (2)", Items.PhotoKaboom, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDK5DTEnemy_EndTrap0: Location(Levels.AngryAztec, "Photo of Aztec Donkey 5DTemple Enemy: End Trap (0)", Items.PhotoKaboom, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDK5DTEnemy_EndTrap1: Location(Levels.AngryAztec, "Photo of Aztec Donkey 5DTemple Enemy: End Trap (1)", Items.PhotoKaboom, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDK5DTEnemy_EndTrap2: Location(Levels.AngryAztec, "Photo of Aztec Donkey 5DTemple Enemy: End Trap (2)", Items.PhotoKaboom, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDK5DTEnemy_EndPath0: Location(Levels.AngryAztec, "Photo of Aztec Donkey 5DTemple Enemy: End Path (0)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDK5DTEnemy_EndPath1: Location(Levels.AngryAztec, "Photo of Aztec Donkey 5DTemple Enemy: End Path (1)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDK5DTEnemy_StartPath: Location(Levels.AngryAztec, "Photo of Aztec Donkey 5DTemple Enemy: Start Path", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDiddy5DTEnemy_EndTrap0: Location(Levels.AngryAztec, "Photo of Aztec Diddy 5DTemple Enemy: End Trap (0)", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDiddy5DTEnemy_EndTrap1: Location(Levels.AngryAztec, "Photo of Aztec Diddy 5DTemple Enemy: End Trap (1)", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDiddy5DTEnemy_EndTrap2: Location(Levels.AngryAztec, "Photo of Aztec Diddy 5DTemple Enemy: End Trap (2)", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDiddy5DTEnemy_StartLeft0: Location(Levels.AngryAztec, "Photo of Aztec Diddy 5DTemple Enemy: Start Left (0)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDiddy5DTEnemy_StartLeft1: Location(Levels.AngryAztec, "Photo of Aztec Diddy 5DTemple Enemy: Start Left (1)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDiddy5DTEnemy_Reward: Location(Levels.AngryAztec, "Photo of Aztec Diddy 5DTemple Enemy: Reward", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecDiddy5DTEnemy_SecondSwitch: Location(Levels.AngryAztec, "Photo of Aztec Diddy 5DTemple Enemy: Second Switch", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLanky5DTEnemy_JoiningPaths: Location(Levels.AngryAztec, "Photo of Aztec Lanky 5DTemple Enemy: Joining Paths", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLanky5DTEnemy_EndTrap: Location(Levels.AngryAztec, "Photo of Aztec Lanky 5DTemple Enemy: End Trap", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLanky5DTEnemy_Reward: Location(Levels.AngryAztec, "Photo of Aztec Lanky 5DTemple Enemy: Reward", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTiny5DTEnemy_StartRightFront: Location(Levels.AngryAztec, "Photo of Aztec Tiny 5DTemple Enemy: Start Right Front", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTiny5DTEnemy_StartLeftBack: Location(Levels.AngryAztec, "Photo of Aztec Tiny 5DTemple Enemy: Start Left Back", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTiny5DTEnemy_StartRightBack: Location(Levels.AngryAztec, "Photo of Aztec Tiny 5DTemple Enemy: Start Right Back", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTiny5DTEnemy_StartLeftFront: Location(Levels.AngryAztec, "Photo of Aztec Tiny 5DTemple Enemy: Start Left Front", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTiny5DTEnemy_Reward0: Location(Levels.AngryAztec, "Photo of Aztec Tiny 5DTemple Enemy: Reward (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTiny5DTEnemy_Reward1: Location(Levels.AngryAztec, "Photo of Aztec Tiny 5DTemple Enemy: Reward (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTiny5DTEnemy_DeadEnd0: Location(Levels.AngryAztec, "Photo of Aztec Tiny 5DTemple Enemy: Dead End (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTiny5DTEnemy_DeadEnd1: Location(Levels.AngryAztec, "Photo of Aztec Tiny 5DTemple Enemy: Dead End (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecChunky5DTEnemy_StartRight: Location(Levels.AngryAztec, "Photo of Aztec Chunky 5DTemple Enemy: Start Right", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecChunky5DTEnemy_StartLeft: Location(Levels.AngryAztec, "Photo of Aztec Chunky 5DTemple Enemy: Start Left", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecChunky5DTEnemy_SecondRight: Location(Levels.AngryAztec, "Photo of Aztec Chunky 5DTemple Enemy: Second Right", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecChunky5DTEnemy_SecondLeft: Location(Levels.AngryAztec, "Photo of Aztec Chunky 5DTemple Enemy: Second Left", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecChunky5DTEnemy_Reward: Location(Levels.AngryAztec, "Photo of Aztec Chunky 5DTemple Enemy: Reward", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLlamaEnemy_KongFreeInstrument: Location(Levels.AngryAztec, "Photo of Aztec Llama Temple Enemy: Kong Free Instrument", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLlamaEnemy_DinoInstrument: Location(Levels.AngryAztec, "Photo of Aztec Llama Temple Enemy: Dino Instrument", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLlamaEnemy_Matching0: Location(Levels.AngryAztec, "Photo of Aztec Llama Temple Enemy: Matching0", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLlamaEnemy_Matching1: Location(Levels.AngryAztec, "Photo of Aztec Llama Temple Enemy: Matching1", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLlamaEnemy_Right: Location(Levels.AngryAztec, "Photo of Aztec Llama Temple Enemy: Right", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLlamaEnemy_Left: Location(Levels.AngryAztec, "Photo of Aztec Llama Temple Enemy: Left", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLlamaEnemy_MelonCrate: Location(Levels.AngryAztec, "Photo of Aztec Llama Temple Enemy: Melon Crate", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecLlamaEnemy_SlamSwitch: Location(Levels.AngryAztec, "Photo of Aztec Llama Temple Enemy: Slam Switch", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating00: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating00", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating01: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating01", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating02: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating02", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating03: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating03", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating04: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating04", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating05: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating05", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating06: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating06", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating07: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating07", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating08: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating08", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating09: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating09", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating10: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating10", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating11: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating11", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating12: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating12", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating13: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating13", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating14: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating14", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_Rotating15: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Rotating15", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_MiniRoom00: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Mini Room00", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_MiniRoom01: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Mini Room01", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_MiniRoom02: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Mini Room02", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_AztecTempleEnemy_MiniRoom03: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Mini Room03", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_GuardRotating0: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Guard Rotating (0)", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_GuardRotating1: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Guard Rotating (1)", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_MainRoom0: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Main Room (0)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_MainRoom1: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Main Room (1)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_MainRoom2: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Main Room (2)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_KongRoom0: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Kong Room (0)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_KongRoom1: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Kong Room (1)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_KongRoom2: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Kong Room (2)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_KongRoom3: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Kong Room (3)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_KongRoom4: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Kong Room (4)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_AztecTempleEnemy_Underwater: Location(Levels.AngryAztec, "Photo of Aztec Tiny Temple Enemy: Underwater", Items.PhotoShuri, Types.EnemyPhoto, Kongs.any, []),

    Locations.KremKap_FactoryMainEnemy_CandyCranky0: Location(Levels.FranticFactory, "Photo of Factory Enemy: Candy Cranky (0)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_CandyCranky1: Location(Levels.FranticFactory, "Photo of Factory Enemy: Candy Cranky (1)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_LobbyLeft: Location(Levels.FranticFactory, "Photo of Factory Enemy: Lobby Left", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_LobbyRight: Location(Levels.FranticFactory, "Photo of Factory Enemy: Lobby Right", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_StorageRoom: Location(Levels.FranticFactory, "Photo of Factory Enemy: Storage Room", Items.PhotoZingerRobo, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_BlockTower0: Location(Levels.FranticFactory, "Photo of Factory Enemy: Block Tower (0)", Items.PhotoMrDice1, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_BlockTower1: Location(Levels.FranticFactory, "Photo of Factory Enemy: Block Tower (1)", Items.PhotoSirDomino, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_BlockTower2: Location(Levels.FranticFactory, "Photo of Factory Enemy: Block Tower (2)", Items.PhotoMrDice1, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_TunnelToHatch: Location(Levels.FranticFactory, "Photo of Factory Enemy: Tunnel To Hatch", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_TunnelToProd0: Location(Levels.FranticFactory, "Photo of Factory Enemy: Tunnel To Prod (0)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_TunnelToProd1: Location(Levels.FranticFactory, "Photo of Factory Enemy: Tunnel To Prod (1)", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_TunnelToBlockTower: Location(Levels.FranticFactory, "Photo of Factory Enemy: Tunnel To Block Tower", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_TunnelToRace0: Location(Levels.FranticFactory, "Photo of Factory Enemy: Tunnel To Race (0)", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_TunnelToRace1: Location(Levels.FranticFactory, "Photo of Factory Enemy: Tunnel To Race (1)", Items.PhotoZingerRobo, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_LowWarp4: Location(Levels.FranticFactory, "Photo of Factory Enemy: Low Warp 4", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_DiddySwitch: Location(Levels.FranticFactory, "Photo of Factory Enemy: Diddy Switch", Items.PhotoZingerRobo, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_ToBlockTowerTunnel: Location(Levels.FranticFactory, "Photo of Factory Enemy: To Block Tower Tunnel", Items.PhotoZingerRobo, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_DarkRoom0: Location(Levels.FranticFactory, "Photo of Factory Enemy: Dark Room (0)", Items.PhotoZingerRobo, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryMainEnemy_DarkRoom1: Location(Levels.FranticFactory, "Photo of Factory Enemy: Dark Room (1)", Items.PhotoZingerRobo, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM0: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM0", Items.PhotoMrDice0, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM1: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM1", Items.PhotoSirDomino, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM2: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM2", Items.PhotoSirDomino, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM3: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM3", Items.PhotoMrDice0, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM4: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM4", Items.PhotoMrDice0, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM5: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM5", Items.PhotoRuler, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM6: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM6", Items.PhotoRuler, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM7: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM7", Items.PhotoMrDice1, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM8: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM8", Items.PhotoSirDomino, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_BHDM9: Location(Levels.FranticFactory, "Photo of Factory Enemy: BHDM9", Items.PhotoSirDomino, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_1342Gauntlet0: Location(Levels.FranticFactory, "Photo of Factory Enemy: 1342Gauntlet0", Items.PhotoZingerRobo, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_1342Gauntlet1: Location(Levels.FranticFactory, "Photo of Factory Enemy: 1342Gauntlet1", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_1342Gauntlet2: Location(Levels.FranticFactory, "Photo of Factory Enemy: 1342Gauntlet2", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_3124Gauntlet0: Location(Levels.FranticFactory, "Photo of Factory Enemy: 3124Gauntlet0", Items.PhotoMrDice1, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_3124Gauntlet1: Location(Levels.FranticFactory, "Photo of Factory Enemy: 3124Gauntlet1", Items.PhotoSirDomino, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_3124Gauntlet2: Location(Levels.FranticFactory, "Photo of Factory Enemy: 3124Gauntlet2", Items.PhotoMrDice1, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_4231Gauntlet0: Location(Levels.FranticFactory, "Photo of Factory Enemy: 4231Gauntlet0", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_FactoryMainEnemy_4231Gauntlet1: Location(Levels.FranticFactory, "Photo of Factory Enemy: 4231Gauntlet1", Items.PhotoRoboKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_FactoryLobbyEnemy_Enemy0: Location(Levels.DKIsles, "Photo of Frantic Factory Lobby Enemy: Enemy (0)", Items.PhotoZingerRobo, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonMainEnemy_ChestRoom0: Location(Levels.GloomyGalleon, "Photo of Galleon Enemy: Chest Room (0)", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonMainEnemy_ChestRoom1: Location(Levels.GloomyGalleon, "Photo of Galleon Enemy: Chest Room (1)", Items.PhotoKaboom, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonMainEnemy_NearVineCannon: Location(Levels.GloomyGalleon, "Photo of Galleon Enemy: Near Vine Cannon", Items.PhotoKaboom, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonMainEnemy_CrankyCannon: Location(Levels.GloomyGalleon, "Photo of Galleon Enemy: Cranky Cannon", Items.PhotoKaboom, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonMainEnemy_Submarine: Location(Levels.GloomyGalleon, "Photo of Galleon Enemy: Submarine", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonMainEnemy_5DS0: Location(Levels.GloomyGalleon, "Photo of Galleon Enemy: 5DS0", Items.PhotoShuri, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonMainEnemy_5DS1: Location(Levels.GloomyGalleon, "Photo of Galleon Enemy: 5DS1", Items.PhotoShuri, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonMainEnemy_PeanutTunnel: Location(Levels.GloomyGalleon, "Photo of Galleon Enemy: Peanut Tunnel", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonMainEnemy_CoconutTunnel: Location(Levels.GloomyGalleon, "Photo of Galleon Enemy: Coconut Tunnel", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonLighthouseEnemy_Enemy0: Location(Levels.GloomyGalleon, "Photo of Galleon Lighthouse Enemy: Enemy (0)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonLighthouseEnemy_Enemy1: Location(Levels.GloomyGalleon, "Photo of Galleon Lighthouse Enemy: Enemy (1)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon5DSDLCEnemy_Diddy: Location(Levels.GloomyGalleon, "Photo of Galleon5DShip Diddy Lanky Chunky Enemy: Diddy", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon5DSDLCEnemy_Chunky: Location(Levels.GloomyGalleon, "Photo of Galleon5DShip Diddy Lanky Chunky Enemy: Chunky", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon5DSDLCEnemy_Lanky: Location(Levels.GloomyGalleon, "Photo of Galleon5DShip Diddy Lanky Chunky Enemy: Lanky", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon5DSDTEnemy_DK0: Location(Levels.GloomyGalleon, "Photo of Galleon5DShip DKTiny Enemy: DK0", Items.PhotoShuri, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon5DSDTEnemy_DK1: Location(Levels.GloomyGalleon, "Photo of Galleon5DShip DKTiny Enemy: DK1", Items.PhotoShuri, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon5DSDTEnemy_DK2: Location(Levels.GloomyGalleon, "Photo of Galleon5DShip DKTiny Enemy: DK2", Items.PhotoShuri, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon5DSDTEnemy_TinyCage: Location(Levels.GloomyGalleon, "Photo of Galleon5DShip DKTiny Enemy: Tiny Cage", Items.PhotoShuri, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon5DSDTEnemy_TinyBed: Location(Levels.GloomyGalleon, "Photo of Galleon5DShip DKTiny Enemy: Tiny Bed", Items.PhotoShuri, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon2DSEnemy_Tiny0: Location(Levels.GloomyGalleon, "Photo of Galleon2DShip Enemy: Tiny0", Items.PhotoGimpfish, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Galleon2DSEnemy_Tiny1: Location(Levels.GloomyGalleon, "Photo of Galleon2DShip Enemy: Tiny1", Items.PhotoGimpfish, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonSubEnemy_Enemy0: Location(Levels.GloomyGalleon, "Photo of Galleon Submarine Enemy: Enemy0", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonSubEnemy_Enemy1: Location(Levels.GloomyGalleon, "Photo of Galleon Submarine Enemy: Enemy1", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonSubEnemy_Enemy2: Location(Levels.GloomyGalleon, "Photo of Galleon Submarine Enemy: Enemy2", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_GalleonSubEnemy_Enemy3: Location(Levels.GloomyGalleon, "Photo of Galleon Submarine Enemy: Enemy3", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_HollowTree0: Location(Levels.FungiForest, "Photo of Forest Enemy: Hollow Tree (0)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_HollowTree1: Location(Levels.FungiForest, "Photo of Forest Enemy: Hollow Tree (1)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_HollowTreeEntrance: Location(Levels.FungiForest, "Photo of Forest Enemy: Hollow Tree Entrance", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_TreeMelonCrate0: Location(Levels.FungiForest, "Photo of Forest Enemy: Tree Melon Crate (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_TreeMelonCrate1: Location(Levels.FungiForest, "Photo of Forest Enemy: Tree Melon Crate (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_TreeMelonCrate2: Location(Levels.FungiForest, "Photo of Forest Enemy: Tree Melon Crate (2)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_AppleGauntlet0: Location(Levels.FungiForest, "Photo of Forest Enemy: Apple Gauntlet (0)", Items.PhotoTomato, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_AppleGauntlet1: Location(Levels.FungiForest, "Photo of Forest Enemy: Apple Gauntlet (1)", Items.PhotoTomato, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_AppleGauntlet2: Location(Levels.FungiForest, "Photo of Forest Enemy: Apple Gauntlet (2)", Items.PhotoTomato, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_AppleGauntlet3: Location(Levels.FungiForest, "Photo of Forest Enemy: Apple Gauntlet (3)", Items.PhotoTomato, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearBeanstalk0: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Beanstalk (0)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearBeanstalk1: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Beanstalk (1)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_GreenTunnel: Location(Levels.FungiForest, "Photo of Forest Enemy: Green Tunnel", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearLowWarp5: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Low Warp 5", Items.PhotoMushroomMan, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearPinkTunnelBounceTag: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Pink Tunnel Bounce Tag", Items.PhotoMushroomMan, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearGMRocketbarrel: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Giant Mushroom Rocketbarrel", Items.PhotoMushroomMan, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_BetweenYellowTunnelAndRB: Location(Levels.FungiForest, "Photo of Forest Enemy: Between Yellow Tunnel And RB", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearCranky: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Cranky", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearPinkTunnelGM: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Pink Tunnel Giant Mushroom", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_GMRearTag: Location(Levels.FungiForest, "Photo of Forest Enemy: Giant Mushroom Rear Tag", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearFacePuzzle: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Face Puzzle", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearCrown: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Crown", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearHighWarp5: Location(Levels.FungiForest, "Photo of Forest Enemy: Near High Warp 5", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_TopOfMushroom: Location(Levels.FungiForest, "Photo of Forest Enemy: Top Of Mushroom", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearAppleDropoff: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Apple Dropoff", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearDKPortal: Location(Levels.FungiForest, "Photo of Forest Enemy: Near DKPortal", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearWellTag: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Well Tag", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_YellowTunnel0: Location(Levels.FungiForest, "Photo of Forest Enemy: Yellow Tunnel (0)", Items.PhotoMushroomMan, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_YellowTunnel1: Location(Levels.FungiForest, "Photo of Forest Enemy: Yellow Tunnel (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_YellowTunnel2: Location(Levels.FungiForest, "Photo of Forest Enemy: Yellow Tunnel (2)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_YellowTunnel3: Location(Levels.FungiForest, "Photo of Forest Enemy: Yellow Tunnel (3)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearSnide: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Snide", Items.PhotoMushroomMan, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearIsoCoin: Location(Levels.FungiForest, "Photo of Forest Enemy: Near the hidden Rainbow Coin", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearBBlast: Location(Levels.FungiForest, "Photo of Forest Enemy: Near BBlast", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearDarkAttic: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Dark Attic", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearWellExit: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Well Exit", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_NearBlueTunnel: Location(Levels.FungiForest, "Photo of Forest Enemy: Near Blue Tunnel", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_Thornvine0: Location(Levels.FungiForest, "Photo of Forest Enemy: Thornvine (0)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_Thornvine1: Location(Levels.FungiForest, "Photo of Forest Enemy: Thornvine (1)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_Thornvine2: Location(Levels.FungiForest, "Photo of Forest Enemy: Thornvine (2)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMainEnemy_ThornvineEntrance: Location(Levels.FungiForest, "Photo of Forest Enemy: Thornvine Entrance", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestAnthillEnemy_Gauntlet0: Location(Levels.FungiForest, "Photo of Forest Anthill Enemy: Gauntlet (0)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestAnthillEnemy_Gauntlet1: Location(Levels.FungiForest, "Photo of Forest Anthill Enemy: Gauntlet (1)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestAnthillEnemy_Gauntlet2: Location(Levels.FungiForest, "Photo of Forest Anthill Enemy: Gauntlet (2)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestAnthillEnemy_Gauntlet3: Location(Levels.FungiForest, "Photo of Forest Anthill Enemy: Gauntlet (3)", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestWinchEnemy_Enemy: Location(Levels.FungiForest, "Photo of Forest Winch Room Enemy: Enemy", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestThornBarnEnemy_Enemy: Location(Levels.FungiForest, "Photo of Forest Thornvine Barn Enemy: Enemy", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMillFrontEnemy_Enemy: Location(Levels.FungiForest, "Photo of Forest Mill Front Enemy: Enemy", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestMillRearEnemy_Enemy: Location(Levels.FungiForest, "Photo of Forest Mill Back Enemy: Enemy", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestGMEnemy_AboveNightDoor: Location(Levels.FungiForest, "Photo of Forest Giant Mushroom Enemy: Above Night Door", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestGMEnemy_Path0: Location(Levels.FungiForest, "Photo of Forest Giant Mushroom Enemy: Path (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestGMEnemy_Path1: Location(Levels.FungiForest, "Photo of Forest Giant Mushroom Enemy: Path (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_ForestLankyAtticEnemy_Gauntlet0: Location(Levels.FungiForest, "Photo of Forest Mill Attic Enemy: Gauntlet0", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_ForestLankyAtticEnemy_Gauntlet1: Location(Levels.FungiForest, "Photo of Forest Mill Attic Enemy: Gauntlet1", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_ForestLankyAtticEnemy_Gauntlet2: Location(Levels.FungiForest, "Photo of Forest Mill Attic Enemy: Gauntlet2", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestLeapEnemy_Enemy0: Location(Levels.FungiForest, "Photo of Forest Lanky Zingers Room Enemy: Enemy (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestLeapEnemy_Enemy1: Location(Levels.FungiForest, "Photo of Forest Lanky Zingers Room Enemy: Enemy (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_ForestFacePuzzleEnemy_Enemy: Location(Levels.FungiForest, "Photo of Forest Chunky Face Room Enemy: Enemy", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_ForestSpiderEnemy_Gauntlet0: Location(Levels.FungiForest, "Photo of Forest Spider Enemy: Gauntlet0", Items.PhotoSpiderSmall, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_ForestSpiderEnemy_Gauntlet1: Location(Levels.FungiForest, "Photo of Forest Spider Enemy: Gauntlet1", Items.PhotoSpiderSmall, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_ForestSpiderEnemy_Gauntlet2: Location(Levels.FungiForest, "Photo of Forest Spider Enemy: Gauntlet2", Items.PhotoSpiderSmall, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CavesMainEnemy_Start: Location(Levels.CrystalCaves, "Photo of Caves Enemy: Start", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CavesMainEnemy_NearIceCastle: Location(Levels.CrystalCaves, "Photo of Caves Enemy: Near Ice Castle", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CavesMainEnemy_Outside5DC: Location(Levels.CrystalCaves, "Photo of Caves Enemy: Outside 5DC", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CavesMainEnemy_1DCWaterfall: Location(Levels.CrystalCaves, "Photo of Caves Enemy: 1DC Waterfall", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CavesMainEnemy_NearFunky: Location(Levels.CrystalCaves, "Photo of Caves Enemy: Near Funky", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CavesMainEnemy_NearSnide: Location(Levels.CrystalCaves, "Photo of Caves Enemy: Near Snide", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CavesMainEnemy_NearBonusRoom: Location(Levels.CrystalCaves, "Photo of Caves Enemy: Near Bonus Room", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CavesMainEnemy_1DCHeadphones: Location(Levels.CrystalCaves, "Photo of Caves Enemy: 1DC Headphones", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CavesMainEnemy_GiantKosha: Location(Levels.CrystalCaves, "Photo of Caves Enemy: Giant Kosha", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Caves5DIDKEnemy_Right: Location(Levels.CrystalCaves, "Photo of Caves Donkey Igloo Enemy: Right", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Caves5DIDKEnemy_Left: Location(Levels.CrystalCaves, "Photo of Caves Donkey Igloo Enemy: Left", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DILankyEnemy_First0: Location(Levels.CrystalCaves, "Photo of Caves Lanky Igloo Enemy: First (0)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DILankyEnemy_First1: Location(Levels.CrystalCaves, "Photo of Caves Lanky Igloo Enemy: First (1)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DILankyEnemy_Second0: Location(Levels.CrystalCaves, "Photo of Caves Lanky Igloo Enemy: Second (0)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DILankyEnemy_Second1: Location(Levels.CrystalCaves, "Photo of Caves Lanky Igloo Enemy: Second (1)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DILankyEnemy_Second2: Location(Levels.CrystalCaves, "Photo of Caves Lanky Igloo Enemy: Second (2)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Caves5DITinyEnemy_BigEnemy: Location(Levels.CrystalCaves, "Photo of Caves Tiny Igloo Enemy: Big Enemy", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DIChunkyEnemy_Gauntlet00: Location(Levels.CrystalCaves, "Photo of Caves Chunky Igloo Enemy: Gauntlet00", Items.PhotoFireballGlasses, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DIChunkyEnemy_Gauntlet01: Location(Levels.CrystalCaves, "Photo of Caves Chunky Igloo Enemy: Gauntlet01", Items.PhotoFireballGlasses, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DIChunkyEnemy_Gauntlet02: Location(Levels.CrystalCaves, "Photo of Caves Chunky Igloo Enemy: Gauntlet02", Items.PhotoFireballGlasses, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DIChunkyEnemy_Gauntlet03: Location(Levels.CrystalCaves, "Photo of Caves Chunky Igloo Enemy: Gauntlet03", Items.PhotoFireballGlasses, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DIChunkyEnemy_Gauntlet04: Location(Levels.CrystalCaves, "Photo of Caves Chunky Igloo Enemy: Gauntlet04", Items.PhotoFireballGlasses, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_Caves1DCEnemy_Near: Location(Levels.CrystalCaves, "Photo of Caves Lanky Cabin Enemy: Near", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves1DCEnemy_Far: Location(Levels.CrystalCaves, "Photo of Caves Lanky Cabin Enemy: Far", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDKEnemy_Gauntlet0: Location(Levels.CrystalCaves, "Photo of Caves Donkey Cabin Enemy: Gauntlet0", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDKEnemy_Gauntlet1: Location(Levels.CrystalCaves, "Photo of Caves Donkey Cabin Enemy: Gauntlet1", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDKEnemy_Gauntlet2: Location(Levels.CrystalCaves, "Photo of Caves Donkey Cabin Enemy: Gauntlet2", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDKEnemy_Gauntlet3: Location(Levels.CrystalCaves, "Photo of Caves Donkey Cabin Enemy: Gauntlet3", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDKEnemy_Gauntlet4: Location(Levels.CrystalCaves, "Photo of Caves Donkey Cabin Enemy: Gauntlet4", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDKEnemy_Gauntlet5: Location(Levels.CrystalCaves, "Photo of Caves Donkey Cabin Enemy: Gauntlet5", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyLowEnemy_CloseRight: Location(Levels.CrystalCaves, "Photo of Caves Diddy Lower Cabin Enemy: Close Right", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyLowEnemy_FarRight: Location(Levels.CrystalCaves, "Photo of Caves Diddy Lower Cabin Enemy: Far Right", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyLowEnemy_CloseLeft: Location(Levels.CrystalCaves, "Photo of Caves Diddy Lower Cabin Enemy: Close Left", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyLowEnemy_FarLeft: Location(Levels.CrystalCaves, "Photo of Caves Diddy Lower Cabin Enemy: Far Left", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyLowEnemy_Center0: Location(Levels.CrystalCaves, "Photo of Caves Diddy Lower Cabin Enemy: Center0", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyLowEnemy_Center1: Location(Levels.CrystalCaves, "Photo of Caves Diddy Lower Cabin Enemy: Center1", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyLowEnemy_Center2: Location(Levels.CrystalCaves, "Photo of Caves Diddy Lower Cabin Enemy: Center2", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyLowEnemy_Center3: Location(Levels.CrystalCaves, "Photo of Caves Diddy Lower Cabin Enemy: Center3", Items.PhotoKlobber, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyUpperEnemy_Enemy0: Location(Levels.CrystalCaves, "Photo of Caves Diddy Upper Cabin Enemy: Enemy (0)", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCDiddyUpperEnemy_Enemy1: Location(Levels.CrystalCaves, "Photo of Caves Diddy Upper Cabin Enemy: Enemy (1)", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCTinyEnemy_Gauntlet0: Location(Levels.CrystalCaves, "Photo of Caves Tiny Cabin Enemy: Gauntlet0", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCTinyEnemy_Gauntlet1: Location(Levels.CrystalCaves, "Photo of Caves Tiny Cabin Enemy: Gauntlet1", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCTinyEnemy_Gauntlet2: Location(Levels.CrystalCaves, "Photo of Caves Tiny Cabin Enemy: Gauntlet2", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCTinyEnemy_Gauntlet3: Location(Levels.CrystalCaves, "Photo of Caves Tiny Cabin Enemy: Gauntlet3", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_Caves5DCTinyEnemy_Gauntlet4: Location(Levels.CrystalCaves, "Photo of Caves Tiny Cabin Enemy: Gauntlet4", Items.PhotoKlaptrapPurple, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_NearBridge0: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Near Bridge (0)", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_NearBridge1: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Near Bridge (1)", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_WoodenExtrusion0: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Wooden Extrusion (0)", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_WoodenExtrusion1: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Wooden Extrusion (1)", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_NearShed: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Near Shed", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_NearLibrary: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Near Library", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_NearTower: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Near Tower", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_MuseumSteps: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Museum Steps", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_NearLowCave: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Near Low Cave", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_PathToLowKasplat: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Path To Low Kasplat", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_LowTnS: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Low TnS", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_PathToDungeon: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Path To Dungeon", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMainEnemy_NearHeadphones: Location(Levels.CreepyCastle, "Photo of Castle Enemy: Near Headphones", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLobbyEnemy_Left: Location(Levels.DKIsles, "Photo of Creepy Castle Lobby Enemy: Left", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLobbyEnemy_FarRight: Location(Levels.DKIsles, "Photo of Creepy Castle Lobby Enemy: Far Right", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLobbyEnemy_NearRight: Location(Levels.DKIsles, "Photo of Creepy Castle Lobby Enemy: Near Right", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleBallroomEnemy_Board00: Location(Levels.CreepyCastle, "Photo of Castle Ballroom Enemy: Board00", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleBallroomEnemy_Board01: Location(Levels.CreepyCastle, "Photo of Castle Ballroom Enemy: Board01", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleBallroomEnemy_Board02: Location(Levels.CreepyCastle, "Photo of Castle Ballroom Enemy: Board02", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleBallroomEnemy_Board03: Location(Levels.CreepyCastle, "Photo of Castle Ballroom Enemy: Board03", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleBallroomEnemy_Board04: Location(Levels.CreepyCastle, "Photo of Castle Ballroom Enemy: Board04", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleBallroomEnemy_Start: Location(Levels.CreepyCastle, "Photo of Castle Ballroom Enemy: Start", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleDungeonEnemy_FaceRoom: Location(Levels.CreepyCastle, "Photo of Castle Dungeon Enemy: Face Room", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleDungeonEnemy_ChairRoom: Location(Levels.CreepyCastle, "Photo of Castle Dungeon Enemy: Chair Room", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleDungeonEnemy_OutsideLankyRoom: Location(Levels.CreepyCastle, "Photo of Castle Dungeon Enemy: Outside Lanky Room", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleShedEnemy_Gauntlet00: Location(Levels.CreepyCastle, "Photo of Castle Shed Enemy: Gauntlet00", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleShedEnemy_Gauntlet01: Location(Levels.CreepyCastle, "Photo of Castle Shed Enemy: Gauntlet01", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleShedEnemy_Gauntlet02: Location(Levels.CreepyCastle, "Photo of Castle Shed Enemy: Gauntlet02", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleShedEnemy_Gauntlet03: Location(Levels.CreepyCastle, "Photo of Castle Shed Enemy: Gauntlet03", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleShedEnemy_Gauntlet04: Location(Levels.CreepyCastle, "Photo of Castle Shed Enemy: Gauntlet04", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLowCaveEnemy_NearCrypt: Location(Levels.CreepyCastle, "Photo of Castle Lower Cave Enemy: Near Crypt", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLowCaveEnemy_StairRight: Location(Levels.CreepyCastle, "Photo of Castle Lower Cave Enemy: Stair Right", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLowCaveEnemy_StairLeft: Location(Levels.CreepyCastle, "Photo of Castle Lower Cave Enemy: Stair Left", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLowCaveEnemy_NearMausoleum: Location(Levels.CreepyCastle, "Photo of Castle Lower Cave Enemy: Near Mausoleum", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLowCaveEnemy_NearFunky: Location(Levels.CreepyCastle, "Photo of Castle Lower Cave Enemy: Near Funky", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLowCaveEnemy_NearTag: Location(Levels.CreepyCastle, "Photo of Castle Lower Cave Enemy: Near Tag", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_DiddyCoffin0: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Diddy Coffin (0)", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_DiddyCoffin1: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Diddy Coffin (1)", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_DiddyCoffin2: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Diddy Coffin (2)", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_DiddyCoffin3: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Diddy Coffin (3)", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_ChunkyCoffin0: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Chunky Coffin (0)", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_ChunkyCoffin1: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Chunky Coffin (1)", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_ChunkyCoffin2: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Chunky Coffin (2)", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_ChunkyCoffin3: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Chunky Coffin (3)", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_MinecartEntry: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Minecart Entry", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_Fork: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Fork", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_NearDiddy: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Near Diddy", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleCryptEnemy_NearChunky: Location(Levels.CreepyCastle, "Photo of Castle Crypt Enemy: Near Chunky", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMausoleumEnemy_TinyPath: Location(Levels.CreepyCastle, "Photo of Castle Mausoleum Enemy: Tiny Path", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMausoleumEnemy_LankyPath0: Location(Levels.CreepyCastle, "Photo of Castle Mausoleum Enemy: Lanky Path (0)", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMausoleumEnemy_LankyPath1: Location(Levels.CreepyCastle, "Photo of Castle Mausoleum Enemy: Lanky Path (1)", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleUpperCaveEnemy_NearDungeon: Location(Levels.CreepyCastle, "Photo of Castle Upper Cave Enemy: Near Dungeon", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleUpperCaveEnemy_Pit: Location(Levels.CreepyCastle, "Photo of Castle Upper Cave Enemy: Pit", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleUpperCaveEnemy_NearPit: Location(Levels.CreepyCastle, "Photo of Castle Upper Cave Enemy: Near Pit", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleUpperCaveEnemy_NearEntrance: Location(Levels.CreepyCastle, "Photo of Castle Upper Cave Enemy: Near Entrance", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleKKOEnemy_CenterEnemy: Location(Levels.CreepyCastle, "Photo of Castle Boss Enemy: Center Enemy", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleKKOEnemy_WaterEnemy00: Location(Levels.CreepyCastle, "Photo of Castle Boss Enemy: Water Enemy00", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleKKOEnemy_WaterEnemy01: Location(Levels.CreepyCastle, "Photo of Castle Boss Enemy: Water Enemy01", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleKKOEnemy_WaterEnemy02: Location(Levels.CreepyCastle, "Photo of Castle Boss Enemy: Water Enemy02", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleKKOEnemy_WaterEnemy03: Location(Levels.CreepyCastle, "Photo of Castle Boss Enemy: Water Enemy03", Items.PhotoPufftup, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleLibraryEnemy_Gauntlet00: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Gauntlet00", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleLibraryEnemy_Gauntlet01: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Gauntlet01", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleLibraryEnemy_Gauntlet02: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Gauntlet02", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleLibraryEnemy_Gauntlet03: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Gauntlet03", Items.PhotoKrossbones, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_Corridor00: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Corridor00", Items.PhotoBook, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_Corridor01: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Corridor01", Items.PhotoBook, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_Corridor02: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Corridor02", Items.PhotoBook, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_Corridor03: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Corridor03", Items.PhotoBook, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_Corridor04: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Corridor04", Items.PhotoBook, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_Corridor05: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Corridor05", Items.PhotoBook, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_ForkLeft0: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Fork Left (0)", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_ForkLeft1: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Fork Left (1)", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_ForkCenter: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Fork Center", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleLibraryEnemy_ForkRight: Location(Levels.CreepyCastle, "Photo of Castle Library Enemy: Fork Right", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMuseumEnemy_MainFloor0: Location(Levels.CreepyCastle, "Photo of Castle Museum Enemy: Main Floor (0)", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMuseumEnemy_MainFloor1: Location(Levels.CreepyCastle, "Photo of Castle Museum Enemy: Main Floor (1)", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMuseumEnemy_MainFloor2: Location(Levels.CreepyCastle, "Photo of Castle Museum Enemy: Main Floor (2)", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMuseumEnemy_MainFloor3: Location(Levels.CreepyCastle, "Photo of Castle Museum Enemy: Main Floor (3)", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleMuseumEnemy_Start: Location(Levels.CreepyCastle, "Photo of Castle Museum Enemy: Start", Items.PhotoKosha, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTowerEnemy_Gauntlet0: Location(Levels.CreepyCastle, "Photo of Castle Tower Enemy: Gauntlet0", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTowerEnemy_Gauntlet1: Location(Levels.CreepyCastle, "Photo of Castle Tower Enemy: Gauntlet1", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTowerEnemy_Gauntlet2: Location(Levels.CreepyCastle, "Photo of Castle Tower Enemy: Gauntlet2", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTowerEnemy_Gauntlet3: Location(Levels.CreepyCastle, "Photo of Castle Tower Enemy: Gauntlet3", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTowerEnemy_Gauntlet4: Location(Levels.CreepyCastle, "Photo of Castle Tower Enemy: Gauntlet4", Items.PhotoGhost, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTrashEnemy_Gauntlet0: Location(Levels.CreepyCastle, "Photo of Castle Trash Can Enemy: Gauntlet0", Items.PhotoBug, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTrashEnemy_Gauntlet1: Location(Levels.CreepyCastle, "Photo of Castle Trash Can Enemy: Gauntlet1", Items.PhotoBug, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTrashEnemy_Gauntlet2: Location(Levels.CreepyCastle, "Photo of Castle Trash Can Enemy: Gauntlet2", Items.PhotoBug, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTrashEnemy_Gauntlet3: Location(Levels.CreepyCastle, "Photo of Castle Trash Can Enemy: Gauntlet3", Items.PhotoBug, Types.EnemyPhoto, Kongs.any, []),
    # Locations.KremKap_CastleTrashEnemy_Gauntlet4: Location(Levels.CreepyCastle, "Photo of Castle Trash Can Enemy: Gauntlet4", Items.PhotoBug, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleTreeEnemy_StartRoom0: Location(Levels.CreepyCastle, "Photo of Castle Tree Enemy: Start Room (0)", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_CastleTreeEnemy_StartRoom1: Location(Levels.CreepyCastle, "Photo of Castle Tree Enemy: Start Room (1)", Items.PhotoBat, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_Start0: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Start (0)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_Start1: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Start (1)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_Hill: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Hill", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_SwitchRoom0: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Switch Room (0)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_SwitchRoom1: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Switch Room (1)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_MiniRoom0: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Mini Room (0)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_MiniRoom1: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Mini Room (1)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_MiniRoom2: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Mini Room (2)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_MiniRoom3: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Mini Room (3)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_DKRoom: Location(Levels.HideoutHelm, "Photo of Helm Enemy: DKRoom", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_ChunkyRoom0: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Chunky Room (0)", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_ChunkyRoom1: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Chunky Room (1)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_TinyRoom: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Tiny Room", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_LankyRoom0: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Lanky Room (0)", Items.PhotoKlump, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_LankyRoom1: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Lanky Room (1)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_DiddyRoom0: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Diddy Room (0)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_DiddyRoom1: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Diddy Room (1)", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_NavRight: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Nav Right", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_HelmMainEnemy_NavLeft: Location(Levels.HideoutHelm, "Photo of Helm Enemy: Nav Left", Items.PhotoKlaptrapGreen, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_PineappleCage0: Location(Levels.DKIsles, "Photo of Isles Enemy: Pineapple Cage (0)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_FungiCannon0: Location(Levels.DKIsles, "Photo of Isles Enemy: Fungi Cannon (0)", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_JapesEntrance: Location(Levels.DKIsles, "Photo of Isles Enemy: Japes Entrance", Items.PhotoBeaverBlue, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_MonkeyportPad: Location(Levels.DKIsles, "Photo of Isles Enemy: Monkeyport Pad", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_UpperFactoryPath: Location(Levels.DKIsles, "Photo of Isles Enemy: Upper Factory Path", Items.PhotoKremling, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_NearAztec: Location(Levels.DKIsles, "Photo of Isles Enemy: Near Aztec", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_FungiCannon1: Location(Levels.DKIsles, "Photo of Isles Enemy: Fungi Cannon (1)", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_PineappleCage1: Location(Levels.DKIsles, "Photo of Isles Enemy: Pineapple Cage (1)", Items.PhotoZingerCharger, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_LowerFactoryPath0: Location(Levels.DKIsles, "Photo of Isles Enemy: Lower Factory Path (0)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    Locations.KremKap_IslesMainEnemy_LowerFactoryPath1: Location(Levels.DKIsles, "Photo of Isles Enemy: Lower Factory Path (1)", Items.PhotoZingerLime, Types.EnemyPhoto, Kongs.any, []),
    # Half Medals
    Locations.JapesDonkeyHalfMedal: Location(Levels.JungleJapes, "Japes Donkey Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.donkey),
    Locations.JapesDiddyHalfMedal: Location(Levels.JungleJapes, "Japes Diddy Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.diddy),
    Locations.JapesLankyHalfMedal: Location(Levels.JungleJapes, "Japes Lanky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.lanky),
    Locations.JapesTinyHalfMedal: Location(Levels.JungleJapes, "Japes Tiny Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.tiny),
    Locations.JapesChunkyHalfMedal: Location(Levels.JungleJapes, "Japes Chunky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.chunky),
    Locations.AztecDonkeyHalfMedal: Location(Levels.AngryAztec, "Aztec Donkey Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.donkey),
    Locations.AztecDiddyHalfMedal: Location(Levels.AngryAztec, "Aztec Diddy Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.diddy),
    Locations.AztecLankyHalfMedal: Location(Levels.AngryAztec, "Aztec Lanky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.lanky),
    Locations.AztecTinyHalfMedal: Location(Levels.AngryAztec, "Aztec Tiny Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.tiny),
    Locations.AztecChunkyHalfMedal: Location(Levels.AngryAztec, "Aztec Chunky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.chunky),
    Locations.FactoryDonkeyHalfMedal: Location(Levels.FranticFactory, "Factory Donkey Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.donkey),
    Locations.FactoryDiddyHalfMedal: Location(Levels.FranticFactory, "Factory Diddy Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.diddy),
    Locations.FactoryLankyHalfMedal: Location(Levels.FranticFactory, "Factory Lanky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.lanky),
    Locations.FactoryTinyHalfMedal: Location(Levels.FranticFactory, "Factory Tiny Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.tiny),
    Locations.FactoryChunkyHalfMedal: Location(Levels.FranticFactory, "Factory Chunky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.chunky),
    Locations.GalleonDonkeyHalfMedal: Location(Levels.GloomyGalleon, "Galleon Donkey Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.donkey),
    Locations.GalleonDiddyHalfMedal: Location(Levels.GloomyGalleon, "Galleon Diddy Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.diddy),
    Locations.GalleonLankyHalfMedal: Location(Levels.GloomyGalleon, "Galleon Lanky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.lanky),
    Locations.GalleonTinyHalfMedal: Location(Levels.GloomyGalleon, "Galleon Tiny Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.tiny),
    Locations.GalleonChunkyHalfMedal: Location(Levels.GloomyGalleon, "Galleon Chunky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.chunky),
    Locations.ForestDonkeyHalfMedal: Location(Levels.FungiForest, "Forest Donkey Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.donkey),
    Locations.ForestDiddyHalfMedal: Location(Levels.FungiForest, "Forest Diddy Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.diddy),
    Locations.ForestLankyHalfMedal: Location(Levels.FungiForest, "Forest Lanky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.lanky),
    Locations.ForestTinyHalfMedal: Location(Levels.FungiForest, "Forest Tiny Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.tiny),
    Locations.ForestChunkyHalfMedal: Location(Levels.FungiForest, "Forest Chunky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.chunky),
    Locations.CavesDonkeyHalfMedal: Location(Levels.CrystalCaves, "Caves Donkey Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.donkey),
    Locations.CavesDiddyHalfMedal: Location(Levels.CrystalCaves, "Caves Diddy Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.diddy),
    Locations.CavesLankyHalfMedal: Location(Levels.CrystalCaves, "Caves Lanky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.lanky),
    Locations.CavesTinyHalfMedal: Location(Levels.CrystalCaves, "Caves Tiny Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.tiny),
    Locations.CavesChunkyHalfMedal: Location(Levels.CrystalCaves, "Caves Chunky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.chunky),
    Locations.CastleDonkeyHalfMedal: Location(Levels.CreepyCastle, "Castle Donkey Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.donkey),
    Locations.CastleDiddyHalfMedal: Location(Levels.CreepyCastle, "Castle Diddy Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.diddy),
    Locations.CastleLankyHalfMedal: Location(Levels.CreepyCastle, "Castle Lanky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.lanky),
    Locations.CastleTinyHalfMedal: Location(Levels.CreepyCastle, "Castle Tiny Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.tiny),
    Locations.CastleChunkyHalfMedal: Location(Levels.CreepyCastle, "Castle Chunky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.chunky),
    Locations.IslesDonkeyHalfMedal: Location(Levels.DKIsles, "Isles Donkey Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.donkey),
    Locations.IslesDiddyHalfMedal: Location(Levels.DKIsles, "Isles Diddy Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.diddy),
    Locations.IslesLankyHalfMedal: Location(Levels.DKIsles, "Isles Lanky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.lanky),
    Locations.IslesTinyHalfMedal: Location(Levels.DKIsles, "Isles Tiny Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.tiny),
    Locations.IslesChunkyHalfMedal: Location(Levels.DKIsles, "Isles Chunky Half-Medal", Items.HalfMedal, Types.HalfMedal, Kongs.chunky),
}

TrainingBarrelLocations = {
    Locations.IslesSwimTrainingBarrel,
    Locations.IslesVinesTrainingBarrel,
    Locations.IslesBarrelsTrainingBarrel,
    Locations.IslesOrangesTrainingBarrel,
}
DonkeyMoveLocations = {
    Locations.BaboonBlast,
    Locations.StrongKong,
    Locations.GorillaGrab,
    Locations.CoconutGun,
    Locations.Bongos,
    Locations.DonkeyGalleonPotion,
    Locations.DonkeyForestPotion,
    Locations.DonkeyCavesPotion,
    Locations.DonkeyCastlePotion,
    Locations.DonkeyAztecGun,
    Locations.DonkeyFactoryGun,
    Locations.DonkeyGalleonGun,
    Locations.DonkeyForestGun,
    Locations.DonkeyCavesGun,
    Locations.DonkeyCastleGun,
    Locations.DonkeyFactoryInstrument,
    Locations.DonkeyGalleonInstrument,
    Locations.DonkeyCavesInstrument,
    Locations.DonkeyCastleInstrument,
    Locations.DonkeyIslesPotion,
}
DiddyMoveLocations = {
    Locations.ChimpyCharge,
    Locations.RocketbarrelBoost,
    Locations.SimianSpring,
    Locations.PeanutGun,
    Locations.Guitar,
    Locations.DiddyGalleonPotion,
    Locations.DiddyForestPotion,
    Locations.DiddyCavesPotion,
    Locations.DiddyCastlePotion,
    Locations.DiddyAztecGun,
    Locations.DiddyFactoryGun,
    Locations.DiddyGalleonGun,
    Locations.DiddyForestGun,
    Locations.DiddyCavesGun,
    Locations.DiddyCastleGun,
    Locations.DiddyFactoryInstrument,
    Locations.DiddyGalleonInstrument,
    Locations.DiddyCavesInstrument,
    Locations.DiddyCastleInstrument,
    Locations.DiddyIslesPotion,
}
LankyMoveLocations = {
    Locations.Orangstand,
    Locations.BaboonBalloon,
    Locations.OrangstandSprint,
    Locations.GrapeGun,
    Locations.Trombone,
    Locations.LankyAztecPotion,
    Locations.LankyGalleonPotion,
    Locations.LankyForestPotion,
    Locations.LankyCastlePotion,
    Locations.LankyAztecGun,
    Locations.LankyFactoryGun,
    Locations.LankyGalleonGun,
    Locations.LankyForestGun,
    Locations.LankyCavesGun,
    Locations.LankyCastleGun,
    Locations.LankyFactoryInstrument,
    Locations.LankyGalleonInstrument,
    Locations.LankyCavesInstrument,
    Locations.LankyCastleInstrument,
    Locations.LankyIslesPotion,
}
TinyMoveLocations = {
    Locations.MiniMonkey,
    Locations.PonyTailTwirl,
    Locations.Monkeyport,
    Locations.FeatherGun,
    Locations.Saxophone,
    Locations.TinyAztecPotion,
    Locations.TinyGalleonPotion,
    Locations.TinyForestPotion,
    Locations.TinyCastlePotion,
    Locations.TinyAztecGun,
    Locations.TinyFactoryGun,
    Locations.TinyGalleonGun,
    Locations.TinyForestGun,
    Locations.TinyCavesGun,
    Locations.TinyCastleGun,
    Locations.TinyFactoryInstrument,
    Locations.TinyGalleonInstrument,
    Locations.TinyCavesInstrument,
    Locations.TinyCastleInstrument,
    Locations.TinyIslesPotion,
    Locations.CameraAndShockwave,
}
ChunkyMoveLocations = {
    Locations.HunkyChunky,
    Locations.PrimatePunch,
    Locations.GorillaGone,
    Locations.PineappleGun,
    Locations.Triangle,
    Locations.ChunkyAztecPotion,
    Locations.ChunkyGalleonPotion,
    Locations.ChunkyForestPotion,
    Locations.ChunkyCastlePotion,
    Locations.ChunkyAztecGun,
    Locations.ChunkyFactoryGun,
    Locations.ChunkyGalleonGun,
    Locations.ChunkyForestGun,
    Locations.ChunkyCavesGun,
    Locations.ChunkyCastleGun,
    Locations.ChunkyFactoryInstrument,
    Locations.ChunkyGalleonInstrument,
    Locations.ChunkyCavesInstrument,
    Locations.ChunkyCastleInstrument,
    Locations.ChunkyIslesPotion,
}
SharedMoveLocations = {
    Locations.SimianSlam,
    Locations.SuperSimianSlam,
    Locations.SuperDuperSimianSlam,
    Locations.SniperSight,
    Locations.HomingAmmo,
    Locations.AmmoBelt1,
    Locations.AmmoBelt2,
    Locations.MusicUpgrade1,
    Locations.ThirdMelon,
    Locations.MusicUpgrade2,
    Locations.SharedJapesPotion,
    Locations.SharedJapesGun,
    Locations.SharedAztecPotion,
    Locations.SharedAztecGun,
    Locations.SharedAztecInstrument,
    Locations.SharedFactoryPotion,
    Locations.SharedFactoryInstrument,
    Locations.SharedGalleonPotion,
    Locations.SharedGalleonGun,
    Locations.SharedCavesPotion,
    Locations.IslesSwimTrainingBarrel,
    Locations.IslesVinesTrainingBarrel,
    Locations.IslesBarrelsTrainingBarrel,
    Locations.IslesOrangesTrainingBarrel,
    Locations.CameraAndShockwave,
}
SharedShopLocations = {
    Locations.SimianSlam,
    Locations.SuperSimianSlam,
    Locations.SuperDuperSimianSlam,
    Locations.SniperSight,
    Locations.HomingAmmo,
    Locations.AmmoBelt1,
    Locations.AmmoBelt2,
    Locations.MusicUpgrade1,
    Locations.ThirdMelon,
    Locations.MusicUpgrade2,
    Locations.SharedJapesPotion,
    Locations.SharedJapesGun,
    Locations.SharedAztecPotion,
    Locations.SharedAztecGun,
    Locations.SharedAztecInstrument,
    Locations.SharedFactoryPotion,
    Locations.SharedFactoryInstrument,
    Locations.SharedGalleonPotion,
    Locations.SharedGalleonGun,
    Locations.SharedCavesPotion,
}
PreGivenLocations = {
    Locations.IslesFirstMove,
    Locations.IslesClimbing,
    Locations.PreGiven_Location00,
    Locations.PreGiven_Location01,
    Locations.PreGiven_Location02,
    Locations.PreGiven_Location03,
    Locations.PreGiven_Location04,
    Locations.PreGiven_Location05,
    Locations.PreGiven_Location06,
    Locations.PreGiven_Location07,
    Locations.PreGiven_Location08,
    Locations.PreGiven_Location09,
    Locations.PreGiven_Location10,
    Locations.PreGiven_Location11,
    Locations.PreGiven_Location12,
    Locations.PreGiven_Location13,
    Locations.PreGiven_Location14,
    Locations.PreGiven_Location15,
    Locations.PreGiven_Location16,
    Locations.PreGiven_Location17,
    Locations.PreGiven_Location18,
    Locations.PreGiven_Location19,
    Locations.PreGiven_Location20,
    Locations.PreGiven_Location21,
    Locations.PreGiven_Location22,
    Locations.PreGiven_Location23,
    Locations.PreGiven_Location24,
    Locations.PreGiven_Location25,
    Locations.PreGiven_Location26,
    Locations.PreGiven_Location27,
    Locations.PreGiven_Location28,
    Locations.PreGiven_Location29,
    Locations.PreGiven_Location30,
    Locations.PreGiven_Location31,
    Locations.PreGiven_Location32,
    Locations.PreGiven_Location33,
    Locations.PreGiven_Location34,
    Locations.PreGiven_Location35,
    Locations.PreGiven_Location36,
}

WrinklyHintLocations = {
    Locations.JapesDonkeyDoor,
    Locations.JapesDiddyDoor,
    Locations.JapesLankyDoor,
    Locations.JapesTinyDoor,
    Locations.JapesChunkyDoor,
    Locations.AztecDonkeyDoor,
    Locations.AztecDiddyDoor,
    Locations.AztecLankyDoor,
    Locations.AztecTinyDoor,
    Locations.AztecChunkyDoor,
    Locations.FactoryDonkeyDoor,
    Locations.FactoryDiddyDoor,
    Locations.FactoryLankyDoor,
    Locations.FactoryTinyDoor,
    Locations.FactoryChunkyDoor,
    Locations.GalleonDonkeyDoor,
    Locations.GalleonDiddyDoor,
    Locations.GalleonLankyDoor,
    Locations.GalleonTinyDoor,
    Locations.GalleonChunkyDoor,
    Locations.ForestDonkeyDoor,
    Locations.ForestDiddyDoor,
    Locations.ForestLankyDoor,
    Locations.ForestTinyDoor,
    Locations.ForestChunkyDoor,
    Locations.CavesDonkeyDoor,
    Locations.CavesDiddyDoor,
    Locations.CavesLankyDoor,
    Locations.CavesTinyDoor,
    Locations.CavesChunkyDoor,
    Locations.CastleDonkeyDoor,
    Locations.CastleDiddyDoor,
    Locations.CastleLankyDoor,
    Locations.CastleTinyDoor,
    Locations.CastleChunkyDoor,
}

ProgressiveHintLocations = {
    Locations.ProgressiveHint_01,
    Locations.ProgressiveHint_02,
    Locations.ProgressiveHint_03,
    Locations.ProgressiveHint_04,
    Locations.ProgressiveHint_05,
    Locations.ProgressiveHint_06,
    Locations.ProgressiveHint_07,
    Locations.ProgressiveHint_08,
    Locations.ProgressiveHint_09,
    Locations.ProgressiveHint_10,
    Locations.ProgressiveHint_11,
    Locations.ProgressiveHint_12,
    Locations.ProgressiveHint_13,
    Locations.ProgressiveHint_14,
    Locations.ProgressiveHint_15,
    Locations.ProgressiveHint_16,
    Locations.ProgressiveHint_17,
    Locations.ProgressiveHint_18,
    Locations.ProgressiveHint_19,
    Locations.ProgressiveHint_20,
    Locations.ProgressiveHint_21,
    Locations.ProgressiveHint_22,
    Locations.ProgressiveHint_23,
    Locations.ProgressiveHint_24,
    Locations.ProgressiveHint_25,
    Locations.ProgressiveHint_26,
    Locations.ProgressiveHint_27,
    Locations.ProgressiveHint_28,
    Locations.ProgressiveHint_29,
    Locations.ProgressiveHint_30,
    Locations.ProgressiveHint_31,
    Locations.ProgressiveHint_32,
    Locations.ProgressiveHint_33,
    Locations.ProgressiveHint_34,
    Locations.ProgressiveHint_35,
}

# Dictionary to speed up lookups of related shop locations
ShopLocationReference = {}
ShopLocationReference[Levels.JungleJapes] = {}
ShopLocationReference[Levels.JungleJapes][VendorType.Cranky] = [
    Locations.BaboonBlast,
    Locations.ChimpyCharge,
    Locations.Orangstand,
    Locations.MiniMonkey,
    Locations.HunkyChunky,
    Locations.SharedJapesPotion,
]
ShopLocationReference[Levels.JungleJapes][VendorType.Funky] = [
    Locations.CoconutGun,
    Locations.PeanutGun,
    Locations.GrapeGun,
    Locations.FeatherGun,
    Locations.PineappleGun,
    Locations.SharedJapesGun,
]
ShopLocationReference[Levels.AngryAztec] = {}
ShopLocationReference[Levels.AngryAztec][VendorType.Cranky] = [
    Locations.StrongKong,
    Locations.RocketbarrelBoost,
    Locations.LankyAztecPotion,
    Locations.TinyAztecPotion,
    Locations.ChunkyAztecPotion,
    Locations.SharedAztecPotion,
]
ShopLocationReference[Levels.AngryAztec][VendorType.Candy] = [
    Locations.Bongos,
    Locations.Guitar,
    Locations.Trombone,
    Locations.Saxophone,
    Locations.Triangle,
    Locations.SharedAztecInstrument,
]
ShopLocationReference[Levels.AngryAztec][VendorType.Funky] = [
    Locations.DonkeyAztecGun,
    Locations.DiddyAztecGun,
    Locations.LankyAztecGun,
    Locations.TinyAztecGun,
    Locations.ChunkyAztecGun,
    Locations.SharedAztecGun,
]
ShopLocationReference[Levels.FranticFactory] = {}
ShopLocationReference[Levels.FranticFactory][VendorType.Cranky] = [
    Locations.GorillaGrab,
    Locations.SimianSpring,
    Locations.BaboonBalloon,
    Locations.PonyTailTwirl,
    Locations.PrimatePunch,
    Locations.SharedFactoryPotion,
]
ShopLocationReference[Levels.FranticFactory][VendorType.Candy] = [
    Locations.DonkeyFactoryInstrument,
    Locations.DiddyFactoryInstrument,
    Locations.LankyFactoryInstrument,
    Locations.TinyFactoryInstrument,
    Locations.ChunkyFactoryInstrument,
    Locations.SharedFactoryInstrument,
]
ShopLocationReference[Levels.FranticFactory][VendorType.Funky] = [
    Locations.DonkeyFactoryGun,
    Locations.DiddyFactoryGun,
    Locations.LankyFactoryGun,
    Locations.TinyFactoryGun,
    Locations.ChunkyFactoryGun,
    Locations.AmmoBelt1,
]
ShopLocationReference[Levels.GloomyGalleon] = {}
ShopLocationReference[Levels.GloomyGalleon][VendorType.Cranky] = [
    Locations.DonkeyGalleonPotion,
    Locations.DiddyGalleonPotion,
    Locations.LankyGalleonPotion,
    Locations.TinyGalleonPotion,
    Locations.ChunkyGalleonPotion,
    Locations.SharedGalleonPotion,
]
ShopLocationReference[Levels.GloomyGalleon][VendorType.Candy] = [
    Locations.DonkeyGalleonInstrument,
    Locations.DiddyGalleonInstrument,
    Locations.LankyGalleonInstrument,
    Locations.TinyGalleonInstrument,
    Locations.ChunkyGalleonInstrument,
    Locations.MusicUpgrade1,
]
ShopLocationReference[Levels.GloomyGalleon][VendorType.Funky] = [
    Locations.DonkeyGalleonGun,
    Locations.DiddyGalleonGun,
    Locations.LankyGalleonGun,
    Locations.TinyGalleonGun,
    Locations.ChunkyGalleonGun,
    Locations.SharedGalleonGun,
]
ShopLocationReference[Levels.FungiForest] = {}
ShopLocationReference[Levels.FungiForest][VendorType.Cranky] = [
    Locations.DonkeyForestPotion,
    Locations.DiddyForestPotion,
    Locations.LankyForestPotion,
    Locations.TinyForestPotion,
    Locations.ChunkyForestPotion,
    Locations.SuperSimianSlam,
]
ShopLocationReference[Levels.FungiForest][VendorType.Funky] = [
    Locations.DonkeyForestGun,
    Locations.DiddyForestGun,
    Locations.LankyForestGun,
    Locations.TinyForestGun,
    Locations.ChunkyForestGun,
    Locations.HomingAmmo,
]
ShopLocationReference[Levels.CrystalCaves] = {}
ShopLocationReference[Levels.CrystalCaves][VendorType.Cranky] = [
    Locations.DonkeyCavesPotion,
    Locations.DiddyCavesPotion,
    Locations.OrangstandSprint,
    Locations.Monkeyport,
    Locations.GorillaGone,
    Locations.SharedCavesPotion,
]
ShopLocationReference[Levels.CrystalCaves][VendorType.Candy] = [
    Locations.DonkeyCavesInstrument,
    Locations.DiddyCavesInstrument,
    Locations.LankyCavesInstrument,
    Locations.TinyCavesInstrument,
    Locations.ChunkyCavesInstrument,
    Locations.ThirdMelon,
]
ShopLocationReference[Levels.CrystalCaves][VendorType.Funky] = [
    Locations.DonkeyCavesGun,
    Locations.DiddyCavesGun,
    Locations.LankyCavesGun,
    Locations.TinyCavesGun,
    Locations.ChunkyCavesGun,
    Locations.AmmoBelt2,
]
ShopLocationReference[Levels.CreepyCastle] = {}
ShopLocationReference[Levels.CreepyCastle][VendorType.Cranky] = [
    Locations.DonkeyCastlePotion,
    Locations.DiddyCastlePotion,
    Locations.LankyCastlePotion,
    Locations.TinyCastlePotion,
    Locations.ChunkyCastlePotion,
    Locations.SuperDuperSimianSlam,
]
ShopLocationReference[Levels.CreepyCastle][VendorType.Candy] = [
    Locations.DonkeyCastleInstrument,
    Locations.DiddyCastleInstrument,
    Locations.LankyCastleInstrument,
    Locations.TinyCastleInstrument,
    Locations.ChunkyCastleInstrument,
    Locations.MusicUpgrade2,
]
ShopLocationReference[Levels.CreepyCastle][VendorType.Funky] = [
    Locations.DonkeyCastleGun,
    Locations.DiddyCastleGun,
    Locations.LankyCastleGun,
    Locations.TinyCastleGun,
    Locations.ChunkyCastleGun,
    Locations.SniperSight,
]
ShopLocationReference[Levels.DKIsles] = {}
ShopLocationReference[Levels.DKIsles][VendorType.Cranky] = [Locations.DonkeyIslesPotion, Locations.DiddyIslesPotion, Locations.LankyIslesPotion, Locations.TinyIslesPotion, Locations.ChunkyIslesPotion, Locations.SimianSlam]
