"""Stores the data for the locations of banana coins."""

from __future__ import annotations

from typing import TYPE_CHECKING

from randomizer.Enums.Events import Events
from randomizer.Enums.Kongs import Kongs
from randomizer.Enums.Levels import Levels
from randomizer.Enums.Regions import Regions
from randomizer.Enums.Settings import DamageAmount
from randomizer.Enums.Switches import Switches
from randomizer.Enums.Time import Time
from randomizer.Enums.Maps import Maps
from randomizer.Enums.Collectibles import Collectibles

FACTORY_LEDGE_COINS_HEIGHT = 178.5
CAVES_WATER_HEIGHT = 20


class BananaCoinGroup:
    """Stores data for each group of coins."""

    def __init__(
        self,
        *,
        group=0,
        name="No Location",
        map_id=0,
        konglist=[],
        region=None,
        logic=None,
        vanilla=False,
        locations=[],
    ) -> None:
        """Initialize with given parameters."""
        self.group = group
        self.name = name
        self.map = map_id
        self.kongs = konglist
        self.locations = locations  # 4 numbers: {float scale, int x, y, z}
        self.region = region
        self.vanilla = vanilla
        if logic is None:
            self.logic = lambda _: True
        else:
            self.logic = logic
        self.placed_type = Collectibles.coin if vanilla else None


BananaCoinGroupList = {
    Levels.DKIsles: [
        BananaCoinGroup(
            group=1,
            map_id=Maps.TrainingGrounds,
            name="Around tree by training barrels",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TrainingGrounds,
            locations=[
                [1.0, 886, 58, 714],
                [1.0, 826, 58, 654],
                [1.0, 766, 58, 714],
                [1.0, 826, 58, 774],
            ],
        ),
        BananaCoinGroup(
            group=2,
            map_id=Maps.TrainingGrounds,
            name="On left treetop by Cranky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TrainingGrounds,
            logic=lambda l: l.climbing,
            locations=[
                [1.0, 889, 265, 1775],
                [1.0, 871, 265, 1790],
            ],
        ),
        BananaCoinGroup(
            group=3,
            map_id=Maps.TrainingGrounds,
            name="On right treetop by Cranky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TrainingGrounds,
            logic=lambda l: l.climbing,
            locations=[
                [1.0, 768, 272, 1683],
                [1.0, 758, 272, 1704],
            ],
        ),
        BananaCoinGroup(
            group=4,
            map_id=Maps.TrainingGrounds,
            name="On cliff top by Cranky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TrainingGrounds,
            logic=lambda l: l.climbing,
            locations=[
                [1.0, 1061, 253, 1853],
                [1.0, 1048, 248, 1822],
                [1.0, 1087, 251, 1824],
            ],
        ),
        BananaCoinGroup(
            group=5,
            map_id=Maps.TrainingGrounds,
            name="Bottom of the pool",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TrainingGrounds,
            logic=lambda l: l.swim,
            locations=[
                [1.2, 1853, -175, 1157],
                [1.2, 1863, -175, 1157],
                [1.2, 1853, -175, 1147],
            ],
        ),
        BananaCoinGroup(
            group=6,
            map_id=Maps.TrainingGrounds,
            name="Banana hoard",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TrainingGrounds,
            logic=lambda l: (l.can_use_vines or l.CanMoonkick()) and l.climbing,
            locations=[
                [1.0, 2547, 191, 1036],
                [1.0, 2497, 191, 986],
                [1.0, 2447, 191, 1036],
                [1.0, 2497, 191, 1086],
            ],
        ),
        BananaCoinGroup(
            group=8,
            map_id=Maps.Treehouse,
            name="In tire in DK's house",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Treehouse,
            locations=[
                [1.0, 258, 90, 552],
                [1.0, 263, 90, 552],
            ],
        ),
        BananaCoinGroup(
            group=9,
            map_id=Maps.Treehouse,
            name="By the door in DK's house",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Treehouse,
            locations=[
                [1.0, 404, 85, 441],
                [1.0, 390, 85, 452],
                [1.0, 407, 85, 461],
            ],
        ),
        BananaCoinGroup(
            group=10,
            map_id=Maps.Isles,
            name="On beach, in corner by path to Aztec lobby",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesMain,
            locations=[
                [1.0, 2499, 509, 1866],
                [1.0, 2504, 509, 1896],
                [1.0, 2482, 507, 1888],
            ],
        ),
        BananaCoinGroup(
            group=11,
            map_id=Maps.Isles,
            name="In water below waterfall",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesMain,
            logic=lambda l: l.swim,
            locations=[
                [1.2, 3036, 405, 1213],
                [1.2, 3000, 405, 1210],
                [1.2, 2964, 405, 1206],
                [1.2, 2928, 405, 1203],
                [1.2, 2892, 405, 1200],
            ],
        ),
        BananaCoinGroup(
            group=12,
            map_id=Maps.Isles,
            name="On top of tree between Japes Lobby and Fungi cannon",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesMain,
            logic=lambda l: l.climbing,
            locations=[
                [1.0, 3507, 695, 1375],
                [1.0, 3501, 698, 1395],
                [1.0, 3514, 697, 1393],
            ],
        ),
        BananaCoinGroup(
            group=13,
            map_id=Maps.Isles,
            name="On slope on path to Aztec lobby",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesHill,
            locations=[
                [1.0, 2522, 812, 1378],
                [1.0, 2546, 848, 1303],
                [1.0, 2584, 886, 1218],
            ],
        ),
        BananaCoinGroup(
            group=14,
            map_id=Maps.Isles,
            name="At the bottom of Aztec steps",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesMainUpper,
            locations=[
                [1.0, 3379, 901, 1343],
                [1.0, 3393, 901, 1374],
                [1.0, 3407, 901, 1405],
                [1.0, 3420, 901, 1435],
                [1.0, 3434, 901, 1466],
            ],
        ),
        BananaCoinGroup(
            group=15,
            map_id=Maps.Isles,
            name="In little alcove in DK's ear in front of Aztec lobby",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesMainUpper,
            locations=[
                [1.0, 3208, 1000, 1610],
                [1.0, 3206, 1000, 1592],
                [1.0, 3224, 1000, 1600],
            ],
        ),
        BananaCoinGroup(
            group=16,
            map_id=Maps.Isles,
            name="On top of Aztec lobby",
            konglist=[Kongs.diddy],
            region=Regions.AztecLobbyRoof,
            locations=[
                [1.0, 3515, 1170, 1800],
                [1.0, 3457, 1170, 1773],
                [1.0, 3501, 1170, 1668],
                [1.0, 3560, 1170, 1695],
            ],
        ),
        BananaCoinGroup(
            group=17,
            map_id=Maps.Isles,
            name="On top of K. Lumsy's prison",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.KremIsle,
            logic=lambda l: (
                (l.hasMoveSwitchsanity(Switches.IslesMonkeyport) and ((l.istiny and l.twirl) or (l.settings.damage_amount != DamageAmount.ohko and not l.IsHardFallDamage())))
                or ((l.settings.open_lobbies or Events.GalleonKeyTurnedIn in l.Events) and Events.IslesDiddyBarrelSpawn in l.Events and l.isdiddy and l.jetpack)
                or (
                    (l.settings.open_lobbies or Events.AztecKeyTurnedIn in l.Events)
                    and ((l.isdonkey and (not l.isKrushaAdjacent(Kongs.donkey)) and l.settings.damage_amount != DamageAmount.ohko and not l.IsHardFallDamage()) or (l.istiny and l.twirl))
                )
            )
            or l.CanMoonkick(),
            locations=[
                [1.0, 3343, 712, 3376],
                [1.0, 3318, 710, 3427],
                [1.0, 3375, 711, 3450],
                [1.0, 3397, 711, 3396],
            ],
        ),
        BananaCoinGroup(
            group=18,
            map_id=Maps.Isles,
            name="On top of Fungi lobby",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.CabinIsle,
            locations=[
                [1.0, 2492, 1621, 891],
                [1.0, 2508, 1621, 866],
                [1.0, 2524, 1621, 841],
            ],
        ),
        BananaCoinGroup(
            group=19,
            map_id=Maps.Isles,
            name="Hanging off the edge behind Fungi lobby",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CabinIsle,
            locations=[
                [1.0, 2413, 1482, 785],
                [1.0, 2400, 1482, 805],
                [1.0, 2387, 1482, 825],
            ],
        ),
        BananaCoinGroup(
            group=20,
            map_id=Maps.Isles,
            name="Around palm tree on lone fairy island",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesMain,
            locations=[
                [1.0, 1104, 500, 1418],
                [1.0, 1090, 500, 1374],
                [1.0, 1042, 500, 1374],
                [1.0, 1028, 500, 1418],
                [1.0, 1066, 500, 1446],
            ],
        ),
        BananaCoinGroup(
            group=21,
            map_id=Maps.Isles,
            name="By main warps",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesMain,
            locations=[
                [1.0, 2803, 500, 2367],
                [1.0, 2792, 500, 2334],
                [1.0, 2756, 500, 2334],
                [1.0, 2745, 500, 2367],
                [1.0, 2774, 500, 2388],
            ],
        ),
        BananaCoinGroup(
            group=22,
            map_id=Maps.Isles,
            name="Back of Fairy Isle",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesMain,
            locations=[
                [1.0, 695, 493, 2363],
                [1.0, 688, 492, 2349],
                [1.0, 681, 492, 2369],
            ],
        ),
        BananaCoinGroup(
            group=23,
            map_id=Maps.BananaFairyRoom,
            name="On lilypads in front pools",
            konglist=[Kongs.tiny],
            region=Regions.BananaFairyRoom,
            locations=[
                [1.0, 396, 60, 818],
                [1.0, 335, 60, 747],
                [1.0, 341, 60, 273],
                [1.0, 443, 60, 186],
            ],
        ),
        BananaCoinGroup(
            group=24,
            map_id=Maps.BananaFairyRoom,
            name="On lilypads in back pools",
            konglist=[Kongs.tiny],
            region=Regions.BananaFairyRoom,
            locations=[
                [1.0, 873, 60, 220],
                [1.0, 925, 60, 260],
                [1.0, 949, 60, 756],
                [1.0, 861, 60, 820],
            ],
        ),
        BananaCoinGroup(
            group=25,
            map_id=Maps.Isles,
            name="Back of Krem Isle, near propellors",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.KremIsle,
            locations=[
                [1.0, 1840, 600, 4110],
                [1.0, 1833, 600, 4079],
                [1.0, 1862, 600, 4106],
            ],
        ),
        BananaCoinGroup(
            group=26,
            map_id=Maps.Isles,
            name="Behind pillar on path to Factory lobby",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.KremIsleBeyondLift,
            locations=[
                [1.0, 2103, 845, 3756],
                [1.0, 2142, 828, 3821],
                [1.0, 2036, 847, 3774],
            ],
        ),
        BananaCoinGroup(
            group=27,
            map_id=Maps.Isles,
            name="Around pillar behind Factory lobby",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.KremIsleBeyondLift,
            locations=[
                [1.0, 2492, 1200, 3858],
                [1.0, 2458, 1200, 3851],
                [1.0, 2433, 1200, 3875],
                [1.0, 2457, 1200, 3908],
                [1.0, 2489, 1200, 3895],
            ],
        ),
        BananaCoinGroup(
            group=28,
            map_id=Maps.Isles,
            name="To the right of the staircase to Factory lobby",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.KremIsleBeyondLift,
            locations=[
                [1.0, 2419, 1200, 3410],
                [1.0, 2420, 1200, 3371],
                [1.0, 2406, 1200, 3391],
            ],
        ),
        BananaCoinGroup(
            group=29,
            map_id=Maps.Isles,
            name="Around Saxophone pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.KremIsleTopLevel,
            locations=[
                [1.0, 2389, 1720, 3866],
                [1.0, 2367, 1720, 3822],
                [1.0, 2326, 1720, 3850],
                [1.0, 2347, 1720, 3886],
            ],
        ),
        BananaCoinGroup(
            group=30,
            map_id=Maps.IslesSnideRoom,
            name="To the right of Snide's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesSnideRoom,
            locations=[
                [1.0, 369, 0, 490],
                [1.0, 344, 0, 486],
                [1.0, 356, 0, 462],
            ],
        ),
        BananaCoinGroup(
            group=31,
            map_id=Maps.IslesSnideRoom,
            name="To the left of Snide's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IslesSnideRoom,
            locations=[
                [1.0, 530, 0, 492],
                [1.0, 554, 0, 488],
                [1.0, 544, 0, 466],
            ],
        ),
        BananaCoinGroup(
            group=32,
            map_id=Maps.KLumsy,
            name="Around a candle to the left",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Prison,
            locations=[
                [1.0, 1522, 95, 815],
                [1.0, 1533, 95, 788],
                [1.0, 1499, 95, 771],
                [1.0, 1488, 95, 794],
            ],
        ),
        BananaCoinGroup(
            group=33,
            map_id=Maps.KLumsy,
            name="Around a candle to the right",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Prison,
            locations=[
                [1.0, 521, 95, 1192],
                [1.0, 482, 95, 1178],
                [1.0, 472, 95, 1217],
                [1.0, 514, 95, 1229],
            ],
        ),
        BananaCoinGroup(
            group=34,
            map_id=Maps.JungleJapesLobby,
            name="Next to Japes lobby entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesLobby,
            locations=[
                [1.0, 330, 0, 202],
                [1.0, 312, 0, 214],
                [1.0, 289, 0, 218],
            ],
        ),
        BananaCoinGroup(
            group=35,
            map_id=Maps.JungleJapesLobby,
            name="Next to a torch on the left",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesLobby,
            locations=[
                [1.0, 683, 0, 733],
                [1.0, 700, 0, 723],
                [1.0, 716, 0, 718],
            ],
        ),
        BananaCoinGroup(
            group=36,
            map_id=Maps.AngryAztecLobby,
            name="Around the front pillar",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecLobby,
            locations=[
                [1.0, 321, 0, 267],
                [1.0, 367, 0, 299],
                [1.0, 400, 0, 255],
                [1.0, 356, 0, 224],
            ],
        ),
        BananaCoinGroup(
            group=37,
            map_id=Maps.AngryAztecLobby,
            name="Around the back pillar",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecLobby,
            locations=[
                [1.0, 649, 0, 297],
                [1.0, 682, 0, 254],
                [1.0, 601, 0, 263],
                [1.0, 634, 0, 225],
            ],
        ),
        BananaCoinGroup(
            group=38,
            map_id=Maps.AngryAztecLobby,
            name="On the left stairs",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecLobby,
            locations=[
                [1.0, 757, 18, 563],
                [1.0, 736, 32, 584],
                [1.0, 712, 55, 627],
            ],
        ),
        BananaCoinGroup(
            group=39,
            map_id=Maps.AngryAztecLobby,
            name="In the back room (left)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecLobby,
            logic=lambda l: l.hasMoveSwitchsanity(Switches.IslesAztecLobbyFeather, False) or l.CanPhase(),
            locations=[
                [1.0, 1148, 0, 624],
                [1.0, 1126, 0, 632],
                [1.0, 1123, 0, 581],
            ],
        ),
        BananaCoinGroup(
            group=40,
            map_id=Maps.AngryAztecLobby,
            name="In the back room (right)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecLobby,
            logic=lambda l: l.hasMoveSwitchsanity(Switches.IslesAztecLobbyFeather, False) or l.CanPhase(),
            locations=[
                [1.0, 895, 0, 640],
                [1.0, 920, 0, 658],
                [1.0, 925, 0, 621],
            ],
        ),
        BananaCoinGroup(
            group=41,
            map_id=Maps.FranticFactoryLobby,
            name="In tag barrel corner",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryLobby,
            locations=[
                [1.0, 705, 0, 180],
                [1.0, 705, 0, 146],
                [1.0, 666, 0, 147],
            ],
        ),
        BananaCoinGroup(
            group=42,
            map_id=Maps.FranticFactoryLobby,
            name="On top level in corner",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryLobby,
            logic=lambda l: ((l.grab and l.donkey) or (l.monkey_maneuvers and (l.isdiddy or l.istiny))),
            locations=[
                [1.0, 632, 133, 571],
                [1.0, 608, 133, 562],
                [1.0, 623, 133, 550],
            ],
        ),
        BananaCoinGroup(
            group=43,
            map_id=Maps.FranticFactoryLobby,
            name="On top level at the end",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryLobby,
            logic=lambda l: ((l.grab and l.donkey) or (l.monkey_maneuvers and (l.isdiddy or l.istiny))),
            locations=[
                [1.0, 707, 133, 326],
                [1.0, 677, 133, 326],
                [1.0, 647, 133, 326],
            ],
        ),
        BananaCoinGroup(
            group=44,
            map_id=Maps.GloomyGalleonLobby,
            name="In water on left side",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonLobbyEntrance,
            locations=[
                [1.3, 995, 10, 778],
                [1.3, 990, 10, 771],
                [1.3, 984, 10, 785],
            ],
        ),
        BananaCoinGroup(
            group=45,
            map_id=Maps.GloomyGalleonLobby,
            name="On left side",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonLobby,
            locations=[
                [1.0, 944, 159, 1006],
                [1.0, 968, 159, 989],
                [1.0, 982, 159, 966],
            ],
        ),
        BananaCoinGroup(
            group=46,
            map_id=Maps.GloomyGalleonLobby,
            name="On right side",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonLobby,
            locations=[
                [1.0, 464, 119, 886],
                [1.0, 480, 119, 909],
                [1.0, 496, 119, 933],
            ],
        ),
        BananaCoinGroup(
            group=47,
            map_id=Maps.GloomyGalleonLobby,
            name="Around golden banana",
            konglist=[Kongs.tiny],
            region=Regions.GloomyGalleonLobby,
            logic=lambda l: l.swim and ((l.mini and l.twirl and l.tiny and l.CanSlamSwitch(Levels.GloomyGalleon, 2) and l.chunky) or l.CanPhaseswim()) and (not l.IsLavaWater() or l.Melons >= 3),
            locations=[
                [1.0, 832, 99, 202],
                [1.0, 815, 99, 253],
                [1.0, 873, 100, 245],
            ],
        ),
        BananaCoinGroup(
            group=48,
            map_id=Maps.FungiForestLobby,
            name="On wooden box",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestLobby,
            locations=[
                [1.0, 360, 46, 731],
                [1.0, 345, 46, 722],
                [1.0, 349, 45, 740],
            ],
        ),
        BananaCoinGroup(
            group=49,
            map_id=Maps.FungiForestLobby,
            name="On flour bag",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestLobby,
            locations=[
                [1.0, 182, 11, 677],
                [1.0, 169, 12, 661],
            ],
        ),
        BananaCoinGroup(
            group=50,
            map_id=Maps.FungiForestLobby,
            name="On metal box",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestLobby,
            locations=[
                [1.0, 443, 29, 633],
                [1.0, 435, 29, 630],
                [1.0, 436, 29, 640],
            ],
        ),
        BananaCoinGroup(
            group=51,
            map_id=Maps.FungiForestLobby,
            name="In corner by B. Locker",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestLobby,
            locations=[
                [1.0, 162, 0, 188],
                [1.0, 150, 0, 196],
                [1.0, 139, 0, 204],
            ],
        ),
        BananaCoinGroup(
            group=52,
            map_id=Maps.CrystalCavesLobby,
            name="Around blue switch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesLobby,
            locations=[
                [1.0, 957, 4, 351],
                [1.0, 917, 3, 325],
                [1.0, 895, 4, 372],
                [1.0, 932, 5, 386],
            ],
        ),
        BananaCoinGroup(
            group=53,
            map_id=Maps.CrystalCavesLobby,
            name="Beyond wall to lava GB",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesLobby,
            logic=lambda l: (l.punch and l.chunky) or l.CanPhase() or l.ledgeclip,
            locations=[
                [1.0, 502, 13, 396],
                [1.0, 521, 13, 398],
                [1.0, 536, 13, 400],
            ],
        ),
        BananaCoinGroup(
            group=54,
            map_id=Maps.CrystalCavesLobby,
            name="Around Lanky blueprint room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesLobby,
            logic=lambda l: (l.punch and l.chunky) or l.CanPhase() or l.ledgeclip,
            locations=[
                [1.0, 1472, 13, 497],
                [1.0, 1446, 13, 636],
                [1.0, 1561, 13, 737],
                [1.0, 1698, 13, 737],
                [1.0, 1806, 13, 638],
            ],
        ),
        BananaCoinGroup(
            group=55,
            map_id=Maps.CrystalCavesLobby,
            name="Near left ledge",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesLobby,
            locations=[
                [1.0, 1011, 118, 490],
                [1.0, 1054, 118, 481],
                [1.0, 1090, 118, 488],
            ],
        ),
        BananaCoinGroup(
            group=56,
            map_id=Maps.CreepyCastleLobby,
            name="Around entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleLobby,
            locations=[
                [1.0, 682, 60, 79],
                [1.0, 579, 60, 39],
                [1.0, 474, 60, 81],
                [1.0, 458, 60, 197],
                [1.0, 696, 60, 193],
            ],
        ),
        BananaCoinGroup(
            group=57,
            map_id=Maps.CreepyCastleLobby,
            name="In alcove to the right",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleLobby,
            locations=[
                [1.0, 506, 71, 593],
                [1.0, 470, 71, 659],
            ],
        ),
        BananaCoinGroup(
            group=58,
            map_id=Maps.CreepyCastleLobby,
            name="In alcove to the left",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleLobby,
            locations=[
                [1.0, 690, 71, 696],
                [1.0, 657, 71, 766],
            ],
        ),
        BananaCoinGroup(
            group=59,
            map_id=Maps.CreepyCastleLobby,
            name="On top of center piece",
            konglist=[Kongs.lanky],
            region=Regions.CreepyCastleLobby,
            logic=lambda l: l.chunky and l.barrels and l.islanky and l.balloon,
            locations=[
                [1.0, 629, 245, 650],
                [1.0, 528, 245, 650],
                [1.0, 580, 245, 738],
            ],
        ),
        BananaCoinGroup(
            group=60,
            map_id=Maps.CreepyCastleLobby,
            name="On wooden posts",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleLobby,
            locations=[
                [1.0, 671, 151, 1092],
                [1.0, 485, 150, 1095],
            ],
        ),
        BananaCoinGroup(
            group=61,
            map_id=Maps.HideoutHelmLobby,
            name="Around Gorilla Gone pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.HideoutHelmLobby,
            locations=[
                [1.0, 417, 191, 330],
                [1.0, 456, 191, 371],
                [1.0, 490, 191, 337],
                [1.0, 459, 191, 301],
            ],
        ),
        BananaCoinGroup(
            group=62,
            map_id=Maps.TrainingGrounds,
            name="Rear Tunnel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TrainingGrounds,
            locations=[
                [1.0, 1252.088623046875, 37.38066482543945, 2169.115234375],
                [1.0, 1175.427490234375, 36.610477447509766, 2199.226318359375],
                [1.0, 1251.8765869140625, 37.47738265991211, 2235.137451171875],
            ],
            vanilla=True,
        ),
    ],
    Levels.JungleJapes: [
        BananaCoinGroup(
            group=1,
            map_id=Maps.JungleJapes,
            name="Next to the DK portal",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 802, 280, 1004],
                [1.0, 818, 280, 992],
                [1.0, 816, 280, 1013],
            ],
        ),
        BananaCoinGroup(
            group=2,
            map_id=Maps.JungleJapes,
            name="Next to the bush to the right",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 1114, 285, 638],
                [1.0, 1095, 284, 631],
                [1.0, 1117, 282, 610],
            ],
        ),
        BananaCoinGroup(
            group=3,
            map_id=Maps.JungleJapes,
            name="Next to the bush to the left",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 1006, 280, 270],
                [1.0, 1024, 280, 262],
                [1.0, 1021, 280, 291],
            ],
        ),
        BananaCoinGroup(
            group=4,
            map_id=Maps.JungleJapes,
            name="In the first tunnel to the left",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 2020, 280, 372],
                [1.0, 2000, 280, 341],
                [1.0, 1973, 280, 311],
            ],
        ),
        BananaCoinGroup(
            group=5,
            map_id=Maps.JungleJapes,
            name="In the Diddy GB cave room around a torch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondPeanutGate,
            locations=[
                [1.0, 2369, 280, 681],
                [1.0, 2367, 280, 650],
                [1.0, 2342, 280, 644],
            ],
        ),
        BananaCoinGroup(
            group=6,
            map_id=Maps.JungleJapes,
            name="In the first tunnel past peanut gate to the left",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 1683, 280, 950],
                [1.0, 1720, 280, 917],
                [1.0, 1771, 280, 894],
            ],
        ),
        BananaCoinGroup(
            group=7,
            map_id=Maps.JungleJapes,
            name="Around far W1",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 1510, 280, 1235],
                [1.0, 1482, 280, 1187],
                [1.0, 1544, 280, 1194],
            ],
        ),
        BananaCoinGroup(
            group=8,
            map_id=Maps.JungleJapes,
            name="Around low W2",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 1514, 280, 1447],
                [1.0, 1464, 280, 1446],
                [1.0, 1492, 280, 1393],
            ],
        ),
        BananaCoinGroup(
            group=9,
            map_id=Maps.JungleJapes,
            name="Against wall near W1 and W2",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 1381, 280, 1323],
                [1.0, 1346, 280, 1334],
                [1.0, 1316, 280, 1351],
            ],
        ),
        BananaCoinGroup(
            group=10,
            map_id=Maps.JungleJapes,
            name="Against wall between W1W2 and underground",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 1856, 288, 1334],
                [1.0, 1807, 293, 1331],
                [1.0, 1755, 286, 1325],
            ],
        ),
        BananaCoinGroup(
            group=11,
            map_id=Maps.JungleJapes,
            name="Around Chunky underground entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 2227, 280, 1229],
                [1.0, 2345, 280, 1101],
                [1.0, 2447, 280, 1099],
                [1.0, 2504, 280, 1357],
            ],
        ),
        BananaCoinGroup(
            group=12,
            map_id=Maps.JungleJapes,
            name="In corner behind W3 beehive side",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 2546, 283, 1581],
                [1.0, 2563, 292, 1520],
                [1.0, 2510, 284, 1477],
            ],
        ),
        BananaCoinGroup(
            group=13,
            map_id=Maps.JungleJapes,
            name="Against wall across from DK's kasplat alcove",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate1,
            locations=[
                [1.0, 2757, 334, 1923],
                [1.0, 2798, 334, 1935],
                [1.0, 2861, 334, 1947],
            ],
        ),
        BananaCoinGroup(
            group=14,
            map_id=Maps.JungleJapes,
            name="Against wall across from Tiny's kasplat alcove",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate1,
            locations=[
                [1.0, 3058, 438, 2484],
                [1.0, 3047, 433, 2428],
                [1.0, 3037, 427, 2367],
            ],
        ),
        BananaCoinGroup(
            group=15,
            map_id=Maps.JungleJapes,
            name="In front of each feather switch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate1,
            locations=[
                [1.0, 2693, 539, 2957],
                [1.0, 2655, 539, 2867],
            ],
        ),
        BananaCoinGroup(
            group=16,
            map_id=Maps.JungleJapes,
            name="Around W5 in beehive area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 2554, 539, 2959],
                [1.0, 2592, 539, 2979],
                [1.0, 2606, 539, 2935],
                [1.0, 2568, 539, 2924],
            ],
        ),
        BananaCoinGroup(
            group=17,
            map_id=Maps.JungleJapes,
            name="On left log in beehive area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 2659, 564, 3280],
                [1.0, 2630, 563, 3364],
                [1.0, 2612, 563, 3445],
                [1.0, 2545, 563, 3479],
                [1.0, 2470, 563, 3510],
            ],
        ),
        BananaCoinGroup(
            group=18,
            map_id=Maps.JungleJapes,
            name="On right log in beehive area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 2498, 564, 2785],
                [1.0, 2446, 564, 2738],
                [1.0, 2389, 563, 2744],
                [1.0, 2321, 563, 2760],
                [1.0, 2264, 564, 2774],
            ],
        ),
        BananaCoinGroup(
            group=19,
            map_id=Maps.JungleJapes,
            name="To the right of the beehive",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 1880, 573, 2780],
                [1.0, 1908, 573, 2768],
                [1.0, 1908, 573, 2790],
            ],
        ),
        BananaCoinGroup(
            group=20,
            map_id=Maps.JungleJapes,
            name="To the left of the beehive",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 1812, 573, 2937],
                [1.0, 1783, 573, 2956],
                [1.0, 1822, 573, 2960],
            ],
        ),
        BananaCoinGroup(
            group=21,
            map_id=Maps.JungleJapes,
            name="Beehive area around front big tree",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 2429, 551, 3031],
                [1.0, 2419, 551, 3085],
                [1.0, 2377, 551, 3072],
                [1.0, 2383, 551, 3030],
            ],
        ),
        BananaCoinGroup(
            group=22,
            map_id=Maps.JungleJapes,
            name="Beehive area around left big tree",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 2440, 551, 3191],
                [1.0, 2477, 551, 3202],
                [1.0, 2461, 551, 3243],
                [1.0, 2427, 551, 3229],
            ],
        ),
        BananaCoinGroup(
            group=23,
            map_id=Maps.JungleJapes,
            name="Beehive area around back big tree",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 2261, 551, 3266],
                [1.0, 2228, 551, 3302],
                [1.0, 2192, 551, 3267],
                [1.0, 2225, 551, 3234],
            ],
        ),
        BananaCoinGroup(
            group=24,
            map_id=Maps.JungleJapes,
            name="Beehive area around right big tree",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 2234, 551, 3055],
                [1.0, 2258, 551, 3019],
                [1.0, 2290, 551, 3037],
                [1.0, 2272, 551, 3075],
            ],
        ),
        BananaCoinGroup(
            group=25,
            map_id=Maps.JungleJapes,
            name="On riverbed near tag barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            logic=lambda l: l.swim,
            locations=[
                [1.3, 1908, 163, 1625],
                [1.3, 1733, 163, 1603],
                [1.3, 1572, 163, 1627],
            ],
        ),
        BananaCoinGroup(
            group=26,
            map_id=Maps.JungleJapes,
            name="In front of Tiny's caged GB",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesMain,
            locations=[
                [1.0, 1379, 280, 1913],
                [1.0, 1360, 280, 1902],
                [1.0, 1343, 280, 1888],
            ],
        ),
        BananaCoinGroup(
            group=27,
            map_id=Maps.JungleJapes,
            name="Against wall to the right near Painting room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesMain,
            locations=[
                [1.0, 960, 288, 1624],
                [1.0, 930, 288, 1661],
                [1.0, 879, 288, 1686],
            ],
        ),
        BananaCoinGroup(
            group=28,
            map_id=Maps.JungleJapes,
            name="In corner behind W3 painting room side",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesMain,
            locations=[
                [1.0, 981, 288, 2254],
                [1.0, 1056, 288, 2238],
                [1.0, 1035, 288, 2163],
            ],
        ),
        BananaCoinGroup(
            group=29,
            map_id=Maps.JungleJapes,
            name="Around middle tree tag barrel side",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 1903, 287, 1516],
                [1.0, 1937, 289, 1472],
                [1.0, 1893, 282, 1448],
                [1.0, 1858, 285, 1493],
            ],
        ),
        BananaCoinGroup(
            group=30,
            map_id=Maps.JungleJapes,
            name="In front of Diddy's caged GB",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesHillTop,
            locations=[
                [1.0, 2230, 520, 2112],
                [1.0, 2237, 520, 2138],
                [1.0, 2244, 520, 2161],
            ],
        ),
        BananaCoinGroup(
            group=31,
            map_id=Maps.JungleJapes,
            name="Around tree in front of Snide's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesHillTop,
            locations=[
                [1.0, 2156, 680, 2298],
                [1.0, 2124, 680, 2314],
                [1.0, 2138, 680, 2342],
                [1.0, 2164, 680, 2332],
            ],
        ),
        BananaCoinGroup(
            group=32,
            map_id=Maps.JungleJapes,
            name="In front of Chunky's caged GB",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesHillTop,
            locations=[
                [1.0, 2302, 680, 2293],
                [1.0, 2286, 680, 2268],
                [1.0, 2272, 680, 2245],
            ],
        ),
        BananaCoinGroup(
            group=33,
            map_id=Maps.JungleJapes,
            name="On slope between Snide's and mountain",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesHillTop,
            locations=[
                [1.0, 1958, 687, 2545],
                [1.0, 1923, 706, 2546],
                [1.0, 1887, 727, 2549],
                [1.0, 1843, 753, 2553],
                [1.0, 1795, 781, 2556],
            ],
        ),
        BananaCoinGroup(
            group=34,
            map_id=Maps.JungleJapes,
            name="Behind melon crate",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesHillTop,
            locations=[
                [1.0, 1611, 790, 2646],
                [1.0, 1577, 790, 2645],
                [1.0, 1543, 790, 2638],
            ],
        ),
        BananaCoinGroup(
            group=35,
            map_id=Maps.JungleJapes,
            name="On slope between mountain and Diddy cage",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesHillTop,
            locations=[
                [1.0, 1383, 797, 2542],
                [1.0, 1327, 821, 2541],
                [1.0, 1267, 845, 2538],
            ],
        ),
        BananaCoinGroup(
            group=36,
            map_id=Maps.JungleJapes,
            name="Around tree in front of Diddy's cage",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesHillTop,
            locations=[
                [1.0, 1007, 852, 2367],
                [1.0, 972, 852, 2376],
                [1.0, 987, 852, 2416],
                [1.0, 1021, 852, 2403],
            ],
        ),
        BananaCoinGroup(
            group=37,
            map_id=Maps.JungleJapes,
            name="In front of T&S portal near Diddy kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesTnSAlcove,
            locations=[
                [1.0, 740, 538, 2361],
                [1.0, 756, 538, 2353],
                [1.0, 742, 538, 2340],
            ],
        ),
        BananaCoinGroup(
            group=38,
            map_id=Maps.JungleJapes,
            name="In pit near Diddy kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1099, 230, 2603],
                [1.0, 1149, 230, 2615],
                [1.0, 1230, 230, 2610],
            ],
        ),
        BananaCoinGroup(
            group=39,
            map_id=Maps.JungleJapes,
            name="Around tunnel W4",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1575, 280, 2244],
                [1.0, 1626, 280, 2239],
                [1.0, 1594, 280, 2185],
            ],
        ),
        BananaCoinGroup(
            group=40,
            map_id=Maps.JungleJapes,
            name="Around small torch in W4 tunnel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1530, 280, 2572],
                [1.0, 1497, 280, 2576],
                [1.0, 1500, 280, 2601],
                [1.0, 1528, 277, 2614],
            ],
        ),
        BananaCoinGroup(
            group=41,
            map_id=Maps.JungleJapes,
            name="Around small torch near Lanky kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1738, 210, 3086],
                [1.0, 1711, 210, 3062],
                [1.0, 1734, 210, 3040],
            ],
        ),
        BananaCoinGroup(
            group=42,
            map_id=Maps.JungleJapes,
            name="Against right wall in Lanky kasplat room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 2013, 250, 3065],
                [1.0, 1974, 250, 3036],
                [1.0, 1935, 250, 2982],
            ],
        ),
        BananaCoinGroup(
            group=43,
            map_id=Maps.JungleJapes,
            name="Against left wall in Lanky kasplat room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1848, 250, 2690],
                [1.0, 1832, 250, 2751],
                [1.0, 1849, 250, 2811],
            ],
        ),
        BananaCoinGroup(
            group=44,
            map_id=Maps.JungleJapes,
            name="On top of right slippery slope",
            konglist=[Kongs.lanky],
            region=Regions.JapesUselessSlope,
            locations=[
                [1.0, 2214, 338, 3280],
                [1.0, 2247, 338, 3268],
                [1.0, 2275, 338, 3244],
            ],
        ),
        BananaCoinGroup(
            group=45,
            map_id=Maps.JungleJapes,
            name="Around Tiny hut",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 2218, 280, 3989],
                [1.0, 2267, 280, 3922],
                [1.0, 2270, 280, 3856],
                [1.0, 2207, 280, 3806],
            ],
        ),
        BananaCoinGroup(
            group=46,
            map_id=Maps.JungleJapes,
            name="Around Lanky hut",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 2127, 280, 4266],
                [1.0, 2106, 280, 4331],
                [1.0, 2040, 280, 4391],
                [1.0, 1956, 280, 4399],
            ],
        ),
        BananaCoinGroup(
            group=47,
            map_id=Maps.JungleJapes,
            name="Around Donkey hut",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1440, 280, 4409],
                [1.0, 1353, 280, 4400],
                [1.0, 1296, 280, 4353],
                [1.0, 1267, 280, 4270],
            ],
        ),
        BananaCoinGroup(
            group=48,
            map_id=Maps.JungleJapes,
            name="Around Diddy hut",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1177, 280, 3990],
                [1.0, 1127, 280, 3938],
                [1.0, 1130, 280, 3873],
                [1.0, 1181, 280, 3804],
            ],
        ),
        BananaCoinGroup(
            group=49,
            map_id=Maps.JungleJapes,
            name="Around Cranky W4",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1672, 280, 3868],
                [1.0, 1698, 280, 3919],
                [1.0, 1723, 280, 3875],
            ],
        ),
        BananaCoinGroup(
            group=50,
            map_id=Maps.JungleJapes,
            name="Around far tree in Cranky area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1780, 280, 4215],
                [1.0, 1799, 280, 4238],
                [1.0, 1820, 280, 4215],
                [1.0, 1802, 280, 4202],
            ],
        ),
        BananaCoinGroup(
            group=51,
            map_id=Maps.JungleJapes,
            name="In front of Rambi gate",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 1259, 280, 3392],
                [1.0, 1241, 280, 3363],
                [1.0, 1224, 280, 3326],
                [1.0, 1200, 280, 3291],
            ],
        ),
        BananaCoinGroup(
            group=52,
            map_id=Maps.JungleJapes,
            name="Around melon crate in boulder room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondRambiGate,
            locations=[
                [1.0, 744, 280, 3773],
                [1.0, 738, 280, 3726],
                [1.0, 701, 280, 3731],
            ],
        ),
        BananaCoinGroup(
            group=53,
            map_id=Maps.JungleJapes,
            name="Around torch near fairy room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondRambiGate,
            locations=[
                [1.0, 1113, 278, 3300],
                [1.0, 1133, 280, 3321],
                [1.0, 1150, 280, 3310],
            ],
        ),
        BananaCoinGroup(
            group=54,
            map_id=Maps.JungleJapes,
            name="Around torch in fairy room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondRambiGate,
            locations=[
                [1.0, 682, 240, 3125],
                [1.0, 719, 240, 3104],
                [1.0, 725, 243, 3141],
            ],
        ),
        BananaCoinGroup(
            group=55,
            map_id=Maps.JungleJapes,
            name="Left side of fairy room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondRambiGate,
            locations=[
                [1.0, 528, 236, 3168],
                [1.0, 489, 222, 3146],
                [1.0, 451, 208, 3131],
                [1.0, 407, 192, 3121],
            ],
        ),
        BananaCoinGroup(
            group=56,
            map_id=Maps.JungleJapes,
            name="Right side of fairy room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondRambiGate,
            locations=[
                [1.0, 772, 235, 2869],
                [1.0, 758, 218, 2834],
                [1.0, 742, 198, 2798],
            ],
        ),
        BananaCoinGroup(
            group=57,
            map_id=Maps.JungleJapes,
            name="Around torch in the middle of fairy room on water surface",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondRambiGate,
            locations=[
                [1.0, 461, 192, 2823],
                [1.0, 452, 192, 2782],
                [1.0, 419, 192, 2787],
                [1.0, 418, 192, 2817],
            ],
        ),
        BananaCoinGroup(
            group=58,
            map_id=Maps.JapesUnderGround,
            name="Right side towards the end of the cave",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesCatacomb,
            locations=[
                [1.0, 727, 20, 412],
                [1.0, 739, 20, 388],
                [1.0, 749, 20, 404],
            ],
        ),
        BananaCoinGroup(
            group=59,
            map_id=Maps.JapesUnderGround,
            name="Left side towards the end of the cave",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesCatacomb,
            locations=[
                [1.0, 879, 20, 411],
                [1.0, 899, 20, 408],
                [1.0, 890, 20, 393],
            ],
        ),
        BananaCoinGroup(
            group=60,
            map_id=Maps.JapesUnderGround,
            name="Around center platform",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesCatacomb,
            locations=[
                [1.0, 456, 20, 871],
                [1.0, 422, 20, 794],
                [1.0, 501, 20, 781],
            ],
        ),
        BananaCoinGroup(
            group=61,
            map_id=Maps.JapesLankyCave,
            name="Next to entrance",
            konglist=[Kongs.lanky, Kongs.tiny],
            region=Regions.JapesLankyCave,
            locations=[
                [1.0, 319, 25, 99],
                [1.0, 295, 25, 73],
                [1.0, 268, 25, 44],
            ],
        ),
        BananaCoinGroup(
            group=62,
            map_id=Maps.JapesMountain,
            name="On right minecart near entrance",
            konglist=[Kongs.diddy],
            region=Regions.Mine,
            locations=[
                [1.0, 533, 140, 532],
                [1.0, 493, 140, 529],
                [1.0, 457, 140, 531],
            ],
        ),
        BananaCoinGroup(
            group=63,
            map_id=Maps.JapesMountain,
            name="In treadmills room front left corner",
            konglist=[Kongs.diddy],
            region=Regions.Mine,
            logic=lambda l: l.CanSlamSwitch(Levels.JungleJapes, 1) or l.CanPhase(),
            locations=[
                [1.0, 241, 100, 1238],
                [1.0, 257, 100, 1230],
            ],
        ),
        BananaCoinGroup(
            group=64,
            map_id=Maps.JapesMountain,
            name="On coal pile in second room",
            konglist=[Kongs.diddy],
            region=Regions.Mine,
            locations=[
                [1.0, 718, 134, 1237],
            ],
        ),
        BananaCoinGroup(
            group=65,
            map_id=Maps.JapesTinyHive,
            name="Behind Door 1",
            konglist=[Kongs.tiny],
            region=Regions.TinyHive,
            locations=[
                [1.8, 1173, 226, 1469],
                [1.8, 1171, 226, 1380],
            ],
        ),
        BananaCoinGroup(
            group=66,
            map_id=Maps.JapesTinyHive,
            name="Around final room",
            konglist=[Kongs.tiny],
            region=Regions.TinyHive,
            logic=lambda l: l.CanSlamSwitch(Levels.JungleJapes, 1) or l.CanPhase() or l.generalclips,
            locations=[
                [1.8, 2080, 254, 1704],
                [1.8, 2592, 254, 1530],
                [1.8, 2597, 254, 1320],
                [1.8, 2084, 254, 1146],
            ],
        ),
        BananaCoinGroup(
            group=67,
            map_id=Maps.JapesMountain,
            name="Behind a barrel near the HI-LO Machine",
            konglist=[Kongs.diddy],
            region=Regions.Mine,
            logic=lambda l: (l.charge and l.isdiddy) or l.CanPhase(),
            locations=[
                [0.9600000381469727, 496.6444396972656, 100.67699432373047, 1627.4833984375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=68,
            map_id=Maps.JapesMountain,
            name="Under the Conveyors",
            konglist=[Kongs.diddy],
            region=Regions.Mine,
            logic=lambda l: (l.CanSlamSwitch(Levels.JungleJapes, 1) and l.isdiddy) or l.CanPhase(),
            locations=[
                [1.0, 132.3656463623047, 143.6666717529297, 896.5993041992188],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=69,
            map_id=Maps.JapesMountain,
            name="On the bridge",
            konglist=[Kongs.diddy],
            region=Regions.Mine,
            logic=lambda l: l.peanut or l.monkey_maneuvers,
            locations=[
                [1.0, 498.7564392089844, 146.62681579589844, 198.3944091796875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=70,
            map_id=Maps.JapesMountain,
            name="On a coal stack",
            konglist=[Kongs.diddy],
            region=Regions.Mine,
            locations=[
                [1.0, 908.790283203125, 105.0, 887.046875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=71,
            map_id=Maps.JungleJapes,
            name="Near entrance portal",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 999.6126098632812, 280.0, 1012.6856079101562],
                [1.0, 1012.5659790039062, 280.0, 1009.3839721679688],
                [1.0, 1002.0214233398438, 280.0, 998.895263671875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=72,
            map_id=Maps.JungleJapes,
            name="Main Area Underwater (1)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            logic=lambda l: l.swim,
            locations=[
                [1.0, 1161.7161865234375, 160.0, 1444.52587890625],
                [1.0, 1180.0701904296875, 160.0, 1449.4384765625],
                [1.0, 1173.1038818359375, 160.0, 1425.298828125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=73,
            map_id=Maps.JungleJapes,
            name="Around the Baboon Blast Pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.chunky],
            region=Regions.JapesBlastPadPlatform,
            locations=[
                [1.0, 2433.09033203125, 530.0, 1123.9302978515625],
                [1.0, 2488.2314453125, 530.0, 1157.3780517578125],
                [1.0, 2437.709716796875, 530.0, 1183.588623046875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=74,
            map_id=Maps.JungleJapes,
            name="Start of first tunnel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            locations=[
                [1.0, 1632.14697265625, 280.0, 292.7493591308594],
                [1.0, 1651.8271484375, 280.0, 300.2619934082031],
                [1.0, 1632.5484619140625, 280.0, 309.4810485839844],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=75,
            map_id=Maps.JungleJapes,
            name="Diddy Cavern (1)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondPeanutGate,
            locations=[
                [1.0, 2409.144775390625, 285.0, 617.8091430664062],
                [1.0, 2471.943115234375, 280.0, 372.3658142089844],
                [1.0, 2350.5, 280.0, 515.3485717773438],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=76,
            map_id=Maps.JungleJapes,
            name="Fairy Pool",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondRambiGate,
            logic=lambda l: l.swim,
            locations=[
                [1.0, 424.1111755371094, 140.0, 2682.5078125],
                [1.0, 353.4961242675781, 140.0, 2737.464599609375],
                [1.0, 582.4089965820312, 140.0, 2689.986572265625],
                [1.0, 315.656982421875, 140.0, 2952.7177734375],
                [1.0, 305.4899597167969, 140.0, 2836.795166015625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=77,
            map_id=Maps.JungleJapes,
            name="Near Rambi Box",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            logic=lambda l: l.hasMoveSwitchsanity(Switches.JapesRambi, False) or l.CanPhase(),
            locations=[
                [1.0, 1703.703125, 280.0, 4452.66162109375],
                [1.0, 1654.2239990234375, 280.0, 4448.4892578125],
                [1.0, 1743.1768798828125, 280.0, 4453.4970703125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=78,
            map_id=Maps.JungleJapes,
            name="Diddy Cavern (2)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondPeanutGate,
            locations=[
                [1.0, 2608.202880859375, 280.0, 390.9048156738281],
                [1.0, 2667.58056640625, 280.0, 504.7973937988281],
                [1.0, 2618.726806640625, 280.0, 618.5264282226562],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=79,
            map_id=Maps.JungleJapes,
            name="Near Diddy Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 823.560791015625, 280.0, 2682.6923828125],
                [1.0, 883.4852905273438, 280.0, 2726.369384765625],
                [1.0, 954.63037109375, 280.0, 2728.579345703125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=80,
            map_id=Maps.JungleJapes,
            name="Near Tiny Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate1,
            locations=[
                [1.0, 3225.389892578125, 437.8333435058594, 2456.965087890625],
                [1.0, 3248.022216796875, 437.8333435058594, 2389.616455078125],
                [1.0, 3205.435791015625, 437.8333435058594, 2324.978759765625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=81,
            map_id=Maps.JungleJapes,
            name="In between the vines near Diddy Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            logic=lambda l: l.can_use_vines,
            locations=[
                [1.0, 1121.0938720703125, 318.16082763671875, 2603.72119140625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=82,
            map_id=Maps.JungleJapes,
            name="Near DK Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate1,
            locations=[
                [1.0, 2701.797607421875, 334.6666564941406, 2109.7900390625],
                [1.0, 2757.901123046875, 334.6666564941406, 2162.4404296875],
                [1.0, 2826.635498046875, 334.6666564941406, 2140.668701171875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=83,
            map_id=Maps.JungleJapes,
            name="Cannon Path",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesCannonPlatform,
            locations=[
                [1.0, 1222.9649658203125, 804.0, 2200.499755859375],
                [1.0, 1225.0323486328125, 780.0, 2188.296142578125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=84,
            map_id=Maps.JungleJapes,
            name="Near Lanky Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondCoconutGate2,
            locations=[
                [1.0, 2189.70703125, 250.6666717529297, 2835.56640625],
                [1.0, 2160.501708984375, 250.6666717529297, 2747.009521484375],
                [1.0, 2085.108642578125, 250.6666717529297, 2665.501220703125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=85,
            map_id=Maps.JungleJapes,
            name="Main Area Underwater (2)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            logic=lambda l: l.swim,
            locations=[
                [1.0, 2351.8994140625, 160.0, 1797.2802734375],
                [1.0, 2356.33642578125, 160.0, 1807.9783935546875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=86,
            map_id=Maps.JungleJapes,
            name="Main Area Underwater (3)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JungleJapesStart,
            logic=lambda l: l.swim,
            locations=[
                [1.0, 2242.110595703125, 160.0, 1989.25830078125],
                [1.0, 2256.636474609375, 160.0, 1965.8919677734375],
                [1.0, 2228.9140625, 160.0, 1961.3966064453125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=87,
            map_id=Maps.JungleJapes,
            name="Behind the stump",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesBeyondFeatherGate,
            locations=[
                [1.0, 2075.8359375, 539.3333129882812, 3584.9404296875],
                [1.0, 1980.7818603515625, 539.3333129882812, 3573.294189453125],
                [1.0, 1978.1563720703125, 539.3333129882812, 3488.201171875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=88,
            map_id=Maps.JungleJapes,
            name="On top of the mountain",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesTopOfMountain,
            locations=[
                [1.0, 1601.7271728515625, 990.0, 2377.057861328125],
                [1.0, 1666.3236083984375, 989.08349609375, 2413.665283203125],
                [1.0, 1653.80224609375, 989.3388061523438, 2481.564208984375],
                [1.0, 1593.6490478515625, 989.266845703125, 2491.558349609375],
                [1.0, 1553.7437744140625, 989.64111328125, 2435.263671875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=89,
            map_id=Maps.JungleJapes,
            name="Near Snide's HQ",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesHillTop,
            locations=[
                [1.0, 2279.154296875, 680.0, 2382.241943359375],
                [1.0, 2295.2626953125, 680.0, 2424.940673828125],
                [1.0, 2302.979248046875, 680.0, 2400.536376953125],
                [1.0, 2269.52783203125, 680.0, 2422.548583984375],
                [1.0, 2259.3857421875, 680.0, 2396.973876953125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=90,
            map_id=Maps.JapesTinyHive,
            name="Behind Door 2",
            konglist=[Kongs.tiny],
            region=Regions.TinyHive,
            logic=lambda l: ((l.CanSlamSwitch(Levels.JungleJapes, 1) and (l.saxophone or l.oranges)) or l.CanPhase() or l.generalclips) and l.istiny,
            locations=[
                [1.7199993133544922, 1425.05810546875, 233.0, 923.6173706054688],
                [1.7199993133544922, 1332.7794189453125, 233.0, 924.2100219726562],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=91,
            map_id=Maps.JapesLankyCave,
            name="On Pegs",
            konglist=[Kongs.lanky, Kongs.tiny],
            region=Regions.JapesLankyCave,
            locations=[
                [1.0, 369.7975769042969, 112.0, 299.9463195800781],
                [1.0000009536743164, 59.98158645629883, 112.0, 304.1812438964844],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=92,
            map_id=Maps.JapesUnderGround,
            name="Behind Entrance Cannon",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.JapesCatacomb,
            locations=[
                [1.0, 417.4634094238281, 20.0, 98.96123504638672],
                [1.0, 468.3440246582031, 20.0, 41.87655258178711],
                [1.0, 470.413330078125, 20.0, 155.2516632080078],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=93,
            map_id=Maps.JapesUnderGround,
            name="Near Underground Reward",
            konglist=[Kongs.chunky],
            region=Regions.JapesCatacomb,
            logic=lambda l: (l.can_use_vines and l.pineapple and l.ischunky) or l.CanPhase(),
            locations=[
                [1.0, 127.6156005859375, 20.0, 1017.9220581054688],
                [1.0, 79.51215362548828, 20.0, 1009.2542114257812],
                [1.0, 119.65310668945312, 20.0, 1062.73486328125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=94,
            map_id=Maps.JapesBaboonBlast,
            name="Between cannons in the Baboon Blast Course",
            konglist=[Kongs.donkey],
            region=Regions.JapesBaboonBlast,
            locations=[
                [1.0, 1896.4229736328125, 830.4993896484375, 1890.2225341796875],
                [1.0, 1106.90234375, 708.71044921875, 1444.036865234375],
            ],
            vanilla=True,
        ),
    ],
    Levels.AngryAztec: [
        BananaCoinGroup(
            group=1,
            map_id=Maps.AngryAztec,
            name="In front of Chunky vase room door",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BetweenVinesByPortal,
            locations=[
                [1.0, 651, 120, 903],
                [1.0, 652, 120, 846],
                [1.0, 650, 120, 793],
            ],
        ),
        BananaCoinGroup(
            group=2,
            map_id=Maps.AngryAztec,
            name="In front of Donkey kasplat room door",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AztecTunnelBeforeOasis,
            locations=[
                [1.0, 1423, 120, 1028],
                [1.0, 1377, 120, 1030],
                [1.0, 1319, 120, 1029],
            ],
        ),
        BananaCoinGroup(
            group=3,
            map_id=Maps.AngryAztec,
            name="Left side across the sand from Candy's store",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 2197, 120, 658],
                [1.0, 2183, 120, 675],
                [1.0, 2208, 120, 676],
            ],
        ),
        BananaCoinGroup(
            group=4,
            map_id=Maps.AngryAztec,
            name="Right side of Candy's store",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 2448, 120, 476],
                [1.0, 2461, 120, 492],
                [1.0, 2442, 120, 497],
            ],
        ),
        BananaCoinGroup(
            group=5,
            map_id=Maps.AngryAztec,
            name="Next to quicksand close to llama cage",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 2234, 120, 1187],
                [1.0, 2242, 120, 1205],
                [1.0, 2246, 120, 1188],
            ],
        ),
        BananaCoinGroup(
            group=6,
            map_id=Maps.AngryAztec,
            name="On gentle slope of llama cage",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 2319, 165, 1513],
                [1.0, 2267, 202, 1546],
                [1.0, 2206, 163, 1384],
                [1.0, 2172, 197, 1431],
            ],
        ),
        BananaCoinGroup(
            group=7,
            map_id=Maps.AngryAztec,
            name="Around Oasis W1",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 2344, 120, 729],
                [1.0, 2344, 120, 776],
                [1.0, 2393, 120, 752],
            ],
        ),
        BananaCoinGroup(
            group=8,
            map_id=Maps.AngryAztec,
            name="Each side of Tiny Temple stairs",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 2928, 153, 730],
                [1.0, 2909, 153, 708],
                [1.0, 3024, 153, 863],
                [1.0, 3042, 153, 881],
            ],
        ),
        BananaCoinGroup(
            group=9,
            map_id=Maps.AngryAztec,
            name="Behind guitar door",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecConnectorTunnel,
            locations=[
                [1.0, 2727, 120, 1503],
                [1.0, 2619, 120, 1505],
            ],
        ),
        BananaCoinGroup(
            group=10,
            map_id=Maps.AngryAztec,
            name="On boulder switch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecConnectorTunnel,
            locations=[
                [1.0, 3296, 140, 1857],
                [1.0, 3245, 140, 1855],
                [1.0, 3297, 140, 1808],
                [1.0, 3245, 140, 1804],
            ],
        ),
        BananaCoinGroup(
            group=11,
            map_id=Maps.AngryAztec,
            name="Right side of Cranky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecConnectorTunnel,
            locations=[
                [1.0, 2724, 120, 2430],
                [1.0, 2711, 120, 2458],
            ],
        ),
        BananaCoinGroup(
            group=12,
            map_id=Maps.AngryAztec,
            name="Against wall near Chunky bonus GB",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecConnectorTunnel,
            locations=[
                [1.0, 3888, 120, 2513],
                [1.0, 3923, 120, 2513],
                [1.0, 3962, 120, 2510],
            ],
        ),
        BananaCoinGroup(
            group=13,
            map_id=Maps.AngryAztec,
            name="In shady corner to the right of Llama temple",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3213, 170, 3120],
                [1.0, 3192, 161, 3128],
                [1.0, 3195, 164, 3152],
            ],
        ),
        BananaCoinGroup(
            group=14,
            map_id=Maps.AngryAztec,
            name="Around W4 by other warps",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3407, 120, 3415],
                [1.0, 3388, 122, 3357],
                [1.0, 3437, 122, 3364],
            ],
        ),
        BananaCoinGroup(
            group=15,
            map_id=Maps.AngryAztec,
            name="Around W3 by warps",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3483, 122, 3371],
                [1.0, 3534, 122, 3377],
                [1.0, 3501, 120, 3424],
            ],
        ),
        BananaCoinGroup(
            group=16,
            map_id=Maps.AngryAztec,
            name="Around rocketbarrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3700, 190, 3621],
                [1.0, 3763, 190, 3624],
                [1.0, 3799, 190, 3563],
                [1.0, 3764, 190, 3510],
            ],
        ),
        BananaCoinGroup(
            group=17,
            map_id=Maps.AngryAztec,
            name="By quicksand near tower",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3921, 120, 3387],
                [1.0, 3911, 120, 3402],
                [1.0, 3901, 120, 3423],
            ],
        ),
        BananaCoinGroup(
            group=18,
            map_id=Maps.AngryAztec,
            name="Behind tower",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 4576, 80, 2908],
                [1.0, 4556, 80, 2887],
                [1.0, 4529, 80, 2880],
            ],
        ),
        BananaCoinGroup(
            group=19,
            map_id=Maps.AngryAztec,
            name="On ramps on stairs leading to tower",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 4190, 91, 3430],
                [1.0, 4211, 121, 3382],
                [1.0, 4259, 84, 3484],
                [1.0, 4294, 126, 3419],
            ],
        ),
        BananaCoinGroup(
            group=20,
            map_id=Maps.AngryAztec,
            name="Middle of Snide's platform",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 4077, 120, 4425],
                [1.0, 4033, 120, 4402],
                [1.0, 4075, 120, 4368],
            ],
        ),
        BananaCoinGroup(
            group=21,
            map_id=Maps.AngryAztec,
            name="On corner of path near Snide's tag barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3755, 120, 3805],
                [1.0, 3725, 120, 3804],
                [1.0, 3704, 120, 3803],
            ],
        ),
        BananaCoinGroup(
            group=22,
            map_id=Maps.AngryAztec,
            name="Left wall by Funky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3019, 120, 4423],
                [1.0, 3027, 120, 4466],
                [1.0, 3025, 120, 4517],
                [1.0, 3018, 120, 4558],
            ],
        ),
        BananaCoinGroup(
            group=23,
            map_id=Maps.AngryAztec,
            name="Left corner near quicksand by Llama temple",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 2696, 162, 3258],
                [1.0, 2715, 160, 3249],
                [1.0, 2721, 162, 3271],
            ],
        ),
        BananaCoinGroup(
            group=24,
            map_id=Maps.AngryAztec,
            name="Left of Llama temple stairs",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 2895, 160, 3223],
                [1.0, 2890, 160, 3203],
                [1.0, 2873, 160, 3211],
            ],
        ),
        BananaCoinGroup(
            group=25,
            map_id=Maps.AngryAztec,
            name="Right of Llama temple stairs",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3026, 160, 3167],
                [1.0, 3037, 160, 3184],
                [1.0, 3050, 160, 3163],
            ],
        ),
        BananaCoinGroup(
            group=26,
            map_id=Maps.AngryAztec,
            name="Front of 5 door temple",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 2208, 180, 3719],
                [1.0, 2218, 180, 3696],
                [1.0, 2234, 180, 3665],
            ],
        ),
        BananaCoinGroup(
            group=27,
            map_id=Maps.AngryAztec,
            name="Top of 5 door temple side stairs",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 2023, 280, 3802],
                [1.0, 1942, 280, 3761],
            ],
        ),
        BananaCoinGroup(
            group=28,
            map_id=Maps.AngryAztec,
            name="On roof of 5 door temple",
            konglist=[Kongs.diddy],
            region=Regions.AngryAztecMain,
            logic=lambda l: l.jetpack,
            locations=[
                [1.0, 2154, 384, 3690],
                [1.0, 2173, 385, 3650],
                [1.0, 2195, 384, 3612],
            ],
        ),
        BananaCoinGroup(
            group=29,
            map_id=Maps.AngryAztec,
            name="On roof of Llama Temple",
            konglist=[Kongs.diddy],
            region=Regions.AngryAztecMain,
            logic=lambda l: l.jetpack,
            locations=[
                [1.0, 2921, 437, 2883],
                [1.0, 2844, 437, 2907],
                [1.0, 2863, 437, 2995],
                [1.0, 2958, 437, 2980],
            ],
        ),
        BananaCoinGroup(
            group=30,
            map_id=Maps.AngryAztec,
            name="Around far W5",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AztecDonkeyQuicksandCave,
            locations=[
                [1.0, 2750, 120, 4754],
                [1.0, 2775, 120, 4721],
                [1.0, 2809, 120, 4754],
                [1.0, 2774, 120, 4782],
            ],
        ),
        BananaCoinGroup(
            group=31,
            map_id=Maps.AztecBaboonBlast,
            name="Inside the Baboon Blast Course",
            konglist=[Kongs.donkey],
            region=Regions.AztecBaboonBlast,
            locations=[
                [1.0, 558, 650, 1000],
                [1.0, 716, 647, 776],
                [1.0, 1002, 643, 632],
            ],
        ),
        BananaCoinGroup(
            group=32,
            map_id=Maps.AztecChunky5DTemple,
            name="First Switch",
            konglist=[Kongs.chunky],
            region=Regions.ChunkyTempleEntrance,
            locations=[
                [1.0, 641, 65, 383],
            ],
        ),
        BananaCoinGroup(
            group=33,
            map_id=Maps.AztecChunky5DTemple,
            name="On torches",
            konglist=[Kongs.chunky],
            region=Regions.ChunkyTemple,
            locations=[
                [1.0, 689, 143, 1008],
                [1.0, 589, 143, 1014],
            ],
        ),
        BananaCoinGroup(
            group=34,
            map_id=Maps.AztecChunky5DTemple,
            name="On stairs",
            konglist=[Kongs.chunky],
            region=Regions.ChunkyTemple,
            locations=[
                [1.0, 643, 94, 1688],
                [1.0, 642, 103, 1713],
                [1.0, 641, 113, 1730],
            ],
        ),
        BananaCoinGroup(
            group=35,
            map_id=Maps.AztecDiddy5DTemple,
            name="Behind some pillars",
            konglist=[Kongs.diddy],
            region=Regions.DiddyTemple,
            logic=lambda l: (l.peanut and l.isdiddy) or l.CanPhase(),
            locations=[
                [1.0, 479, 57, 599],
                [1.0, 381, 57, 600],
                [1.0, 681, 57, 955],
            ],
        ),
        BananaCoinGroup(
            group=35,
            map_id=Maps.AztecDiddy5DTemple,
            name="Behind some pillars",
            konglist=[Kongs.diddy],
            region=Regions.DiddyTempleDeadEndRight,
            locations=[
                [1.0, 161, 57, 955],
            ],
        ),
        BananaCoinGroup(
            group=36,
            map_id=Maps.AztecDonkey5DTemple,
            name="Final pathway",
            konglist=[Kongs.donkey],
            region=Regions.DonkeyTemple,
            logic=lambda l: (l.coconut and l.isdonkey) or l.CanPhase(),
            locations=[
                [1.0, 365, 66, 978],
                [1.0, 370, 86, 1017],
                [1.0, 469, 86, 1015],
                [1.0, 469, 66, 975],
            ],
        ),
        BananaCoinGroup(
            group=37,
            map_id=Maps.AztecLanky5DTemple,
            name="Starting Staircase",
            konglist=[Kongs.lanky],
            region=Regions.LankyTemple,
            locations=[
                [1.0, 707, 34, 486],
                [1.0, 213, 34, 486],
            ],
        ),
        BananaCoinGroup(
            group=38,
            map_id=Maps.AztecLanky5DTemple,
            name="Final Corridors",
            konglist=[Kongs.lanky],
            region=Regions.LankyTemple,
            logic=lambda l: (l.grape and l.islanky) or l.CanPhase(),
            locations=[
                [1.0, 812, 85, 1207],
                [1.0, 107, 85, 1207],
            ],
        ),
        BananaCoinGroup(
            group=39,
            map_id=Maps.AztecTiny5DTemple,
            name="Fairy Side Room",
            konglist=[Kongs.tiny],
            region=Regions.TinyTemple,
            logic=lambda l: (l.istiny and l.mini) or l.CanPhase(),
            locations=[
                [1.0, 1207, 47, 583],
                [1.0, 1206, 47, 727],
                [1.0, 1054, 47, 726],
                [1.0, 1058, 47, 582],
            ],
        ),
        BananaCoinGroup(
            group=40,
            map_id=Maps.AztecTiny5DTemple,
            name="Other Side Room",
            konglist=[Kongs.tiny],
            region=Regions.TinyTemple,
            logic=lambda l: (l.istiny and l.mini) or l.CanPhase(),
            locations=[
                [1.0, 59, 47, 610],
                [1.0, 73, 47, 597],
                [1.0, 83, 47, 607],
                [1.0, 73, 47, 618],
            ],
        ),
        BananaCoinGroup(
            group=41,
            map_id=Maps.AztecLlamaTemple,
            name="Left Bannister",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LlamaTemple,
            locations=[
                [1.0, 2017, 416, 2728],
                [1.0, 1976, 441, 2730],
                [1.0, 1935, 466, 2728],
                [1.0, 1889, 494, 2729],
                [1.0, 1857, 513, 2728],
            ],
        ),
        BananaCoinGroup(
            group=42,
            map_id=Maps.AztecLlamaTemple,
            name="Right Bannister",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LlamaTemple,
            locations=[
                [1.0, 2016, 416, 2075],
                [1.0, 1973, 443, 2074],
                [1.0, 1930, 469, 2071],
                [1.0, 1892, 492, 2074],
                [1.0, 1857, 513, 2073],
            ],
        ),
        BananaCoinGroup(
            group=43,
            map_id=Maps.AztecLlamaTemple,
            name="Matching Game Room",
            konglist=[Kongs.lanky],
            region=Regions.LlamaTempleMatching,
            locations=[
                [1.0, 1093, 641, 2580],
                [1.0, 1113, 641, 2591],
                [1.0, 1115, 641, 2572],
            ],
        ),
        BananaCoinGroup(
            group=44,
            map_id=Maps.AztecLlamaTemple,
            name="Around the right lava room platform",
            konglist=[Kongs.tiny],
            region=Regions.LlamaTempleBack,
            logic=lambda l: (l.CanSlamSwitch(Levels.AngryAztec, 1) or (l.twirl and l.monkey_maneuvers)) and l.istiny,
            locations=[
                [1.0, 945, 420, 3708],
                [1.0, 852, 420, 3659],
                [1.0, 795, 420, 3756],
                [1.0, 886, 420, 3807],
            ],
        ),
        BananaCoinGroup(
            group=45,
            map_id=Maps.AztecLlamaTemple,
            name="Around the left lava room platform",
            konglist=[Kongs.tiny],
            region=Regions.LlamaTempleBack,
            logic=lambda l: (l.CanSlamSwitch(Levels.AngryAztec, 1) or (l.twirl and l.monkey_maneuvers)) and l.istiny,
            locations=[
                [1.0, 1442, 420, 3997],
                [1.0, 1380, 420, 4102],
                [1.0, 1472, 420, 4146],
                [1.0, 1531, 420, 4044],
            ],
        ),
        BananaCoinGroup(
            group=46,
            map_id=Maps.AztecTinyTemple,
            name="Around the torch in the starting area",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TempleStart,
            locations=[
                [1.0, 1799, 283, 597],
                [1.0, 1784, 283, 614],
                [1.0, 1805, 283, 626],
                [1.0, 1816, 283, 610],
            ],
        ),
        # BananaCoinGroup(  # Removed in favour of a DoorLocation
        #     group=47,
        #     map_id=Maps.AztecTinyTemple,
        #     name="On frog ledge",
        #     konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
        #     region=Regions.TempleStart,
        #     locations=[
        #         [1.0, 1820, 389, 1132],
        #         [1.0, 1834, 389, 1117],
        #     ],
        # ),
        BananaCoinGroup(
            group=48,
            map_id=Maps.AztecTinyTemple,
            name="Around Lanky Pedestal",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TempleVultureRoom,
            locations=[
                [1.0, 1467, 360, 2140],
                [1.0, 1435, 375, 2156],
                [1.0, 1417, 360, 2191],
                [1.0, 1437, 375, 2220],
                [1.0, 1469, 360, 2239],
                [1.0, 1499, 375, 2222],
                [1.0, 1516, 360, 2189],
                [1.0, 1499, 375, 2161],
            ],
        ),
        BananaCoinGroup(
            group=49,
            map_id=Maps.AztecTinyTemple,
            name="On KONG Letters",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.TempleKONGRoom,
            locations=[
                [1.0, 758, 412, 1703],
                [1.0, 354, 412, 1822],
                [1.0, 92, 412, 1567],
                [1.0, 92, 429, 1342],
            ],
        ),
        BananaCoinGroup(
            group=50,
            map_id=Maps.AztecTinyTemple,
            name="Underwater tunnels",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TempleUnderwater,
            locations=[
                [1.0, 1479, 126, 1708],
                [1.0, 1271, 129, 1468],
                [1.0, 1462, 134, 1211],
            ],
        ),
        BananaCoinGroup(
            group=51,
            map_id=Maps.AztecTinyTemple,
            name="On Diddy's Guitar Pad",
            konglist=[Kongs.diddy],
            region=Regions.TempleGuitarPad,
            locations=[
                [1.0, 1367.7171630859375, 538.5, 578.090576171875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=52,
            map_id=Maps.AztecTinyTemple,
            name="Around the Upper Lanky Pedestal",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TempleVultureRoom,
            locations=[
                [1.0, 1465.5142822265625, 375.3333435058594, 2161.44775390625],
                [1.0, 1499.1553955078125, 375.3333435058594, 2192.168212890625],
                [1.0, 1471.25146484375, 375.3333435058594, 2218.7431640625],
                [1.0, 1437.2344970703125, 375.3333435058594, 2190.0244140625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=53,
            map_id=Maps.AztecTinyTemple,
            name="Starting Corner (1)",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TempleStart,
            locations=[
                [1.0, 1186.2073974609375, 289.8311767578125, 658.03173828125],
                [1.0, 1184.0921630859375, 289.85321044921875, 630.0655517578125],
                [1.0, 1213.3585205078125, 289.65740966796875, 626.4440307617188],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=54,
            map_id=Maps.AztecTinyTemple,
            name="Starting Corner (2)",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TempleStart,
            locations=[
                [1.0, 1189.968505859375, 289.7919921875, 1034.4820556640625],
                [1.0, 1221.5074462890625, 289.4634704589844, 1034.4150390625],
                [1.0, 1223.5050048828125, 289.5545349121094, 1068.6181640625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=55,
            map_id=Maps.AztecTinyTemple,
            name="Near Kong Cage",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.TempleKONGRoom,
            locations=[
                [1.0, 289.41265869140625, 344.8333435058594, 1129.9959716796875],
                [1.0, 307.7870788574219, 344.8333435058594, 1131.9871826171875],
                [1.0, 313.0320739746094, 344.8333435058594, 1150.5093994140625],
                [1.0, 293.04608154296875, 344.8333435058594, 1165.4097900390625],
                [1.0, 278.5104675292969, 344.8333435058594, 1150.9351806640625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=56,
            map_id=Maps.AztecDonkey5DTemple,
            name="Near 2nd Switch",
            konglist=[Kongs.donkey],
            region=Regions.DonkeyTemple,
            logic=lambda l: (l.coconut and l.isdonkey) or l.CanPhase(),
            locations=[
                [1.0, 767.441162109375, 58.0, 880.442626953125],
                [1.0, 686.1561889648438, 58.0, 881.2146606445312],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=57,
            map_id=Maps.AztecLlamaTemple,
            name="Around Instrument Pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LlamaTemple,
            locations=[
                [1.0, 2222.318115234375, 458.1666564941406, 1558.25],
                [1.0, 2263.00244140625, 458.1666564941406, 1549.9932861328125],
                [1.0, 2218.9794921875, 458.1666564941406, 1597.1627197265625],
                [1.0, 2279.378173828125, 458.1666564941406, 1591.06982421875],
                [1.0, 2248.489013671875, 458.1666564941406, 1616.8902587890625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=58,
            map_id=Maps.AztecLlamaTemple,
            name="Near Tag Barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LlamaTemple,
            locations=[
                [1.0, 1690.3333740234375, 641.3333129882812, 2402.224853515625],
                [1.0, 1771.9189453125, 641.3333129882812, 2403.727294921875],
                [1.0, 1838.7867431640625, 641.3333129882812, 2404.957763671875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=59,
            map_id=Maps.AztecLlamaTemple,
            name="Matching Game Room Vines",
            konglist=[Kongs.lanky],
            region=Regions.LlamaTempleMatching,
            logic=lambda l: l.can_use_vines,
            locations=[
                [1.0, 900.880859375, 691.3333129882812, 2207.316650390625],
                [1.0, 827.7755737304688, 691.3333129882812, 2168.489501953125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=60,
            map_id=Maps.AztecDiddy5DTemple,
            name="Dead Ends",
            konglist=[Kongs.diddy],
            region=Regions.DiddyTemple,
            logic=lambda l: (l.peanut and l.isdiddy) or l.CanPhase(),
            locations=[
                [1.0, 1057.60205078125, 50.5, 569.7957153320312],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=60,
            map_id=Maps.AztecDiddy5DTemple,
            name="Dead Ends",
            konglist=[Kongs.diddy],
            region=Regions.DiddyTempleDeadEndRight,
            locations=[
                [1.0, 116.1932373046875, 88.0, 1077.492431640625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            # Under the first switch
            group=61,
            map_id=Maps.AztecTiny5DTemple,
            name="Under Feather Switches and at Dead Ends",
            konglist=[Kongs.tiny],
            region=Regions.TinyTempleEntrance,
            locations=[
                [1.0, 645.4479370117188, 20.5, 370.3265075683594],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            # Under the second switch
            group=61,
            map_id=Maps.AztecTiny5DTemple,
            name="Under Feather Switches and at Dead Ends",
            konglist=[Kongs.tiny],
            region=Regions.TinyTemple,
            locations=[
                [1.0, 641.8150024414062, 87.0, 1198.519775390625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            # Dead end coin
            group=61,
            map_id=Maps.AztecTiny5DTemple,
            name="Under Feather Switches and at Dead Ends",
            konglist=[Kongs.tiny],
            region=Regions.TinyTemple,
            logic=lambda l: (l.feather and l.istiny) or l.CanPhase(),
            locations=[
                [1.0, 331.8021545410156, 138.66665649414062, 1419.3331298828125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=62,
            map_id=Maps.AztecChunky5DTemple,
            name="Around the temple",
            konglist=[Kongs.chunky],
            region=Regions.ChunkyTemple,
            locations=[
                [1.0, 639.6046752929688, 85.0, 1242.0648193359375],
                [1.0, 642.6415405273438, 122.66666412353516, 1924.775390625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=62,
            map_id=Maps.AztecChunky5DTemple,
            name="Around the temple",
            konglist=[Kongs.chunky],
            region=Regions.ChunkyTemple,
            logic=lambda l: (l.pineapple and l.ischunky) or l.CanPhase(),
            locations=[
                [1.0, 341.9595031738281, 122.66666412353516, 1988.3603515625],
                [1.0, 1179.842041015625, 160.1666717529297, 2103.085205078125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=63,
            map_id=Maps.AngryAztec,
            name="Near Cranky's Lab",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecConnectorTunnel,
            locations=[
                [1.0, 2774.681640625, 120.0, 2645.0517578125],
                [1.0, 2787.124267578125, 120.0, 2614.707275390625],
                [1.0, 2752.337890625, 120.0, 2613.15673828125],
                [1.0, 2739.4716796875, 120.0, 2645.649658203125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=64,
            map_id=Maps.AngryAztec,
            name="Outside Funky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 2933.618408203125, 120.0, 4443.46630859375],
                [1.0, 2946.123291015625, 120.0, 4413.31396484375],
                [1.0, 2898.611328125, 120.0, 4438.72216796875],
                [1.0, 2923.883056640625, 120.0, 4392.27001953125],
                [1.0, 2894.731689453125, 120.0, 4403.57275390625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=65,
            map_id=Maps.AngryAztec,
            name="Around Oasis Warp 2",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 2914.220703125, 153.3333282470703, 876.552978515625],
                [1.0, 2856.9140625, 153.39370727539062, 865.5851440429688],
                [1.0, 2883.7529296875, 153.36155700683594, 893.75830078125],
                [1.0, 2868.736083984375, 153.38079833984375, 839.609375],
                [1.0, 2910.05078125, 153.3333282470703, 842.4771118164062],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=66,
            map_id=Maps.AngryAztec,
            name="Under Snide's HQ Tag Barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3949.1318359375, 190.0, 4063.035888671875],
                [1.0, 3954.7568359375, 190.0, 4007.091552734375],
                [1.0, 4009.227783203125, 190.0, 4041.890625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=67,
            map_id=Maps.AngryAztec,
            name="On posts near Llama",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 2249.796630859375, 233.85943603515625, 1518.5277099609375],
                [1.0, 2196.356689453125, 233.86122131347656, 1459.8997802734375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=68,
            map_id=Maps.AngryAztec,
            name="At the end of the quicksand tunnel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AztecDonkeyQuicksandCave,
            locations=[
                [1.0, 2757.255859375, 120.0, 4645.79736328125],
                [1.0, 2659.03515625, 120.0, 4701.876953125],
                [1.0, 2701.979736328125, 120.0, 4855.4052734375],
                [1.0, 2792.623291015625, 120.0, 4870.29736328125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=69,
            map_id=Maps.AngryAztec,
            name="Around the Giant Boulder",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecConnectorTunnel,
            locations=[
                [1.0, 3630.039794921875, 120.0, 2513.942626953125],
                [1.0, 3483.240234375, 120.0, 2413.07958984375],
                [1.0, 3642.2255859375, 120.0, 2288.22265625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=70,
            map_id=Maps.AngryAztec,
            name="Around the Bonus Barrel Cage",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecConnectorTunnel,
            locations=[
                [1.0, 4155.48193359375, 120.0, 2408.625244140625],
                [1.0, 4329.55419921875, 120.0, 2405.08984375],
                [1.0, 4245.7080078125, 122.54806518554688, 2498.574462890625],
                [1.0, 4235.62353515625, 120.0, 2319.407958984375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=71,
            map_id=Maps.AngryAztec,
            name="Vines near Snide's HQ",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            logic=lambda l: (l.can_use_vines and l.climbing) or (l.jetpack and l.isdiddy),
            locations=[
                [1.0, 3899.343505859375, 312.0, 4292.08837890625],
                [1.0, 3992.395751953125, 306.0, 4411.8349609375],
                [1.0, 4014.3642578125, 226.1666717529297, 4584.63134765625],
                [1.0, 4026.322998046875, 260.0, 4510.701171875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=72,
            map_id=Maps.AngryAztec,
            name="Around Hunky Chunky Barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecConnectorTunnel,
            locations=[
                [1.0, 3138.619384765625, 120.0, 1580.4622802734375],
                [1.0, 3263.6005859375, 120.0, 1436.5548095703125],
                [1.0, 3396.42578125, 120.0, 1585.50244140625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=73,
            map_id=Maps.AngryAztec,
            name="Behind 5-Door Temple",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 2005.3177490234375, 300.0, 3501.9248046875],
                [1.0, 1894.5625, 200.2208251953125, 3624.213623046875],
                [1.0, 1944.41796875, 300.0, 3627.410400390625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=74,
            map_id=Maps.AngryAztec,
            name="Under Oasis Tag",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 2455.719482421875, 115.5, 983.2329711914062],
                [1.0, 2450.566650390625, 115.5, 935.1344604492188],
                [1.0, 2399.508056640625, 115.5, 937.164306640625],
                [1.0, 2407.984619140625, 115.5, 994.1201171875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=75,
            map_id=Maps.AngryAztec,
            name="Around Totem W5",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecMain,
            locations=[
                [1.0, 3482.4248046875, 120.0, 4518.00146484375],
                [1.0, 3456.824462890625, 120.0, 4501.80029296875],
                [1.0, 3462.391845703125, 120.0, 4461.54345703125],
                [1.0, 3508.263427734375, 120.0, 4497.9326171875],
                [1.0, 3500.856201171875, 120.0, 4460.08740234375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=76,
            map_id=Maps.AngryAztec,
            name="Around Tiny Temple",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AngryAztecOasis,
            locations=[
                [1.0, 3036.075927734375, 214.230224609375, 580.3154907226562],
                [1.0, 3138.151611328125, 213.0538330078125, 509.662353515625],
                [1.0, 3301.577880859375, 213.2405548095703, 720.41259765625],
                [1.0, 3197.343994140625, 214.15843200683594, 797.4346313476562],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=77,
            map_id=Maps.AngryAztec,
            name="Blueprint Cavern",
            konglist=[Kongs.donkey, Kongs.tiny],
            region=Regions.AngryAztecOasis,
            logic=lambda l: (l.hasMoveSwitchsanity(Switches.AztecBlueprintDoor, False) or l.CanPhase()) and ((l.strongKong and l.isdonkey) or (l.twirl and l.istiny)),
            locations=[
                [1.0, 1240.0001220703125, 120.0, 747.2548217773438],
                [1.0, 1219.4495849609375, 120.0, 730.4656372070312],
                [1.0, 1219.2955322265625, 120.0, 754.693603515625],
            ],
            vanilla=True,
        ),
    ],
    Levels.FranticFactory: [
        BananaCoinGroup(
            group=0,
            map_id=Maps.FranticFactory,
            name="Near Portal",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryStart,
            locations=[
                [1.0, 1343, 830, 2688],
                [1.0, 1342, 827, 2776],
                [1.0, 1180, 827, 2776],
                [1.0, 1182, 830, 2684],
            ],
        ),
        BananaCoinGroup(
            group=1,
            map_id=Maps.FranticFactory,
            name="Behind Lobby W1",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryStart,
            locations=[
                [1.0, 1135, 830, 2462],
                [1.0, 1170, 830, 2462],
            ],
        ),
        BananaCoinGroup(
            group=2,
            map_id=Maps.FranticFactory,
            name="Behind Lobby W2",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryStart,
            locations=[
                [1.0, 1356, 830, 2461],
                [1.0, 1392, 830, 2461],
            ],
        ),
        BananaCoinGroup(
            group=3,
            map_id=Maps.FranticFactory,
            name="On the Clock",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.FranticFactoryStart,
            locations=[
                [1.0, 1282, 943, 2031],
                [1.0, 1243, 943, 2032],
                [1.0, 1265, 943, 2032],
            ],
        ),
        BananaCoinGroup(
            group=4,
            map_id=Maps.FranticFactory,
            name="Around Lobby W3",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryStart,
            locations=[
                [1.0, 1461, 842, 2175],
                [1.0, 1418, 842, 2195],
                [1.0, 1412, 842, 2157],
            ],
        ),
        BananaCoinGroup(
            group=5,
            map_id=Maps.FranticFactory,
            name="Around the doorways in the Lobby",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryStart,
            locations=[
                [1.0, 1021, 830, 2436],
                [1.0, 1022, 830, 2569],
                [1.0, 1483, 830, 2570],
                [1.0, 1483, 830, 2438],
            ],
        ),
        BananaCoinGroup(
            group=6,
            map_id=Maps.FranticFactory,
            name="Top window to Production Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryStart,
            locations=[
                [1.0, 609, 804, 1734],
                [1.0, 646, 804, 1732],
                [1.0, 682, 804, 1735],
            ],
        ),
        BananaCoinGroup(
            group=7,
            map_id=Maps.FranticFactory,
            name="Down Hatch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.AlcoveBeyondHatch,
            locations=[
                [1.0, 609, 443, 1853],
                [1.0, 644, 443, 1840],
                [1.0, 682, 443, 1855],
            ],
        ),
        BananaCoinGroup(
            group=8,
            map_id=Maps.FranticFactory,
            name="Bottom of the Hatch Pole",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LowerCore,
            locations=[
                [1.0, 642, 167, 1909],
                [1.0, 657, 167, 1888],
                [1.0, 641, 167, 1882],
            ],
        ),
        BananaCoinGroup(
            group=9,
            map_id=Maps.FranticFactory,
            name="Corner of tunnel to Production Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LowerCore,
            locations=[
                [1.0, 31, 5, 1308],
                [1.0, 47, 5, 1308],
                [1.0, 32, 5, 1332],
            ],
        ),
        BananaCoinGroup(
            group=10,
            map_id=Maps.FranticFactory,
            name="On low boxes in lower Production Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MiddleCore,
            locations=[
                [1.0, 473, 108, 1488],
                [1.0, 493, 202, 1211],
                [1.0, 781, 108, 1180],
                [1.0, 753, 200, 1460],
            ],
        ),
        BananaCoinGroup(
            group=11,
            map_id=Maps.FranticFactory,
            name="On a pipe in lower Production Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MiddleCore,
            locations=[
                [1.0, 900, 154, 1337],
                [1.0, 785, 154, 1336],
                [1.0, 887, 156, 1500],
            ],
        ),
        BananaCoinGroup(
            group=12,
            map_id=Maps.FranticFactory,
            name="On the spinning section in Lower Production Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.SpinningCore,
            logic=lambda l: l.climbing,
            locations=[
                [1.0, 569, 308, 1392],
                [1.0, 685, 308, 1388],
                [1.0, 681, 308, 1285],
                [1.0, 576, 308, 1279],
            ],
        ),
        BananaCoinGroup(
            group=13,
            map_id=Maps.FranticFactory,
            name="On the boxes at the top of production Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.UpperCore,
            locations=[
                [1.0, 783, 677, 972],
                [1.0, 473, 677, 972],
            ],
        ),
        BananaCoinGroup(
            group=14,
            map_id=Maps.FranticFactory,
            name="Around the production room Simian Spring pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.UpperCore,
            locations=[
                [1.0, 542, 608, 1705],
                [1.0, 558, 607, 1686],
                [1.0, 541, 608, 1669],
                [1.0, 519, 607, 1686],
            ],
        ),
        BananaCoinGroup(
            group=15,
            map_id=Maps.FranticFactory,
            name="On various boxes around Production Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.UpperCore,
            locations=[
                [1.0, 282, 479, 1484],
                [1.0, 262, 478, 1480],
                [1.0, 984, 321, 1163],
                [1.0, 993, 322, 1184],
            ],
        ),
        BananaCoinGroup(
            group=16,
            map_id=Maps.FranticFactory,
            name="On the archway to Storage Room",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.MiddleCore,
            locations=[
                [1.0, 1007, 100, 1272],
                [1.0, 1007, 117, 1312],
                [1.0, 1007, 99, 1355],
            ],
        ),
        BananaCoinGroup(
            group=17,
            map_id=Maps.FranticFactory,
            name="Around the BBlast pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondHatch,
            locations=[
                [1.0, 1296, 6, 674],
                [1.0, 1266, 6, 647],
                [1.0, 1293, 6, 618],
                [1.0, 1321, 6, 645],
            ],
        ),
        BananaCoinGroup(
            group=18,
            map_id=Maps.FranticFactory,
            name="Inbetween the storage room boxes",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondHatch,
            locations=[
                [1.0, 1045, 6, 897],
                [1.0, 1060, 6, 900],
            ],
        ),
        BananaCoinGroup(
            group=19,
            map_id=Maps.FranticFactory,
            name="Behind a storage room box",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondHatch,
            locations=[
                [1.0, 969, 46, 870],
            ],
        ),
        BananaCoinGroup(
            group=20,
            map_id=Maps.FranticFactory,
            name="On boxes near Cranky's Lab",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondHatch,
            locations=[
                [1.0, 327, 185, 910],
                [1.0, 285, 205, 909],
                [1.0, 282, 180, 876],
            ],
        ),
        BananaCoinGroup(
            group=21,
            map_id=Maps.FranticFactory,
            name="On boxes near Candy's Music Shop",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondHatch,
            locations=[
                [1.0, 289, 194, 537],
                [1.0, 330, 205, 548],
            ],
        ),
        BananaCoinGroup(
            group=22,
            map_id=Maps.FranticFactory,
            name="Behind some boxes near Cranky's",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.BeyondHatch,
            locations=[
                [1.0, 472, 249, 912],
            ],
        ),
        BananaCoinGroup(
            group=23,
            map_id=Maps.FranticFactory,
            name="Around the top of the pole to Arcade",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FactoryArcadeTunnel,
            locations=[
                [1.0, 1309, 1118, 932],
                [1.0, 1251, 1118, 930],
                [1.0, 1230, 1118, 885],
                [1.0, 1259, 1118, 836],
                [1.0, 1313, 1118, 843],
                [1.0, 1339, 1118, 886],
            ],
        ),
        BananaCoinGroup(
            group=24,
            map_id=Maps.FranticFactory,
            name="Arcade Room Poles",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FactoryArcadeTunnel,
            locations=[
                [1.0, 1626, 1106, 1482],
                [1.0, 1669, 1106, 1527],
                [1.0, 1625, 1106, 1268],
                [1.0, 1668, 1106, 1226],
            ],
        ),
        BananaCoinGroup(
            group=25,
            map_id=Maps.FranticFactory,
            name="On top of the Arcade Machine",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FactoryArcadeTunnel,
            locations=[
                [1.0, 1849, 1165, 1226],
                [1.0, 1872, 1166, 1227],
            ],
        ),
        BananaCoinGroup(
            group=26,
            map_id=Maps.FranticFactory,
            name="Snide Window",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryStart,
            locations=[
                [1.0, 1680, 836, 2298],
                [1.0, 1707, 836, 2297],
                [1.0, 1743, 837, 2299],
            ],
        ),
        BananaCoinGroup(
            group=27,
            map_id=Maps.FranticFactory,
            name="Above the pole to Block Tower Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 1996, 1038, 2438],
                [1.0, 2044, 1038, 2439],
                [1.0, 2074, 1038, 2387],
                [1.0, 2043, 1038, 2346],
                [1.0, 1988, 1038, 2343],
                [1.0, 1960, 1038, 2389],
            ],
        ),
        BananaCoinGroup(
            group=28,
            map_id=Maps.FranticFactory,
            name="Behind some elevator blocks (1)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 2257, 1026, 1603],
                [1.0, 2152, 1026, 1604],
                [1.0, 2059, 1026, 1604],
            ],
        ),
        BananaCoinGroup(
            group=29,
            map_id=Maps.FranticFactory,
            name="On a box in Testing Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 1999, 1071, 1152],
                [1.0, 1994, 1071, 1207],
            ],
        ),
        BananaCoinGroup(
            group=30,
            map_id=Maps.FranticFactory,
            name="On some boxes near Funky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 1331, 1133, 636],
                [1.0, 1297, 1153, 605],
                [1.0, 1328, 1113, 588],
            ],
        ),
        BananaCoinGroup(
            group=31,
            map_id=Maps.FranticFactory,
            name="On block tower (1)",
            konglist=[Kongs.diddy],
            region=Regions.Testing,
            logic=lambda l: ((l.spring or l.CanMoontail()) and l.isdiddy),
            locations=[
                [1.0, 2372, 1266, 1381],
                [1.0, 2427, 1286, 1379],
                [1.0, 2457, 1306, 1376],
            ],
        ),
        BananaCoinGroup(
            group=32,
            map_id=Maps.FranticFactory,
            name="On block tower (2)",
            konglist=[Kongs.diddy],
            region=Regions.Testing,
            logic=lambda l: ((l.spring or l.CanMoontail()) and l.isdiddy),
            locations=[
                [1.0, 2414, 1426, 1343],
                [1.0, 2386, 1326, 1258],
                [1.0, 2364, 1306, 1255],
            ],
        ),
        BananaCoinGroup(
            group=33,
            map_id=Maps.FranticFactory,
            name="Near dartboard entry",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 2406, 1106, 846],
                [1.0, 2407, 1106, 974],
                [1.0, 2291, 1106, 846],
                [1.0, 2245, 1106, 920],
                [1.0, 2244, 1106, 970],
            ],
        ),
        BananaCoinGroup(
            group=34,
            map_id=Maps.FranticFactory,
            name="In Dartboard Mini Tunnel",
            konglist=[Kongs.tiny],
            region=Regions.Testing,
            logic=lambda l: (l.istiny and l.mini) or l.CanPhase(),
            locations=[
                [1.0, 2435, 1106, 784],
                [1.0, 2344, 1106, 789],
            ],
        ),
        BananaCoinGroup(
            group=35,
            map_id=Maps.FranticFactory,
            name="In Dartboard Room",
            konglist=[Kongs.tiny],
            region=Regions.Testing,
            logic=lambda l: (l.istiny and l.mini) or l.CanPhase(),
            locations=[
                [1.0, 2576, 1106, 846],
                [1.0, 2577, 1106, 959],
                [1.0, 2429, 1106, 844],
                [1.0, 2431, 1106, 957],
            ],
        ),
        BananaCoinGroup(
            group=36,
            map_id=Maps.FranticFactory,
            name="On the archway to R&D",
            konglist=[Kongs.lanky],
            region=Regions.Testing,
            logic=lambda l: (l.balloon or l.monkey_maneuvers) and l.islanky,
            locations=[
                [1.0, 2786, 1119, 1091],
                [1.0, 2787, 1136, 1135],
                [1.0, 2788, 1117, 1180],
            ],
        ),
        BananaCoinGroup(
            group=37,
            map_id=Maps.FranticFactory,
            name="On the number game board",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 2642, 1033, 1803],
                [1.0, 2640, 1033, 1842],
                [1.0, 2676, 1033, 1837],
                [1.0, 2676, 1033, 1803],
            ],
        ),
        BananaCoinGroup(
            group=38,
            map_id=Maps.FranticFactory,
            name="Behind some elevator blocks (2)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 2786, 1026, 1289],
                [1.0, 2786, 1026, 1422],
                [1.0, 2784, 1026, 1548],
            ],
        ),
        BananaCoinGroup(
            group=39,
            map_id=Maps.FranticFactory,
            name="Around a light in R&D",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.RandD,
            locations=[
                [1.0, 4019, 1264, 1340],
                [1.0, 4009, 1264, 1370],
                [1.0, 4019, 1264, 1386],
            ],
        ),
        BananaCoinGroup(
            group=40,
            map_id=Maps.FranticFactory,
            name="On a ledge in R&D",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.RandD,
            locations=[
                [1.0, 3944, 1322, 610],
                [1.0, 3934, 1322, 653],
                [1.0, 3923, 1322, 710],
                [1.0, 3912, 1322, 754],
            ],
        ),
        BananaCoinGroup(
            group=41,
            map_id=Maps.FranticFactory,
            name="In Piano Room",
            konglist=[Kongs.lanky],
            region=Regions.RandD,
            logic=lambda l: (l.trombone and l.islanky) or l.CanAccessRNDRoom(),
            locations=[
                [1.0, 3287, 1288, 502],
                [1.0, 3274, 1264, 518],
                [1.0, 3393, 1288, 357],
            ],
        ),
        BananaCoinGroup(
            group=42,
            map_id=Maps.FranticFactory,
            name="In Diddy R&D Room",
            konglist=[Kongs.diddy],
            region=Regions.RandDUpper,
            logic=lambda l: (l.guitar and l.isdiddy) or l.CanAccessRNDRoom(),
            locations=[
                [1.0, 4175, 1336, 444],
                [1.0, 4187, 1336, 463],
                [1.0, 4194, 1336, 429],
            ],
        ),
        BananaCoinGroup(
            group=43,
            map_id=Maps.FranticFactory,
            name="In Chunky R&D Room",
            konglist=[Kongs.chunky],
            region=Regions.RandDUpper,
            logic=lambda l: (l.punch and l.triangle and l.ischunky) or l.CanAccessRNDRoom(),
            locations=[
                [1.0, 4570, 1336, 1400],
                [1.0, 4525, 1336, 1366],
                [1.0, 4478, 1336, 1536],
                [1.0, 4429, 1336, 1505],
            ],
        ),
        BananaCoinGroup(
            group=44,
            map_id=Maps.FranticFactory,
            name="In Car Race Lobby",
            konglist=[Kongs.tiny],
            region=Regions.FactoryTinyRaceLobby,
            locations=[
                [1.0, 3590, 1264, 1350],
                [1.0, 3579, 1264, 1316],
                [1.0, 3557, 1264, 1285],
                [1.0, 3526, 1264, 1274],
                [1.0, 3505, 1264, 1301],
                [1.0, 3512, 1264, 1342],
            ],
        ),
        BananaCoinGroup(
            group=45,
            map_id=Maps.FranticFactory,
            name="On Power Shed Platform",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ChunkyRoomPlatform,
            locations=[
                [1.0, 1498, 126, 755],
                [1.0, 1497, 126, 791],
                [1.0, 1502, 126, 833],
                [1.0, 1439, 126, 836],
                [1.0, 1398, 126, 834],
            ],
        ),
        BananaCoinGroup(
            group=46,
            map_id=Maps.FactoryTinyRace,
            name="In Car Race",
            konglist=[Kongs.tiny],
            region=Regions.FactoryTinyRace,
            locations=[
                [1.0, 2747, 5, 1201],
                [1.0, 2983, 99, 1531],
                [1.0, 1920, 5, 3070],
            ],
        ),
        BananaCoinGroup(
            group=47,
            map_id=Maps.FactoryCrusher,
            name="In the Crusher Room near the GB",
            konglist=[Kongs.donkey],
            region=Regions.InsideCore,
            logic=lambda l: (l.strongKong and l.isdonkey) or l.generalclips,
            locations=[
                [1.0, 169, 0, 66],
                [1.0, 69, 0, 69],
                [1.0, 69, 0, 146],
                [1.0, 170, 0, 147],
            ],
        ),
        BananaCoinGroup(
            group=48,
            map_id=Maps.FactoryPowerHut,
            name="Inside the Power Hut",
            konglist=[Kongs.donkey],
            region=Regions.PowerHut,
            locations=[
                [1.0, 49, 0, 50],
                [1.0, 67, 0, 48],
                [1.0, 49, 0, 67],
            ],
        ),
        BananaCoinGroup(
            group=49,
            map_id=Maps.FranticFactory,
            name="On spring small ledge",
            konglist=[Kongs.diddy],
            region=Regions.BeyondHatch,
            logic=lambda l: ((l.spring or l.CanMoontail()) and l.isdiddy) or l.CanPhase(),
            locations=[
                [1.0, 1065.8306884765625, FACTORY_LEDGE_COINS_HEIGHT, 482.5064697265625],
                [1.0, 1099.849853515625, FACTORY_LEDGE_COINS_HEIGHT, 482.2723693847656],
                [1.0, 1081.9705810546875, FACTORY_LEDGE_COINS_HEIGHT, 481.5174865722656],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=50,
            map_id=Maps.FranticFactory,
            name="Around Snide's HQ W3",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 1527.0517578125, 810.6666870117188, 2140.05322265625],
                [1.0, 1527.3900146484375, 811.8082885742188, 2186.71630859375],
                [1.0, 1565.6845703125, 810.6666870117188, 2155.104736328125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=51,
            map_id=Maps.FranticFactory,
            name="On steps near Block Tower",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 2090.16748046875, 1068.1666259765625, 1375.4637451171875],
                [1.0, 2150.828369140625, 1037.0, 1376.0411376953125],
                [1.0, 2028.3016357421875, 1099.5, 1376.48095703125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=52,
            map_id=Maps.FranticFactory,
            name="Outside R&D Window",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.RandD,
            locations=[
                [1.0, 3720.285888671875, 1263.935302734375, 1383.4241943359375],
                [1.0, 3736.643310546875, 1263.8917236328125, 1396.1673583984375],
                [1.0, 3701.911865234375, 1263.9837646484375, 1370.1102294921875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=53,
            map_id=Maps.FranticFactory,
            name="Around Storage W1",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondHatch,
            locations=[
                [1.0, 1538.5792236328125, 6.5, 551.8562622070312],
                [1.0, 1603.740234375, 6.5, 534.7921142578125],
                [1.0, 1591.6131591796875, 6.5, 492.4469909667969],
                [1.0, 1572.01513671875, 6.5, 562.488037109375],
                [1.0, 1542.2894287109375, 6.5, 505.4068908691406],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=54,
            map_id=Maps.FranticFactory,
            name="Lower Production Room Window",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LowerCore,
            locations=[
                [1.0, 600.1427001953125, 167.8333282470703, 1741.869140625],
                [1.0, 682.5888671875, 167.8333282470703, 1740.31591796875],
                [1.0, 641.4027709960938, 167.8333282470703, 1740.0592041015625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=55,
            map_id=Maps.FranticFactory,
            name="Above chute to Power Shed",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.RandDUpper,
            locations=[
                [1.0, 4658.2060546875, 1335.6666259765625, 851.6416625976562],
                [1.0, 4700.2529296875, 1335.6666259765625, 939.56689453125],
                [1.0, 4660.82177734375, 1335.6666259765625, 1022.7540893554688],
                [1.0, 4524.70263671875, 1335.6666259765625, 1004.2959594726562],
                [1.0, 4538.9462890625, 1335.6666259765625, 856.9103393554688],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=56,
            map_id=Maps.FranticFactory,
            name="Around Hatch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FranticFactoryStart,
            locations=[
                [1.0, 586.1878051757812, 804.1666870117188, 1968.2698974609375],
                [1.0, 642.8145751953125, 804.1666870117188, 1814.87548828125],
                [1.0, 726.1378784179688, 804.1666870117188, 1859.43408203125],
                [1.0, 689.337646484375, 804.1666870117188, 1974.71484375],
                [1.0, 554.8397827148438, 804.1666870117188, 1860.2259521484375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=57,
            map_id=Maps.FranticFactory,
            name="Around Tiny's Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondHatch,
            locations=[
                [1.0, 1200.7178955078125, 6.5, 177.23988342285156],
                [1.0, 1286.6082763671875, 6.5, 103.50585174560547],
                [1.0, 1397.8260498046875, 6.5, 177.35182189941406],
                [1.0, 1364.7738037109375, 6.5, 306.8700256347656],
                [1.0, 1216.9498291015625, 6.5, 299.6664123535156],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=58,
            map_id=Maps.FranticFactory,
            name="Around pole to Block Tower Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 2078.63916015625, 863.5, 2485.390625],
                [1.0, 2081.285888671875, 863.5, 2385.789794921875],
                [1.0, 2080.683349609375, 863.5, 2288.223876953125],
                [1.0, 1977.2904052734375, 863.5, 2271.684326171875],
                [1.0, 1976.7010498046875, 863.5, 2499.165771484375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=59,
            map_id=Maps.FranticFactory,
            name="Above pole to Snide's HQ",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 1569.6783447265625, 1025.5, 2045.39306640625],
                [1.0, 1505.2171630859375, 1025.5, 2092.47314453125],
                [1.0, 1631.87255859375, 1025.5, 2094.25341796875],
                [1.0, 1509.5843505859375, 1025.5, 2167.59716796875],
                [1.0, 1638.0440673828125, 1025.5, 2169.43505859375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=60,
            map_id=Maps.FranticFactory,
            name="Around high W4",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.SpinningCore,
            locations=[
                [1.0, 931.4605102539062, 422.5, 1553.5030517578125],
                [1.0, 893.4967651367188, 422.5, 1561.4439697265625],
                [1.0, 874.394775390625, 422.5, 1529.4300537109375],
                [1.0, 900.1627807617188, 422.5, 1502.59912109375],
                [1.0, 932.0536499023438, 422.5, 1515.7684326171875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=61,
            map_id=Maps.FranticFactory,
            name="Below pole to R&D",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 3107.675537109375, 1026.6666259765625, 965.2498168945312],
                [1.0, 3205.883544921875, 1026.6666259765625, 971.2708129882812],
                [1.0, 3237.093994140625, 1026.6666259765625, 1058.982421875],
                [1.0, 3250.356201171875, 1026.6666259765625, 1160.2457275390625],
                [1.0, 3150.5048828125, 1026.6666259765625, 1199.4373779296875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=62,
            map_id=Maps.FranticFactory,
            name="Above pole to R&D",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.RandD,
            locations=[
                [1.0, 3137.290771484375, 1264.0, 1144.1612548828125],
                [1.0, 3219.395263671875, 1264.0, 1132.8262939453125],
                [1.0, 3245.653564453125, 1264.0, 1072.7725830078125],
                [1.0, 3200.698486328125, 1264.0, 1008.4885864257812],
                [1.0, 3120.655029296875, 1264.0, 1013.32177734375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=63,
            map_id=Maps.FranticFactory,
            name="On elevators to upper production room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.SpinningCore,
            locations=[
                [1.0, 1001.6917114257812, 406.5, 1353.4505615234375],
                [1.0, 998.500732421875, 470.0, 1297.3572998046875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=63,
            map_id=Maps.FranticFactory,
            name="On elevators to upper production room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.UpperCore,
            locations=[
                [1.0, 1001.7814331054688, 520.0, 1239.797119140625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=64,
            map_id=Maps.FranticFactory,
            name="Behind a box in Block Tower room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 2472.408935546875, 1026.6666259765625, 986.9763793945312],
                [1.0, 2471.919677734375, 1026.6666259765625, 1011.7045288085938],
                [1.0, 2519.369140625, 1026.6666259765625, 1010.618408203125],
                [1.0, 2519.845947265625, 1026.6666259765625, 984.8517456054688],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=65,
            map_id=Maps.FranticFactory,
            name="Alcoves in Block Tower Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            logic=lambda _: True,
            locations=[
                [1.0, 2155.634521484375, 1129.673095703125, 1630.73291015625],
                [1.0, 2055.782958984375, 1128.6666259765625, 1632.19921875],
                [1.0, 2795.863037109375, 1132.6766357421875, 1545.44873046875],
                [1.0, 2810.158935546875, 1128.3333740234375, 1287.1773681640625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=66,
            map_id=Maps.FranticFactory,
            name="Around the spinning section in Production Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.SpinningCore,
            locations=[
                [1.0, 677.56005859375, 338.6666564941406, 1279.873291015625],
                [1.0, 569.2798461914062, 338.6666564941406, 1282.5758056640625],
                [1.0, 682.74951171875, 338.6666564941406, 1390.1435546875],
                [1.0, 567.9761352539062, 338.6666564941406, 1393.18798828125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=67,
            map_id=Maps.FranticFactory,
            name="Under a grate in R&D",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.RandDUpper,
            logic=lambda l: (l.grab and l.donkey) or l.CanAccessRNDRoom(),
            locations=[
                [0.8400001525878906, 4089.829833984375, 1313.7386474609375, 780.3251342773438],
                [0.8400001525878906, 4045.697998046875, 1313.7386474609375, 809.5780639648438],
                [0.8400001525878906, 4057.638916015625, 1313.7386474609375, 735.4220581054688],
                [0.8400001525878906, 4016.709228515625, 1313.7386474609375, 771.4345092773438],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=68,
            map_id=Maps.FranticFactory,
            name="Near Number Game",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Testing,
            locations=[
                [1.0, 2530.98828125, 1002.0, 1659.1370849609375],
                [1.0, 2518.415283203125, 1002.0, 1662.5087890625],
                [1.0, 2528.357666015625, 1002.0, 1675.18017578125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=69,
            map_id=Maps.FranticFactory,
            name="Around Arcade Tunnel Bonus",
            konglist=[Kongs.chunky],
            region=Regions.FactoryArcadeTunnel,
            logic=lambda l: (l.punch and l.ischunky) or l.CanPhase(),
            locations=[
                [1.0, 1343.939208984375, 1106.5, 1885.8740234375],
                [1.0, 1178.9105224609375, 1106.5, 1893.6072998046875],
                [1.0, 1266.9810791015625, 1106.5, 1930.1778564453125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=70,
            map_id=Maps.FranticFactory,
            name="On boxes to Candy and Cranky",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BeyondHatch,
            locations=[
                [1.0, 977.3386840820312, 126.5, 709.9769897460938],
                [1.0, 1007.8041381835938, 94.83333587646484, 778.486328125],
                [1.0, 1096.5106201171875, 26.5, 742.3613891601562],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=73,
            map_id=Maps.FranticFactory,
            name="Near Tiny Twirl Bonus",
            konglist=[Kongs.tiny],
            region=Regions.UpperCore,
            logic=lambda l: l.istiny and l.twirl,
            locations=[
                [1.0, 376.8532409667969, 858.5, 1608.5509033203125],
                [1.0, 386.0055236816406, 858.5, 1584.20751953125],
                [1.0, 400.1599426269531, 858.5, 1600.9425048828125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=74,
            map_id=Maps.FactoryPowerHut,
            name="On top of doorway",
            konglist=[Kongs.donkey],
            region=Regions.PowerHut,
            locations=[
                [0.940000057220459, 119.7491226196289, 47.0, 56.72202682495117],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=75,
            map_id=Maps.FactoryCrusher,
            name="Crusher Start",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.InsideCore,
            locations=[
                [1.0, 528.8129272460938, 0, 528.6017456054688],
                [1.0, 513.3380126953125, 0, 525.7289428710938],
                [1.0, 526.6267700195312, 0, 511.1326904296875],
            ],
            vanilla=True,
        ),
    ],
    Levels.GloomyGalleon: [
        BananaCoinGroup(
            group=0,
            map_id=Maps.GloomyGalleon,
            name="Corner of Lanky Slam Switch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 2629, 1620, 2878],
                [1.0, 2617, 1620, 2862],
                [1.0, 2612, 1620, 2889],
            ],
        ),
        BananaCoinGroup(
            group=1,
            map_id=Maps.GloomyGalleon,
            name="Around near W1",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 2821, 1670, 3234],
                [1.0, 2847, 1671, 3254],
                [1.0, 2864, 1675, 3234],
                [1.0, 2844, 1673, 3214],
            ],
        ),
        BananaCoinGroup(
            group=2,
            map_id=Maps.GloomyGalleon,
            name="Around near W2",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 2887, 1670, 3351],
                [1.0, 2889, 1670, 3386],
                [1.0, 2923, 1670, 3384],
                [1.0, 2922, 1670, 3350],
            ],
        ),
        BananaCoinGroup(
            group=3,
            map_id=Maps.GloomyGalleon,
            name="On top of GB Chest",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 3561, 1725, 3883],
                [1.0, 3539, 1725, 3899],
                [1.0, 3517, 1725, 3915],
            ],
        ),
        BananaCoinGroup(
            group=4,
            map_id=Maps.GloomyGalleon,
            name="On top of headphones chest",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 3668, 1729, 3837],
                [1.0, 3668, 1729, 3806],
                [1.0, 3672, 1729, 3780],
            ],
        ),
        BananaCoinGroup(
            group=5,
            map_id=Maps.GloomyGalleon,
            name="On top of fairy chest",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 3563, 1723, 3720],
                [1.0, 3544, 1723, 3701],
                [1.0, 3523, 1723, 3683],
            ],
        ),
        BananaCoinGroup(
            group=5,
            map_id=Maps.GloomyGalleon,
            name="Around crossroads tag barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 2469, 1618, 3310],
                [1.0, 2470, 1618, 3158],
                [1.0, 2607, 1618, 3148],
                [1.0, 2609, 1618, 3315],
            ],
        ),
        BananaCoinGroup(
            group=6,
            map_id=Maps.GloomyGalleon,
            name="On broken ship near Cranky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 3085, 1790, 3471],
                [1.0, 3093, 1790, 3450],
                [1.0, 3076, 1790, 3436],
                [1.0, 3055, 1790, 3438],
                [1.0, 3056, 1790, 3468],
            ],
        ),
        BananaCoinGroup(
            group=7,
            map_id=Maps.GloomyGalleon,
            name="On Broken Ship railings",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 3147, 1835, 3526],
                [1.0, 3147, 1835, 3431],
                [1.0, 3146, 1835, 3336],
                [1.0, 2997, 1835, 3328],
                [1.0, 2999, 1835, 3423],
                [1.0, 3001, 1835, 3532],
            ],
        ),
        BananaCoinGroup(
            group=8,
            map_id=Maps.GloomyGalleon,
            name="Around Tiny's Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GalleonPastVines,
            locations=[
                [1.0, 2843, 1890, 2931],
                [1.0, 2804, 1890, 2914],
                [1.0, 2775, 1890, 2936],
                [1.0, 2763, 1890, 2974],
                [1.0, 2776, 1890, 3006],
                [1.0, 2819, 1890, 3016],
            ],
        ),
        BananaCoinGroup(
            group=9,
            map_id=Maps.GloomyGalleon,
            name="Around the T&S Portal near Cranky",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GalleonPastVines,
            locations=[
                [1.0, 3423, 1895, 3060],
                [1.0, 3389, 1890, 3124],
            ],
        ),
        BananaCoinGroup(
            group=10,
            map_id=Maps.GloomyGalleon,
            name="Underneath Cranky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            logic=lambda l: (l.punch and l.chunky) or l.CanPhase() or l.CanSkew(False, False),
            locations=[
                [1.0, 3334, 1670, 2555],
                [1.0, 3240, 1670, 2500],
                [1.0, 3225, 1670, 2377],
                [1.0, 3346, 1670, 2348],
                [1.0, 3391, 1670, 2464],
            ],
        ),
        BananaCoinGroup(
            group=11,
            map_id=Maps.GloomyGalleon,
            name="In the corner of the main room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 2940, 1672, 2830],
                [1.0, 3005, 1688, 2830],
                [1.0, 2887, 1688, 2877],
            ],
        ),
        BananaCoinGroup(
            group=12,
            map_id=Maps.GloomyGalleon,
            name="Near Tiny's Slam Switch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 2216, 1620, 2662],
                [1.0, 2214, 1620, 2679],
                [1.0, 2235, 1620, 2661],
            ],
        ),
        BananaCoinGroup(
            group=13,
            map_id=Maps.GloomyGalleon,
            name="Around 5-Door Ship Exterior",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ShipyardUnderwater,
            locations=[
                [1.0, 2192, 1169, 2148],
                [1.0, 2604, 1183, 2130],
                [1.0, 2603, 1189, 1889],
                [1.0, 2191, 1173, 1906],
            ],
        ),
        BananaCoinGroup(
            group=14,
            map_id=Maps.GloomyGalleon,
            name="Above W4 Platform",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Shipyard,
            locations=[
                [1.0, 2709, 1606, 1474],
                [1.0, 2822, 1606, 1470],
                [1.0, 2824, 1606, 1394],
                [1.0, 2707, 1606, 1394],
            ],
        ),
        BananaCoinGroup(
            group=15,
            map_id=Maps.GloomyGalleon,
            name="Near the 2-Door Ship",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ShipyardUnderwater,
            locations=[
                [1.0, 1369, 1094, 1957],
                [1.0, 1390, 1034, 1970],
            ],
        ),
        BananaCoinGroup(
            group=16,
            map_id=Maps.GloomyGalleon,
            name="Above Candy's Platform",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Shipyard,
            locations=[
                [1.0, 2911, 1623, 694],
                [1.0, 2908, 1623, 413],
                [1.0, 2739, 1623, 406],
                [1.0, 2715, 1623, 669],
            ],
        ),
        BananaCoinGroup(
            group=17,
            map_id=Maps.GloomyGalleon,
            name="Seal Race Entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Shipyard,
            logic=lambda l: Events.WaterRaised in l.Events,
            locations=[
                [1.0, 3361, 1640, 81],
                [1.0, 3407, 1640, 110],
            ],
        ),
        BananaCoinGroup(
            group=18,
            map_id=Maps.GloomyGalleon,
            name="Mech Fish Grate",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ShipyardUnderwater,
            locations=[
                [1.0, 3118, 66, 600],
                [1.0, 2981, 67, 569],
                [1.0, 3010, 66, 431],
                [1.0, 3148, 66, 464],
            ],
        ),
        BananaCoinGroup(
            group=19,
            map_id=Maps.GloomyGalleon,
            name="Near BBlast Pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthousePlatform,
            locations=[
                [1.0, 1703, 1660, 4010],
                [1.0, 1697, 1660, 4025],
                [1.0, 1695, 1661, 4070],
                [1.0, 1695, 1660, 4093],
            ],
        ),
        BananaCoinGroup(
            group=20,
            map_id=Maps.GloomyGalleon,
            name="Around DK's slam switch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthousePlatform,
            locations=[
                [1.0, 1517, 1609, 3868],
                [1.0, 1496, 1609, 3894],
                [1.0, 1517, 1609, 3916],
                [1.0, 1537, 1609, 3896],
            ],
        ),
        BananaCoinGroup(
            group=21,
            map_id=Maps.GloomyGalleon,
            name="Around Lighthouse W5",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthousePlatform,
            locations=[
                [1.0, 1370, 1610, 3950],
                [1.0, 1380, 1609, 3992],
                [1.0, 1416, 1609, 3984],
                [1.0, 1406, 1609, 3945],
            ],
        ),
        BananaCoinGroup(
            group=22,
            map_id=Maps.GloomyGalleon,
            name="Snide Alcove Edge",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseSnideAlcove,
            locations=[
                [1.0, 2142, 1610, 4548],
                [1.0, 2079, 1610, 4611],
                [1.0, 2028, 1610, 4666],
                [1.0, 1963, 1610, 4722],
                [1.0, 1903, 1610, 4780],
            ],
        ),
        BananaCoinGroup(
            group=23,
            map_id=Maps.GloomyGalleon,
            name="Blueprint Water Alcove",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseSurface,
            locations=[
                [1.0, 752, 1574, 4213],
                [1.0, 584, 1574, 4201],
                [1.0, 530, 1574, 4047],
                [1.0, 581, 1574, 3890],
                [1.0, 759, 1574, 3870],
            ],
        ),
        BananaCoinGroup(
            group=24,
            map_id=Maps.GloomyGalleon,
            name="Lighthouse Underwater corner (Circle)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseUnderwater,
            locations=[
                [1.0, 1407, 639, 3376],
                [1.0, 1405, 651, 3436],
                [1.0, 1357, 630, 3372],
            ],
        ),
        BananaCoinGroup(
            group=25,
            map_id=Maps.GloomyGalleon,
            name="Lighthouse Underwater corner (Stack)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseUnderwater,
            logic=lambda _: True,
            locations=[
                [1.0, 2209, 931, 4174],
                [1.0, 2209, 861, 4174],
                [1.0, 2205, 796, 4174],
                [1.0, 2205, 724, 4174],
            ],
        ),
        BananaCoinGroup(
            group=26,
            map_id=Maps.GloomyGalleon,
            name="Mermaid Palace Roof",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseUnderwater,
            locations=[
                [1.0, 887, 398, 4619],
                [1.0, 718, 393, 4368],
            ],
        ),
        BananaCoinGroup(
            group=27,
            map_id=Maps.GalleonSickBay,
            name="Behind the Cannon",
            konglist=[Kongs.chunky],
            region=Regions.SickBay,
            locations=[
                [1.0, 614, 20, 89],
            ],
        ),
        BananaCoinGroup(
            group=28,
            map_id=Maps.GalleonSickBay,
            name="In the corner before the grate",
            konglist=[Kongs.chunky],
            region=Regions.SickBay,
            locations=[
                [1.0, 750, 20, 953],
                [1.0, 717, 20, 964],
                [1.0, 751, 20, 913],
            ],
        ),
        BananaCoinGroup(
            group=29,
            map_id=Maps.GalleonLighthouse,
            name="Lighthouse Entry",
            konglist=[Kongs.donkey],
            region=Regions.Lighthouse,
            locations=[
                [1.0, 400, 0, 164],
                [1.0, 394, 0, 136],
                [1.0, 501, 0, 140],
                [1.0, 499, 0, 164],
            ],
        ),
        BananaCoinGroup(
            group=30,
            map_id=Maps.GalleonLighthouse,
            name="Lighthouse Climb (1)",
            konglist=[Kongs.donkey],
            region=Regions.LighthouseAboveLadder,
            locations=[
                [1.0, 449, 460, 365],
                [1.0, 449, 500, 365],
                [1.0, 449, 540, 365],
            ],
        ),
        BananaCoinGroup(
            group=31,
            map_id=Maps.GalleonLighthouse,
            name="Lighthouse Climb (2)",
            konglist=[Kongs.donkey],
            region=Regions.LighthouseAboveLadder,
            locations=[
                [1.0, 381, 640, 533],
                [1.0, 381, 590, 533],
                [1.0, 381, 540, 533],
            ],
        ),
        BananaCoinGroup(
            group=32,
            map_id=Maps.GalleonLighthouse,
            name="Top of Lighthouse",
            konglist=[Kongs.donkey],
            region=Regions.LighthouseAboveLadder,
            locations=[
                [1.0, 553, 720, 401],
                [1.0, 515, 720, 362],
            ],
        ),
        BananaCoinGroup(
            group=33,
            map_id=Maps.GalleonMechafish,
            name="Near Pillars",
            konglist=[Kongs.diddy],
            region=Regions.Mechafish,
            locations=[
                [1.0, 522, 46, 695],
                [1.0, 510, 39, 676],
                [1.0, 541, 39, 650],
                [1.0, 136, 39, 479],
                [1.0, 160, 35, 459],
                [1.0, 133, 39, 424],
            ],
        ),
        BananaCoinGroup(
            group=34,
            map_id=Maps.GalleonMermaidRoom,
            name="On fork in Mermaid Room",
            konglist=[Kongs.tiny],
            region=Regions.MermaidRoom,
            locations=[
                [1.0, 431, 159, 163],
            ],
        ),
        BananaCoinGroup(
            group=35,
            map_id=Maps.GalleonSealRace,
            name="Seal Race Start",
            konglist=[Kongs.donkey],
            region=Regions.SealRace,
            locations=[
                [1.0, 1660, 51, 1748],
                [1.0, 1656, 46, 1446],
            ],
        ),
        BananaCoinGroup(
            group=36,
            map_id=Maps.Galleon5DShipDiddyLankyChunky,
            name="Above Diddy Cannons",
            konglist=[Kongs.diddy],
            region=Regions.GuitarShip,
            locations=[
                [1.0, 1585, 74, 746],
                [1.0, 1588, 74, 1063],
            ],
        ),
        BananaCoinGroup(
            group=37,
            map_id=Maps.Galleon5DShipDiddyLankyChunky,
            name="On Diddy Barrel",
            konglist=[Kongs.diddy],
            region=Regions.GuitarShip,
            locations=[
                [1.0, 1150, 1, 811],
                [1.0, 1115, 50, 811],
            ],
        ),
        BananaCoinGroup(
            group=38,
            map_id=Maps.Galleon5DShipDiddyLankyChunky,
            name="On Lanky Barrel",
            konglist=[Kongs.lanky],
            region=Regions.TromboneShip,
            locations=[
                [1.0, 984, 47, 1587],
                [1.0, 1056, 1, 1581],
            ],
        ),
        BananaCoinGroup(
            group=39,
            map_id=Maps.Galleon5DShipDiddyLankyChunky,
            name="On Lanky Bench",
            konglist=[Kongs.lanky],
            region=Regions.TromboneShip,
            locations=[
                [1.0, 787, 52, 1586],
                [1.0, 894, 60, 1588],
            ],
        ),
        BananaCoinGroup(
            group=40,
            map_id=Maps.Galleon5DShipDiddyLankyChunky,
            name="Near Chunky Boxes",
            konglist=[Kongs.chunky],
            region=Regions.TriangleShip,
            locations=[
                [1.0, 1585, 41, 1322],
                [1.0, 1563, 42, 1322],
                [1.0, 1584, 38, 1252],
            ],
        ),
        BananaCoinGroup(
            group=41,
            map_id=Maps.Galleon5DShipDKTiny,
            name="Corner of DK Bonus Room",
            konglist=[Kongs.donkey],
            region=Regions.BongosShip,
            locations=[
                [1.0, 1215, 31, 494],
                [1.0, 1190, 31, 473],
                [1.0, 1160, 31, 471],
            ],
        ),
        BananaCoinGroup(
            group=42,
            map_id=Maps.Galleon5DShipDKTiny,
            name="Bed in DK Ship",
            konglist=[Kongs.donkey],
            region=Regions.BongosShip,
            locations=[
                [1.0, 317, 36, 466],
                [1.0, 314, 34, 530],
                [1.0, 329, 35, 547],
            ],
        ),
        BananaCoinGroup(
            group=43,
            map_id=Maps.Galleon2DShip,
            name="Corner of Tiny Entrance",
            konglist=[Kongs.tiny],
            region=Regions.TinyShip,
            locations=[
                [1.0, 55, 1, 65],
                [1.0, 628, 1, 67],
                [1.0, 625, 1, 311],
            ],
        ),
        BananaCoinGroup(
            group=44,
            map_id=Maps.Galleon2DShip,
            name="Corners of Tiny's Main Room",
            konglist=[Kongs.tiny],
            region=Regions.TinyShip,
            locations=[
                [1.0, 55, 1, 577],
                [1.0, 54, 1, 922],
                [1.0, 1001, 1, 920],
            ],
        ),
        BananaCoinGroup(
            group=45,
            map_id=Maps.Galleon2DShip,
            name="Start of Lanky's Room",
            konglist=[Kongs.lanky],
            region=Regions.LankyShip,
            locations=[
                [1.0, 2579, 1, 799],
                [1.0, 2574, 6, 702],
            ],
        ),
        BananaCoinGroup(
            group=46,
            map_id=Maps.GalleonSubmarine,
            name="Behind the desk",
            konglist=[Kongs.tiny],
            region=Regions.Submarine,
            locations=[
                [1.0, 440, 2, 859],
                [1.0, 438, 3, 748],
                [1.0, 437, 2, 626],
            ],
        ),
        BananaCoinGroup(
            group=47,
            map_id=Maps.GalleonSubmarine,
            name="In the corner",
            konglist=[Kongs.tiny],
            region=Regions.Submarine,
            locations=[
                [1.0, 764, 1, 1098],
                [1.0, 760, 2, 1055],
                [1.0, 795, 1, 1098],
            ],
        ),
        BananaCoinGroup(
            group=48,
            map_id=Maps.GalleonTreasureChest,
            name="Around the keyhole",
            konglist=[Kongs.tiny],
            region=Regions.TinyChest,
            locations=[
                [1.0, 1306, 501, 1910],
                [1.0, 1198, 509, 1912],
                [1.0, 1218, 783, 1914],
                [1.0, 1302, 776, 1908],
            ],
        ),
        BananaCoinGroup(
            group=49,
            map_id=Maps.GloomyGalleon,
            name="On Mermaid Palace Shells",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseUnderwater,
            locations=[
                [1.0, 1039.66650390625, 148.3982391357422, 4191.85595703125],
                [1.0, 1136.514404296875, 149.2041473388672, 4299.54296875],
                [1.0, 894.3958129882812, 149.93450927734375, 4160.53369140625],
                [1.0, 1186.3199462890625, 148.81101989746094, 4439.0556640625],
                [1.0, 1155.2606201171875, 149.1146240234375, 4568.43603515625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=50,
            map_id=Maps.GloomyGalleon,
            name="Behind Enguarde Door",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseEnguardeDoor,
            locations=[
                [1.0, 1004.564697265625, 1460.0, 4952.32666015625],
                [1.0, 996.052978515625, 1460.0, 4959.49755859375],
                [1.0, 989.5244140625, 1460.0, 4943.47216796875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=51,
            map_id=Maps.GloomyGalleon,
            name="On Rafter near Funky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Shipyard,
            logic=lambda l: Events.WaterRaised in l.Events or l.CanMoonkick(),
            locations=[
                [1.0, 3645.180419921875, 1648.24267578125, 1823.3914794921875],
                [1.0, 3509.1142578125, 1637.95849609375, 1949.4278564453125],
                [1.0, 3544.777587890625, 1636.423828125, 1973.5443115234375],
                [1.0, 3674.367919921875, 1647.2513427734375, 1854.24853515625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=52,
            map_id=Maps.GloomyGalleon,
            name="Around Cranky's Lab",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 3215.933837890625, 1790.0, 2387.90576171875],
                [1.0, 3402.533447265625, 1791.8359375, 2475.832275390625],
                [1.0, 3348.2275390625, 1803.8118896484375, 2553.47021484375],
                [1.0, 3231.945556640625, 1790.0, 2492.75732421875],
                [1.0, 3365.507080078125, 1790.0, 2348.406982421875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=53,
            map_id=Maps.GloomyGalleon,
            name="Tunnel to Chests Area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GloomyGalleonStart,
            locations=[
                [1.0, 3035.781982421875, 1670.0, 3837.2080078125],
                [1.0, 3013.817138671875, 1670.0, 3841.261474609375],
                [1.0, 3030.2734375, 1670.0, 3858.253173828125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=54,
            map_id=Maps.GloomyGalleon,
            name="Around Cactus",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ShipyardUnderwater,
            locations=[
                [1.0, 4304.78076171875, 1290.0, 1048.9107666015625],
                [1.0, 4396.90380859375, 1292.69140625, 1109.5670166015625],
                [1.0, 4350.005859375, 1296.0, 966.343017578125],
                [1.0, 4453.4365234375, 1290.0, 1008.7509765625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=55,
            map_id=Maps.GloomyGalleon,
            name="On Seal Cage",
            konglist=[Kongs.diddy],
            region=Regions.LighthousePlatform,
            logic=lambda l: l.jetpack and l.isdiddy,
            locations=[
                [1.0, 2166.177978515625, 1851.5, 4070.290771484375],
                [1.0, 2176.599365234375, 1851.5, 4082.485595703125],
                [1.0, 2181.55859375, 1851.5, 4066.402587890625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=56,
            map_id=Maps.GloomyGalleon,
            name="Around Lighthouse W1",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthousePlatform,
            locations=[
                [1.0, 1523.8978271484375, 1610.8031005859375, 4194.34814453125],
                [1.0, 1551.3348388671875, 1610.5323486328125, 4224.71142578125],
                [1.0, 1491.71923828125, 1610.3836669921875, 4223.92041015625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=57,
            map_id=Maps.GloomyGalleon,
            name="Behind Cannon Game Room gate",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GalleonBeyondPineappleGate,
            locations=[
                [1.0, 1653.0989990234375, 1523.3648681640625, 2975.35546875],
                [1.0, 1653.99072265625, 1524.5118408203125, 2988.74609375],
                [1.0, 1633.55322265625, 1520.7216796875, 2985.838134765625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=58,
            map_id=Maps.GloomyGalleon,
            name="Under Lighthouse Enguarde Crate",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseUnderwater,
            locations=[
                [1.0, 1998.7457275390625, 963.54345703125, 3699.846923828125],
                [1.0, 1981.14306640625, 961.11328125, 3694.167236328125],
                [1.0, 1985.8995361328125, 955.9008178710938, 3702.193115234375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=59,
            map_id=Maps.GloomyGalleon,
            name="Cannon Game Room Corner",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GalleonBeyondPineappleGate,
            locations=[
                [1.0, 1423.9552001953125, 1537.68408203125, 2654.234619140625],
                [1.0, 1433.69140625, 1538.6029052734375, 2675.133056640625],
                [1.0, 1418.5946044921875, 1538.5130615234375, 2675.447998046875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=60,
            map_id=Maps.GloomyGalleon,
            name="Inside Shipyard Chest (1)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ShipyardUnderwater,
            logic=lambda l: Events.ShipyardEnguarde in l.Events or l.CanPhaseswim(),
            locations=[
                [1.0, 1980.0323486328125, 650.776611328125, 2230.565185546875],
                [1.0, 1983.3388671875, 649.1094360351562, 2241.46826171875],
                [1.0, 1980.2039794921875, 648.0787353515625, 2247.15625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=61,
            map_id=Maps.GloomyGalleon,
            name="Inside Shipyard Chest (2)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ShipyardUnderwater,
            logic=lambda l: Events.ShipyardEnguarde in l.Events or l.CanPhaseswim(),
            locations=[
                [1.0, 2428.680908203125, 614.0368041992188, 1582.86962890625],
                [1.0, 2429.461181640625, 615.6483154296875, 1587.7398681640625],
                [1.0, 2430.287109375, 618.4383544921875, 1596.88671875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=62,
            map_id=Maps.GloomyGalleon,
            name="Inside Shipyard Chest (3)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ShipyardUnderwater,
            logic=lambda l: Events.ShipyardEnguarde in l.Events or l.CanPhaseswim(),
            locations=[
                [1.0, 3012.022216796875, 574.6257934570312, 1582.55908203125],
                [1.0, 3010.5576171875, 574.5258178710938, 1578.3270263671875],
                [1.0, 3008.313720703125, 574.3427734375, 1577.35888671875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=63,
            map_id=Maps.GloomyGalleon,
            name="Inside Shipyard Chest (4)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ShipyardUnderwater,
            logic=lambda l: Events.ShipyardEnguarde in l.Events or l.CanPhaseswim(),
            locations=[
                [1.0, 3016.286865234375, 101.37421417236328, 739.2562866210938],
                [1.0, 3021.080078125, 101.37618255615234, 739.508544921875],
                [1.0, 3025.873779296875, 101.37813568115234, 739.7607421875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=64,
            map_id=Maps.GloomyGalleon,
            name="Around Cactus Top",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Shipyard,
            locations=[
                [1.0, 4389.86669921875, 1650.0, 1060.6160888671875],
                [1.0, 4415.75244140625, 1650.0, 1023.9403686523438],
                [1.0, 4386.0537109375, 1650.0, 996.3543090820312],
                [1.0, 4351.97802734375, 1650.0, 1015.3634643554688],
                [1.0, 4353.00830078125, 1650.0, 1054.1190185546875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=65,
            map_id=Maps.GloomyGalleon,
            name="Under Mermaid Tag Barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseUnderwater,
            locations=[
                [1.0, 901.1618041992188, 328.4149475097656, 3913.3544921875],
                [1.0, 922.6913452148438, 326.8333435058594, 3920.7138671875],
                [1.0, 928.5752563476562, 326.8333435058594, 3949.272216796875],
                [1.0, 907.58203125, 326.8333435058594, 3957.786865234375],
                [1.0, 892.7731323242188, 326.8333435058594, 3935.9453125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=66,
            map_id=Maps.GloomyGalleon,
            name="Down the deep hole",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LighthouseUnderwater,
            locations=[
                [1.0, 1863.68603515625, 100.0, 4505.087890625],
                [1.0, 1865.4154052734375, 100.0, 4518.79248046875],
                [1.0, 1852.2203369140625, 100.0, 4513.89013671875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=67,
            map_id=Maps.GalleonSickBay,
            name="On Barrels near the spinning totem",
            konglist=[Kongs.chunky],
            region=Regions.SickBay,
            logic=lambda l: (l.punch and l.ischunky) or l.CanPhase(),
            locations=[
                [1.0, 265.6437072753906, 116.50603485107422, 1026.2169189453125],
                [1.0, 308.7044982910156, 67.35160064697266, 1026.9432373046875],
                [1.0399999618530273, 200.70228576660156, 69.09432983398438, 1024.726318359375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=68,
            map_id=Maps.Galleon5DShipDiddyLankyChunky,
            name="Under Lanky Bench",
            konglist=[Kongs.lanky],
            region=Regions.TromboneShip,
            locations=[
                [1.0, 827.8541870117188, 0, 1559.57421875],
                [1.0, 852.068359375, 0, 1560.33251953125],
                [1.0, 839.7125854492188, 0, 1540.63232421875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=69,
            map_id=Maps.Galleon5DShipDiddyLankyChunky,
            name="Near Chunky Treasure Chest",
            konglist=[Kongs.chunky],
            region=Regions.TriangleShip,
            locations=[
                [1.0, 1584.1849365234375, 0, 1496.3468017578125],
                [1.0, 1582.7076416015625, 0, 1482.98876953125],
                [1.0, 1571.2100830078125, 0, 1493.0616455078125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=70,
            map_id=Maps.GalleonTreasureChest,
            name="On the coin floor",
            konglist=[Kongs.tiny],
            region=Regions.TinyChest,
            locations=[
                [3.0199995040893555, 1287.526123046875, 121.51451873779297, 1489.7413330078125],
                [3.0199990272521973, 1286.575439453125, 122.8635482788086, 846.6179809570312],
                [3.0199990272521973, 942.6312866210938, 83.83333587646484, 1151.8057861328125],
                [3.0199990272521973, 1615.16064453125, 168.4228057861328, 1188.8363037109375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=71,
            map_id=Maps.GalleonMermaidRoom,
            name="In the corner",
            konglist=[Kongs.tiny],
            region=Regions.MermaidRoom,
            locations=[
                [1.0, 395.8691711425781, -2.3596036434173584, 354.9566650390625],
                [1.0, 416.7943420410156, -3.0050084590911865, 344.0810546875],
                [1.0, 396.5152587890625, -1.045554280281067, 334.0840148925781],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=72,
            map_id=Maps.Galleon5DShipDKTiny,
            name="On boxes near bed",
            konglist=[Kongs.tiny],
            region=Regions.SaxophoneShip,
            locations=[
                [1.1999998092651367, 706.3814086914062, 3.0, 1091.90478515625],
                [1.119999885559082, 672.1433715820312, 37.0, 1094.9642333984375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=73,
            map_id=Maps.Galleon5DShipDKTiny,
            name="At a dead end",
            konglist=[Kongs.donkey],
            region=Regions.BongosShip,
            locations=[
                [1.0, 153.89437866210938, 0, 1258.3231201171875],
                [1.0, 162.85096740722656, 0, 1252.719482421875],
                [1.0, 157.5319061279297, 0, 1243.0726318359375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=74,
            map_id=Maps.Galleon2DShip,
            name="In various chests",
            konglist=[Kongs.lanky],
            region=Regions.LankyShip,
            locations=[
                [1.0, 1552.4559326171875, 15.0, 612.2144165039062],
                [1.0, 2409.232666015625, 15.0, 863.0134887695312],
                [1.0, 2189.744873046875, 15.0, 622.9729614257812],
                [1.0, 1867.0992431640625, 26.16666603088379, 996.242431640625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=75,
            map_id=Maps.GalleonBaboonBlast,
            name="In the Baboon Blast Course",
            konglist=[Kongs.donkey],
            region=Regions.GalleonBaboonBlast,
            locations=[
                [1.0, 982.7968139648438, 531.0634155273438, 634.3992309570312],
                [1.0, 675.5610961914062, 535.4956665039062, 764.8893432617188],
            ],
            vanilla=True,
        ),
    ],
    Levels.FungiForest: [
        BananaCoinGroup(
            group=0,
            map_id=Maps.FungiForest,
            name="Left of Green Tunnel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 2607, 200, 1614],
                [1.0, 2594, 202, 1641],
                [1.0, 2591, 200, 1608],
            ],
        ),
        BananaCoinGroup(
            group=1,
            map_id=Maps.FungiForest,
            name="Around apple dropoff zone",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 3025, 206, 1753],
                [1.0, 2890, 211, 1878],
                [1.0, 2872, 227, 2052],
                [1.0, 2987, 211, 2210],
                [1.0, 3183, 191, 2265],
                [1.0, 3282, 182, 2163],
            ],
        ),
        BananaCoinGroup(
            group=2,
            map_id=Maps.FungiForest,
            name="Right of blue tunnel entry",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 3246, 166, 2594],
                [1.0, 3240, 166, 2615],
                [1.0, 3223, 166, 2588],
            ],
        ),
        BananaCoinGroup(
            group=3,
            map_id=Maps.FungiForest,
            name="Around well stump",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 2238, 102, 3071],
                [1.0, 2281, 105, 3054],
                [1.0, 2265, 109, 3008],
                [1.0, 2221, 103, 3024],
            ],
        ),
        BananaCoinGroup(
            group=4,
            map_id=Maps.FungiForest,
            name="On well roof edges",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            logic=lambda l: (l.climbing and (l.can_use_vines or l.CanMoonkick())) or (l.jetpack and l.isdiddy),
            locations=[
                [1.0, 2306, 268, 3133],
                [1.0, 2248, 273, 3051],
                [1.0, 2145, 277, 3031],
            ],
        ),
        BananaCoinGroup(
            group=5,
            map_id=Maps.FungiForest,
            name="Mushroom Stem",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            logic=lambda l: l.climbing or (l.jetpack and l.isdiddy),
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 2204, 214, 2699],
                [1.0, 2200, 254, 2706],
                [1.0, 2197, 291, 2722],
                [1.0, 2194, 331, 2720],
            ],
        ),
        BananaCoinGroup(
            group=6,
            map_id=Maps.FungiForest,
            name="On Clock Pendulum",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            logic=lambda l: l.climbing,
            locations=[
                [1.0, 2531, 457, 2239],
                [1.0, 2531, 441, 2304],
                [1.0, 2530, 451, 2391],
            ],
        ),
        BananaCoinGroup(
            group=7,
            map_id=Maps.FungiForest,
            name="On Clock Torches",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 2538, 664, 2263],
                [1.0, 2538, 664, 2387],
            ],
        ),
        BananaCoinGroup(
            group=8,
            map_id=Maps.FungiForest,
            name="Level Entry",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 2209, 177, 1596],
                [1.0, 2229, 180, 1594],
                [1.0, 2226, 182, 1615],
            ],
        ),
        BananaCoinGroup(
            group=9,
            map_id=Maps.FungiForest,
            name="On bouncy mushroom in clock area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 2035, 448, 2093],
                [1.0, 2035, 409, 2093],
                [1.0, 2035, 353, 2093],
            ],
        ),
        BananaCoinGroup(
            group=10,
            map_id=Maps.FungiForest,
            name="Left of pink tunnel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 1735, 179, 2078],
            ],
        ),
        BananaCoinGroup(
            group=11,
            map_id=Maps.FungiForest,
            name="Near bouncy mushroom tag in the Giant Mushroom area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GiantMushroomArea,
            locations=[
                [1.0, 1749, 234, 930],
                [1.0, 1736, 234, 943],
                [1.0, 1730, 234, 918],
            ],
        ),
        BananaCoinGroup(
            group=12,
            map_id=Maps.FungiForest,
            name="Around a torch in the outer Giant Mushroom area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GiantMushroomArea,
            locations=[
                [1.0, 1340, 186, 546],
                [1.0, 1339, 184, 525],
                [1.0, 1320, 186, 537],
            ],
        ),
        BananaCoinGroup(
            group=13,
            map_id=Maps.FungiForest,
            name="Above a patch of grass in the Giant Mushroom area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GiantMushroomArea,
            locations=[
                [1.0, 225, 200, 1299],
                [1.0, 237, 200, 1273],
                [1.0, 250, 200, 1303],
            ],
        ),
        BananaCoinGroup(
            group=14,
            map_id=Maps.FungiForest,
            name="On the pink tunnel rim on the Giant Mushroom side",
            konglist=[Kongs.diddy],
            region=Regions.GiantMushroomArea,
            logic=lambda l: l.jetpack and l.isdiddy,
            locations=[
                [1.0, 1583, 303, 1591],
                [1.0, 1608, 331, 1536],
                [1.0, 1646, 303, 1502],
            ],
        ),
        BananaCoinGroup(
            group=15,
            map_id=Maps.FungiForest,
            name="Behind the lowest ladder in the Giant Mushroom area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomLowerExterior,
            locations=[
                [1.0, 464, 389, 931],
                [1.0, 384, 389, 867],
            ],
        ),
        BananaCoinGroup(
            group=16,
            map_id=Maps.FungiForest,
            name="Around the lower platform in the Giant Mushroom Area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomLowerExterior,
            locations=[
                [1.0, 709, 389, 780],
                [1.0, 629, 389, 905],
                [1.0, 1037, 389, 746],
                [1.0, 1149, 389, 832],
                [1.0, 1059, 389, 1200],
                [1.0, 1112, 389, 1132],
            ],
        ),
        BananaCoinGroup(
            group=17,
            map_id=Maps.FungiForest,
            name="Behind the second lowest ladder in the Giant Mushroom Area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomBlastLevelExterior,
            locations=[
                [1.0, 705, 589, 1256],
                [1.0, 727, 589, 1273],
            ],
        ),
        BananaCoinGroup(
            group=18,
            map_id=Maps.FungiForest,
            name="Above the second lowest ladder in the Giant Mushroom Area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomUpperMidExterior,
            locations=[
                [1.0, 679, 779, 1365],
                [1.0, 622, 779, 1323],
            ],
        ),
        BananaCoinGroup(
            group=19,
            map_id=Maps.FungiForest,
            name="On the entryways to the 3 top rooms in the Giant Mushroom Area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomUpperExterior,
            locations=[
                [1.0, 841, 1250, 1428],
                [1.0, 1282, 1250, 876],
                [1.0, 968, 1250, 626],
            ],
        ),
        BananaCoinGroup(
            group=20,
            map_id=Maps.FungiForest,
            name="On the Giant Mushroom top",
            konglist=[Kongs.diddy, Kongs.lanky],
            region=Regions.MushroomUpperExterior,
            logic=lambda l: (l.jetpack and l.isdiddy) or (l.islanky and l.handstand),
            locations=[
                [1.0, 1078, 1432, 1177],
                [1.0, 1142, 1429, 1034],
                [1.0, 1142, 1420, 931],
                [1.0, 1042, 1430, 833],
                [1.0, 942, 1446, 827],
                [1.0, 774, 1407, 772],
                [1.0, 672, 1410, 873],
                [1.0, 671, 1428, 979],
                [1.0, 631, 1400, 1139],
                [1.0, 789, 1433, 1225],
                [1.0, 872, 1466, 1182],
                [1.0, 975, 1415, 1282],
            ],
        ),
        BananaCoinGroup(
            group=21,
            map_id=Maps.FungiForest,
            name="On the Rabbit Race roof",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.HollowTreeArea,
            locations=[
                [1.0, 2316, 168, 3809],
                [1.0, 2311, 175, 3761],
                [1.0, 2362, 173, 3807],
                [1.0, 2312, 171, 3860],
                [1.0, 2265, 178, 3816],
            ],
        ),
        BananaCoinGroup(
            group=22,
            map_id=Maps.FungiForest,
            name="Around the Saxophone pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.HollowTreeArea,
            locations=[
                [1.0, 1255, 291, 4595],
                [1.0, 1274, 294, 4620],
                [1.0, 1305, 291, 4592],
                [1.0, 1277, 288, 4569],
            ],
        ),
        BananaCoinGroup(
            group=24,
            map_id=Maps.FungiForest,
            name="On the stump at the start of the hollow tree area",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.HollowTreeArea,
            locations=[
                [1.0, 1287, 286, 2918],
                [1.0, 1287, 248, 2918],
            ],
        ),
        BananaCoinGroup(
            group=25,
            map_id=Maps.FungiForest,
            name="Around the shop in the Worm area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.WormArea,
            locations=[
                [1.0, 3509, 177, 246],
                [1.0, 3398, 184, 296],
                [1.0, 3149, 183, 225],
                [1.0, 3020, 170, 82],
            ],
        ),
        BananaCoinGroup(
            group=27,
            map_id=Maps.FungiForest,
            name="Beanstalk Fence",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.WormArea,
            locations=[
                [1.0, 2076, 259, 767],
                [1.0, 1904, 255, 714],
                [1.0, 1911, 254, 475],
                [1.0, 2027, 245, 407],
            ],
        ),
        BananaCoinGroup(
            group=28,
            map_id=Maps.FungiForest,
            name="On the Green Tunnel Roof",
            konglist=[Kongs.tiny],
            region=Regions.WormArea,
            logic=lambda l: l.twirl and l.istiny,
            locations=[
                [1.0, 3158, 328, 1197],
                [1.0, 3101, 362, 1175],
                [1.0, 3037, 323, 1164],
            ],
        ),
        BananaCoinGroup(
            group=29,
            map_id=Maps.FungiForest,
            name="On a mushroom in the Worm Area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.WormArea,
            locations=[
                [1.0, 3469, 378, 526],
                [1.0, 3516, 378, 447],
                [1.0, 3533, 378, 509],
                [1.0, 3459, 379, 462],
            ],
        ),
        BananaCoinGroup(
            group=30,
            map_id=Maps.FungiForest,
            name="On the waterwheel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ForestTopOfMill,
            locations=[
                [1.0, 4117, 208, 3369],
                [1.0, 4108, 259, 3390],
                [1.0, 4061, 287, 3427],
                [1.0, 4015, 263, 3465],
                [1.0, 3997, 206, 3482],
            ],
        ),
        BananaCoinGroup(
            group=31,
            map_id=Maps.FungiForest,
            name="In the Snide's area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MillArea,
            locations=[
                [1.0, 3337, 270, 3714],
                [1.0, 3312, 243, 3603],
                [1.0, 3287, 235, 3462],
                [1.0, 3251, 268, 3363],
            ],
            logic=lambda l: (l.TimeAccess(Regions.MillArea, Time.Day)),
        ),
        BananaCoinGroup(
            group=32,
            map_id=Maps.FungiForest,
            name="Near the entrance to the Dark Rafters",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MillArea,
            locations=[
                [1.0, 3400, 272, 4510],
                [1.0, 3451, 272, 4582],
                [1.0, 3459, 272, 4472],
                [1.0, 3503, 272, 4550],
            ],
        ),
        BananaCoinGroup(
            group=33,
            map_id=Maps.FungiForest,
            name="Hanging off the mill roof",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MillArea,
            locations=[
                [1.0, 4392, 360, 3800],
                [1.0, 4414, 360, 3779],
                [1.0, 4445, 360, 3748],
            ],
        ),
        BananaCoinGroup(
            group=34,
            map_id=Maps.FungiForest,
            name="In front of the mill in the air",
            konglist=[Kongs.lanky],
            region=Regions.MillArea,
            logic=lambda l: l.balloon and l.islanky,
            locations=[
                [1.0, 4012, 330, 3773],
                [1.0, 4080, 330, 3846],
                [1.0, 4148, 330, 3910],
            ],
        ),
        BananaCoinGroup(
            group=35,
            map_id=Maps.FungiForest,
            name="Behind the well exit",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MillArea,
            locations=[
                [1.0, 5450, 230, 3714],
                [1.0, 5433, 230, 3687],
                [1.0, 5421, 230, 3724],
            ],
        ),
        BananaCoinGroup(
            group=36,
            map_id=Maps.FungiForest,
            name="Near the thornvine barn",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.ThornvineArea,
            locations=[
                [1.0, 4022, 115, 2252],
                [1.0, 4020, 115, 2221],
                [1.0, 4032, 115, 2234],
            ],
        ),
        BananaCoinGroup(
            group=37,
            map_id=Maps.ForestAnthill,
            name="Around the cannon",
            konglist=[Kongs.tiny],
            region=Regions.Anthill,
            locations=[
                [1.0, 444, 205, 707],
                [1.0, 409, 205, 795],
                [1.0, 377, 205, 725],
                [1.0, 479, 205, 765],
            ],
        ),
        BananaCoinGroup(
            group=38,
            map_id=Maps.ForestWinchRoom,
            name="On flour bags",
            konglist=[Kongs.diddy],
            region=Regions.WinchRoom,
            locations=[
                [1.0, 162, 18, 417],
                [1.0, 180, 13, 427],
                [1.0, 84, 25, 425],
            ],
        ),
        BananaCoinGroup(
            group=39,
            map_id=Maps.ForestRafters,
            name="On the torch at the start",
            konglist=[Kongs.diddy],
            region=Regions.MillRafters,
            locations=[
                [1.0, 367, 44, 40],
            ],
        ),
        BananaCoinGroup(
            group=40,
            map_id=Maps.ForestRafters,
            name="On various rafters",
            konglist=[Kongs.diddy],
            region=Regions.MillRafters,
            logic=lambda l: l.guitar and l.isdiddy,
            locations=[
                [1.0, 247, 0, 561],
                [1.0, 246, -25, 481],
                [1.0, 558, -25, 482],
                [1.0, 555, 0, 363],
            ],
        ),
        BananaCoinGroup(
            group=41,
            map_id=Maps.ForestThornvineBarn,
            name="On the fence",
            konglist=[Kongs.donkey],
            region=Regions.ThornvineBarn,
            locations=[
                [1.0, 172, 70, 104],
                [1.0, 175, 70, 152],
                [1.0, 177, 69, 196],
            ],
        ),
        BananaCoinGroup(
            group=42,
            map_id=Maps.ForestThornvineBarn,
            name="Behind some boxes",
            konglist=[Kongs.donkey],
            region=Regions.ThornvineBarn,
            locations=[
                [1.0, 559, 4, 208],
                [1.0, 592, 4, 527],
                [1.0, 94, 4, 648],
            ],
        ),
        BananaCoinGroup(
            group=43,
            map_id=Maps.ForestMillFront,
            name="On the levers",
            konglist=[Kongs.donkey],
            region=Regions.GrinderRoom,
            logic=lambda l: l.isdonkey and (l.CanSlamSwitch(Levels.FungiForest, 2) or l.CanPhase() or l.generalclips),
            locations=[
                [1.0, 560, 1, 159],
                [1.0, 500, 1, 163],
                [1.0, 441, 1, 160],
            ],
        ),
        BananaCoinGroup(
            group=44,
            map_id=Maps.ForestMillFront,
            name="In front of the keg painting",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GrinderRoom,
            locations=[
                [1.0, 421, 70, 636],
            ],
        ),
        BananaCoinGroup(
            group=45,
            map_id=Maps.ForestMillFront,
            name="On support beams",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.GrinderRoom,
            locations=[
                [1.0, 26, 137, 237],
                [1.0, 26, 137, 199],
            ],
        ),
        BananaCoinGroup(
            group=46,
            map_id=Maps.ForestGiantMushroom,
            name="At the bottom of ramps",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomLower,
            locations=[
                [1.0, 159, 0, 525],
                [1.0, 52, 1, 526],
                [1.0, 741, 0, 531],
                [1.0, 844, 2, 538],
            ],
        ),
        BananaCoinGroup(
            group=47,
            map_id=Maps.ForestGiantMushroom,
            name="At the dead ends at the bottom",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomLowerBetweenLadders,
            locations=[
                [1.0, 110, 278, 529],
                [1.0, 790, 278, 533],
            ],
        ),
        BananaCoinGroup(
            group=48,
            map_id=Maps.ForestGiantMushroom,
            name="Under a pathway",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomUpperVineFloor,
            locations=[
                [1.0, 823, 1169, 394],
                [1.0, 697, 1168, 356],
                [1.0, 735, 1169, 472],
            ],
        ),
        BananaCoinGroup(
            group=49,
            map_id=Maps.ForestMillAttic,
            name="Behind the starting box",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MillAttic,
            locations=[
                [1.0, 438, 0, 136],
                [1.0, 440, 0, 107],
            ],
        ),
        BananaCoinGroup(
            group=50,
            map_id=Maps.ForestLankyZingersRoom,
            name="In the shadows",
            konglist=[Kongs.lanky],
            region=Regions.MushroomLankyZingersRoom,
            locations=[
                [1.0, 411, 0, 431],
                [1.0, 373, 0, 396],
                [1.0, 347, 0, 476],
            ],
        ),
        BananaCoinGroup(
            group=51,
            map_id=Maps.ForestMillBack,
            name="On the question mark box",
            konglist=[Kongs.tiny, Kongs.chunky],
            region=Regions.MillChunkyTinyArea,
            locations=[
                [1.0, 603, 66, 188],
                [1.0, 649, 65, 191],
                [1.0, 648, 65, 139],
                [1.0, 596, 66, 136],
            ],
        ),
        BananaCoinGroup(
            group=52,
            map_id=Maps.ForestMillBack,
            name="Inside a keg",
            konglist=[Kongs.chunky],
            region=Regions.MillChunkyTinyArea,
            logic=lambda l: l.barrels,
            locations=[
                [1.0, 475, 0, 239],
            ],
        ),
        BananaCoinGroup(
            group=53,
            map_id=Maps.ForestMillBack,
            name="On boxes near spider",
            konglist=[Kongs.tiny, Kongs.chunky],
            region=Regions.MillChunkyTinyArea,
            locations=[
                [1.0, 215, 51, 121],
                [1.0, 146, 61, 110],
                [1.0, 212, 62, 171],
            ],
        ),
        BananaCoinGroup(
            group=54,
            map_id=Maps.FungiForest,
            name="On Thorny Hedge",
            konglist=[Kongs.donkey],
            region=Regions.ThornvineArea,
            logic=lambda l: l.isdonkey and l.strongKong,
            locations=[
                [1.0, 3749.895263671875, 210.0, 1661.8482666015625],
                [1.0, 3759.383056640625, 210.0, 1654.7012939453125],
                [1.0, 3738.197021484375, 210.0, 1664.0355224609375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=55,
            map_id=Maps.FungiForest,
            name="Around BBlast Pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomBlastLevelExterior,
            locations=[
                [1.0, 324.719970703125, 589.3333129882812, 973.1728515625],
                [1.0, 321.9471130371094, 589.3333129882812, 922.9804077148438],
                [1.0, 376.1307678222656, 589.3333129882812, 960.1115112304688],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=56,
            map_id=Maps.FungiForest,
            name="Behind Clock",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 2345.25341796875, 603.6666870117188, 2325.597412109375],
                [1.0, 2331.1953125, 603.6666870117188, 2335.5341796875],
                [1.0, 2331.8251953125, 603.6666870117188, 2319.580078125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=57,
            map_id=Maps.FungiForest,
            name="On Mushroom near Mill far tag",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MillArea,
            logic=lambda l: l.climbing,
            locations=[
                [1.0, 4624.845703125, 369.134765625, 4055.068115234375],
                [1.0, 4616.716796875, 369.8291015625, 4052.482666015625],
                [1.0, 4618.90478515625, 369.34490966796875, 4057.671875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=58,
            map_id=Maps.FungiForest,
            name="Around Crown Pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomUpperExterior,
            locations=[
                [1.0, 1233.3140869140625, 1079.3333740234375, 1357.3380126953125],
                [1.0, 1302.4144287109375, 1079.3333740234375, 1312.1282958984375],
                [1.0, 1243.2852783203125, 1079.3333740234375, 1259.4718017578125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=59,
            map_id=Maps.FungiForest,
            name="Around Tiny Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomLowerExterior,
            locations=[
                [1.0, 1174.5614013671875, 389.3333435058594, 547.6068115234375],
                [1.0, 1268.591796875, 389.3333435058594, 580.3642578125],
                [1.0, 1283.2174072265625, 389.3333435058594, 760.4766235351562],
                [1.0, 1315.27294921875, 389.3333435058594, 670.2959594726562],
                [1.0, 1080.0330810546875, 389.3333435058594, 602.4680786132812],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=60,
            map_id=Maps.FungiForest,
            name="On Mushroom near Dark Rafters Barn",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MillArea,
            logic=lambda l: l.climbing,
            locations=[
                [1.0, 3714.3486328125, 412.55731201171875, 3885.161376953125],
                [1.0, 3717.993408203125, 411.7261962890625, 3893.161376953125],
                [1.0, 3718.025146484375, 410.47149658203125, 3888.161376953125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=61,
            map_id=Maps.FungiForest,
            name="Behind Giant Mushroom",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GiantMushroomArea,
            locations=[
                [1.0, 655.9616088867188, 179.3333282470703, 806.3624877929688],
                [1.0, 651.6785278320312, 179.3333282470703, 787.1334838867188],
                [1.0, 635.9894409179688, 179.3333282470703, 805.6318969726562],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=62,
            map_id=Maps.FungiForest,
            name="Under Hollow Tree",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.HollowTreeArea,
            locations=[
                [1.0, 1274.6341552734375, 249.3333282470703, 3892.971923828125],
                [1.0, 1263.1109619140625, 249.3333282470703, 3894.485595703125],
                [1.0, 1269.507080078125, 249.3333282470703, 3877.209716796875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=63,
            map_id=Maps.FungiForest,
            name="On Well",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            logic=lambda l: ((l.can_use_vines or l.CanMoonkick()) and l.climbing) or (l.jetpack and l.isdiddy),
            locations=[
                [1.0, 2146.2275390625, 295.5009460449219, 3232.423583984375],
                [1.0, 2133.60595703125, 298.7270812988281, 3225.193115234375],
                [1.0, 2148.20556640625, 303.7466735839844, 3215.445068359375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=64,
            map_id=Maps.FungiForest,
            name="On Archway to Giant Mushroom Area",
            konglist=[Kongs.tiny],
            region=Regions.FungiForestStart,
            logic=lambda l: (l.twirl and l.istiny) or l.monkey_maneuvers,
            locations=[
                [1.0, 1785.9215087890625, 334.2199401855469, 1997.81884765625],
                [1.0, 1788.0960693359375, 329.2674865722656, 1982.1993408203125],
                [1.0, 1779.4969482421875, 331.39959716796875, 2010.16796875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=65,
            map_id=Maps.FungiForest,
            name="Inside Hollow Tree night fence",
            konglist=[Kongs.diddy],
            region=Regions.HollowTreeArea,
            logic=lambda l: l.jetpack and l.isdiddy and l.TimeAccess(Regions.HollowTreeArea, Time.Night),
            locations=[
                [1.0, 1282.44775390625, 422.0986633300781, 3757.741943359375],
                [1.0, 1265.284912109375, 423.8885192871094, 3754.115234375],
                [1.0, 1289.3629150390625, 419.2555236816406, 3773.435546875],
                [1.0, 1259.4844970703125, 421.2821960449219, 3774.828125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=66,
            map_id=Maps.FungiForest,
            name="Near Beanstalk Mini Monkey Barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.WormArea,
            locations=[
                [1.0, 2186.608154296875, 223.24549865722656, 243.93670654296875],
                [1.0, 2245.88623046875, 222.2743682861328, 243.90325927734375],
                [1.0, 2274.945556640625, 224.3426055908203, 291.8703308105469],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=67,
            map_id=Maps.FungiForest,
            name="On Mushroom near Mill Well Exit",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MillArea,
            logic=lambda l: l.climbing,
            locations=[
                [1.0, 5004.7548828125, 403.14697265625, 3646.861083984375],
                [1.0, 5003.1103515625, 403.10003662109375, 3650.563720703125],
                [1.0, 5001.9501953125, 402.95123291015625, 3647.419189453125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=68,
            map_id=Maps.FungiForest,
            name="Behind Night Door in the Apple Area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.WormArea,
            logic=lambda l: l.TimeAccess(Regions.WormArea, Time.Night),
            locations=[
                [1.0, 3597.513916015625, 186.8333282470703, 949.5543823242188],
                [1.0, 3602.191162109375, 186.8333282470703, 929.5317993164062],
                [1.0, 3607.429443359375, 186.8333282470703, 913.5304565429688],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=69,
            map_id=Maps.FungiForest,
            name="On switch to face puzzle",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MushroomUpperExterior,
            locations=[
                [1.0, 1310.0802001953125, 1256.0, 858.91650390625],
                [1.0, 1318.0948486328125, 1256.0, 859.779296875],
                [1.0, 1316.613037109375, 1256.0, 867.0820922851562],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=70,
            map_id=Maps.FungiForest,
            name="Behind Well Entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.FungiForestStart,
            locations=[
                [1.0, 2192.89794921875, 121.7884292602539, 3286.261962890625],
                [1.0, 2190.132568359375, 122.10530853271484, 3280.9189453125],
                [1.0, 2185.158447265625, 120.97357940673828, 3278.214111328125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=71,
            map_id=Maps.ForestWinchRoom,
            name="Near the winch machine",
            konglist=[Kongs.diddy],
            region=Regions.WinchRoom,
            locations=[
                [1.0, 309.6408996582031, 0, 93.13311004638672],
                [1.0, 320.85614013671875, 0, 88.46246337890625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=72,
            map_id=Maps.ForestMillAttic,
            name="Behind a box",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.MillAttic,
            locations=[
                [0.880000114440918, 146.0420379638672, 0, 247.82249450683594],
                [0.9200000762939453, 146.01666259765625, 0, 251.927734375],
                [0.9200000762939453, 146.4998779296875, 0, 257.1994323730469],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=73,
            map_id=Maps.ForestThornvineBarn,
            name="Inside the trough",
            konglist=[Kongs.donkey],
            region=Regions.ThornvineBarn,
            locations=[
                [1.0, 359.6155090332031, 15.333333015441895, 636.1107788085938],
                [1.0, 321.60394287109375, 15.333333015441895, 637.284912109375],
                [1.0, 276.3779602050781, 15.333333015441895, 636.2872924804688],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=74,
            map_id=Maps.ForestMillFront,
            name="In the corner near the tag barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GrinderRoom,
            locations=[
                [1.0, 617.2942504882812, 0, 635.030029296875],
                [1.0, 605.1593017578125, 0, 642.5016479492188],
                [1.0, 603.9915161132812, 0, 632.7080688476562],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=75,
            map_id=Maps.ForestMillFront,
            name="In front of the Mini Monkey Hole",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.GrinderRoom,
            locations=[
                [0.9200000762939453, 332.30908203125, 0, 84.87664031982422],
                [0.9200000762939453, 325.3869323730469, -4.736951712906159e-15, 86.29833221435547],
                [0.9200000762939453, 328.9737854003906, 4.736951712906159e-15, 94.87284088134766],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=76,
            map_id=Maps.ForestMillBack,
            name="On various boxes",
            konglist=[Kongs.tiny, Kongs.chunky],
            region=Regions.MillChunkyTinyArea,
            locations=[
                [1.0, 592.6514892578125, 46.20767593383789, 652.8671875],
                [1.0, 560.6297607421875, 31.759504318237305, 654.9306030273438],
                [1.0, 681.8258666992188, 29.83333396911621, 555.6859130859375],
            ],
            vanilla=True,
        ),
    ],
    Levels.CrystalCaves: [
        BananaCoinGroup(
            group=0,
            map_id=Maps.CrystalCaves,
            name="Near the DK Portal",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 1723, -30, 137],
                [1.0, 1766, -30, 173],
                [1.0, 1801, -30, 133],
                [1.0, 1753, -30, 100],
            ],
        ),
        BananaCoinGroup(
            group=1,
            map_id=Maps.CrystalCaves,
            name="Crystal near level entry",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 2260, 36, 299],
                [1.0, 2260, 56, 375],
                [1.0, 2339, 25, 254],
            ],
        ),
        BananaCoinGroup(
            group=2,
            map_id=Maps.CrystalCaves,
            name="Behind the first ice wall",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CavesGGRoom,
            locations=[
                [1.0, 2403, 70, 230],
                [1.0, 2402, 70, 194],
            ],
        ),
        BananaCoinGroup(
            group=3,
            map_id=Maps.CrystalCaves,
            name="Around the lower thick pillar near Ice Castle",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 1708, 93, 935],
                [1.0, 1600, 98, 941],
                [1.0, 1575, 96, 1007],
                [1.0, 1591, 94, 1075],
                [1.0, 1626, 94, 1081],
            ],
        ),
        BananaCoinGroup(
            group=4,
            map_id=Maps.CrystalCaves,
            name="Around Ice Castle Tag",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 2102, 280, 1332],
                [1.0, 2113, 279, 1285],
                [1.0, 2161, 280, 1233],
                [1.0, 2244, 280, 1230],
                [1.0, 2293, 280, 1279],
                [1.0, 2307, 280, 1339],
            ],
        ),
        BananaCoinGroup(
            group=5,
            map_id=Maps.CrystalCaves,
            name="Ice Castle Battlements (1)",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny],
            region=Regions.CrystalCavesMain,
            logic=lambda l: l.isdiddy or (l.islanky and (l.balloon or l.monkey_maneuvers)) or l.istiny,
            locations=[
                [1.0, 2240, 353, 1029],
                [1.0, 2266, 353, 935],
                [1.0, 2171, 353, 911],
                [1.0, 2140, 353, 1003],
            ],
        ),
        BananaCoinGroup(
            group=6,
            map_id=Maps.CrystalCaves,
            name="Ice Castle Battlements (2)",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny],
            region=Regions.CrystalCavesMain,
            logic=lambda l: l.isdiddy or (l.islanky and (l.balloon or l.monkey_maneuvers)) or l.istiny,
            locations=[
                [1.0, 2261, 353, 1007],
                [1.0, 2231, 353, 912],
                [1.0, 2146, 353, 936],
                [1.0, 2169, 353, 1027],
            ],
        ),
        BananaCoinGroup(
            group=7,
            map_id=Maps.CrystalCaves,
            name="On stumps near Ice Castle",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 2061, 151, 1149],
                [1.0, 1949, 172, 1179],
                [1.0, 1921, 197, 1362],
            ],
        ),
        BananaCoinGroup(
            group=8,
            map_id=Maps.CrystalCaves,
            name="Near the small boulder",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 1719, 276, 1025],
                [1.0, 1595, 276, 992],
                [1.0, 1679, 276, 948],
                [1.0, 1632, 276, 1064],
            ],
        ),
        BananaCoinGroup(
            group=9,
            map_id=Maps.CrystalCaves,
            name="Under the bridge near the small boulder",
            konglist=[Kongs.diddy],
            region=Regions.CrystalCavesMain,
            logic=lambda l: l.jetpack and l.isdiddy,
            locations=[
                [1.0, 1518, 270, 1072],
                [1.0, 1426, 300, 1086],
            ],
        ),
        BananaCoinGroup(
            group=10,
            map_id=Maps.CrystalCaves,
            name="Near DK's Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 1674, 285, 766],
                [1.0, 1722, 285, 763],
                [1.0, 1745, 285, 713],
                [1.0, 1646, 285, 739],
            ],
        ),
        BananaCoinGroup(
            group=11,
            map_id=Maps.CrystalCaves,
            name="Hanging off the ledge near Blueprint Cave",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 2925, 224, 946],
                [1.0, 2942, 224, 882],
                [1.0, 3002, 224, 851],
                [1.0, 3085, 224, 919],
                [1.0, 3068, 224, 989],
                [1.0, 3012, 224, 1019],
            ],
        ),
        BananaCoinGroup(
            group=12,
            map_id=Maps.CrystalCaves,
            name="Fences near Ice Castle",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 1841, 124, 1167],
                [1.0, 1784, 112, 1217],
                [1.0, 1449, 95, 1420],
                [1.0, 1407, 95, 1505],
            ],
        ),
        BananaCoinGroup(
            group=13,
            map_id=Maps.CrystalCaves,
            name="Steps up to Cranky's Lab",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 1537, 167, 1991],
                [1.0, 1557, 198, 2114],
                [1.0, 1579, 253, 2244],
            ],
        ),
        BananaCoinGroup(
            group=14,
            map_id=Maps.CrystalCaves,
            name="Around Giant Boulder Shield",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BoulderCave,
            locations=[
                [1.0, 1857, 316, 2546],
                [1.0, 1906, 313, 2574],
                [1.0, 1966, 312, 2553],
                [1.0, 1984, 309, 2502],
                [1.0, 1975, 313, 2467],
                [1.0, 1935, 312, 2437],
                [1.0, 1867, 314, 2454],
                [1.0, 1844, 309, 2496],
            ],
        ),
        BananaCoinGroup(
            group=15,
            map_id=Maps.CrystalCaves,
            name="On a stalagmite",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.BoulderCave,
            locations=[
                [1.0, 2064, 350, 2715],
            ],
        ),
        BananaCoinGroup(
            group=16,
            map_id=Maps.CrystalCaves,
            name="Around a pillar in the boulder cave",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BoulderCave,
            locations=[
                [1.0, 1751, 282, 2661],
                [1.0, 1774, 282, 2717],
                [1.0, 1833, 280, 2692],
                [1.0, 1809, 280, 2636],
            ],
        ),
        BananaCoinGroup(
            group=17,
            map_id=Maps.CrystalCaves,
            name="On a stalagmite in the bonus cave",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CavesBonusCave,
            locations=[
                [1.0, 579, 236, 2411],
            ],
        ),
        BananaCoinGroup(
            group=18,
            map_id=Maps.CrystalCaves,
            name="Around the entrance to the bonus cave",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 886, 53, 1747],
                [1.0, 1046, 71, 1917],
                [1.0, 1106, 133, 2096],
                [1.0, 1091, 178, 2237],
            ],
        ),
        BananaCoinGroup(
            group=19,
            map_id=Maps.CrystalCaves,
            name="On the 5-Door Igloo (Entrances)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IglooArea,
            locations=[
                [1.0, 661, 82, 1385],
                [1.0, 698, 82, 1236],
                [1.0, 564, 82, 1154],
                [1.0, 451, 82, 1252],
                [1.0, 506, 82, 1397],
            ],
        ),
        BananaCoinGroup(
            group=20,
            map_id=Maps.CrystalCaves,
            name="On the 5-Door Igloo (Center)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IglooArea,
            locations=[
                [1.0, 579, 142, 1288],
                [1.0, 586, 142, 1312],
                [1.0, 554, 141, 1309],
                [1.0, 548, 141, 1268],
                [1.0, 586, 140, 1248],
                [1.0, 607, 142, 1285],
            ],
        ),
        BananaCoinGroup(
            group=21,
            map_id=Maps.CrystalCaves,
            name="Around the thin bananaport pillar",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 1217, CAVES_WATER_HEIGHT, 1969],
                [1.0, 1239, CAVES_WATER_HEIGHT, 1901],
                [1.0, 1179, CAVES_WATER_HEIGHT, 1880],
                [1.0, 1150, CAVES_WATER_HEIGHT, 1942],
            ],
        ),
        BananaCoinGroup(
            group=22,
            map_id=Maps.CrystalCaves,
            name="Around the thick bananaport pillar",
            konglist=[Kongs.diddy, Kongs.lanky],
            region=Regions.CavesBlueprintPillar,
            locations=[
                [1.0, 2795, 366, 904],
                [1.0, 2721, 366, 924],
                [1.0, 2731, 366, 1002],
                [1.0, 2806, 366, 982],
            ],
        ),
        BananaCoinGroup(
            group=23,
            map_id=Maps.CrystalCaves,
            name="On boxes near the 5-Door Cabin (1)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CabinArea,
            locations=[
                [1.0, 3545, 295, 1461],
                [1.0, 3645, 285, 1480],
                [1.0, 3695, 300, 1550],
            ],
        ),
        BananaCoinGroup(
            group=24,
            map_id=Maps.CrystalCaves,
            name="On boxes near the 5-Door Cabin (2)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CabinArea,
            locations=[
                [1.0, 3711, 383, 1830],
                [1.0, 3709, 365, 1863],
                [1.0, 3646, 368, 1913],
            ],
        ),
        BananaCoinGroup(
            group=25,
            map_id=Maps.CrystalCaves,
            name="Near the waterfall",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CabinArea,
            locations=[
                [1.0, 2375, 180, 2265],
                [1.0, 2547, 179, 2184],
                [1.0, 2660, 168, 2167],
                [1.0, 2985, 139, 2114],
                [1.0, 3122, 104, 2010],
            ],
        ),
        BananaCoinGroup(
            group=26,
            map_id=Maps.CrystalCaves,
            name="On the doorway to Rotating Room",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CabinArea,
            locations=[
                [1.0, 2775, 340, 2407],
                [1.0, 2806, 340, 2391],
            ],
        ),
        BananaCoinGroup(
            group=27,
            map_id=Maps.CrystalCaves,
            name="Outside the Sprint Cabin",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CabinArea,
            locations=[
                [1.0, 2216, 277, 2051],
                [1.0, 2231, 277, 2047],
                [1.0, 2217, 277, 2060],
            ],
        ),
        BananaCoinGroup(
            group=28,
            map_id=Maps.CrystalCaves,
            name="Near the Sprint Cabin Headphones",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CabinArea,
            locations=[
                [1.0, 2992, 131, 1459],
                [1.0, 3035, 119, 1619],
                [1.0, 3027, 119, 1721],
                [1.0, 3022, 121, 1785],
                [1.0, 2977, 122, 1825],
            ],
        ),
        BananaCoinGroup(
            group=29,
            map_id=Maps.CavesBaboonBlast,
            name="On the path to the first hoop",
            konglist=[Kongs.donkey],
            region=Regions.CavesBaboonBlast,
            locations=[
                [1.0, 963, 500, 1236],
                [1.0, 1054, 497, 1015],
                [1.0, 1137, 432, 814],
            ],
        ),
        BananaCoinGroup(
            group=30,
            map_id=Maps.CavesLankyRace,
            name="On the torches at the start of the beetle race",
            konglist=[Kongs.lanky],
            region=Regions.CavesLankyRace,
            locations=[
                [1.0, 1525, 5150, 491],
                [1.0, 1309, 5150, 277],
                [1.0, 1101, 5150, 491],
            ],
        ),
        BananaCoinGroup(
            group=31,
            map_id=Maps.CavesLankyRace,
            name="In the window in the Caves Beetle Race",
            konglist=[Kongs.lanky],
            region=Regions.CavesLankyRace,
            locations=[
                [1.0, 265, 5156, 2476],
            ],
        ),
        BananaCoinGroup(
            group=32,
            map_id=Maps.CavesFrozenCastle,
            name="Near the Ice Tomato chair",
            konglist=[Kongs.lanky],
            region=Regions.FrozenCastle,
            locations=[
                [1.0, 363, 7, 418],
                [1.0, 413, 7, 282],
            ],
        ),
        BananaCoinGroup(
            group=33,
            map_id=Maps.CavesChunkyIgloo,
            name="Inside the Chunky Igloo TNT",
            konglist=[Kongs.chunky],
            region=Regions.ChunkyIgloo,
            locations=[
                [1.0, 276, 0, 300],
            ],
        ),
        BananaCoinGroup(
            group=34,
            map_id=Maps.CavesDiddyIgloo,
            name="On the barrels in Diddy Igloo",
            konglist=[Kongs.diddy],
            region=Regions.DiddyIgloo,
            locations=[
                [1.0, 269, 15, 205],
                [1.0, 184, 15, 262],
                [1.0, 208, 15, 366],
                [1.0, 307, 15, 408],
                [1.0, 387, 15, 349],
                [1.0, 368, 15, 241],
            ],
        ),
        BananaCoinGroup(
            group=35,
            map_id=Maps.CavesDonkeyIgloo,
            name="At the back of Donkey Igloo",
            konglist=[Kongs.donkey],
            region=Regions.DonkeyIgloo,
            locations=[
                [1.0, 862, 0, 618],
                [1.0, 848, 0, 679],
                [1.0, 718, 0, 826],
                [1.0, 560, 0, 882],
            ],
        ),
        BananaCoinGroup(
            group=36,
            map_id=Maps.CavesLankyIgloo,
            name="On some platforms in Lanky Igloo",
            konglist=[Kongs.lanky],
            region=Regions.LankyIgloo,
            locations=[
                [1.0, 215, 56, 282],
                [1.0, 276, 123, 244],
            ],
        ),
        BananaCoinGroup(
            group=37,
            map_id=Maps.CavesTinyIgloo,
            name="On the swords in Tiny Igloo",
            konglist=[Kongs.tiny],
            region=Regions.TinyIgloo,
            locations=[
                [1.0, 421, 81, 390],
                [1.0, 440, 80, 349],
            ],
        ),
        BananaCoinGroup(
            group=38,
            map_id=Maps.CavesLankyCabin,
            name="Near books in Lanky Cabin",
            konglist=[Kongs.lanky],
            region=Regions.LankyCabin,
            locations=[
                [1.0, 402, 63, 502],
                [1.0, 532, 29, 120],
            ],
        ),
        BananaCoinGroup(
            group=39,
            map_id=Maps.CavesRotatingCabin,
            name="In the corners of Rotating Room",
            konglist=[Kongs.donkey],
            region=Regions.RotatingCabin,
            locations=[
                [1.0, 440, 114, 104],
                [1.0, 191, 114, 103],
                [1.0, 192, 114, 492],
                [1.0, 444, 114, 489],
            ],
        ),
        BananaCoinGroup(
            group=40,
            map_id=Maps.CavesChunkyCabin,
            name="On the targets in Chunky Cabin",
            konglist=[Kongs.chunky],
            region=Regions.ChunkyCabin,
            locations=[
                [1.0, 362, 4, 213],
                [1.0, 441, 4, 299],
                [1.0, 157, 4, 397],
            ],
        ),
        BananaCoinGroup(
            group=41,
            map_id=Maps.CavesDiddyLowerCabin,
            name="On the corners of the center platform in Diddy Lower Cabin",
            konglist=[Kongs.diddy],
            region=Regions.DiddyLowerCabin,
            logic=lambda l: l.jetpack or l.monkey_maneuvers,
            locations=[
                [1.0, 242, 86, 243],
                [1.0, 350, 86, 240],
                [1.0, 356, 86, 356],
                [1.0, 235, 86, 363],
            ],
        ),
        BananaCoinGroup(
            group=42,
            map_id=Maps.CavesDiddyUpperCabin,
            name="In front of the pictures in Diddy Upper Cabin",
            konglist=[Kongs.diddy],
            region=Regions.DiddyUpperCabin,
            logic=lambda l: l.jetpack and (l.guitar or l.oranges) and l.isdiddy and (l.spring or l.CanMoontail()),
            locations=[
                [1.0, 58, 303, 326],
                [1.0, 265, 309, 604],
                [1.0, 542, 305, 394],
            ],
        ),
        BananaCoinGroup(
            group=43,
            map_id=Maps.CavesDonkeyCabin,
            name="In between the holes in Donkey Cabin",
            konglist=[Kongs.donkey],
            region=Regions.DonkeyCabin,
            locations=[
                [1.0, 429, 0, 392],
                [1.0, 417, 0, 275],
                [1.0, 233, 0, 220],
            ],
        ),
        BananaCoinGroup(
            group=44,
            map_id=Maps.CavesTinyCabin,
            name="Around Tiny Cabin Interior",
            konglist=[Kongs.tiny],
            region=Regions.TinyCabin,
            locations=[
                [1.0, 174, 0, 253],
                [1.0, 399, 63, 481],
                [1.0, 401, 0, 113],
            ],
        ),
        BananaCoinGroup(
            group=45,
            map_id=Maps.CrystalCaves,
            name="On stalactite in the Giant Kosha area",
            konglist=[Kongs.tiny],
            region=Regions.GiantKosha,
            locations=[
                [1.0, 1847, 299, 3660],
                [1.0, 1599, 281, 3647],
            ],
        ),
        BananaCoinGroup(
            group=46,
            map_id=Maps.CrystalCaves,
            name="Back of Bonus Cave",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CavesBonusCave,
            logic=lambda l: Events.CavesW3bTagged in l.Events,
            locations=[
                [1.0, 415.590087890625, 180.3333282470703, 2641.762939453125],
                [1.0, 336.6167297363281, 180.3333282470703, 2563.8525390625],
                [1.0, 325.9050598144531, 180.3333282470703, 2477.470703125],
                [1.0, 357.1795959472656, 180.3333282470703, 2384.227783203125],
                [1.0, 455.2051086425781, 180.3333282470703, 2316.06103515625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=47,
            map_id=Maps.CrystalCaves,
            name="Gorilla Gone Cavern",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CavesGGRoom,
            locations=[
                [1.0, 2679.595703125, 13.5, 557.9171752929688],
                [1.0, 2464.411376953125, 13.5, 560.7271118164062],
                [1.0, 2587.134033203125, 13.5, 617.3684692382812],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=48,
            map_id=Maps.CrystalCaves,
            name="Giant Boulder Cavern",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.BoulderCave,
            locations=[
                [1.0, 1993.6322021484375, 280.3333435058594, 2250.162353515625],
                [1.0, 2181.8935546875, 280.3333435058594, 2360.978271484375],
                [1.0, 2215.3583984375, 280.3333435058594, 2572.827392578125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=49,
            map_id=Maps.CrystalCaves,
            name="Giant Kosha Room",
            konglist=[Kongs.tiny],
            region=Regions.GiantKosha,
            locations=[
                [1.0, 1796.47021484375, 257.5393981933594, 3227.193359375],
                [1.0, 1891.2578125, 257.4739074707031, 3218.3046875],
                [1.0, 1836.6007080078125, 257.4121398925781, 3249.552490234375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=50,
            map_id=Maps.CrystalCaves,
            name="In Snide's Cavern",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CavesSnideArea,
            locations=[
                [1.0, 1145.9344482421875, 64.5, 616.4129028320312],
                [1.0, 1112.865966796875, 64.5, 590.035400390625],
                [1.0, 1126.6199951171875, 64.5, 547.1975708007812],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=51,
            map_id=Maps.CrystalCaves,
            name="On Ice Castle",
            konglist=[Kongs.diddy, Kongs.lanky, Kongs.tiny],
            region=Regions.CrystalCavesMain,
            logic=lambda l: ((l.balloon or l.monkey_maneuvers) and l.islanky) or l.isdiddy or l.istiny,
            locations=[
                [1.0, 2261.931884765625, 360.0, 970.6054077148438],
                [1.0, 2202.62353515625, 360.0, 1030.2467041015625],
                [1.0, 2144.980224609375, 360.0, 972.7848510742188],
                [1.0, 2206.591552734375, 360.0, 915.610595703125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=52,
            map_id=Maps.CrystalCaves,
            name="Near Cabin Waterfall",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CabinArea,
            locations=[
                [1.0, 2609.484619140625, 177.01177978515625, 2537.743896484375],
                [1.0, 2623.654296875, 177.18202209472656, 2543.341064453125],
                [1.0, 2614.107666015625, 177.6591339111328, 2556.7900390625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=53,
            map_id=Maps.CrystalCaves,
            name="Behind Igloo Tag Barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IglooArea,
            locations=[
                [1.0, 151.361083984375, CAVES_WATER_HEIGHT, 1419.26611328125],
                [1.0, 156.23382568359375, CAVES_WATER_HEIGHT, 1395.6689453125],
                [1.0, 156.7538604736328, CAVES_WATER_HEIGHT, 1437.8848876953125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=54,
            map_id=Maps.CrystalCaves,
            name="On Slope to Cranky",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 1294.7352294921875, 170.43165588378906, 1491.90673828125],
                [1.0, 1247.3116455078125, 226.2210693359375, 1530.1942138671875],
                [1.0, 1271.0858154296875, 200.39959716796875, 1510.9365234375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=55,
            map_id=Maps.CrystalCaves,
            name="Behind Spike near Igloo Area",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IglooArea,
            locations=[
                [1.0, 775.9146118164062, CAVES_WATER_HEIGHT, 975.7560424804688],
                [1.0, 760.3634643554688, CAVES_WATER_HEIGHT, 967.0896606445312],
                [1.0, 782.792724609375, CAVES_WATER_HEIGHT, 988.7876586914062],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=56,
            map_id=Maps.CrystalCaves,
            name="Near Headphones",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CabinArea,
            locations=[
                [1.0, 2529.18017578125, 188.8098907470703, 1777.48486328125],
                [1.0, 2515.59130859375, 190.7544708251953, 1765.5218505859375],
                [1.0, 2526.3994140625, 188.8497314453125, 1763.8421630859375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=57,
            map_id=Maps.CrystalCaves,
            name="In water near Funky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 3057.88671875, CAVES_WATER_HEIGHT, 975.119873046875],
                [1.0, 3082.9306640625, CAVES_WATER_HEIGHT, 925.3869018554688],
                [1.0, 3044.292724609375, CAVES_WATER_HEIGHT, 893.9777221679688],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=58,
            map_id=Maps.CrystalCaves,
            name="Pillar near Ice Castle",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.CrystalCavesMain,
            logic=lambda l: (l.isdiddy and l.jetpack) or (l.istiny and l.twirl),
            locations=[
                [1.0, 1912.9136962890625, 210.28546142578125, 764.624755859375],
                [1.0, 1917.9136962890625, 210.28546142578125, 766.624755859375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=59,
            map_id=Maps.CrystalCaves,
            name="Behind Caves Igloo W1",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.IglooArea,
            locations=[
                [1.0, 327.6946105957031, CAVES_WATER_HEIGHT, 921.70947265625],
                [1.0, 342.6741943359375, CAVES_WATER_HEIGHT, 910.513427734375],
                [1.0, 329.8152160644531, CAVES_WATER_HEIGHT, 903.6348266601562],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=60,
            map_id=Maps.CrystalCaves,
            name="Entrance to Blueprint Cave",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CrystalCavesMain,
            locations=[
                [1.0, 3143.967529296875, 277.8738098144531, 720.6193237304688],
                [1.0, 3132.0947265625, 278.2353210449219, 710.521728515625],
                [1.0, 3133.3505859375, 274.4537658691406, 722.6437377929688],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=61,
            map_id=Maps.CavesDonkeyCabin,
            name="Above holes",
            konglist=[Kongs.donkey],
            region=Regions.DonkeyCabin,
            locations=[
                [1.0, 445.8771057128906, 16.0, 440.4961242675781],
                [1.0, 172.83599853515625, 15.0, 387.0345764160156],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=62,
            map_id=Maps.CavesDiddyLowerCabin,
            name="In Corner Platforms",
            konglist=[Kongs.diddy],
            region=Regions.DiddyLowerCabin,
            logic=lambda l: l.jetpack or l.monkey_maneuvers,
            locations=[
                [1.0, 477.0457458496094, 86.0, 116.64397430419922],
                [1.0, 486.2208251953125, 86.0, 482.5491027832031],
                [1.0, 120.95458221435547, 86.0, 488.0557556152344],
                [1.0, 113.10247039794922, 86.0, 119.33245086669922],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=63,
            map_id=Maps.CavesFrozenCastle,
            name="On Board",
            konglist=[Kongs.lanky],
            region=Regions.FrozenCastle,
            locations=[
                [1.0, 289.87860107421875, 4.333333969116211, 292.5437316894531],
                [1.0, 279.58258056640625, 4.333333969116211, 319.1732177734375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=64,
            map_id=Maps.CavesBaboonBlast,
            name="In Baboon Blast Course",
            konglist=[Kongs.donkey],
            region=Regions.CavesBaboonBlast,
            logic=lambda _: True,
            locations=[
                [0.7400002479553223, 641.4625854492188, 569.0, 2149.424560546875],
                [1.119999885559082, 1036.116455078125, 292.0, 1675.6175537109375],
                [0.8000001907348633, 1139.65087890625, 324.0, 1113.2296142578125],
            ],
            vanilla=True,
        ),
    ],
    Levels.CreepyCastle: [
        BananaCoinGroup(
            group=0,
            map_id=Maps.CreepyCastle,
            name="Near DK Portal",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 172, 391, 379],
                [1.0, 165, 391, 277],
                [1.0, 230, 391, 212],
                [1.0, 341, 391, 238],
            ],
        ),
        BananaCoinGroup(
            group=1,
            map_id=Maps.CreepyCastle,
            name="Starting pool",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 986, 411, 266],
                [1.0, 969, 387, 265],
                [1.0, 962, 363, 264],
            ],
        ),
        BananaCoinGroup(
            group=2,
            map_id=Maps.CreepyCastle,
            name="Near Low Tag Barrel",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1745, 680, 916],
                [1.0, 1797, 673, 893],
                [1.0, 1869, 674, 1044],
                [1.0, 1847, 673, 1057],
            ],
        ),
        BananaCoinGroup(
            group=3,
            map_id=Maps.CreepyCastle,
            name="High tree at the back of Castle",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            logic=lambda l: l.climbing or (l.jetpack and l.isdiddy),
            locations=[
                [1.0, 1463, 695, 1933],
                [1.0, 1461, 645, 1965],
                [1.0, 1461, 594, 1930],
                [1.0, 1501, 554, 1939],
            ],
        ),
        BananaCoinGroup(
            group=4,
            map_id=Maps.CreepyCastle,
            name="Above a gravestone near lower tunnel entry",  # Reference to 2dos grave push
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1135, 513, 1837],
                [1.0, 1118, 463, 1839],
                [1.0, 1118, 435, 1839],
            ],
        ),
        BananaCoinGroup(
            group=5,
            map_id=Maps.CreepyCastle,
            name="Outside lower tunnel entry",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CastleVeryBottom,
            locations=[
                [1.0, 895, 366, 1837],
                [1.0, 933, 366, 1835],
                [1.0, 1053, 366, 1830],
                [1.0, 1084, 366, 1826],
            ],
        ),
        BananaCoinGroup(
            group=6,
            map_id=Maps.CreepyCastle,
            name="Above retracting platforms",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 637, 725, 1616],
                [1.0, 545, 723, 1591],
            ],
        ),
        BananaCoinGroup(
            group=7,
            map_id=Maps.CreepyCastle,
            name="On a post near the upper tunnel rear entry",
            konglist=[Kongs.diddy],
            region=Regions.CreepyCastleMain,
            logic=lambda l: l.jetpack and l.isdiddy,
            locations=[
                [1.0, 699, 1041, 1728],
                [1.0, 699, 1023, 1728],
                [1.0, 699, 984, 1728],
            ],
        ),
        BananaCoinGroup(
            group=8,
            map_id=Maps.CreepyCastle,
            name="On a series of posts near upper Warp 2",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1399, 907, 1796],
                [1.0, 1777, 922, 1763],
                [1.0, 1935, 921, 1678],
                [1.0, 1956, 922, 1499],
            ],
        ),
        BananaCoinGroup(
            group=9,
            map_id=Maps.CreepyCastle,
            name="Near the wobbly bridge near upper Warp 2",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1336, 962, 676],
                [1.0, 1344, 957, 780],
            ],
        ),
        BananaCoinGroup(
            group=10,
            map_id=Maps.CreepyCastle,
            name="Hanging off the edge near Cranky's",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 719, 1061, 899],
                [1.0, 699, 1068, 939],
                [1.0, 682, 1075, 977],
                [1.0, 664, 1081, 1016],
            ],
        ),
        BananaCoinGroup(
            group=11,
            map_id=Maps.CreepyCastle,
            name="Near Museum Entrance",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 862, 1359, 1526],
                [1.0, 838, 1360, 1560],
            ],
        ),
        BananaCoinGroup(
            group=12,
            map_id=Maps.CreepyCastle,
            name="Near Trash Can",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1429, 1483, 1920],
                [1.0, 1429, 1450, 1920],
            ],
        ),
        BananaCoinGroup(
            group=13,
            map_id=Maps.CreepyCastle,
            name="On the library cloud",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1058, 1487, 1154],
                [1.0, 1028, 1488, 1127],
                [1.0, 1071, 1486, 1140],
            ],
        ),
        BananaCoinGroup(
            group=14,
            map_id=Maps.CreepyCastle,
            name="Around the edge near Library entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 657, 1546, 1065],
                [1.0, 590, 1548, 1099],
                [1.0, 472, 1549, 1230],
                [1.0, 450, 1552, 1299],
                [1.0, 459, 1547, 1477],
                [1.0, 495, 1551, 1542],
                [1.0, 603, 1549, 1647],
                [1.0, 683, 1553, 1675],
            ],
        ),
        BananaCoinGroup(
            group=15,
            map_id=Maps.CreepyCastle,
            name="Top of Castle",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1201, 1741, 1257],
                [1.0, 1217, 1741, 1417],
            ],
        ),
        BananaCoinGroup(
            group=16,
            map_id=Maps.CreepyCastle,
            name="Under Museum Entrance Platform",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 896, 1248, 1578],
                [1.0, 959, 1246, 1546],
            ],
        ),
        BananaCoinGroup(
            group=17,
            map_id=Maps.CreepyCastle,
            name="Outside Greenhouse",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1697, 1391, 1734],
                [1.0, 1499, 1391, 1865],
                [1.0, 1380, 1391, 1809],
            ],
        ),
        BananaCoinGroup(
            group=18,
            map_id=Maps.CreepyCastle,
            name="Around Castle Tree exterior",
            konglist=[Kongs.diddy],
            logic=lambda l: l.jetpack and l.isdiddy,
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1221, 572, 176],
                [1.0, 1176, 638, 240],
                [1.0, 1299, 655, 216],
            ],
        ),
        BananaCoinGroup(
            group=19,
            map_id=Maps.CastleBallroom,
            name="Ballroom Wall Torches",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.Ballroom,
            locations=[
                [1.0, 511, 158, 1384],
                [1.0, 594, 156, 1390],
            ],
        ),
        BananaCoinGroup(
            group=20,
            map_id=Maps.CastleBallroom,
            name="Ballroom Post Torches",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.Ballroom,
            locations=[
                [1.0, 298, 96, 915],
                [1.0, 298, 122, 915],
                [1.0, 798, 96, 918],
                [1.0, 798, 122, 918],
            ],
        ),
        BananaCoinGroup(
            group=21,
            map_id=Maps.CastleBallroom,
            name="Around the tiled Ballroom floor",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.Ballroom,
            locations=[
                [1.0, 773, 45, 321],
                [1.0, 326, 45, 321],
                [1.0, 326, 45, 877],
                [1.0, 772, 45, 880],
            ],
        ),
        BananaCoinGroup(
            group=22,
            map_id=Maps.CastleBallroom,
            name="Ballroom K. Rool Banners",  # In the placement of his eyes
            konglist=[Kongs.diddy],
            region=Regions.Ballroom,
            logic=lambda l: l.jetpack and l.isdiddy,
            locations=[
                [1.0, 960, 414, 591],
                [1.0, 960, 416, 614],
                [1.0, 539, 412, 134],
                [1.0, 564, 417, 134],
                [1.0, 140, 418, 612],
                [1.0, 139, 412, 588],
            ],
        ),
        BananaCoinGroup(
            group=23,
            map_id=Maps.CastleDungeon,
            name="Dungeon Buckets",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Dungeon,
            locations=[
                [1.0, 731, 214, 1883],
                [1.0, 427, 134, 1383],
                [1.0, 424, 134, 2430],
            ],
        ),
        BananaCoinGroup(
            group=24,
            map_id=Maps.CastleDungeon,
            name="Dungeon - Behind Gate (1)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Dungeon,
            logic=lambda l: l.punch and l.chunky or l.CanPhase(),
            locations=[
                [1.0, 312, 115, 2545],
                [1.0, 312, 115, 2500],
            ],
        ),
        BananaCoinGroup(
            group=25,
            map_id=Maps.CastleDungeon,
            name="Dungeon - Behind Gate (2)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Dungeon,
            logic=lambda l: l.punch and l.chunky or l.CanPhase(),
            locations=[
                [1.0, 812, 195, 1780],
                [1.0, 779, 195, 1779],
            ],
        ),
        BananaCoinGroup(
            group=26,
            map_id=Maps.CastleDungeon,
            name="Dungeon - Behind Gate (3)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Dungeon,
            logic=lambda l: l.punch and l.chunky or l.CanPhase(),
            locations=[
                [1.0, 752, 115, 1459],
                [1.0, 755, 115, 1498],
            ],
        ),
        BananaCoinGroup(
            group=27,
            map_id=Maps.CastleDungeon,
            name="Donkey's Dungeon Room",
            konglist=[Kongs.donkey],
            region=Regions.Dungeon,
            logic=lambda l: l.CanSlamSwitch(Levels.CreepyCastle, 3) or l.CanPhase(),
            locations=[
                [1.0, 1266, 181, 1928],
                [1.0, 1395, 176, 1930],
                [1.0, 1387, 176, 2061],
                [1.0, 1264, 181, 2059],
            ],
        ),
        BananaCoinGroup(
            group=28,
            map_id=Maps.CastleDungeon,
            name="Diddy's Dungeon Room",
            konglist=[Kongs.diddy],
            region=Regions.Dungeon,
            logic=lambda l: l.CanSlamSwitch(Levels.CreepyCastle, 3) and l.scope and l.peanut and l.can_use_vines,
            locations=[
                [1.0, 566, 209, 3555],
                [1.0, 566, 152, 3555],
                [1.0, 506, 209, 3561],
                [1.0, 506, 152, 3561],
            ],
        ),
        BananaCoinGroup(
            group=29,
            map_id=Maps.CastleShed,
            name="Shed Interior Corners",
            konglist=[Kongs.chunky],
            region=Regions.Shed,
            locations=[
                [1.0, 469, 0, 171],
                [1.0, 132, 0, 171],
                [1.0, 133, 0, 507],
                [1.0, 467, 0, 506],
            ],
        ),
        BananaCoinGroup(
            group=30,
            map_id=Maps.CastleLowerCave,
            name="Lower Cave hallway start",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LowerCave,
            locations=[
                [1.0, 1217, 200, 1149],
                [1.0, 1016, 200, 1158],
                [1.0, 1015, 200, 1353],
                [1.0, 1235, 200, 1346],
            ],
        ),
        BananaCoinGroup(
            group=31,
            map_id=Maps.CastleLowerCave,
            name="Lower Cave Melon Crate",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LowerCave,
            locations=[
                [1.0, 1890, 320, 1281],
                [1.0, 1869, 320, 1264],
                [1.0, 1891, 320, 1241],
                [1.0, 1910, 320, 1258],
            ],
        ),
        BananaCoinGroup(
            group=32,
            map_id=Maps.CastleLowerCave,
            name="Lower Cave Torch Pillar",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.LowerCave,
            locations=[
                [1.0, 1515, 217, 908],
            ],
        ),
        BananaCoinGroup(
            group=33,
            map_id=Maps.CastleLowerCave,
            name="Near Funky",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LowerCave,
            locations=[
                [1.0, 1403, 200, 605],
                [1.0, 1536, 200, 604],
            ],
        ),
        BananaCoinGroup(
            group=34,
            map_id=Maps.CastleLowerCave,
            name="Near Crypt Entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LowerCave,
            locations=[
                [1.0, 123, 90, 1195],
                [1.0, 336, 90, 1234],
                [1.0, 336, 90, 1305],
                [1.0, 125, 90, 1335],
            ],
        ),
        BananaCoinGroup(
            group=35,
            map_id=Maps.CastleLowerCave,
            name="On Crypt Skull",
            konglist=[Kongs.diddy, Kongs.tiny],
            region=Regions.LowerCave,
            locations=[
                [1.0, 293, 175, 1267],
            ],
        ),
        BananaCoinGroup(
            group=36,
            map_id=Maps.CastleCrypt,
            name="On Wall Torch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.chunky],
            region=Regions.Crypt,
            locations=[
                [1.0, 1133, 215, 1359],
                [1.0, 1133, 241, 1359],
            ],
        ),
        BananaCoinGroup(
            group=37,
            map_id=Maps.CastleCrypt,
            name="On Diddy Coffin",
            konglist=[Kongs.diddy],
            region=Regions.CryptDiddyRoom,
            locations=[
                [1.0, 2009, 68, 288],
                [1.0, 2005, 68, 486],
                [1.0, 2139, 67, 496],
                [1.0, 2141, 68, 292],
            ],
        ),
        BananaCoinGroup(
            group=38,
            map_id=Maps.CastleCrypt,
            name="On incorrect Minecart levers",
            konglist=[Kongs.donkey],
            region=Regions.CryptDonkeyRoom,
            locations=[
                [1.0, 1607, 81, 2323],
                [1.0, 1434, 81, 2318],
                [1.0, 1430, 81, 2226],
            ],
        ),
        BananaCoinGroup(
            group=39,
            map_id=Maps.CastleCrypt,
            name="Inside Chunky's Coffin",
            konglist=[Kongs.chunky],
            region=Regions.CryptChunkyRoom,
            logic=lambda l: (l.Slam or l.CanPhase()) and l.ischunky,
            locations=[
                [1.0, 1000, 160, 2903],
                [1.0, 1000, 160, 2817],
                [1.0, 1155, 160, 2822],
                [1.0, 1155, 160, 2877],
            ],
        ),
        BananaCoinGroup(
            group=40,
            map_id=Maps.CastleMausoleum,
            name="On Goo Hands",
            konglist=[Kongs.tiny],
            region=Regions.Mausoleum,
            logic=lambda l: l.CanSlamSwitch(Levels.CreepyCastle, 3) and l.istiny and l.twirl,
            locations=[
                [1.0, 1176, 181, 2050],
                [1.0, 993, 181, 2065],
                [1.0, 1351, 181, 2088],
                [1.0, 1341, 181, 2304],
                [1.0, 1173, 181, 2215],
                [1.0, 992, 181, 2298],
            ],
        ),
        BananaCoinGroup(
            group=41,
            map_id=Maps.CastleMausoleum,
            name="Between Mausoleum Vines",
            konglist=[Kongs.lanky],
            region=Regions.Mausoleum,
            logic=lambda l: ((l.grape and l.sprint) or l.generalclips or l.CanPhase()) and l.islanky and l.trombone and l.can_use_vines,
            locations=[
                [1.0, 2196, 136, 496],
                [1.0, 2321, 116, 461],
                [1.0, 2382, 129, 506],
                [1.0, 2404, 124, 622],
                [1.0, 2362, 128, 699],
                [1.0, 2301, 124, 747],
            ],
        ),
        BananaCoinGroup(
            group=42,
            map_id=Maps.CastleUpperCave,
            name="Tunnel behind torch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.UpperCave,
            locations=[
                [1.0, 568, 200, 2570],
                [1.0, 547, 200, 2569],
                [1.0, 561, 200, 2555],
            ],
        ),
        BananaCoinGroup(
            group=43,
            map_id=Maps.CastleUpperCave,
            name="Tunnel Rear Entry",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.UpperCave,
            locations=[
                [1.0, 604, 150, 258],
                [1.0, 494, 150, 259],
            ],
        ),
        BananaCoinGroup(
            group=44,
            map_id=Maps.CastleGreenhouse,
            name="Greenhouse Exit Corner",
            konglist=[Kongs.lanky],
            region=Regions.Greenhouse,
            locations=[
                [1.0, 109, 0, 279],
                [1.0, 140, 0, 277],
                [1.0, 120, 0, 257],
            ],
        ),
        BananaCoinGroup(
            group=45,
            map_id=Maps.CastleGreenhouse,
            name="Greenhouse Dead End",
            konglist=[Kongs.lanky],
            region=Regions.Greenhouse,
            locations=[
                [1.0, 324, 0, 802],
                [1.0, 320, 0, 853],
                [1.0, 320, 0, 891],
            ],
        ),
        BananaCoinGroup(
            group=46,
            map_id=Maps.CastleLibrary,
            name="Library Wall Torch",
            konglist=[Kongs.donkey],
            region=Regions.Library,
            locations=[
                [1.0, 794, 156, 770],
                [1.0, 794, 128, 770],
                [1.0, 794, 100, 770],
            ],
        ),
        BananaCoinGroup(
            group=47,
            map_id=Maps.CastleLibrary,
            name="Library inside enemy gauntlet room",
            konglist=[Kongs.donkey],
            region=Regions.Library,
            locations=[
                [1.0, 261, 240, 386],
                [1.0, 211, 240, 340],
                [1.0, 273, 240, 279],
                [1.0, 319, 240, 322],
            ],
        ),
        BananaCoinGroup(
            group=48,
            map_id=Maps.CastleLibrary,
            name="Library Strong Kong Alcove",
            konglist=[Kongs.donkey],
            region=Regions.LibraryPastSlam,
            locations=[
                [1.0, 2040, 180, 207],
                [1.0, 2118, 180, 209],
            ],
        ),
        BananaCoinGroup(
            group=49,
            map_id=Maps.CastleMuseum,
            name="Museum around second skull",
            konglist=[Kongs.chunky],
            region=Regions.Museum,
            locations=[
                [1.0, 1140, 100, 810],
                [1.0, 1100, 100, 809],
                [1.0, 1059, 100, 782],
                [1.0, 1050, 100, 740],
                [1.0, 1059, 100, 697],
                [1.0, 1103, 100, 672],
                [1.0, 1140, 100, 672],
            ],
        ),
        BananaCoinGroup(
            group=50,
            map_id=Maps.CastleMuseum,
            name="Museum entryway to main room",
            konglist=[Kongs.chunky],
            region=Regions.Museum,
            locations=[
                [1.0, 842, 102, 1229],
                [1.0, 841, 199, 1424],
                [1.0, 708, 198, 1422],
                [1.0, 708, 103, 1231],
            ],
        ),
        BananaCoinGroup(
            group=51,
            map_id=Maps.CastleMuseum,
            name="Car Race Building Chimney stack",
            konglist=[Kongs.tiny],
            region=Regions.MuseumBehindGlass,
            locations=[
                [1.0, 374, 285, 1679],
                [1.0, 396, 284, 1643],
                [1.0, 439, 286, 1646],
                [1.0, 454, 285, 1673],
                [1.0, 431, 285, 1712],
                [1.0, 396, 283, 1708],
            ],
        ),
        BananaCoinGroup(
            group=52,
            map_id=Maps.CastleMuseum,
            name="Car Race Second Chimney stack",
            konglist=[Kongs.tiny],
            region=Regions.MuseumBehindGlass,
            locations=[
                [1.0, 393, 301, 1508],
            ],
        ),
        BananaCoinGroup(
            group=53,
            map_id=Maps.CastleMuseum,
            name="Around mystery Museum Pillar",
            konglist=[Kongs.tiny],
            region=Regions.MuseumBehindGlass,
            logic=lambda l: (l.monkeyport or l.CanPhase()) and l.istiny,
            locations=[
                [1.0, 1184, 205, 1468],
                [1.0, 1135, 205, 1450],
                [1.0, 1093, 205, 1466],
                [1.0, 1074, 205, 1514],
                [1.0, 1091, 205, 1559],
                [1.0, 1141, 205, 1579],
                [1.0, 1185, 205, 1561],
            ],
        ),
        BananaCoinGroup(
            group=54,
            map_id=Maps.CastleMuseum,
            name="On mystery Museum Pillar",
            konglist=[Kongs.tiny],
            region=Regions.MuseumBehindGlass,
            logic=lambda l: (l.monkeyport or l.CanPhase()) and l.istiny,
            locations=[
                [1.0, 1125, 258, 1516],
                [1.0, 1149, 258, 1525],
                [1.0, 1143, 258, 1504],
            ],
        ),
        BananaCoinGroup(
            group=55,
            map_id=Maps.CastleTower,
            name="On Wall Torches",
            konglist=[Kongs.lanky],
            region=Regions.Tower,
            logic=lambda l: l.balloon and l.islanky,
            locations=[
                [1.0, 621, 325, 282],
                [1.0, 400, 327, 650],
                [1.0, 186, 329, 267],
            ],
        ),
        BananaCoinGroup(
            group=56,
            map_id=Maps.CastleTrashCan,
            name="Around trash can cheese",
            konglist=[Kongs.tiny],
            region=Regions.TrashCan,
            locations=[
                [1.0, 548, 15, 571],
                [1.0, 588, 42, 735],
                [1.0, 638, 15, 573],
            ],
        ),
        BananaCoinGroup(
            group=57,
            map_id=Maps.CastleTrashCan,
            name="Off the ledge of the trash can cheese",
            konglist=[Kongs.tiny],
            region=Regions.TrashCan,
            locations=[
                [1.0, 621, 82, 577],
                [1.0, 563, 82, 577],
                [1.0, 593, 82, 577],
            ],
        ),
        BananaCoinGroup(
            group=58,
            map_id=Maps.CastleTree,
            name="Underwater in the DK Tree area",
            konglist=[Kongs.donkey],
            region=Regions.CastleTree,
            logic=lambda l: l.swim and ((l.coconut) or l.generalclips or l.CanPhase()) and l.isdonkey,
            locations=[
                [1.0, 1220, 14, 1701],
                [1.0, 1512, 15, 1691],
                [1.0, 1597, 17, 1585],
                [1.0, 1640, 17, 1396],
            ],
        ),
        BananaCoinGroup(
            group=59,
            map_id=Maps.CastleTree,
            name="In Chunky's Tree area",
            konglist=[Kongs.chunky],
            region=Regions.CastleTreePastPunch,
            locations=[
                [1.0, 1066, 350, 395],
                [1.0, 1064, 350, 250],
                [1.0, 896, 350, 273],
                [1.0, 886, 350, 399],
            ],
        ),
        BananaCoinGroup(
            group=60,
            map_id=Maps.CreepyCastle,
            name="Around Tiny Kasplat",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CastleVeryBottom,
            locations=[
                [1.0, 81.56172943115234, 392.0283508300781, 914.9120483398438],
                [1.0, 57.68967819213867, 392.5829772949219, 912.5903930664062],
                [1.0, 74.25150299072266, 391.87158203125, 896.8722534179688],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=61,
            map_id=Maps.CreepyCastle,
            name="Around upper W2",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1906.3798828125, 904.5741577148438, 877.5512084960938],
                [1.0, 1949.6090087890625, 904.45166015625, 893.3656616210938],
                [1.0, 1964.6640625, 904.483154296875, 941.076904296875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=62,
            map_id=Maps.CreepyCastle,
            name="On thin tree near the Tree entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            logic=lambda l: l.climbing or (l.jetpack and l.isdiddy),
            locations=[
                [1.0, 1135.9844970703125, 636.5, 147.59095764160156],
                [1.0, 1133.508544921875, 636.5, 141.38092041015625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=63,
            map_id=Maps.CreepyCastle,
            name="Behind Grave (1)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CastleVeryBottom,
            locations=[
                [1.0, 811.68896484375, 366.3333435058594, 2016.2047119140625],
                [1.0, 823.4928588867188, 366.3333435058594, 1998.4842529296875],
                [1.0, 834.1975708007812, 366.3333435058594, 2010.1474609375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=64,
            map_id=Maps.CreepyCastle,
            name="Hanging off ledge near Castle Tree entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CreepyCastleMain,
            locations=[
                [1.0, 1620.6876220703125, 446.04833984375, 411.7994384765625],
                [1.0, 1601.921875, 442.90478515625, 406.2318115234375],
                [1.0, 1583.5570068359375, 442.8856506347656, 398.0413513183594],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=65,
            map_id=Maps.CreepyCastle,
            name="Behind Grave (2)",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CastleVeryBottom,
            locations=[
                [1.0, 1284.5264892578125, 366.3333435058594, 2060.625732421875],
                [1.0, 1274.8592529296875, 366.3333435058594, 2056.375732421875],
                [1.0, 1294.0469970703125, 366.3333435058594, 2064.810302734375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=66,
            map_id=Maps.CreepyCastle,
            name="On tree near Crypt Entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.CastleVeryBottom,
            logic=lambda l: l.climbing or (l.jetpack and l.isdiddy),
            locations=[
                [1.0, 1705.875, 544.685302734375, 2036.1650390625],
                [1.0, 1708.7713623046875, 540.0343017578125, 2040.3990478515625],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=67,
            map_id=Maps.CreepyCastle,
            name="On Drawbridge",
            konglist=[Kongs.diddy],
            region=Regions.CreepyCastleMain,
            logic=lambda l: l.jetpack and l.isdiddy,
            locations=[
                [1.0, 732.8368530273438, 908.4652709960938, 931.1201782226562],
                [1.0, 598.3995971679688, 908.5, 846.486572265625],
                [1.0, 886.0228881835938, 908.5, 724.0714721679688],
                [1.0, 757.115966796875, 908.5, 626.9812622070312],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=68,
            map_id=Maps.CreepyCastle,
            name="Above Window",
            konglist=[Kongs.diddy],
            region=Regions.CreepyCastleMain,
            logic=lambda l: l.jetpack and l.isdiddy,
            locations=[
                [1.0, 1342.8936767578125, 1141.919677734375, 1008.482666015625],
                [1.0, 1035.1519775390625, 1153.5888671875, 1098.083984375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=69,
            map_id=Maps.CastleBallroom,
            name="Around Monkeyport Pad",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Ballroom,
            locations=[
                [1.0, 533.311767578125, 40.0, 1067.969482421875],
                [1.0, 556.1460571289062, 40.0, 1009.9342651367188],
                [1.0, 584.872802734375, 40.0, 1060.12255859375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=70,
            map_id=Maps.CastleMausoleum,
            name="Around Tiny Slam Pad",
            konglist=[Kongs.tiny],
            region=Regions.Mausoleum,
            logic=lambda l: (l.twirl or (l.monkey_maneuvers and (not l.isKrushaAdjacent(Kongs.tiny))) or l.CanPhase()) and l.istiny,
            locations=[
                [1.0, 1138.8143310546875, 160.0, 1953.1551513671875],
                [1.0, 1201.6041259765625, 160.0, 1949.9952392578125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=71,
            map_id=Maps.CastleMausoleum,
            name="Around Lanky Reward",
            konglist=[Kongs.lanky],
            region=Regions.Mausoleum,
            logic=lambda l: ((l.grape and l.sprint) or l.generalclips or l.CanPhase()) and ((l.trombone and l.can_use_vines) or l.monkey_maneuvers) and l.islanky,
            locations=[
                [1.0, 2219.259521484375, 89.33333587646484, 763.8711547851562],
                [1.0, 2244.122314453125, 89.33333587646484, 739.7951049804688],
                [1.0, 2263.974365234375, 89.33333587646484, 764.0588989257812],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=72,
            map_id=Maps.CastleCrypt,
            name="Inside Chunky Coffins",
            konglist=[Kongs.chunky],
            region=Regions.CryptChunkyRoom,
            logic=lambda l: (l.ischunky and l.punch) or l.CanPhase() or l.generalclips,
            locations=[
                [1.0, 732.0772094726562, 191.0411834716797, 2947.914794921875],
                [1.0, 733.7160034179688, 190.70660400390625, 2951.262939453125],
                [1.0, 729.0805053710938, 191.6529998779297, 2955.185302734375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=73,
            map_id=Maps.CastleCrypt,
            name="Inside Diddy Coffins",
            konglist=[Kongs.diddy],
            region=Regions.CryptDiddyRoom,
            logic=lambda l: (l.isdiddy and l.charge) or l.CanPhase() or l.generalclips,
            locations=[
                [1.0, 2044.2547607421875, 4.043132781982422, 718.1364135742188],
                [1.0, 2056.055419921875, 4.325083255767822, 719.5173950195312],
                [1.0, 2067.984375, 4.53893518447876, 720.5648803710938],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=74,
            map_id=Maps.CastleMuseum,
            name="In front of shields",
            konglist=[Kongs.chunky],
            region=Regions.Museum,
            locations=[
                [1.0, 458.473388671875, 101.0, 814.6547241210938],
                [1.0, 451.050537109375, 100.0, 734.606689453125],
                [1.0, 454.8817138671875, 100.0, 659.3023071289062],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=75,
            map_id=Maps.CastleUpperCave,
            name="Over pit",
            konglist=[Kongs.tiny],
            region=Regions.UpperCave,
            logic=lambda l: l.twirl and l.istiny,
            locations=[
                [1.0, 816.2604370117188, 200.0, 849.3430786132812],
                [1.0, 1040.6785888671875, 200.0, 855.015625],
                [1.0, 929.43310546875, 225.0, 854.7207641601562],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=76,
            map_id=Maps.CastleUpperCave,
            name="Near Candy's Shop",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.UpperCave,
            locations=[
                [1.0, 1033.67529296875, 300.0, 2007.8341064453125],
                [1.0, 1009.177001953125, 300.0, 2008.9534912109375],
                [1.0, 1024.4232177734375, 300.0, 2028.70458984375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=77,
            map_id=Maps.CastleUpperCave,
            name="Outside Dungeon Entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.UpperCave,
            locations=[
                [1.0, 248.86656188964844, 200.0, 1506.45361328125],
                [1.0, 250.5158233642578, 200.0, 1487.8673095703125],
                [1.0, 237.028564453125, 200.0, 1498.52001953125],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=78,
            map_id=Maps.CastleDungeon,
            name="Around Donkey Switch",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Dungeon,
            locations=[
                [1.0, 919.1231079101562, 195.0, 1997.76416015625],
                [1.0, 888.811767578125, 195.0, 2024.5093994140625],
                [1.0, 847.5178833007812, 195.0, 1991.4586181640625],
                [1.0, 884.549072265625, 195.0, 1964.1693115234375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=79,
            map_id=Maps.CastleDungeon,
            name="Inside punchable grate",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.Dungeon,
            logic=lambda l: (l.punch and l.chunky) or l.CanPhase(),
            locations=[
                [1.0, 747.4481811523438, 115.0, 2498.423583984375],
                [1.0, 745.97314453125, 115.0, 2518.90185546875],
                [1.0, 743.9428100585938, 115.0, 2539.614990234375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=80,
            map_id=Maps.CastleDungeon,
            name="Above the acid in Lanky's Room",
            konglist=[Kongs.lanky],
            region=Regions.Dungeon,
            logic=lambda l: (l.CanSlamSwitch(Levels.CreepyCastle, 3) or l.CanPhase()) and l.islanky and l.trombone and l.balloon,
            locations=[
                [1.0, 540.93212890625, 210.0, 620.8506469726562],
                [1.0, 501.813232421875, 210.0, 889.5272827148438],
                [1.0, 540.68798828125, 210.0, 1009.4439086914062],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=81,
            map_id=Maps.CastleTree,
            name="On pathway to Chunky Bonus",
            konglist=[Kongs.chunky],
            region=Regions.CastleTreePastPunch,
            locations=[
                [1.0, 1219.075439453125, 350.0, 321.3106384277344],
                [1.0, 1201.170166015625, 350.0, 328.4452209472656],
                [1.0, 1219.397216796875, 350.0, 334.9942321777344],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=82,
            map_id=Maps.CastleShed,
            name="Around punch box",
            konglist=[Kongs.chunky],
            region=Regions.Shed,
            locations=[
                [1.0, 295.7814636230469, 0, 308.900146484375],
                [1.0, 254.7866973876953, 0, 351.751708984375],
                [1.0, 300.2150573730469, 0, 399.0756530761719],
                [1.0, 352.2215270996094, 0, 356.4903259277344],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=83,
            map_id=Maps.CastleTrashCan,
            name="On the long sides of the cheese",
            konglist=[Kongs.tiny],
            region=Regions.TrashCan,
            locations=[
                [2.059999942779541, 500.1996154785156, 21.977380752563477, 641.8643188476562],
                [2.0, 680.619873046875, 23.86710548400879, 653.20263671875],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=84,
            map_id=Maps.CastleGreenhouse,
            name="At a dead end",
            konglist=[Kongs.lanky],
            region=Regions.Greenhouse,
            logic=lambda _: True,
            locations=[
                [1.2599997520446777, 880.094482421875, 0, 876.8353881835938],
                [1.2599997520446777, 854.2296752929688, 0, 875.391845703125],
                [1.2599997520446777, 835.154052734375, 0, 864.5934448242188],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=85,
            map_id=Maps.CastleLowerCave,
            name="Around the mausoleum entrance",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LowerCave,
            locations=[
                [1.0, 1817.4549560546875, 320.0, 1249.904296875],
                [1.0, 1720.4737548828125, 320.0, 1202.75390625],
                [1.0, 1611.1893310546875, 320.0, 1263.3701171875],
                [1.0, 1701.6748046875, 320.0, 1317.6351318359375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=86,
            map_id=Maps.CastleLowerCave,
            name="Near the Crypt Staircase",
            konglist=[Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky],
            region=Regions.LowerCave,
            locations=[
                [1.0, 656.3650512695312, 90.0, 1271.643798828125],
                [1.0, 676.8861083984375, 90.0, 1261.226806640625],
                [1.0, 655.9323120117188, 90.0, 1251.193115234375],
            ],
            vanilla=True,
        ),
        BananaCoinGroup(
            group=87,
            map_id=Maps.CastleBaboonBlast,
            name="Inside the Baboon Blast Course",
            konglist=[Kongs.donkey],
            region=Regions.CastleBaboonBlast,
            locations=[
                [1.0, 1869.3465576171875, 465.0, 1052.3653564453125],
                [1.0, 2445.34423828125, 510.0, 1870.2607421875],
                [1.0, 1989.99853515625, 630.0, 2237.154052734375],
                [1.0, 1469.6103515625, 445.0, 1960.7777099609375],
                [1.0, 1913.8572998046875, 447.0, 1370.8958740234375],
            ],
            vanilla=True,
        ),
    ],
}
