from typing import NamedTuple, Optional, List
from BaseClasses import Location, Region
from .Helper_Functions import LMRamData


class LMLocationData(NamedTuple):
    region: str
    code: Optional[int]  # used to create ap_id, None for events
    type: str  # type of randomization option/jmp table and group [Chest, Furniture, Furniture, Plant, Boo, GSpeedy (Gold Mouse), BSpeedy (Blue Ghost), Portrait, Toad]
    jmpentry: int = -1  # entry number on the jmp table it belongs to
    access: List[str] = []  # items required to access location, many special cases
    floor: int = 0
    locked_item: Optional[str] = None
    remote_only: bool = None
    update_ram_addr: Optional[list[LMRamData]] = None
    require_poltergust: bool = True
    map_id: list[int] = [2]


class LMLocation(Location):
    game: str = "Luigi's Mansion"
    access: list[str]
    rule_def: str = ""
    locked_item: Optional[str]

    def __init__(self, player: int, name: str, parent: Region, data: LMLocationData):
        address = None if data.code is None else LMLocation.get_apid(data.code)
        super(LMLocation, self).__init__(player, name, address=address, parent=parent)

        self.code = data.code
        self.region = data.region
        self.type = data.type
        self.jmpentry = data.jmpentry
        self.address = self.address
        self.access = data.access
        self.locked_item = data.locked_item

    @staticmethod
    def get_apid(code: int):
        base_id: int = 8000
        return base_id + code


# Notes on type field in location data
# Chest item contents are modified in treasuretable and are always included
# Furniture item contents are modified in furnitureinfo, some are always included
# Plant item contents are modified in furnitureinfo but are only included when Plantsanity is active
# BSpeedy are Blue Ghosts - item contents are modified in
# iyapootable but are only included when SpeedySpirits option is active
# Mouse Are Gold Mice - item contents are modified in
# iyapootable but are only included when GoldMice option is active
# Portrait item contents are modified in characterinfo but are only included when PortraitGhosts option is active
# Toad item contents are unable to be modified without further study
# Boo item contents are unable to be modified without further study. All require Boo Radar

# Base Chests / Locations
BASE_LOCATION_TABLE: dict[str, LMLocationData] = {
    # Give item on Altar lights turning on
    "Luigi's Courage": LMLocationData("Foyer", 708, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFDC, bit_position=1, ram_byte_size=2)], require_poltergust=False),
    "E. Gadd's Gift": LMLocationData("Foyer", 853, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFDC, bit_position=1, ram_byte_size=2)], require_poltergust=False),
    "1F Washroom Toilet": LMLocationData("1F Washroom", 4, "Furniture", 233, [],
        update_ram_addr=[LMRamData(in_game_room_id=16)], require_poltergust=False),
    "Laundry Room Washing Machine": LMLocationData("Laundry Room", 7, "Furniture", 187, [],
        update_ram_addr=[LMRamData(in_game_room_id=5)], require_poltergust=False),
    "Hidden Room Large Chest R": LMLocationData("Hidden Room", 11, "Furniture", 242, [],
        update_ram_addr=[LMRamData(in_game_room_id=1)], require_poltergust=False),
    "Hidden Room Small Chest R Shelf": LMLocationData("Hidden Room", 13, "Furniture", 244, [],
        update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room Small Chest L Floor": LMLocationData("Hidden Room", 14, "Furniture", 246, [],
        update_ram_addr=[LMRamData(in_game_room_id=1)], require_poltergust=False),
    "Rec Room Treadmill Key": LMLocationData("Rec Room", 18, "Furniture", 106, [],
        update_ram_addr=[LMRamData(in_game_room_id=23)], require_poltergust=False),
    "Courtyard Birdhouse": LMLocationData("Courtyard", 20, "Furniture", 146, [],
        update_ram_addr=[LMRamData(in_game_room_id=24)], require_poltergust=False),
    "Sealed Room NW Shelf Chest": LMLocationData("Sealed Room", 29, "Furniture", 532, [],
        update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room NE Shelf Chest": LMLocationData("Sealed Room", 30, "Furniture", 534, [],
        update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room SE Shelf Chest": LMLocationData("Sealed Room", 32, "Furniture", 535, [],
        update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room Table Chest": LMLocationData("Sealed Room", 33, "Furniture", 533, [],
        update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room Upper L Big Chest": LMLocationData("Sealed Room", 35, "Furniture", 528, [],
        update_ram_addr=[LMRamData(in_game_room_id=37)], require_poltergust=False),
    "Sealed Room Upper R Big Chest": LMLocationData("Sealed Room", 37, "Furniture", 530, [],
        update_ram_addr=[LMRamData(in_game_room_id=37)], require_poltergust=False),
    "Armory Gray Chest (left, back Wall)": LMLocationData("Armory", 39, "Furniture", 652, [],
        update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory Gray Chest (middle, back Wall)": LMLocationData("Armory", 41, "Furniture", 654, [],
        update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory Gray Chest (right, back Wall)": LMLocationData("Armory", 42, "Furniture", 655, [],
        update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Telephone Room R2 Chest": LMLocationData("Telephone Room", 45, "Furniture", 682, [],
        update_ram_addr=[LMRamData(in_game_room_id=53)], require_poltergust=False),
    "Huge Flower (Boneyard)":   LMLocationData("Boneyard", 70, "Chest", 8, ["Water Element Medal"],
        update_ram_addr=[LMRamData(0x803CDF66, bit_position=2, in_game_room_id=11, ram_byte_size=2)]),

    "Hidden Room Large Chest L": LMLocationData("Hidden Room", 9, "Furniture", 243, [], remote_only=True,
                                                update_ram_addr=[LMRamData(in_game_room_id=1)], require_poltergust=False),
    "Hidden Room Large Chest C": LMLocationData("Hidden Room", 10, "Furniture", 241, [], remote_only=True,
                                                     update_ram_addr=[LMRamData(in_game_room_id=1)], require_poltergust=False),
    "Hidden Room Small Chest L Shelf": LMLocationData("Hidden Room", 12, "Furniture", 245, [], remote_only=True,
                                                     update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room Small Chest R Floor": LMLocationData("Hidden Room", 15, "Furniture", 247, [], remote_only=True,
                                                     update_ram_addr=[LMRamData(in_game_room_id=1)], require_poltergust=False),
    "Sealed Room Lower Big Chest": LMLocationData("Sealed Room", 34, "Furniture", 529, [], remote_only=True,
                                                  update_ram_addr=[LMRamData(in_game_room_id=37)], require_poltergust=False),
    "Sealed Room Upper C Big Chest": LMLocationData("Sealed Room", 36, "Furniture", 527, [], remote_only=True,
                                                     update_ram_addr=[LMRamData(in_game_room_id=37)], require_poltergust=False),
    "Sealed Room SW Shelf Chest": LMLocationData("Sealed Room", 869, "Furniture", 531, [],
        update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Armory 4th Gray Chest": LMLocationData("Armory", 38, "Furniture", 651, [], remote_only=True,
                                                     update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory 5th Gray Chest": LMLocationData("Armory", 40, "Furniture", 653, [], remote_only=True,
                                                     update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Telephone Room C Chest": LMLocationData("Telephone Room", 43, "Furniture", 680, [], remote_only=True,
                                                     update_ram_addr=[LMRamData(in_game_room_id=53)], require_poltergust=False),
    "Telephone Room R1 Chest": LMLocationData("Telephone Room", 44, "Furniture", 681, [], remote_only=True,
                                                     update_ram_addr=[LMRamData(in_game_room_id=53)], require_poltergust=False),
    "Visit E. Gadd's Gallery": LMLocationData("Foyer", 925, "Map", 0, [], remote_only=True,
                                      update_ram_addr=[LMRamData(0x804D80A4)], require_poltergust=False, map_id=[6]),
    "Complete Training": LMLocationData("Training Room", 926, "Special", 0, [], remote_only=True,
                       update_ram_addr=[LMRamData(0x803D33B2, bit_position=0, in_game_room_id=0)], map_id=[3]),
    "Catch 9 Ghosts in Training": LMLocationData("Training Room", 927, "Special", 0, [], remote_only=True,
                       update_ram_addr=[LMRamData(0x803D33B2, bit_position=1, in_game_room_id=0)], map_id=[3]),
    "Wardrobe Clear Chest": LMLocationData("Wardrobe", 0, "Chest", 22, ["Blackout"],
          update_ram_addr=[LMRamData(0x803CDF9C, bit_position=2, in_game_room_id=41, ram_byte_size=2)]),


    # Special Case
    "Observatory Shoot the Moon": LMLocationData("Observatory", 24, "Special", -1, [], remote_only=True,
        update_ram_addr = [LMRamData(0x803D339F, bit_position=2, in_game_room_id=44)]),

    # Game Event Locations
    "Breaker Box": LMLocationData("Breaker Room", None, "Event", -1, [], locked_item="Blackout"),
    "King Boo": LMLocationData("Secret Altar", None, "Event", -1, [], locked_item="Mario's Painting"),
    "Clockwork Clear": LMLocationData("Clockwork Room", None, "Event", -1, [], locked_item="Defeat Clockwork"),
    "Shivers Spawn": LMLocationData("1F Hallway", None, "Event", -1, [], locked_item="Shivers Spawn")
}

FREESTANDING_KEY_TABLE = {
    "Ghost Foyer Key": LMLocationData("Foyer", 713, "Freestanding", 1, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D3399, bit_position=3, in_game_room_id=2)], require_poltergust=False),
    "Fortune Teller Candles": LMLocationData("Fortune-Teller's Room", 6, "Freestanding", 4, ["Fire Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D33A6, bit_position=6, in_game_room_id=3)]),
    "The Well Key": LMLocationData("The Well", 21, "Freestanding", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFDA, bit_position=1, ram_byte_size=2)]),
    "Wardrobe Shelf Key": LMLocationData("Wardrobe", 50, "Freestanding", 5, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D33A6, bit_position=6, in_game_room_id=41)]),
    "1F Bathroom Shelf Key": LMLocationData("1F Bathroom", 54, "Freestanding", 3, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D33A6, bit_position=6, in_game_room_id=21)]),
}

# Adds all the chests that are spawned after clearing a room of ghosts.
CLEAR_GHOST_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Butler Clear Chest": LMLocationData("Butler's Room", 8, "Chest", 0, ["Fire Element Medal", "Shivers Spawn"],
        update_ram_addr=[LMRamData(0x803CDF50, bit_position=2, in_game_room_id=0, ram_byte_size=2)]),
    "Guest Room Clear Chest": LMLocationData("Guest Room", 46, "Chest", 17, ["Water Element Medal"],
        update_ram_addr = [LMRamData(0x803CDF88, bit_position=2, in_game_room_id=29, ram_byte_size=2)]),
    "Parlor Clear Chest": LMLocationData("Parlor", 47, "Chest", 20, [],
        update_ram_addr=[LMRamData(0x803CDF96, bit_position=2, in_game_room_id=36, ram_byte_size=2)]),
    "Cold Storage Clear Chest": LMLocationData("Cold Storage", 48, "Chest", 33, ["Fire Element Medal"],
        update_ram_addr=[LMRamData(0x803CDFCA, bit_position=2, in_game_room_id=64, ram_byte_size=2)]),
    "Breaker Room Clear Chest": LMLocationData("Breaker Room", 49, "Chest", 36, ["Blackout"],
        update_ram_addr=[LMRamData(0x803CDFD6, bit_position=2, in_game_room_id=69, ram_byte_size=2)]),
    "Twins' Room Clear Chest": LMLocationData("Twins' Room", 25, "Chest", 16, [],
        update_ram_addr=[LMRamData(0x803CDF82, bit_position=2, in_game_room_id=27, ram_byte_size=2)]),
    "Billiards Room Clear Chest": LMLocationData("Billiards Room", 26, "Chest", 9, [],
        update_ram_addr=[LMRamData(0x803CDF68, bit_position=2, in_game_room_id=12, ram_byte_size=2)]),
    "Balcony Clear Chest": LMLocationData("Balcony", 27, "Chest", 31, ["Ice Element Medal"],
        update_ram_addr=[LMRamData(0x803CDFC6, bit_position=2, in_game_room_id=62, ram_byte_size=2)]),
    "Ceramics Studio Clear Chest": LMLocationData("Ceramics Studio", 28, "Chest", 30, ["Ice Element Medal"],
        update_ram_addr=[LMRamData(0x803CDFBE, bit_position=2, in_game_room_id=58, ram_byte_size=2)]),
    "2F Bathroom Clear Chest": LMLocationData("2F Bathroom", 22, "Chest", 24, ["Ice Element Medal"],
        update_ram_addr=[LMRamData(0x803CDFAA, bit_position=2, in_game_room_id=48, ram_byte_size=2)]),
    "Nana's Room Clear Chest": LMLocationData("Nana's Room", 23, "Chest", 25, [],
        update_ram_addr=[LMRamData(0x803CDFAC, bit_position=2, in_game_room_id=49, ram_byte_size=2)]),
    "Rec Room Clear Chest": LMLocationData("Rec Room", 19, "Chest", 13, [],
        update_ram_addr=[LMRamData(0x803CDF7C, bit_position=2, in_game_room_id=23, ram_byte_size=2)]),
    "Conservatory Clear Chest": LMLocationData("Conservatory", 16, "Chest", 12, [],
        update_ram_addr=[LMRamData(0x803CDF7A, bit_position=2, in_game_room_id=22, ram_byte_size=2)]),
    "Dining Room Clear Chest": LMLocationData("Dining Room", 17, "Chest", 6, ["Fire Element Medal"],
        update_ram_addr=[LMRamData(0x803CDF62, bit_position=2, in_game_room_id=8, ram_byte_size=2)]),
    "Fortune Teller Clear Chest": LMLocationData("Fortune-Teller's Room", 5, "Chest", 2, [],
        update_ram_addr=[LMRamData(0x803CDF56, bit_position=2, in_game_room_id=3, ram_byte_size=2)]),
    "Study Clear Chest": LMLocationData("Study", 1, "Chest", 19, [],
        update_ram_addr=[LMRamData(0x803CDF94, bit_position=2, in_game_room_id=35, ram_byte_size=2)]),
    "Master Bedroom Clear Chest": LMLocationData("Master Bedroom", 2, "Chest", 18, [],
        update_ram_addr=[LMRamData(0x803CDF92, bit_position=2, in_game_room_id=34, ram_byte_size=2)]),
    "Nursery Clear Chest": LMLocationData("Nursery", 3, "Chest", 15, [],
        update_ram_addr=[LMRamData(0x803CDF80, bit_position=2, in_game_room_id=26, ram_byte_size=2)]),
    "Pipe Room Clear Chest": LMLocationData("Pipe Room", 64, "Chest", 35, ["Ice Element Medal"],
        update_ram_addr=[LMRamData(0x803CDFD4, bit_position=2, in_game_room_id=68, ram_byte_size=2)]),
}

CLEAR_LOCATION_TABLE = {**CLEAR_GHOST_LOCATION_TABLE, **FREESTANDING_KEY_TABLE}

# Ghost Affected Clear Chests. Rules applied to region entrances
ENEMIZER_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Laundry Room Clear Chest": LMLocationData("Laundry Room", 710, "Chest", 4, [],
        update_ram_addr=[LMRamData(0x803CDF5A, bit_position=2, in_game_room_id=5, ram_byte_size=2)]),
    "Hidden Room Clear Chest": LMLocationData("Hidden Room", 51, "Chest", 1, [],
        update_ram_addr=[LMRamData(0x803CDF52, bit_position=2, in_game_room_id=1, ram_byte_size=2)]),
    "Mirror Room Clear Chest": LMLocationData("Mirror Room", 52, "Chest", 3, [],
        update_ram_addr=[LMRamData(0x803CDF58, bit_position=2, in_game_room_id=4, ram_byte_size=2)]),
    "Kitchen Clear Chest": LMLocationData("Kitchen", 53, "Chest", 5, [],
        update_ram_addr=[LMRamData(0x803CDF60, bit_position=2, in_game_room_id=7, ram_byte_size=2)]),
    "Graveyard Clear Chest": LMLocationData("Graveyard", 711, "Chest", 11, [],
        update_ram_addr=[LMRamData(0x803CDF70, bit_position=2, in_game_room_id=15, ram_byte_size=2)]),
    "Courtyard Clear Chest": LMLocationData("Courtyard", 55, "Chest", 14, [],
        update_ram_addr=[LMRamData(0x803CDF7E, bit_position=2, in_game_room_id=24, ram_byte_size=2)]),
    "Tea Room Clear Chest": LMLocationData("Tea Room", 56, "Chest", 26, [],
        update_ram_addr=[LMRamData(0x803CDFAE, bit_position=2, in_game_room_id=50, ram_byte_size=2)]),
    "2F Washroom Clear Chest": LMLocationData("2F Washroom", 57, "Chest", 23, [],
        update_ram_addr=[LMRamData(0x803CDFA4, bit_position=2, in_game_room_id=45, ram_byte_size=2)]),
    "Projection Room Clear Chest": LMLocationData("Projection Room", 58, "Chest", 10, [],
        update_ram_addr=[LMRamData(0x803CDF6A, bit_position=2, in_game_room_id=13, ram_byte_size=2)]),
    "Safari Room Clear Chest": LMLocationData("Safari Room", 59, "Chest", 29, [],
        update_ram_addr=[LMRamData(0x803CDFB8, bit_position=2, in_game_room_id=55, ram_byte_size=2)]),
    "Ballroom Clear Chest": LMLocationData("Ballroom", 715, "Chest", 7, [],
        update_ram_addr=[LMRamData(0x803CDF64, bit_position=2, in_game_room_id=9, ram_byte_size=2)]),
    "Cellar Clear Chest": LMLocationData("Cellar", 60, "Chest", 34, [],
        update_ram_addr=[LMRamData(0x803CDFCE, bit_position=2, in_game_room_id=66, ram_byte_size=2)]),
    "Roof Clear Chest": LMLocationData("Roof", 61, "Chest", 32, [],
        update_ram_addr=[LMRamData(0x803CDFC8, bit_position=2, in_game_room_id=63, ram_byte_size=2)]),
    "Sealed Room Clear Chest": LMLocationData("Sealed Room", 62, "Chest", 21, [],
        update_ram_addr=[LMRamData(0x803CDF98, bit_position=2, in_game_room_id=37, ram_byte_size=2)]),
    "Armory Clear Chest": LMLocationData("Armory", 63, "Chest", 27, [],
        update_ram_addr=[LMRamData(0x803CDFB0, bit_position=2, in_game_room_id=51, ram_byte_size=2)]),
    "Telephone Room Clear Chest": LMLocationData("Telephone Room", 716, "Chest", 28, [],
        update_ram_addr=[LMRamData(0x803CDFB4, bit_position=2, in_game_room_id=53, ram_byte_size=2)]),
    "Van Gore's Painting (Artist's Studio)": LMLocationData("Artist's Studio", 709, "Furniture", 690, [],
        update_ram_addr=[LMRamData(0x803CDFC2, bit_position=2, in_game_room_id=60, ram_byte_size=2)]),
    "Defeat Chauncey": LMLocationData("Nursery", 924, "Portrait", 77, [], remote_only=True,
            update_ram_addr=[LMRamData(0x803D5DAC, bit_position=0, in_game_room_id=26)], map_id=[2,10]),
    "Defeat Bogmire": LMLocationData("Graveyard", 923, "Portrait", 0, [], remote_only=True,
            update_ram_addr=[LMRamData(0x803D5DBE, bit_position=5, in_game_room_id=15)], map_id=[2,13]),
    "Defeat Boolussus": LMLocationData("Balcony", 922, "Portrait", 56, ["Ice Element Medal"],
                    remote_only=True, update_ram_addr=[LMRamData(0x803D5DBF, bit_position=4, in_game_room_id=62)], map_id=[2,11]),
    "Storage Room Cage Button": LMLocationData("Storage Room", 712, "Special", 256, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D33A2, bit_position=0, in_game_room_id=14)]),
    "Storage Room Wall Button": LMLocationData("Storage Room", 921, "Special", 284, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D339E, bit_position=1, in_game_room_id=14)]),
}

# Adds Toads as locations
TOAD_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Starting Room Toad": LMLocationData("Foyer", 617, "Toad", -1, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D33A6, bit_position=5, in_game_room_id=2)], require_poltergust=False),
    "Foyer Toad": LMLocationData("Foyer", 928, "Toad", -1, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D33A6, bit_position=5, in_game_room_id=2)], require_poltergust=False),
    "Wardrobe Balcony Toad": LMLocationData("Wardrobe Balcony", 618, "Toad", -1, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D33A6, bit_position=5, in_game_room_id=40)], require_poltergust=False),
    "1F Washroom Toad": LMLocationData("1F Washroom", 619, "Toad", -1, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D33A6, bit_position=5, in_game_room_id=16)], require_poltergust=False),
    "Courtyard Toad": LMLocationData("Courtyard", 620, "Toad", -1, [],  remote_only=True,
        update_ram_addr=[LMRamData(0x803D33A6, bit_position=5, in_game_room_id=24)]),
}

# Adds all waterable plants as locations
PLANT_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Wardrobe Balcony Plant R (near Door)": LMLocationData("Wardrobe Balcony", 65, "Plant", 473,
        ["Water Element Medal"], 2, update_ram_addr=[LMRamData(in_game_room_id=40)]),
    "Wardrobe Balcony Plant L (next to Toad)": LMLocationData("Wardrobe Balcony", 66, "Plant", 474,
        ["Water Element Medal"], 2, update_ram_addr=[LMRamData(in_game_room_id=40)]),
    "Wardrobe Balcony Plant C": LMLocationData("Wardrobe Balcony", 67, "Plant", 475, ["Water Element Medal"], 2,
        update_ram_addr=[LMRamData(in_game_room_id=40)]),
    "Wardrobe Balcony Plant L (in Corner)": LMLocationData("Wardrobe Balcony", 68, "Plant", 476,
        ["Water Element Medal"], 2, update_ram_addr=[LMRamData(in_game_room_id=40)]),
    "Master Bedroom Plant": LMLocationData("Master Bedroom", 69, "Plant", 542, ["Water Element Medal"], 2,
        update_ram_addr=[LMRamData(in_game_room_id=34)]),
    "Courtyard SW Plant": LMLocationData("Courtyard", 71, "Plant", 142, ["Water Element Medal"], 1,
        update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard NW Plant": LMLocationData("Courtyard", 72, "Plant", 143, ["Water Element Medal"], 1,
        update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard NE Plant": LMLocationData("Courtyard", 73, "Plant", 144, ["Water Element Medal"], 1,
        update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard SE Plant": LMLocationData("Courtyard", 74, "Plant", 145, ["Water Element Medal"], 1,
        update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Balcony Plant NW Not Near Bench": LMLocationData("Balcony", 75, "Plant", 604, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant E Railing": LMLocationData("Balcony", 76, "Plant", 605, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant NE Near Bench": LMLocationData("Balcony", 77, "Plant", 606, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant NE Corner Near Bench": LMLocationData("Balcony", 78, "Plant", 607, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant SW Corner": LMLocationData("Balcony", 81, "Plant", 610, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant NW Corner": LMLocationData("Balcony", 82, "Plant", 611, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony West Double Plant": LMLocationData("Balcony", 83, "Plant", 612, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Center R": LMLocationData("Balcony", 84, "Plant", 613, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Directly in NE Corner": LMLocationData("Balcony", 872, "Plant", 608, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant SE Corner": LMLocationData("Balcony", 80, "Plant", 609, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Second from Right Plant near NW Bench": LMLocationData("Balcony", 873, "Plant", 615,
        ["Water Element Medal"], 3, update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Furthest Right Plant near NW Bench": LMLocationData("Balcony", 874, "Plant", 614,
        ["Water Element Medal"], 3, update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Second from Right in NE Corner": LMLocationData("Balcony", 875, "Plant", 616, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Center L": LMLocationData("Balcony", 88, "Plant", 712, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Third from Right in NE Corner": LMLocationData("Balcony", 876, "Plant", 713, ["Water Element Medal"], 3,
        update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Sitting Room Plant": LMLocationData("Sitting Room", 91, "Plant", 409, ["Water Element Medal"], 2,
        update_ram_addr=[LMRamData(in_game_room_id=28)]),
    "Guest Room Plant": LMLocationData("Guest Room", 92, "Plant", 417, ["Water Element Medal"], 2,
        update_ram_addr=[LMRamData(in_game_room_id=29)]),
}

CEILING_LOCATION_TABLE ={
    "Heart Door Lamp": LMLocationData("Foyer", 94, "Furniture", 270, [], 1,
                                      update_ram_addr=[LMRamData(in_game_room_id=2)]),
    "Foyer Chandelier": LMLocationData("Foyer", 95, "Furniture", 101, [], 1,
                                       update_ram_addr=[LMRamData(in_game_room_id=2)]),
    "Parlor Chandelier": LMLocationData("Parlor", 104, "Furniture", 342, [], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Anteroom R Chandelier": LMLocationData("Anteroom", 120, "Furniture", 485, [], 2,
                                            update_ram_addr=[LMRamData(in_game_room_id=42)]),
    "Anteroom L Chandelier": LMLocationData("Anteroom", 121, "Furniture", 467, [], 2,
                                            update_ram_addr=[LMRamData(in_game_room_id=42)]),
    "Wardrobe Chandelier": LMLocationData("Wardrobe", 122, "Furniture", 454, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=41)]),
    "Wardrobe Balcony Brazier": LMLocationData("Wardrobe Balcony", 130, "Furniture", 472, [], 2,
                                               update_ram_addr=[LMRamData(in_game_room_id=40)]),
    "Study Chandelier": LMLocationData("Study", 140, "Furniture", 352, [], 2,
                                       update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Master Bedroom Ceiling Fan": LMLocationData("Master Bedroom", 161, "Furniture", 373, [], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=34)]),
    "Nursery Ceiling Lamp": LMLocationData("Nursery", 177, "Furniture", 374, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=26)]),
    "Twins' Room Star Light": LMLocationData("Twins' Room", 183, "Furniture", 423, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=27)]),
    "1F Washroom Light": LMLocationData("1F Washroom", 205, "Furniture", 91, [], 1,
                                        update_ram_addr=[LMRamData(in_game_room_id=16)]),
    "Ballroom West Chandelier": LMLocationData("Ballroom", 207, "Furniture", 173, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=9)]),
    "Ballroom East Chandelier": LMLocationData("Ballroom", 208, "Furniture", 174, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=9)]),
    "Storage Room Ceiling Light": LMLocationData("Storage Room", 216, "Furniture", 163, [], 1,
                                                 update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "Mirror Room West Chandelier": LMLocationData("Mirror Room", 238, "Furniture", 158, [], 1,
                                                  update_ram_addr=[LMRamData(in_game_room_id=4)]),
    "Mirror Room East Chandelier": LMLocationData("Mirror Room", 239, "Furniture", 159, [], 1,
                                                  update_ram_addr=[LMRamData(in_game_room_id=4)]),
    "Laundry Room Ceiling Light": LMLocationData("Laundry Room", 246, "Furniture", 195, [], 1,
                                                 update_ram_addr=[LMRamData(in_game_room_id=5)]),
    "Butler's Room Ceiling Light": LMLocationData("Butler's Room", 257, "Furniture", 228, [], 1,
                                                  update_ram_addr=[LMRamData(in_game_room_id=0)]),
    "Hidden Room Chandelier": LMLocationData("Hidden Room", 267, "Furniture", 240, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Kitchen L Ceiling Light": LMLocationData("Kitchen", 315, "Furniture", 84, [], 1,
                                              update_ram_addr=[LMRamData(in_game_room_id=7)]),
    "Kitchen R Ceiling Light": LMLocationData("Kitchen", 316, "Furniture", 85, [], 1,
                                              update_ram_addr=[LMRamData(in_game_room_id=7)]),
    "Rec Room L Ceiling Light": LMLocationData("Rec Room", 347, "Furniture", 111, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "Rec Room R Ceiling Light": LMLocationData("Rec Room", 348, "Furniture", 110, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "2F Stairwell Ceiling Light": LMLocationData("2F Stairwell", 349, "Furniture", 310, [], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=18)]),
    "Attic Stairwell Ceiling Light": LMLocationData("2F Rear Hallway", 369, "Furniture", 547, [], 3,
                                                 update_ram_addr=[LMRamData(in_game_room_id=33)]),
    "Tea Room Chandelier": LMLocationData("Tea Room", 365, "Furniture", 399, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "2F Washroom Light": LMLocationData("2F Washroom", 383, "Furniture", 442, [], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=45)]),
    "Nana's Room Chandelier": LMLocationData("Nana's Room", 395, "Furniture", 378, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=49)]),
    "Astral Hall Chandelier": LMLocationData("Astral Hall", 403, "Furniture", 446, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=43)]),
    "Billiards Room Ceiling Fan": LMLocationData("Billiards Room", 421, "Furniture", 156, [], 1,
                                                 update_ram_addr=[LMRamData(in_game_room_id=12)]),
    "Projection Room L Ceiling Light": LMLocationData("Projection Room", 427, "Furniture", 199, [], 1,
                                                      update_ram_addr=[LMRamData(in_game_room_id=13)]),
    "Projection Room R Ceiling Light": LMLocationData("Projection Room", 428, "Furniture", 200, [], 1,
                                                      update_ram_addr=[LMRamData(in_game_room_id=13)]),
    "Safari Room L Ceiling Light": LMLocationData("Safari Room", 436, "Furniture", 631, [], 3,
                                                  update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Safari Room R Ceiling Light": LMLocationData("Safari Room", 437, "Furniture", 632, [], 3,
                                                  update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Breaker Room Ceiling Light": LMLocationData("Breaker Room", 446, "Furniture", 43, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=69)]),
    "Cellar Ceiling Light": LMLocationData("Cellar", 460, "Furniture", 4, [],
                                           update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Telephone Room L Ceiling Light": LMLocationData("Telephone Room", 479, "Furniture", 671, [], 3,
                                                     update_ram_addr=[LMRamData(in_game_room_id=53)]),
    "Telephone Room R Ceiling Light": LMLocationData("Telephone Room", 480, "Furniture", 672, [], 3,
                                                     update_ram_addr=[LMRamData(in_game_room_id=53)]),
    "Clockwork Room R Ceiling Light": LMLocationData("Clockwork Room", 486, "Furniture", 596, [], 3,
                                                     update_ram_addr=[LMRamData(in_game_room_id=59)]),
    "Clockwork Room L Ceiling Light": LMLocationData("Clockwork Room", 487, "Furniture", 595, [], 3,
                                                     update_ram_addr=[LMRamData(in_game_room_id=59)]),
    "Armory Ceiling Light": LMLocationData("Armory", 526, "Furniture", 636, [], 3,
                                           update_ram_addr=[LMRamData(in_game_room_id=51)]),
    "Ceramics Studio Ceiling Light": LMLocationData("Ceramics Studio", 533, "Furniture", 657, [], 3,
                                                    update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Sealed Room Chandelier": LMLocationData("Sealed Room", 557, "Furniture", 463, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Artist's Studio Light": LMLocationData("Artist's Studio", 597, "Furniture", 684, [], 3,
                                            update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Basement Stairwell Hanging Light": LMLocationData("Basement Stairwell", 596, "Furniture", 62, [], 1,
                                                       update_ram_addr=[LMRamData(in_game_room_id=67)]),
    "Conservatory Ceiling Light": LMLocationData("Conservatory", 283, "Furniture", 137, [], 1,
                                                 update_ram_addr=[LMRamData(in_game_room_id=22)]),
    "Secret Altar C Chandelier": LMLocationData("Secret Altar", 613, "Furniture", 38, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=73)]),
    "Secret Altar R Chandelier": LMLocationData("Secret Altar", 612, "Furniture", 40, [],
                                                     update_ram_addr=[LMRamData(in_game_room_id=73)]),
    "Secret Altar L Chandelier": LMLocationData("Secret Altar", 614, "Furniture", 39, [],
                                                     update_ram_addr=[LMRamData(in_game_room_id=1)]),
}

DECOR_LOCATION_TABLE = {
    "Foyer Mirror": LMLocationData("Foyer", 96, "Furniture", 208, [], 1,
                                   update_ram_addr=[LMRamData(in_game_room_id=2)], require_poltergust=False),
    "Upper Foyer R Vase": LMLocationData("Foyer", 97, "Furniture", 336, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=31)]),
    "Upper Foyer L Vase": LMLocationData("Foyer", 98, "Furniture", 335, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=31)]),
    "Parlor Green Lamp": LMLocationData("Parlor", 107, "Furniture", 490, [], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Wardrobe Mirror": LMLocationData("Wardrobe", 126, "Furniture", 480, [], 2,
                                      update_ram_addr=[LMRamData(in_game_room_id=41)], require_poltergust=False),
    "Study Lamp": LMLocationData("Study", 138, "Furniture", 349, [], 2,
                                 update_ram_addr=[LMRamData(in_game_room_id=35)], require_poltergust=False),
    "Study Bottled Ship": LMLocationData("Study", 142, "Furniture", 353, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Book (on Desk closer to Door)": LMLocationData("Study", 145, "Furniture", 503, [], 2,
                                                          update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Book (on Desk closer to Bookshelves)": LMLocationData("Study", 146, "Furniture", 504, [], 2,
                                                                 update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Lydia's Vanity (Master Bedroom)": LMLocationData("Master Bedroom", 159, "Furniture", 375, [], 2,
                                                      update_ram_addr=[LMRamData(in_game_room_id=34)], require_poltergust=False),
    "Master Bedroom Lamp": LMLocationData("Master Bedroom", 165, "Furniture", 493, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=34)]),
    "Nursery Vase": LMLocationData("Nursery", 166, "Furniture", 488, [], 2,
                                   update_ram_addr=[LMRamData(in_game_room_id=26)]),
    "Chauncey's Crib (Nursery)": LMLocationData("Nursery", 167, "Furniture", 372, [], 2, remote_only=True,
                                                update_ram_addr=[LMRamData(in_game_room_id=26)], require_poltergust=False),
    "Nursery Toy Blocks R (closer to Door)": LMLocationData("Nursery", 173, "Furniture", 498, [], 2,
                                                            update_ram_addr=[LMRamData(in_game_room_id=26)]),
    "Nursery Toy Blocks R (closer to 4th Wall)": LMLocationData("Nursery", 178, "Furniture", 499, [], 2,
                                                                update_ram_addr=[LMRamData(in_game_room_id=26)]),
    "Twins' Room Tennis Rackets": LMLocationData("Twins' Room", 180, "Furniture", 562, [], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=27)]),
    "Twins' Room Toy Train": LMLocationData("Twins' Room", 184, "Furniture", 430, [], 2,
                                            update_ram_addr=[LMRamData(in_game_room_id=27)]),
    "Twins' Room Toy Car": LMLocationData("Twins' Room", 185, "Furniture", 431, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=27)]),
    "Twins' Room Toy Truck": LMLocationData("Twins' Room", 191, "Furniture", 432, [], 2,
                                            update_ram_addr=[LMRamData(in_game_room_id=27)]),
    "1F Hallway Front Vase L": LMLocationData("1F Hallway", 705, "Furniture", 93, [], 1,
                                              update_ram_addr=[LMRamData(in_game_room_id=6)]),
    "1F Hallway Front Vase C": LMLocationData("1F Hallway", 706, "Furniture", 94, [], 1,
                                              update_ram_addr=[LMRamData(in_game_room_id=6)]),
    "1F Hallway Front Vase R": LMLocationData("1F Hallway", 193, "Furniture", 95, [], 1,
                                              update_ram_addr=[LMRamData(in_game_room_id=6)]),
    "1F Hallway Rear Vase L": LMLocationData("1F Hallway", 194, "Furniture", 102, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=17)]),
    "1F Hallway Rear Vase C": LMLocationData("1F Hallway", 195, "Furniture", 103, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=17)]),
    "1F Hallway Rear Vase R": LMLocationData("1F Hallway", 196, "Furniture", 283, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=17)]),
    "1F Bathroom Sink": LMLocationData("1F Bathroom", 197, "Furniture", 216, [], 1,
                                       update_ram_addr=[LMRamData(in_game_room_id=21)], require_poltergust=False),
    "1F Bathroom Bathtub": LMLocationData("1F Bathroom", 198, "Furniture", 166, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=21)]),
    "1F Washroom Toilet Paper": LMLocationData("1F Washroom", 200, "Furniture", 269, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=16)]),
    "1F Washroom Sink": LMLocationData("1F Washroom", 707, "Furniture", 89, [], 1,
                                       update_ram_addr=[LMRamData(in_game_room_id=16)], require_poltergust=False),
    "1F Washroom Radiator": LMLocationData("1F Washroom", 203, "Furniture", 90, [], 1,
                                           update_ram_addr=[LMRamData(in_game_room_id=16)], require_poltergust=False),
    "Fortune Teller's Room West Pillar": LMLocationData("Fortune-Teller's Room", 232, "Furniture", 297, [], 1,
                                                        update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Fortune Teller's Room East Pillar": LMLocationData("Fortune-Teller's Room", 233, "Furniture", 298, [], 1,
                                                        update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Mirror Room Brazier": LMLocationData("Mirror Room", 235, "Furniture", 302, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=4)], require_poltergust=False),
    "Laundry Room Broom": LMLocationData("Laundry Room", 241, "Furniture", 192, [], 1,
                                         update_ram_addr=[LMRamData(in_game_room_id=5)]),
    "Laundry Room Dustpan": LMLocationData("Laundry Room", 243, "Furniture", 194, [], 1,
                                           update_ram_addr=[LMRamData(in_game_room_id=5)]),
    "Butler's Room Toilet Paper (near Hidden Room)": LMLocationData("Butler's Room", 255, "Furniture", 219, [], 1,
                                                                    update_ram_addr=[LMRamData(in_game_room_id=0)]),
    "Butler's Room Toilet Paper (near Mop n Bucket)": LMLocationData("Butler's Room", 256, "Furniture", 220, [], 1,
                                                                     update_ram_addr=[LMRamData(in_game_room_id=0)]),
    "Hidden Room R Trophy": LMLocationData("Hidden Room", 263, "Furniture", 236, [], 1,
                                           update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room L Trophy": LMLocationData("Hidden Room", 264, "Furniture", 237, [], 1,
                                           update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Conservatory Xylophone Music Stand": LMLocationData("Conservatory", 279, "Furniture", 126, [], 1,
                                                         update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Harp Music Stand": LMLocationData("Conservatory", 280, "Furniture", 127, [], 1,
                                                    update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Xylophone": LMLocationData("Conservatory", 284, "Furniture", 132, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Cello": LMLocationData("Conservatory", 285, "Furniture", 131, [], 1,
                                         update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Left Bongo": LMLocationData("Conservatory", 286, "Furniture", 123, [], 1,
                                              update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Right Bongo": LMLocationData("Conservatory", 287, "Furniture", 122, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Center Bongo": LMLocationData("Conservatory", 288, "Furniture", 121, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Harp": LMLocationData("Conservatory", 289, "Furniture", 133, [], 1,
                                        update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Saxophone": LMLocationData("Conservatory", 290, "Furniture", 305, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Dining Room L Brazier": LMLocationData("Dining Room", 296, "Furniture", 281, [], 1,
                                            update_ram_addr=[LMRamData(in_game_room_id=8)], require_poltergust=False),
    "Dining Room R Brazier": LMLocationData("Dining Room", 297, "Furniture", 282, [], 1,
                                            update_ram_addr=[LMRamData(in_game_room_id=8)], require_poltergust=False),
    "Dining Room Table Mess L (furthest Left)": LMLocationData("Dining Room", 300, "Furniture", 211, [], 1,
                                                               update_ram_addr=[LMRamData(in_game_room_id=8)]),
    "Dining Room Table Mess L (near Luggs)": LMLocationData("Dining Room", 301, "Furniture", 212, [], 1,
                                                            update_ram_addr=[LMRamData(in_game_room_id=8)]),
    "Dining Room Table Mess R (near Candles)": LMLocationData("Dining Room", 302, "Furniture", 213, [], 1,
                                                              update_ram_addr=[LMRamData(in_game_room_id=8)]),
    "Dining Room Table Mess R": LMLocationData("Dining Room", 303, "Furniture", 214, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=8)]),
    "Dining Room Table Mess L (underneath Table)": LMLocationData("Dining Room", 304, "Furniture", 215, [], 1,
                                                                  update_ram_addr=[LMRamData(in_game_room_id=8)]),
    "Kitchen Oven": LMLocationData("Kitchen", 307, "Furniture", 80, ["Fire Element Medal"], 1,
                                   update_ram_addr=[LMRamData(in_game_room_id=7)]),
    "Kitchen Dishwasher": LMLocationData("Kitchen", 314, "Furniture", 82, [], 1, remote_only=True,
                                         update_ram_addr=[LMRamData(in_game_room_id=7)], require_poltergust=False),
    "Spooky's Dog Bowl (Boneyard)": LMLocationData("Boneyard", 317, "Furniture", 97, [], 1,
                                                   update_ram_addr=[LMRamData(in_game_room_id=11)]),
    "Boneyard Faucet": LMLocationData("Boneyard", 318, "Furniture", 98, [], 1,
                                      update_ram_addr=[LMRamData(in_game_room_id=11)], require_poltergust=False),
    "Boneyard Sign": LMLocationData("Boneyard", 319, "Furniture", 280, [], 1,
                                    update_ram_addr=[LMRamData(in_game_room_id=11)], require_poltergust=False),
    "Bogmire's Tombstone (Graveyard)": LMLocationData("Graveyard", 321, "Furniture", 181, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Graveyard L Brazier": LMLocationData("Graveyard", 322, "Furniture", 182, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Graveyard R Brazier": LMLocationData("Graveyard", 323, "Furniture", 183, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Graveyard Tombstone (near Tree stump)": LMLocationData("Graveyard", 324, "Furniture", 175, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Graveyard Tombstone (back row, right side)": LMLocationData("Graveyard", 325, "Furniture", 176, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Graveyard Tombstone (center, right side)": LMLocationData("Graveyard", 327, "Furniture", 178, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Graveyard Tombstone (far right near drain pipe)": LMLocationData("Graveyard", 328, "Furniture", 179, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Graveyard Tombstone (left side, near bogmire)": LMLocationData("Graveyard", 329, "Furniture", 180, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Courtyard Outhouse": LMLocationData("Courtyard", 330, "Furniture", 271, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=24)], require_poltergust=False),
    "Courtyard Male Statue": LMLocationData("Courtyard", 331, "Furniture", 274, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=24)], require_poltergust=False),
    "Courtyard Female Statue": LMLocationData("Courtyard", 332, "Furniture", 273, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=24)], require_poltergust=False),
    "Courtyard R Lamppost": LMLocationData("Courtyard", 333, "Furniture", 275, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=24)], require_poltergust=False),
    "Courtyard L Lamppost": LMLocationData("Courtyard", 334, "Furniture", 276, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=24)], require_poltergust=False),
    "Well Lantern": LMLocationData("The Well", 335, "Furniture", 42, [],
        update_ram_addr=[LMRamData(in_game_room_id=72)]),
    "Rec Room Weight Station": LMLocationData("Rec Room", 336, "Furniture", 109, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=23)], require_poltergust=False),
    "Rec Room Rear Bicycle": LMLocationData("Rec Room", 339, "Furniture", 108, [], 1,
                                            update_ram_addr=[LMRamData(in_game_room_id=23)], require_poltergust=False),
    "Tea Room L Tea Set": LMLocationData("Tea Room", 351, "Furniture", 400, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "Tea Room R Tea Set": LMLocationData("Tea Room", 352, "Furniture", 401, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "Tea Room L Stored China": LMLocationData("Tea Room", 361, "Furniture", 397, [], 2,
                                              update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "Tea Room R Stored China": LMLocationData("Tea Room", 362, "Furniture", 398, [], 2,
                                              update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "2F Rear Hallway Vase L": LMLocationData("2F Rear Hallway", 370, "Furniture", 518, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=32)]),
    "2F Rear Hallway Vase R": LMLocationData("2F Rear Hallway", 371, "Furniture", 519, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=32)]),
    "2F Rear Hallway Vase 3": LMLocationData("2F Rear Hallway", 372, "Furniture", 521, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=46)]),
    "2F Rear Hallway Vase 4": LMLocationData("2F Rear Hallway", 373, "Furniture", 522, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=46)]),
    "2F Rear Hallway Vase 5": LMLocationData("2F Rear Hallway", 374, "Furniture", 523, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=46)]),
    "2F Rear Hallway Vase 6": LMLocationData("2F Rear Hallway", 375, "Furniture", 524, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=46)]),
    "2F Washroom Water Tank (above Toilet)": LMLocationData("2F Washroom", 376, "Furniture", 445, [], 2,
        update_ram_addr=[LMRamData(in_game_room_id=45)], remote_only=True),
    "Tea Room Tea Box": LMLocationData("Tea Room", 366, "Furniture", 481, [], 2,
                                       update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "2F Washroom Radiator": LMLocationData("2F Washroom", 380, "Furniture", 441, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=45)], require_poltergust=False),
    "2F Washroom Toilet Paper": LMLocationData("2F Washroom", 382, "Furniture", 439, [], 2,
                                               update_ram_addr=[LMRamData(in_game_room_id=45)]),
    "2F Washroom Sink": LMLocationData("2F Washroom", 384, "Furniture", 440, [], 2,
                                       update_ram_addr=[LMRamData(in_game_room_id=45)], require_poltergust=False),
    "Petunia's Bathtub (2F Bathroom)": LMLocationData("2F Bathroom", 385, "Furniture", 436, [], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=48)]),
    "2F Bathroom Sink": LMLocationData("2F Bathroom", 387, "Furniture", 500, [], 2,
                                       update_ram_addr=[LMRamData(in_game_room_id=48)], require_poltergust=False),
    "Nana's Room Spinning Wheel": LMLocationData("Nana's Room", 388, "Furniture", 385, [], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=49)], require_poltergust=False),
    "Nana's Room Books": LMLocationData("Nana's Room", 396, "Furniture", 416, [],2,
                                        update_ram_addr=[LMRamData(in_game_room_id=49)]),
    "Nana's Room Shelf Photos": LMLocationData("Nana's Room", 397, "Furniture", 381, [], 2,
                                               update_ram_addr=[LMRamData(in_game_room_id=49)]),
    "Nana's Room Shelf Clock": LMLocationData("Nana's Room", 398, "Furniture", 379, [], 2,
                                              update_ram_addr=[LMRamData(in_game_room_id=49)]),
    "Nana's Room Shelf Pots": LMLocationData("Nana's Room", 399, "Furniture", 380, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=49)]),
    "Nana's Yarn Balls (Nana's Room)": LMLocationData("Nana's Room", 401, "Furniture", 382, [], 2,
                                                      update_ram_addr=[LMRamData(in_game_room_id=49)], require_poltergust=False),
    "Billiards Room Table Drinks (near Chest)": LMLocationData("Billiards Room", 422, "Furniture", 292, [], 1,
                                                               update_ram_addr=[LMRamData(in_game_room_id=12)]),
    "Billiards Room Table Drinks (near Dart Board)": LMLocationData("Billiards Room", 423, "Furniture", 291, [], 1,
                                                                    update_ram_addr=[LMRamData(in_game_room_id=12)]),
    "Projection Room Projector": LMLocationData("Projection Room", 425, "Furniture", 206, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=13)]),
    "Projection Room L Speaker": LMLocationData("Projection Room", 429, "Furniture", 201, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=13)], require_poltergust=False),
    "Projection Room R Speaker": LMLocationData("Projection Room", 430, "Furniture", 202, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=13)], require_poltergust=False),
    "Safari Room Leopard Chest Box": LMLocationData("Safari Room", 433, "Furniture", 633, [], 3,
                                                    update_ram_addr=[LMRamData(in_game_room_id=55)], require_poltergust=False),
    "Balcony West Statue": LMLocationData("Balcony", 443, "Furniture", 621, [], 3,
                                          update_ram_addr=[LMRamData(in_game_room_id=62)], require_poltergust=False),
    "Balcony East Statue": LMLocationData("Balcony", 444, "Furniture", 620, [], 3,
                                          update_ram_addr=[LMRamData(in_game_room_id=62)], require_poltergust=False),
    "Clockwork Room R Puppet": LMLocationData("Clockwork Room", 488, "Furniture", 603, [], 3,
                                              update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Clockwork Room L Puppet": LMLocationData("Clockwork Room", 489, "Furniture", 602, [], 3,
                                              update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Roof Pillar (bottom row, furthest left)": LMLocationData("Roof", 505, "Furniture", 717, [], 4,
                                                              update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (top row, furthest left)": LMLocationData("Roof", 506, "Furniture", 718, [], 4,
                                                           update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (bottom row, second from left)": LMLocationData("Roof", 507, "Furniture", 719, [], 4,
                                                                 update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (top row, second from left)": LMLocationData("Roof", 508, "Furniture", 720, [], 4,
                                                              update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (bottom row, third from left)": LMLocationData("Roof", 509, "Furniture", 721, [], 4,
                                                                update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (top row, third from left)": LMLocationData("Roof", 510, "Furniture", 722, [], 4,
                                                             update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (bottom row, third from right)": LMLocationData("Roof", 511, "Furniture", 723, [], 4,
                                                                 update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (top row, third from right)": LMLocationData("Roof", 512, "Furniture", 724, [], 4,
                                                              update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (bottom row, second from right)": LMLocationData("Roof", 513, "Furniture", 725, [], 4,
                                                                  update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (top row, second from right)": LMLocationData("Roof", 514, "Furniture", 726, [], 4,
                                                               update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (top row, furthest right)": LMLocationData("Roof", 515, "Furniture", 727, [], 4,
                                                            update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Roof Pillar (bottom row, furthest right)": LMLocationData("Roof", 516, "Furniture", 728, [], 4,
                                                               update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Armory Armor Suit R (top)": LMLocationData("Armory", 517, "Furniture", 642, [], 3,
                                                update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory Armor Suit (middle, right)": LMLocationData("Armory", 518, "Furniture", 643, [], 3,
                                                        update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory Armor Suit (middle, left)": LMLocationData("Armory", 519, "Furniture", 644, [], 3,
                                                       update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory Armor Suit (middle, second from left)": LMLocationData("Armory", 520, "Furniture", 645, [], 3,
                                                                   update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory Armor Suit L (right side)": LMLocationData("Armory", 521, "Furniture", 646, [], 3,
                                                       update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Ceramics Studio Ice Vase": LMLocationData("Ceramics Studio", 534, "Furniture", 658, [], 3,
                                               update_ram_addr=[LMRamData(in_game_room_id=58)], require_poltergust=False),
    "Ceramics Studio Vase (bottom right)": LMLocationData("Ceramics Studio", 544, "Furniture", 660, [], 3,
                                                          update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (bottom middle)": LMLocationData("Ceramics Studio", 545, "Furniture", 661, [], 3,
                                                           update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (bottom left)": LMLocationData("Ceramics Studio", 546, "Furniture", 662, [], 3,
                                                         update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (bottom more left)": LMLocationData("Ceramics Studio", 547, "Furniture", 663, [], 3,
                                                              update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (top more left)": LMLocationData("Ceramics Studio", 548, "Furniture", 666, [], 3,
                                                           update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (top left)": LMLocationData("Ceramics Studio", 549, "Furniture", 667, [], 3,
                                                      update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (top middle)": LMLocationData("Ceramics Studio", 550, "Furniture", 668, [], 3,
                                                        update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (top right)": LMLocationData("Ceramics Studio", 551, "Furniture", 669, [], 3,
                                                       update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Sealed Room L Plate": LMLocationData("Sealed Room", 561, "Furniture", 462, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room R Plate": LMLocationData("Sealed Room", 562, "Furniture", 461, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room L Trophy": LMLocationData("Sealed Room", 567, "Furniture", 456, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room R Rear Trophy": LMLocationData("Sealed Room", 568, "Furniture", 457, [], 2,
                                                update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room R Middle Trophy": LMLocationData("Sealed Room", 569, "Furniture", 458, [], 2,
                                                  update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room R Front Trophy": LMLocationData("Sealed Room", 570, "Furniture", 455, [], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Guest Room Vanity": LMLocationData("Guest Room", 579, "Furniture", 421, ["Water Element Medal"], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=29)], require_poltergust=False),
    "Cold Storage Gas Burner": LMLocationData("Cold Storage", 590, "Furniture", 58, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=64)], require_poltergust=False),
    "Artist's Studio Art Supply Shelves": LMLocationData("Artist's Studio", 599, "Furniture", 685, [], 3,
                                                         update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Artist's Studio Biff Bust": LMLocationData("Artist's Studio", 600, "Furniture", 686, [], 3,
                                                update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Artist's Studio Brick Pile": LMLocationData("Artist's Studio", 601, "Furniture", 687, [], 3,
                                                 update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Artist's Studio Art Piece Shelves": LMLocationData("Artist's Studio", 602, "Furniture", 688, [], 3,
                                                        update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Secret Altar Brazier (near front, closest to 4th Wall)": LMLocationData("Secret Altar", 610, "Furniture", 28, [],
        update_ram_addr=[LMRamData(in_game_room_id=73)], require_poltergust=False),
    "Secret Altar Brazier (near front, away from 4th Wall)": LMLocationData("Secret Altar", 611, "Furniture", 29, [],
        update_ram_addr=[LMRamData(in_game_room_id=73)], require_poltergust=False),
    "Armory Armor Suit 6": LMLocationData("Armory", 522, "Furniture", 647, [],  3 ,remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory Armor Suit 7": LMLocationData("Armory", 523, "Furniture", 648, [],  3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory Armor Suit 8": LMLocationData("Armory", 524, "Furniture", 649, [],  3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    "Armory Armor Suit 9": LMLocationData("Armory", 525, "Furniture", 650, [],  3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=51)], require_poltergust=False),
    #        LMLocationData('Ceramics Studio', 'Ceramics Studio Vase 9', 551),
    #        LMLocationData('Ceramics Studio', 'Ceramics Studio Vase 10', 553),
    #        LMLocationData('Ceramics Studio', 'Ceramics Studio Vase 11', 554),
    #        LMLocationData('Ceramics Studio', 'Ceramics Studio Vase 12', 555),
    #        LMLocationData('Ceramics Studio', 'Ceramics Studio Vase 13', 556),
    "Artist's Studio Gold Ghost Easel": LMLocationData("Artist's Studio", 603, "Furniture", 691, [], 3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Artist's Studio Pink Ghost Easel": LMLocationData("Artist's Studio", 604, "Furniture", 692, [], 3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Artist's Studio Blue Ghost Easel": LMLocationData("Artist's Studio", 605, "Furniture", 693, [], 3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Artist's Studio Red Ghost Easel": LMLocationData("Artist's Studio", 606, "Furniture", 694, [], 3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Artist's Studio Shy Guy Ghost Easel": LMLocationData("Artist's Studio", 607, "Furniture", 695, [], 3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Artist's Studio Green Ghost Easel": LMLocationData("Artist's Studio", 608, "Furniture", 696, [], 3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Artist's Studio Purple Ghost Easel": LMLocationData("Artist's Studio", 609, "Furniture", 697, [], 3, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=60)]),
    "Safari Room C Deer Head": LMLocationData("Safari Room", 439, "Furniture", 630, [], 3, remote_only=True,
                                                update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Kitchen Refrigerator": LMLocationData("Kitchen", 308, "Furniture", 78, [], 1, remote_only=True,
                                                update_ram_addr=[LMRamData(in_game_room_id=7)], require_poltergust=False),
    "Graveyard Tombstone 3": LMLocationData("Graveyard", 326, "Furniture", 177, [], 1, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Rec Room Front Bicycle": LMLocationData("Rec Room", 338, "Furniture", 107, [], 1, remote_only=True,
                                             update_ram_addr=[LMRamData(in_game_room_id=23)], require_poltergust=False),
    "2F Washroom Toilet": LMLocationData("2F Washroom", 378, "Furniture", 443, [], 2, remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=45)], require_poltergust=False),
}

CANDLES_LOCATION_TABLE = {
    "Mirror Room West Candlestick": LMLocationData("Mirror Room", 236, "Furniture", 306, [], 1,
                                                   update_ram_addr=[LMRamData(in_game_room_id=4)], require_poltergust=False),
    "Mirror Room East Candlestick": LMLocationData("Mirror Room", 237, "Furniture", 307, [], 1,
                                                   update_ram_addr=[LMRamData(in_game_room_id=4)], require_poltergust=False),
    "Fortune Teller's Room West Candlestick": LMLocationData("Fortune-Teller's Room", 228, "Furniture", 119, [], 1,
                                                             update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Fortune Teller's Room East Candlestick": LMLocationData("Fortune-Teller's Room", 229, "Furniture", 293, [], 1,
                                                             update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Parlor Table Candle": LMLocationData("Parlor", 108, "Furniture", 556, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Parlor West Three Candles": LMLocationData("Parlor", 109, "Furniture", 511, [], 2,
                                                update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Parlor East Three Candles": LMLocationData("Parlor", 110, "Furniture", 512, [], 2,
                                                update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Foyer R Candlestick": LMLocationData("Foyer", 714, "Furniture", 99, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=2)], require_poltergust=False),
    "Foyer L Candlestick": LMLocationData("Foyer", 99, "Furniture", 100, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=2)]),
    "Hidden Room R Candlestick": LMLocationData("Hidden Room", 265, "Furniture", 238, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room L Candlestick": LMLocationData("Hidden Room", 266, "Furniture", 239, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Dining Room R Candles and Plates": LMLocationData("Dining Room", 305, "Furniture", 287, [], 1,
                                                       update_ram_addr=[LMRamData(in_game_room_id=8)]),
    "Dining Room L Candles": LMLocationData("Dining Room", 306, "Furniture", 288, [], 1,
                                            update_ram_addr=[LMRamData(in_game_room_id=8)]),
    "Tea Room L Candlestick": LMLocationData("Tea Room", 367, "Furniture", 546, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "Astral Hall North Candlabra": LMLocationData("Astral Hall", 404, "Furniture", 447, [], 2,
                                                  update_ram_addr=[LMRamData(in_game_room_id=43)], require_poltergust=False),
    "Astral Hall East Candlabra": LMLocationData("Astral Hall", 405, "Furniture", 448, [], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=43)], require_poltergust=False),
    "Astral Hall West Candlabra": LMLocationData("Astral Hall", 406, "Furniture", 449, [], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=43)], require_poltergust=False),
    "Astral Hall SW Candlabra": LMLocationData("Astral Hall", 407, "Furniture", 450, [], 2,
                                               update_ram_addr=[LMRamData(in_game_room_id=43)], require_poltergust=False),
    "Astral Hall SE Candlabra": LMLocationData("Astral Hall", 408, "Furniture", 451, [], 2,
                                               update_ram_addr=[LMRamData(in_game_room_id=43)], require_poltergust=False),
    "Cellar Back Candle": LMLocationData("Cellar", 473, "Furniture", 10, [],
                                         update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Left Candle": LMLocationData("Cellar", 474, "Furniture", 11, [],
                                         update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Roof Campfire": LMLocationData("Roof", 850, "Furniture", 729, [], 4,
                                    update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Sealed Room L Mantle Candles": LMLocationData("Sealed Room", 563, "Furniture", 459, [], 2,
                                                   update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room R Mantle Candles": LMLocationData("Sealed Room", 564, "Furniture", 460, [], 2,
                                                   update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room L Table Candles": LMLocationData("Sealed Room", 565, "Furniture", 465, [], 2,
                                                  update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room R Table Candles": LMLocationData("Sealed Room", 566, "Furniture", 466, [], 2,
                                                  update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Secret Altar L Candles": LMLocationData("Secret Altar", 615, "Furniture", 36, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=73)]),
    "Secret Altar R Candles": LMLocationData("Secret Altar", 616, "Furniture", 37, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=73)]),
}

HANGABLES_LOCATION_TABLE = {
    "Parlor Painting L (closer to 4th Wall)": LMLocationData("Parlor", 112, "Furniture", 344, [], 2,
                                                             update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Parlor Painting L (near China Cabinet)": LMLocationData("Parlor", 113, "Furniture", 343, [], 2,
                                                             update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Parlor Center Painting": LMLocationData("Parlor", 114, "Furniture", 347, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Parlor Painting R (near Anteroom Door)": LMLocationData("Parlor", 115, "Furniture", 346, [], 2,
                                                             update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Parlor Painting R (closer to 4th Wall)": LMLocationData("Parlor", 116, "Furniture", 345, [], 2,
                                                             update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Wardrobe Purple Dress": LMLocationData("Wardrobe", 131, "Furniture", 452, [], 2,
                                            update_ram_addr=[LMRamData(in_game_room_id=41)]),
    "Wardrobe Red Dress": LMLocationData("Wardrobe", 132, "Furniture", 505, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=41)]),
    "Wardrobe Blue Shirt": LMLocationData("Wardrobe", 133, "Furniture", 506, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=41)]),
    "Wardrobe Brown Suit": LMLocationData("Wardrobe", 134, "Furniture", 507, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=41)]),
    "Wardrobe Black Shirt": LMLocationData("Wardrobe", 135, "Furniture", 508, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=41)]),
    "Wardrobe Green Jacket": LMLocationData("Wardrobe", 136, "Furniture", 509, [], 2,
                                            update_ram_addr=[LMRamData(in_game_room_id=41)]),
    "Wardrobe Yellow Dress": LMLocationData("Wardrobe", 137, "Furniture", 510, [], 2,
                                            update_ram_addr=[LMRamData(in_game_room_id=41)]),
    "2F Washroom Mirror": LMLocationData("2F Washroom", 377, "Furniture", 544, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=45)]),
    "1F Washroom Mirror": LMLocationData("1F Washroom", 201, "Furniture", 234, [], 1,
                                         update_ram_addr=[LMRamData(in_game_room_id=16)]),
    "Study Painting L (closer to 4th Wall)": LMLocationData("Study", 147, "Furniture", 553, [], 2,
                                                            update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Painting L (near Bookshelves)": LMLocationData("Study", 148, "Furniture", 552, [], 2,
                                                          update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Painting R (near Bookshelves)": LMLocationData("Study", 149, "Furniture", 555, [], 2,
                                                          update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Painting R (closer to 4th Wall)": LMLocationData("Study", 150, "Furniture", 554, [], 2,
                                                            update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Master Bedroom Painting L": LMLocationData("Master Bedroom", 162, "Furniture", 361, [], 2,
                                                update_ram_addr=[LMRamData(in_game_room_id=34)]),
    "Master Bedroom Painting R (near Window)": LMLocationData("Master Bedroom", 163, "Furniture", 363, [], 2,
                                                              update_ram_addr=[LMRamData(in_game_room_id=34)]),
    "Master Bedroom Painting R (closer to 4th Wall)": LMLocationData("Master Bedroom", 164, "Furniture", 362, [], 2,
                                                                     update_ram_addr=[LMRamData(in_game_room_id=34)]),
    "Nursery Wall Picture": LMLocationData("Nursery", 174, "Furniture", 368, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=26)]),
    "Nursery Picture R (closer to Door)": LMLocationData("Nursery", 175, "Furniture", 369, [], 2,
                                                         update_ram_addr=[LMRamData(in_game_room_id=26)]),
    "Nursery Picture R (closer to 4th Wall)": LMLocationData("Nursery", 176, "Furniture", 370, [], 2,
                                                             update_ram_addr=[LMRamData(in_game_room_id=26)]),
    "Twins' Room Dartboard": LMLocationData("Twins' Room", 181, "Furniture", 424, [], 2,
                                            update_ram_addr=[LMRamData(in_game_room_id=27)]),
    "Twins' Room Picture L (farthest from Globe)": LMLocationData("Twins' Room", 187, "Furniture", 551, [], 2,
                                                                  update_ram_addr=[LMRamData(in_game_room_id=27)]),
    "Twins' Room Picture L (closest to Globe)": LMLocationData("Twins' Room", 188, "Furniture", 549, [], 2,
                                                               update_ram_addr=[LMRamData(in_game_room_id=27)]),
    "1F Hallway Painting": LMLocationData("1F Hallway", 704, "Furniture", 104, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=17)]),
    "1F Washroom Picture": LMLocationData("1F Washroom", 202, "Furniture", 235, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=16)]),
    "Ballroom Portrait": LMLocationData("Ballroom", 210, "Furniture", 172, [], 1 ,
                                        update_ram_addr=[LMRamData(in_game_room_id=9)]),
    "Mirror Room Picture": LMLocationData("Mirror Room", 240, "Furniture", 210, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=4)]),
    "Hidden Room Painting Middle C": LMLocationData("Hidden Room", 268, "Furniture", 248, [], 1,
                                                    update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room Painting R": LMLocationData("Hidden Room", 269, "Furniture", 249, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room Painting L": LMLocationData("Hidden Room", 270, "Furniture", 250, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room Painting Middle L": LMLocationData("Hidden Room", 271, "Furniture", 299, [], 1,
                                                    update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room Painting Middle R": LMLocationData("Hidden Room", 272, "Furniture", 300, [], 1,
                                                    update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Conservatory Picture L": LMLocationData("Conservatory", 273, "Furniture", 124, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=22)]),
    "Conservatory Picture C": LMLocationData("Conservatory", 274, "Furniture", 125, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=22)]),
    "Conservatory Picture R (hangs Higher)": LMLocationData("Conservatory", 275, "Furniture", 135, [], 1,
                                                            update_ram_addr=[LMRamData(in_game_room_id=22)]),
    "Conservatory Picture R (hangs Lower)": LMLocationData("Conservatory", 276, "Furniture", 136, [], 1,
                                                           update_ram_addr=[LMRamData(in_game_room_id=22)]),
    "Dining Room Picture": LMLocationData("Dining Room", 298, "Furniture", 315, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=8)]),
    "Graveyard Downspout": LMLocationData("Graveyard", 320, "Furniture", 186, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=15)]),
    "2F Rear Hallway Picture": LMLocationData("2F Rear Hallway", 368, "Furniture", 520, [], 2,
                                              update_ram_addr=[LMRamData(in_game_room_id=47)]),
    "2F Washroom Picture": LMLocationData("2F Washroom", 381, "Furniture", 444, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=45)]),
    "Nana's Room Painting": LMLocationData("Nana's Room", 400, "Furniture", 388, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=49)]),
    "Billiards Room L Portrait (near Chest)": LMLocationData("Billiards Room", 414, "Furniture", 153, [], 1,
                                                             update_ram_addr=[LMRamData(in_game_room_id=12)]),
    "Billiards Room L Portrait (near Dart Board)": LMLocationData("Billiards Room", 415, "Furniture", 155, [], 1,
                                                                  update_ram_addr=[LMRamData(in_game_room_id=12)]),
    "Billiards Room C Portrait": LMLocationData("Billiards Room", 416, "Furniture", 152, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=12)]),
    "Billiards Room R Portrait": LMLocationData("Billiards Room", 417, "Furniture", 154, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=12)]),
    "Billiards Room Dart Board": LMLocationData("Billiards Room", 418, "Furniture", 157, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=12)]),
    "Projection Room Projection Screen": LMLocationData("Projection Room", 431, "Furniture", 204, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=13)]),
    "Rec Room SW Crest": LMLocationData("Rec Room", 340, "Furniture", 112, [], 1,
                                        update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "Rec Room NW Crest": LMLocationData("Rec Room", 341, "Furniture", 113, [], 1,
                                        update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "Rec Room SE Crest": LMLocationData("Rec Room", 342, "Furniture", 114, [], 1,
                                        update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "Rec Room NE Crest": LMLocationData("Rec Room", 719, "Furniture", 115, [], 1,
                                        update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "Rec Room L Sandbag": LMLocationData("Rec Room", 344, "Furniture", 253, [], 1,
                                         update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "Rec Room R Sandbag": LMLocationData("Rec Room", 345, "Furniture", 252, [], 1,
                                         update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "Rec Room C Sandbag": LMLocationData("Rec Room", 346, "Furniture", 251, [], 1,
                                         update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "Armory Wall Shield": LMLocationData("Armory", 527, "Furniture", 637, [], 3,
                                         update_ram_addr=[LMRamData(in_game_room_id=51)]),
    "Safari Room L Deer Head": LMLocationData("Safari Room", 438, "Furniture", 635, [], 3,
                                              update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Safari Room R Deer Head": LMLocationData("Safari Room", 440, "Furniture", 634, [], 3,
                                              update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Sealed Room Mirror": LMLocationData("Sealed Room", 558, "Furniture", 557, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room Picture": LMLocationData("Sealed Room", 559, "Furniture", 526, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sitting Room Picture": LMLocationData("Sitting Room", 576, "Furniture", 411, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=28)]),
    "Guest Room Painting": LMLocationData("Guest Room", 581, "Furniture", 418, ["Water Element Medal"], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=29)]),
    "Basement Stairwell Wall Lamp": LMLocationData("Basement Stairwell", 595, "Furniture", 63, [],
                                                   update_ram_addr=[LMRamData(in_game_room_id=67)]),
}

SEATING_LOCATION_TABLE = {
    "Parlor Chair": LMLocationData("Parlor", 100, "Furniture", 338, [], 2,
                                   update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Parlor Couch": LMLocationData("Parlor", 102, "Furniture", 341, [], 2,
                                   update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Study Chair": LMLocationData("Study", 144, "Furniture", 350, [], 2, remote_only=True,
                                  update_ram_addr=[LMRamData(in_game_room_id=35)], require_poltergust=False),
    "Master Bedroom Stool": LMLocationData("Master Bedroom", 158, "Furniture", 376, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=34)], require_poltergust=False),
    "Nursery Chair L": LMLocationData("Nursery", 170, "Furniture", 365, [], 2,
                                      update_ram_addr=[LMRamData(in_game_room_id=26)], require_poltergust=False),
    "Nursery Chair R": LMLocationData("Nursery", 171, "Furniture", 366, [], 2,
                                      update_ram_addr=[LMRamData(in_game_room_id=26)], require_poltergust=False),
    "Twins' Room Chair L (closest to 4th Wall)": LMLocationData("Twins' Room", 189, "Furniture", 428, [], 2,
                                                                update_ram_addr=[LMRamData(in_game_room_id=27)], require_poltergust=False),
    "Twins' Room Chair L (farthest from Desk)": LMLocationData("Twins' Room", 190, "Furniture", 429, [], 2,
                                                               update_ram_addr=[LMRamData(in_game_room_id=27)], require_poltergust=False),
    "Ballroom Chair L (closest to Left Wall)": LMLocationData("Ballroom", 211, "Furniture", 168, [], 1,
                                                              update_ram_addr=[LMRamData(in_game_room_id=9)], require_poltergust=False),
    "Ballroom Chair L (furthest from Left Wall)": LMLocationData("Ballroom", 212, "Furniture", 169, [], 1,
                                                                 update_ram_addr=[LMRamData(in_game_room_id=9)], require_poltergust=False),
    "Ballroom Chair R (furthest from Right Wall)": LMLocationData("Ballroom", 213, "Furniture", 170, [], 1,
                                                                  update_ram_addr=[LMRamData(in_game_room_id=9)], require_poltergust=False),
    "Ballroom Chair R (closest from Right Wall)": LMLocationData("Ballroom", 214, "Furniture", 171, [], 1 ,
                                                                 update_ram_addr=[LMRamData(in_game_room_id=9)], require_poltergust=False),
    "Storage Room Chair Stack": LMLocationData("Storage Room", 217, "Furniture", 162, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=14)], require_poltergust=False),
    "Fortune Teller's Chair": LMLocationData("Fortune-Teller's Room", 230, "Furniture", 116, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Butler's Room Stool (candle on Top)": LMLocationData("Butler's Room", 262, "Furniture", 313, [], 1 ,
                                                          update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Butler's Room Stool (near Sewing Machine)": LMLocationData("Butler's Room", 250, "Furniture", 223, [], 1,
                                                                update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Butler's Room Stool (middle table, R)": LMLocationData("Butler's Room", 251, "Furniture", 224, [], 1,
                                                            update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Butler's Room Stool (middle table, C)": LMLocationData("Butler's Room", 252, "Furniture", 225, [], 1,
                                                            update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Butler's Room Stool (close to Left Wall)": LMLocationData("Butler's Room", 253, "Furniture", 226, [], 1,
                                                               update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Conservatory Harp Chair": LMLocationData("Conservatory", 277, "Furniture", 128, [], 1,
                                              update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Cello Chair": LMLocationData("Conservatory", 278, "Furniture", 129, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Conservatory Piano Bench": LMLocationData("Conservatory", 282, "Furniture", 120, [], 1, remote_only=True,
                                               update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Dining Room L Chair": LMLocationData("Dining Room", 292, "Furniture", 86, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=8)], require_poltergust=False),
    "Dining Room R Chair": LMLocationData("Dining Room", 293, "Furniture", 87, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=8)]),
    "Mr' Lugg's Stool (Dining Room)": LMLocationData("Dining Room", 852, "Furniture", 167, [], 1,
                                                     update_ram_addr=[LMRamData(in_game_room_id=8)], require_poltergust=False),
    "Tea Room Chair (left table, left side near Door)": LMLocationData("Tea Room", 353, "Furniture", 389, [], 2,
                                                                       update_ram_addr=[LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Tea Room Chair (left table, right side near Door)": LMLocationData("Tea Room", 354, "Furniture", 390, [], 2,
                                                                        update_ram_addr=[
                                                                            LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Tea Room Chair (right table, left side near Door)": LMLocationData("Tea Room", 355, "Furniture", 391, [], 2,
                                                                        update_ram_addr=[
                                                                            LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Tea Room Chair (right table, right side near Door)": LMLocationData("Tea Room", 356, "Furniture", 392, [], 2,
                                                                         update_ram_addr=[
                                                                             LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Tea Room Chair (right table, right side near Window)": LMLocationData("Tea Room", 357, "Furniture", 393, [], 2,
                                                                           update_ram_addr=[
                                                                               LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Tea Room Chair (right table, left side near Window)": LMLocationData("Tea Room", 358, "Furniture", 394, [], 2,
                                                                          update_ram_addr=[
                                                                              LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Tea Room Chair (left table, right side near Window)": LMLocationData("Tea Room", 359, "Furniture", 395, [], 2,
                                                                          update_ram_addr=[
                                                                              LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Tea Room Chair (left table, left side near Window)": LMLocationData("Tea Room", 360, "Furniture", 396, [], 2,
                                                                         update_ram_addr=[
                                                                             LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Nana's Room Loveseat": LMLocationData("Nana's Room", 391, "Furniture", 387, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=49)], require_poltergust=False),
    "Nana's Room Chair": LMLocationData("Nana's Room", 392, "Furniture", 386, [], 2, remote_only=True,
                                        update_ram_addr=[LMRamData(in_game_room_id=49)], require_poltergust=False),
    "Observatory Chair": LMLocationData("Observatory", 718, "Furniture", 470, [], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=44)], require_poltergust=False),
    "Billiards Room C Chair": LMLocationData("Billiards Room", 411, "Furniture", 149, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=12)], require_poltergust=False),
    "Billiards Room L Chair": LMLocationData("Billiards Room", 412, "Furniture", 150, [], 1,
                                             update_ram_addr=[LMRamData(in_game_room_id=12)], require_poltergust=False),
    "Billiards Room R Chair": LMLocationData("Billiards Room", 413, "Furniture", 151, [], 1, remote_only=True,
                                             update_ram_addr=[LMRamData(in_game_room_id=12)], require_poltergust=False),
    "Safari Room Chair": LMLocationData("Safari Room", 432, "Furniture", 627, [], 3,
                                        update_ram_addr=[LMRamData(in_game_room_id=55)], require_poltergust=False),
    "Balcony West Bench": LMLocationData("Balcony", 441, "Furniture", 617, [], 3,
                                         update_ram_addr=[LMRamData(in_game_room_id=62)], require_poltergust=False),
    "Balcony East Bench": LMLocationData("Balcony", 442, "Furniture", 618, [], 3,
                                         update_ram_addr=[LMRamData(in_game_room_id=62)], require_poltergust=False),
    "Telephone Room Couch": LMLocationData("Telephone Room", 478, "Furniture", 674, [], 3,
                                           update_ram_addr=[LMRamData(in_game_room_id=53)], require_poltergust=False),
    "Clockwork Room L Chair": LMLocationData("Clockwork Room", 490, "Furniture", 600, [], 3,
                                             update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Clockwork Room R Chair": LMLocationData("Clockwork Room", 491, "Furniture", 597, [], 3,
                                             update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Clockwork Room C Chair": LMLocationData("Clockwork Room", 492, "Furniture", 601, [], 3,
                                             update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Clockwork Room R Stool": LMLocationData("Clockwork Room", 493, "Furniture", 598, [], 3,
                                             update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Clockwork Room L Stool": LMLocationData("Clockwork Room", 494, "Furniture", 599, [], 3,
                                             update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Ceramics Studio Chair": LMLocationData("Ceramics Studio", 532, "Furniture", 659, [], 3,
                                            update_ram_addr=[LMRamData(in_game_room_id=58)], require_poltergust=False),
    "Sitting Room Couch": LMLocationData("Sitting Room", 573, "Furniture", 415, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=28)], require_poltergust=False),
    "Sitting Room Chair": LMLocationData("Sitting Room", 574, "Furniture", 414, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=28)], require_poltergust=False),
    "Guest Room Chair": LMLocationData("Guest Room", 577, "Furniture", 422, ["Water Element Medal"], 2,
                                       update_ram_addr=[LMRamData(in_game_room_id=29)]),
    "Artist's Studio Stools": LMLocationData("Artist's Studio", 598, "Furniture", 689, [], 3,
                                             update_ram_addr=[LMRamData(in_game_room_id=60)], require_poltergust=False),
    "Storage Room Chair": LMLocationData("Storage Room", 209, "Furniture", 161, [], 1, remote_only=True,
                                         update_ram_addr=[LMRamData(in_game_room_id=14)], require_poltergust=False),
}

SURFACES_LOCATION_TABLE = {
    "Parlor Wine Table": LMLocationData("Parlor", 101, "Furniture", 339, [], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Parlor Table": LMLocationData("Parlor", 103, "Furniture", 340, [], 2,
                                   update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Study Desk": LMLocationData("Study", 143, "Furniture", 351, [], 2,
                                 update_ram_addr=[LMRamData(in_game_room_id=35)], require_poltergust=False),
    "Master Bedroom Bed": LMLocationData("Master Bedroom", 141, "Furniture", 360, [], 2 ,
                                         update_ram_addr=[LMRamData(in_game_room_id=34)], require_poltergust=False),
    "Nursery Table": LMLocationData("Nursery", 172, "Furniture", 371, [], 2,
                                    update_ram_addr=[LMRamData(in_game_room_id=26)], require_poltergust=False),
    "Twins' Room Desk": LMLocationData("Twins' Room", 186, "Furniture", 427, [], 2,
                                       update_ram_addr=[LMRamData(in_game_room_id=27)], require_poltergust=False),
    "Clairvoya's Crystal Ball (Fortune-Teller's Room)": LMLocationData("Fortune-Teller's Room", 231, "Furniture", 117,
                                                                       [], 1,
                                                                       update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Butler's Room Table": LMLocationData("Butler's Room", 249, "Furniture", 221, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Butler's Room Ironing Board": LMLocationData("Butler's Room", 260, "Furniture", 229, [], 1,
                                                  update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Butler's Room Sewing Machine": LMLocationData("Butler's Room", 261, "Furniture", 222, [], 1 ,
                                                   update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Melody's Piano (Conservatory)": LMLocationData("Conservatory", 281, "Furniture", 130, [], 1 ,
                                                    update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Dining Room Table": LMLocationData("Dining Room", 299, "Furniture", 105, [], 1,
                                        update_ram_addr=[LMRamData(in_game_room_id=8)], require_poltergust=False),
    "Kitchen Table": LMLocationData("Kitchen", 312, "Furniture", 83, [], 1,
                                    update_ram_addr=[LMRamData(in_game_room_id=7)]),
    "Nana's Room Sewing Machine": LMLocationData("Nana's Room", 389, "Furniture", 384, [], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=49)], require_poltergust=False),
    "Nana's Room Coffee Table": LMLocationData("Nana's Room", 393, "Furniture", 383, [], 2,
                                               update_ram_addr=[LMRamData(in_game_room_id=49)], require_poltergust=False),
    "Slim's Pool Table (Billiards Room)": LMLocationData("Billiards Room", 409, "Furniture", 147, [], 1,
                                                         update_ram_addr=[LMRamData(in_game_room_id=12)], require_poltergust=False),
    "Billiards Room Chess Table": LMLocationData("Billiards Room", 410, "Furniture", 148, [], 1,
                                                 update_ram_addr=[LMRamData(in_game_room_id=12)], require_poltergust=False),
    "Projection Room Projector Table": LMLocationData("Projection Room", 426, "Furniture", 203, [], 1,
                                                      update_ram_addr=[LMRamData(in_game_room_id=13)]),
    "Telephone Room Table": LMLocationData("Telephone Room", 477, "Furniture", 673, [], 3,
                                           update_ram_addr=[LMRamData(in_game_room_id=53)], require_poltergust=False),
    "Clockwork Room Bee-Hat Clocks": LMLocationData("Clockwork Room", 495, "Furniture", 656, [], 3,
                                                    update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Clockwork Room Woman-Windmill Clocks": LMLocationData("Clockwork Room", 496, "Furniture", 622,  [], 3,
                                                           update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Clockwork Room Cuckoo Clock": LMLocationData("Clockwork Room", 497, "Furniture", 624, [], 3,
                                                  update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Clockwork Room Table with Head": LMLocationData("Clockwork Room", 498, "Furniture", 623, [], 3,
                                                     update_ram_addr=[LMRamData(in_game_room_id=59)], require_poltergust=False),
    "Sealed Room Table": LMLocationData("Sealed Room", 560, "Furniture", 464, [], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=37)], require_poltergust=False),
    "Sitting Room Table": LMLocationData("Sitting Room", 575, "Furniture", 412, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=28)], require_poltergust=False),
    "Sue Pea's Bed (Guest Room)": LMLocationData("Guest Room", 578, "Furniture", 420, ["Water Element Medal"], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=29)]),
    "Tea Room West Table": LMLocationData("Tea Room", 862, "Furniture", 538, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "Tea Room East Table": LMLocationData("Tea Room", 863, "Furniture", 539, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "Twins' Room Bunk Bed": LMLocationData("Twins' Room", 182, "Furniture", 425, [], 2, remote_only=True,
                                           update_ram_addr=[LMRamData(in_game_room_id=27)], require_poltergust=False),
    "Breaker Room Table": LMLocationData("Breaker Room", 445, "Furniture", 44, [], remote_only=True,
                                         update_ram_addr=[LMRamData(in_game_room_id=69)]),
}

STORAGE_LOCATION_TABLE = {
    "Study Hat Rack": LMLocationData("Study", 139, "Furniture", 348, [], 2,
                                     update_ram_addr=[LMRamData(in_game_room_id=35)], require_poltergust=False),
    "Wardrobe Shoe Rack": LMLocationData("Wardrobe", 124, "Furniture", 453, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=41)], require_poltergust=False),
    "Wardrobe Hat Stand": LMLocationData("Wardrobe", 125, "Furniture", 495, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=41)], require_poltergust=False),
    "Study Bookshelf (far left)": LMLocationData("Study", 151, "Furniture", 354, [], 2,
                                                 update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Bookshelf (second from left)": LMLocationData("Study", 152, "Furniture", 355, [], 2,
                                                         update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Bookshelf (third from left)": LMLocationData("Study", 153, "Furniture", 356, [], 2,
                                                        update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Bookshelf (third from right)": LMLocationData("Study", 154, "Furniture", 357, [], 2,
                                                         update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Bookshelf (second from right)": LMLocationData("Study", 155, "Furniture", 358, [], 2,
                                                          update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Bookshelf (far right)": LMLocationData("Study", 156, "Furniture", 359, [], 2,
                                                  update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "1F Washroom Water Tank (above Toilet)": LMLocationData("1F Washroom", 206, "Furniture", 92, [], 1,
        update_ram_addr=[LMRamData(in_game_room_id=16)], remote_only=True),
    "1F Bathroom Shelf": LMLocationData("1F Bathroom", 199, "Furniture", 304, [], 1,
                                        update_ram_addr=[LMRamData(in_game_room_id=21)]),
    "Storage Room Bucket": LMLocationData("Storage Room", 215, "Furniture", 165, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=14)], require_poltergust=False),
    "Storage Room Crate (bottom row, L)": LMLocationData("Storage Room", 218, "Furniture", 259, [], 1,
                                                         update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "Storage Room Crate (bottom row, back L)": LMLocationData("Storage Room", 219, "Furniture", 260, [], 1,
                                                              update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "Storage Room Crate (bottom row, back R)": LMLocationData("Storage Room", 220, "Furniture", 261, [], 1,
                                                              update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "Storage Room Crate (bottom row, C)": LMLocationData("Storage Room", 221, "Furniture", 262, [], 1,
                                                         update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "Storage Room Crate (bottom row, R)": LMLocationData("Storage Room", 222, "Furniture", 263, [], 1,
                                                         update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "Storage Room Crate (middle row, R)": LMLocationData("Storage Room", 223, "Furniture", 264, [], 1,
                                                         update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "Storage Room Crate (middle row, L)": LMLocationData("Storage Room", 224, "Furniture", 265, [], 1,
                                                         update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "Storage Room Crate (top row)": LMLocationData("Storage Room", 225, "Furniture", 266, [], 1,
                                                   update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "Laundry Room Bucket": LMLocationData("Laundry Room", 242, "Furniture", 193, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=5)]),
    "Laundry Room West Soap Shelf": LMLocationData("Laundry Room", 247, "Furniture", 189, [], 1,
                                                   update_ram_addr=[LMRamData(in_game_room_id=5)]),
    "Laundry Room East Soap Shelf": LMLocationData("Laundry Room", 248, "Furniture", 190, [], 1,
                                                   update_ram_addr=[LMRamData(in_game_room_id=5)]),
    "Laundry Room Laundry Basket": LMLocationData("Laundry Room", 244, "Furniture", 191, [], 1,
                                                  update_ram_addr=[LMRamData(in_game_room_id=5)], require_poltergust=False),
    "Butler's Room Mop Bucket": LMLocationData("Butler's Room", 254, "Furniture", 218, [], 1,
                                               update_ram_addr=[LMRamData(in_game_room_id=0)]),
    "Butler's Room Bucket": LMLocationData("Butler's Room", 258, "Furniture", 227, [], 1,
                                           update_ram_addr=[LMRamData(in_game_room_id=0)]),
    "Kitchen L Food Shelf": LMLocationData("Kitchen", 310, "Furniture", 76, [], 1,
                                           update_ram_addr=[LMRamData(in_game_room_id=7)]),
    "Kitchen R Food Shelf": LMLocationData("Kitchen", 311, "Furniture", 77, [], 1,
                                           update_ram_addr=[LMRamData(in_game_room_id=7)]),
    "Kitchen Utensils Rack": LMLocationData("Kitchen", 313, "Furniture", 81, [], 1,
                                            update_ram_addr=[LMRamData(in_game_room_id=7)]),
    "Tea Room Ice Bucket": LMLocationData("Tea Room", 350, "Furniture", 550, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "2F Washroom Shelf": LMLocationData("2F Washroom", 379, "Furniture", 438, [], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=45)]),
    "Safari Room Top Boxes": LMLocationData("Safari Room", 434, "Furniture", 626, [], 3,
                                            update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Safari Room Bottom Boxes": LMLocationData("Safari Room", 435, "Furniture", 625, [], 3,
                                               update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Breaker Room Metal Drum L": LMLocationData("Breaker Room", 447, "Furniture", 45, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=69)], require_poltergust=False),
    "Breaker Room Metal Drum R": LMLocationData("Breaker Room", 448, "Furniture", 46, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=69)], require_poltergust=False),
    "Breaker Room Barrel (bottom row, closest to back Wall)": LMLocationData("Breaker Room", 449, "Furniture", 47, [],
                                                                             update_ram_addr=[
                                                                                 LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (bottom row, second from back Wall)": LMLocationData("Breaker Room", 450, "Furniture", 48, [],
                                                                              update_ram_addr=[
                                                                                  LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (bottom row, third from back Wall)": LMLocationData("Breaker Room", 451, "Furniture", 49, [],
                                                                             update_ram_addr=[
                                                                                 LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (bottom row, third from 4th Wall)": LMLocationData("Breaker Room", 452, "Furniture", 50, [],
                                                                            update_ram_addr=[
                                                                                LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (bottom row, second from 4th Wall)": LMLocationData("Breaker Room", 453, "Furniture", 51, [],
                                                                             update_ram_addr=[
                                                                                 LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (bottom row, closest to 4th Wall)": LMLocationData("Breaker Room", 454, "Furniture", 52, [],
                                                                            update_ram_addr=[
                                                                                LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (top row, closest to 4th Wall)": LMLocationData("Breaker Room", 455, "Furniture", 53, [],
                                                                         update_ram_addr=[
                                                                             LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (top row, second from 4th Wall)": LMLocationData("Breaker Room", 456, "Furniture", 54, [],
                                                                          update_ram_addr=[
                                                                              LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (top row, middle)": LMLocationData("Breaker Room", 457, "Furniture", 55, [],
                                                            update_ram_addr=[LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (top row, second from back Wall)": LMLocationData("Breaker Room", 458, "Furniture", 56, [],
                                                                           update_ram_addr=[
                                                                               LMRamData(in_game_room_id=69)]),
    "Breaker Room Barrel (top row, closest to back Wall)": LMLocationData("Breaker Room", 459, "Furniture", 57, [],
                                                                          update_ram_addr=[
                                                                              LMRamData(in_game_room_id=69)]),
    "Cellar Crate (along back wall)": LMLocationData("Cellar", 461, "Furniture", 12, [],
                                                     update_ram_addr=[LMRamData(in_game_room_id=66)], require_poltergust=False),
    "Cellar Center Shelf Boxes": LMLocationData("Cellar", 465, "Furniture", 19, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Center High Shelf": LMLocationData("Cellar", 466, "Furniture", 13, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Center Middle Shelf": LMLocationData("Cellar", 467, "Furniture", 15, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Center Low Shelf": LMLocationData("Cellar", 468, "Furniture", 17, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Right Shelf Boxes": LMLocationData("Cellar", 469, "Furniture", 20, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Right High Shelf": LMLocationData("Cellar", 470, "Furniture", 14, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Right Middle Shelf": LMLocationData("Cellar", 471, "Furniture", 16, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Right Low Shelf": LMLocationData("Cellar", 472, "Furniture", 18, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar C Shelf": LMLocationData("Cellar", 475, "Furniture", 7, [],
                                     update_ram_addr=[LMRamData(in_game_room_id=66)], require_poltergust=False),
    "Cellar R Shelf": LMLocationData("Cellar", 476, "Furniture", 8, [],
                                     update_ram_addr=[LMRamData(in_game_room_id=66)], require_poltergust=False),
    "Telephone Room R Shelf": LMLocationData("Telephone Room", 483, "Furniture", 677, [], 3 ,
                                             update_ram_addr=[LMRamData(in_game_room_id=53)]),
    "Telephone Room L Shelf": LMLocationData("Telephone Room", 484, "Furniture", 676, [], 3,
                                             update_ram_addr=[LMRamData(in_game_room_id=53)]),
    "Telephone Room C Shelf": LMLocationData("Telephone Room", 485, "Furniture", 675, [], 3,
                                             update_ram_addr=[LMRamData(in_game_room_id=53)]),
    "Armory Crates (near Ceramics Door)": LMLocationData("Armory", 528, "Furniture", 638, [], 3,
                                                         update_ram_addr=[LMRamData(in_game_room_id=51)]),
    "Armory Crates (front left corner)": LMLocationData("Armory", 529, "Furniture", 639, [], 3,
                                                        update_ram_addr=[LMRamData(in_game_room_id=51)]),
    "Armory Crates (middle left)": LMLocationData("Armory", 530, "Furniture", 640, [], 3,
                                                  update_ram_addr=[LMRamData(in_game_room_id=51)]),
    "Armory Crates (back left corner)": LMLocationData("Armory", 531, "Furniture", 641, [], 3,
                                                       update_ram_addr=[LMRamData(in_game_room_id=51)]),
    "Ceramics Studio Boxes (top, back Wall)": LMLocationData("Ceramics Studio", 536, "Furniture", 664, [], 3,
                                                             update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Boxes (bottom, back Wall)": LMLocationData("Ceramics Studio", 537, "Furniture", 665, [], 3,
                                                                update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Boxes (left corner near 4th Wall)": LMLocationData("Ceramics Studio", 538, "Furniture", 670, [], 3,
        update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Sitting Room Hat Stand": LMLocationData("Sitting Room", 571, "Furniture", 413, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=28)], require_poltergust=False),
    "Cold Storage Barrel L": LMLocationData("Cold Storage", 584, "Furniture", 24, [],
                                            update_ram_addr=[LMRamData(in_game_room_id=64)], require_poltergust=False),
    "Cold Storage Barrel R": LMLocationData("Cold Storage", 585, "Furniture", 27, [],
                                            update_ram_addr=[LMRamData(in_game_room_id=64)], require_poltergust=False),
    "Cold Storage Crate (closest to Sir Weston)": LMLocationData("Cold Storage", 586, "Furniture", 2, [],
                                                                 update_ram_addr=[LMRamData(in_game_room_id=64)], require_poltergust=False),
    "Cold Storage Crate (closest to 4th Wall)": LMLocationData("Cold Storage", 587, "Furniture", 3, [],
                                                               update_ram_addr=[LMRamData(in_game_room_id=64)], require_poltergust=False),
    "Cold Storage Rear Shelf": LMLocationData("Cold Storage", 588, "Furniture", 0, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=64)]),
    "Cold Storage Front Shelf": LMLocationData("Cold Storage", 589, "Furniture", 1, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=64)]),
    "Weston's Camp Gear (Cold Storage)": LMLocationData("Cold Storage", 591, "Furniture", 74, [],
                                                        update_ram_addr=[LMRamData(in_game_room_id=64)]),
    "Pipe Room Ice Barrel": LMLocationData("Pipe Room", 592, "Furniture", 65, [],
                                           update_ram_addr=[LMRamData(in_game_room_id=68)], require_poltergust=False),
    "Pipe Room Crate": LMLocationData("Pipe Room", 593, "Furniture", 21, [],
                                      update_ram_addr=[LMRamData(in_game_room_id=68)], require_poltergust=False),
    "Pipe Room Bucket": LMLocationData("Pipe Room", 594, "Furniture", 22, [],
                                       update_ram_addr=[LMRamData(in_game_room_id=68)]),
    "Cellar NW Crate": LMLocationData("Cellar", 462, "Furniture", 67, [], remote_only=True,
                                      update_ram_addr=[LMRamData(in_game_room_id=66)], require_poltergust=False),
    "Cellar Barrel (back wall)": LMLocationData("Cellar", 463, "Furniture", 5, [], remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=66)], require_poltergust=False),
    "Cellar Barrel (by exit)": LMLocationData("Cellar", 464, "Furniture", 6, [], remote_only=True,
                                                    update_ram_addr=[LMRamData(in_game_room_id=66)], require_poltergust=False),
}

DRAWERS_LOCATION_TABLE = {
    "Foyer Dresser": LMLocationData("Foyer", 93, "Furniture", 207, [], 1,
                                    update_ram_addr=[LMRamData(in_game_room_id=2)], require_poltergust=False),
    "Parlor China Cabinet": LMLocationData("Parlor", 105, "Furniture", 337, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Parlor Book Hutch": LMLocationData("Parlor", 106, "Furniture", 494, [], 2,
                                        update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Parlor End Table": LMLocationData("Parlor", 111, "Furniture", 489, [], 2,
                                       update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Sitting Room Water Pot": LMLocationData("Sitting Room", 572, "Furniture", 410, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=28)], require_poltergust=False),
    "Anteroom R End Table": LMLocationData("Anteroom", 117, "Furniture", 486, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=42)], require_poltergust=False),
    "Anteroom C End Table": LMLocationData("Anteroom", 118, "Furniture", 514, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=42)], require_poltergust=False),
    "Anteroom L End Table": LMLocationData("Anteroom", 119, "Furniture", 513, [], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=42)], require_poltergust=False),
    "Wardrobe Center Dresser": LMLocationData("Wardrobe", 128, "Furniture", 478, [], 2,
                                              update_ram_addr=[LMRamData(in_game_room_id=41)], require_poltergust=False),
    "Master Bedroom Dresser": LMLocationData("Master Bedroom", 157, "Furniture", 491, [], 2,
                                             update_ram_addr=[LMRamData(in_game_room_id=34)], require_poltergust=False),
    "Master Bedroom End Table": LMLocationData("Master Bedroom", 160, "Furniture", 492, [], 2,
                                               update_ram_addr=[LMRamData(in_game_room_id=34)], require_poltergust=False),
    "Nursery Heart Dresser": LMLocationData("Nursery", 169, "Furniture", 367, [], 2,
                                            update_ram_addr=[LMRamData(in_game_room_id=26)], require_poltergust=False),
    "Twins' Room Dresser": LMLocationData("Twins' Room", 192, "Furniture", 515, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=27)], require_poltergust=False),
    "1F Washroom Cabinet": LMLocationData("1F Washroom", 204, "Furniture", 232, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=16)]),
    "Fortune Teller's Room West Dresser": LMLocationData("Fortune-Teller's Room", 226, "Furniture", 196, [], 1,
                                                         update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Fortune Teller's Room East Dresser": LMLocationData("Fortune-Teller's Room", 227, "Furniture", 209, [], 1,
                                                         update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Mirror Room Drawer": LMLocationData("Mirror Room", 234, "Furniture", 286, [], 1,
                                         update_ram_addr=[LMRamData(in_game_room_id=4)], require_poltergust=False),
    "Laundry Room Cabinet": LMLocationData("Laundry Room", 245, "Furniture", 188, [], 1,
                                           update_ram_addr=[LMRamData(in_game_room_id=5)], require_poltergust=False),
    "Butler's Room Cabinet": LMLocationData("Butler's Room", 259, "Furniture", 217, [], 1,
                                            update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Conservatory Cabinet": LMLocationData("Conservatory", 291, "Furniture", 197, [], 1,
                                           update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Dining Room R China Hutch": LMLocationData("Dining Room", 295, "Furniture", 279, [], 1,
                                                update_ram_addr=[LMRamData(in_game_room_id=8)], require_poltergust=False),
    "Kitchen China Hutch": LMLocationData("Kitchen", 309, "Furniture", 79, [], 1,
                                          update_ram_addr=[LMRamData(in_game_room_id=7)], require_poltergust=False),
    "Rec Room Drawer": LMLocationData("Rec Room", 337, "Furniture", 198, [], 1,
                                      update_ram_addr=[LMRamData(in_game_room_id=23)], require_poltergust=False),
    "Tea Room L Drawers": LMLocationData("Tea Room", 363, "Furniture", 516, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Tea Room R Drawers": LMLocationData("Tea Room", 364, "Furniture", 517, [], 2,
                                         update_ram_addr=[LMRamData(in_game_room_id=50)], require_poltergust=False),
    "Nana's Room Dresser": LMLocationData("Nana's Room", 390, "Furniture", 482, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=49)], require_poltergust=False),
    "Astral Hall Right Dresser": LMLocationData("Astral Hall", 394, "Furniture", 484, [], 2,
                                                update_ram_addr=[LMRamData(in_game_room_id=43)], require_poltergust=False),
    "Astral Hall Left Dresser": LMLocationData("Astral Hall", 402, "Furniture", 483, [], 2,
                                               update_ram_addr=[LMRamData(in_game_room_id=43)], require_poltergust=False),
    "Observatory Dresser": LMLocationData("Observatory", 717, "Furniture", 469, [], 2,
                                          update_ram_addr=[LMRamData(in_game_room_id=44)], require_poltergust=False),
    "Billiards Room End Table (near Chest)": LMLocationData("Billiards Room", 419, "Furniture", 290, [], 1,
                                                            update_ram_addr=[LMRamData(in_game_room_id=12)], require_poltergust=False),
    "Billiards Room End Table (near Dart Board)": LMLocationData("Billiards Room", 420, "Furniture", 289, [], 1,
                                                                 update_ram_addr=[LMRamData(in_game_room_id=12)], require_poltergust=False),
    "Projection Room Cabinet": LMLocationData("Projection Room", 424, "Furniture", 205, [], 1,
                                              update_ram_addr=[LMRamData(in_game_room_id=13)], require_poltergust=False),
    "Telephone Room L Cabinet": LMLocationData("Telephone Room", 481, "Furniture", 678, [], 3,
                                               update_ram_addr=[LMRamData(in_game_room_id=53)], require_poltergust=False),
    "Telephone Room R Cabinet": LMLocationData("Telephone Room", 482, "Furniture", 679, [], 3,
                                               update_ram_addr=[LMRamData(in_game_room_id=53)], require_poltergust=False),
    "Guest Room End Table": LMLocationData("Guest Room", 580, "Furniture", 497, ["Water Element Medal"], 2,
                                           update_ram_addr=[LMRamData(in_game_room_id=29)]),
    "Wardrobe West Dresser": LMLocationData("Wardrobe", 127, "Furniture", 479, [], 2, remote_only=True,
                                                update_ram_addr=[LMRamData(in_game_room_id=41)], require_poltergust=False),
    "Wardrobe Right Dresser": LMLocationData("Wardrobe", 871, "Furniture", 477, [], 2, remote_only=True,
                                             update_ram_addr=[LMRamData(in_game_room_id=41)], require_poltergust=False),
    "Dining Room L China Hutch": LMLocationData("Dining Room", 294, "Furniture", 278, [], 1, remote_only=True,
                                                update_ram_addr=[LMRamData(in_game_room_id=8)], require_poltergust=False),
}

# Adds the myriad of shakable objects or objects that react to elements such as fire as locations
FURNITURE_LOCATION_TABLE: dict[str, LMLocationData] = {
    **PLANT_LOCATION_TABLE,
    **CEILING_LOCATION_TABLE,
    **HANGABLES_LOCATION_TABLE,
    **CANDLES_LOCATION_TABLE,
    **SEATING_LOCATION_TABLE,
    **SURFACES_LOCATION_TABLE,
    **STORAGE_LOCATION_TABLE,
    **DRAWERS_LOCATION_TABLE,
    **DECOR_LOCATION_TABLE
}

BASEMENT_LOCS = dict((name, loc_data) for (name, loc_data) in FURNITURE_LOCATION_TABLE.items() if loc_data.floor == 0)

FIRST_FLOOR_LOCS = dict((name, loc_data) for (name, loc_data) in FURNITURE_LOCATION_TABLE.items() if loc_data.floor == 1)

SECOND_FLOOR_LOCS = dict((name, loc_data) for (name, loc_data) in FURNITURE_LOCATION_TABLE.items() if loc_data.floor == 2)

ATTIC_LOCS = dict((name, loc_data) for (name, loc_data) in FURNITURE_LOCATION_TABLE.items() if loc_data.floor == 3)

ROOF_LOCS = dict((name, loc_data) for (name, loc_data) in FURNITURE_LOCATION_TABLE.items() if loc_data.floor == 4)


TREASURES_LOCATION_TABLE = {
    "Heart Door Lamp": LMLocationData("Foyer", 94, "Furniture", 270, [],
        update_ram_addr=[LMRamData(in_game_room_id=2)]),
    "Foyer Chandelier": LMLocationData("Foyer", 95, "Furniture", 101, [],
        update_ram_addr=[LMRamData(in_game_room_id=2)]),
    "Parlor Book Hutch": LMLocationData("Parlor", 106, "Furniture", 494, [],
        update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Parlor Chandelier": LMLocationData("Parlor", 104, "Furniture", 342, [],
        update_ram_addr=[LMRamData(in_game_room_id=36)]),
    "Parlor China Cabinet": LMLocationData("Parlor", 105, "Furniture", 337, [],
        update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Parlor End Table": LMLocationData("Parlor", 111, "Furniture", 489, [],
        update_ram_addr=[LMRamData(in_game_room_id=36)], require_poltergust=False),
    "Anteroom L End Table": LMLocationData("Anteroom", 119, "Furniture", 513, [],
        update_ram_addr=[LMRamData(in_game_room_id=42)], require_poltergust=False),
    "Anteroom R Chandelier": LMLocationData("Anteroom", 120, "Furniture", 485, [],
        update_ram_addr=[LMRamData(in_game_room_id=42)]),
    "Anteroom L Chandelier": LMLocationData("Anteroom", 121, "Furniture", 467, [],
        update_ram_addr=[LMRamData(in_game_room_id=42)]),
    "Anteroom C End Table": LMLocationData("Anteroom", 118, "Furniture", 514, [],
        update_ram_addr=[LMRamData(in_game_room_id=42)], require_poltergust=False),
    "Wardrobe Center Dresser": LMLocationData("Wardrobe", 128, "Furniture", 478, [],
        update_ram_addr=[LMRamData(in_game_room_id=41)], require_poltergust=False),
    "Study Chandelier": LMLocationData("Study", 140, "Furniture", 352, [],
        update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Book (on Desk closer to Door)": LMLocationData("Study", 145, "Furniture", 503, [],
        update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Study Book (on Desk closer to Bookshelves)": LMLocationData("Study", 146, "Furniture", 504, [],
        update_ram_addr=[LMRamData(in_game_room_id=35)]),
    "Master Bedroom End Table": LMLocationData("Master Bedroom", 160, "Furniture", 492, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=34)], require_poltergust=False),
    "Master Bedroom Ceiling Fan": LMLocationData("Master Bedroom", 161, "Furniture", 373, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=34)]),
    "Master Bedroom Dresser": LMLocationData("Master Bedroom", 157, "Furniture", 491, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=34)], require_poltergust=False),
    "Nursery Ceiling Lamp": LMLocationData("Nursery", 177, "Furniture", 374, [],
                                           update_ram_addr=[LMRamData(in_game_room_id=26)]),
    "1F Hallway Front Vase L": LMLocationData("1F Hallway", 705, "Furniture", 93, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=6)]),
    "1F Hallway Front Vase R": LMLocationData("1F Hallway", 193, "Furniture", 95, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=6)]),
    "1F Hallway Rear Vase L": LMLocationData("1F Hallway", 194, "Furniture", 102, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=17)]),
    "1F Hallway Rear Vase R": LMLocationData("1F Hallway", 196, "Furniture", 283, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=17)]),
    "Ballroom West Chandelier": LMLocationData("Ballroom", 207, "Furniture", 173, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=9)]),
    "Ballroom East Chandelier": LMLocationData("Ballroom", 208, "Furniture", 174, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=9)]),
    "Storage Room Bucket": LMLocationData("Storage Room", 215, "Furniture", 165, [],
                                          update_ram_addr=[LMRamData(in_game_room_id=14)]),
    "1F Washroom Cabinet": LMLocationData("1F Washroom", 204, "Furniture", 232, [],
                                          update_ram_addr=[LMRamData(in_game_room_id=16)]),
    "1F Washroom Light": LMLocationData("1F Washroom", 205, "Furniture", 91, [],
                                        update_ram_addr=[LMRamData(in_game_room_id=16)]),
    "Basement Stairwell Hanging Light": LMLocationData("Basement Stairwell", 596, "Furniture", 62, [],
                                                       update_ram_addr=[LMRamData(in_game_room_id=67)]),
    "Breaker Room Ceiling Light": LMLocationData("Breaker Room", 446, "Furniture", 43, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=69)]),
    "Fortune Teller's Room West Dresser": LMLocationData("Fortune-Teller's Room", 226, "Furniture", 196, [],
                                                         update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Fortune Teller's Room East Dresser": LMLocationData("Fortune-Teller's Room", 227, "Furniture", 209, [],
                                                         update_ram_addr=[LMRamData(in_game_room_id=3)], require_poltergust=False),
    "Laundry Room Bucket": LMLocationData("Laundry Room", 242, "Furniture", 193, [],
                                          update_ram_addr=[LMRamData(in_game_room_id=5)]),
    "Laundry Room Ceiling Light": LMLocationData("Laundry Room", 246, "Furniture", 195, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=5)]),
    "Laundry Room Cabinet": LMLocationData("Laundry Room", 245, "Furniture", 188, [],
                                           update_ram_addr=[LMRamData(in_game_room_id=5)], require_poltergust=False),
    "Butler's Room Bucket": LMLocationData("Butler's Room", 258, "Furniture", 227, [],
                                           update_ram_addr=[LMRamData(in_game_room_id=0)]),
    "Butler's Room Ceiling Light": LMLocationData("Butler's Room", 257, "Furniture", 228, [],
                                                  update_ram_addr=[LMRamData(in_game_room_id=0)]),
    "Butler's Room Cabinet": LMLocationData("Butler's Room", 259, "Furniture", 217, [],
                                            update_ram_addr=[LMRamData(in_game_room_id=0)], require_poltergust=False),
    "Hidden Room R Trophy": LMLocationData("Hidden Room", 263, "Furniture", 236, [],
                                           update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room L Trophy": LMLocationData("Hidden Room", 264, "Furniture", 237, [],
                                           update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Hidden Room Chandelier": LMLocationData("Hidden Room", 267, "Furniture", 240, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=1)]),
    "Conservatory Ceiling Light": LMLocationData("Conservatory", 283, "Furniture", 137, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=22)]),
    "Conservatory Cabinet": LMLocationData("Conservatory", 291, "Furniture", 197, [],
                                           update_ram_addr=[LMRamData(in_game_room_id=22)], require_poltergust=False),
    "Dining Room R China Hutch": LMLocationData("Dining Room", 295, "Furniture", 279, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=8)], require_poltergust=False),
    "Kitchen Oven": LMLocationData("Kitchen", 307, "Furniture", 80, ["Fire Element Medal"],
                                   update_ram_addr=[LMRamData(in_game_room_id=7)]),
    "Kitchen China Hutch": LMLocationData("Kitchen", 309, "Furniture", 79, [],
                                          update_ram_addr=[LMRamData(in_game_room_id=7)], require_poltergust=False),
    "Kitchen R Ceiling Light": LMLocationData("Kitchen", 316, "Furniture", 85, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=7)]),
    "Graveyard Downspout": LMLocationData("Graveyard", 320, "Furniture", 186, [],
                                          update_ram_addr=[LMRamData(in_game_room_id=15)], require_poltergust=False),
    "Courtyard Female Statue": LMLocationData("Courtyard", 332, "Furniture", 273, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=24)], require_poltergust=False),
    "Rec Room Drawer": LMLocationData("Rec Room", 337, "Furniture", 198, [],
                                      update_ram_addr=[LMRamData(in_game_room_id=23)], require_poltergust=False),
    "Rec Room L Ceiling Light": LMLocationData("Rec Room", 347, "Furniture", 111, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=23)]),
    "2F Stairwell Ceiling Light": LMLocationData("2F Stairwell", 349, "Furniture", 310, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=18)]),
    "Tea Room Chandelier": LMLocationData("Tea Room", 365, "Furniture", 399, [],
                                          update_ram_addr=[LMRamData(in_game_room_id=50)]),
    "Tea Room R Drawers": LMLocationData("Tea Room", 364, "Furniture", 517, [],
                                         update_ram_addr=[LMRamData(in_game_room_id=50)], require_poltergust=False),
    "2F Rear Hallway Vase R": LMLocationData("2F Rear Hallway", 371, "Furniture", 519, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=32)]),
    "2F Rear Hallway Vase 3": LMLocationData("2F Rear Hallway", 372, "Furniture", 521, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=46)]),
    "2F Rear Hallway Vase 5": LMLocationData("2F Rear Hallway", 374, "Furniture", 523, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=46)]),
    "2F Rear Hallway Vase 6": LMLocationData("2F Rear Hallway", 375, "Furniture", 524, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=46)]),
    "Nana's Room Dresser": LMLocationData("Nana's Room", 390, "Furniture", 482, [],
                                          update_ram_addr=[LMRamData(in_game_room_id=49)], require_poltergust=False),
    "Nana's Room Chandelier": LMLocationData("Nana's Room", 395, "Furniture", 378, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=49)]),
    "2F Washroom Light": LMLocationData("2F Washroom", 383, "Furniture", 442, [],
                                        update_ram_addr=[LMRamData(in_game_room_id=45)]),
    "Astral Hall Left Dresser": LMLocationData("Astral Hall", 402, "Furniture", 483, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=43)], require_poltergust=False),
    "Astral Hall Chandelier": LMLocationData("Astral Hall", 403, "Furniture", 446, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=43)]),
    "Observatory Dresser": LMLocationData("Observatory", 717, "Furniture", 469, [],
                                          update_ram_addr=[LMRamData(in_game_room_id=44)], require_poltergust=False),
    "Billiards Room End Table (near Chest)": LMLocationData("Billiards Room", 419, "Furniture", 290, [],
                                                            update_ram_addr=[LMRamData(in_game_room_id=12)], require_poltergust=False),
    "Billiards Room End Table (near Dart Board)": LMLocationData("Billiards Room", 420, "Furniture", 289, [],
                                                                 update_ram_addr=[LMRamData(in_game_room_id=12)], require_poltergust=False),
    "Billiards Room Ceiling Fan": LMLocationData("Billiards Room", 421, "Furniture", 156, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=12)]),
    "Projection Room R Ceiling Light": LMLocationData("Projection Room", 428, "Furniture", 200, [],
                                                      update_ram_addr=[LMRamData(in_game_room_id=13)]),
    "Projection Room Cabinet": LMLocationData("Projection Room", 424, "Furniture", 205, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=13)]),
    "Twins' Room Star Light": LMLocationData("Twins' Room", 183, "Furniture", 423, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=27)]),
    "Twins' Room Dresser": LMLocationData("Twins' Room", 192, "Furniture", 515, [],
                                          update_ram_addr=[LMRamData(in_game_room_id=27)], require_poltergust=False),
    "Safari Room L Ceiling Light": LMLocationData("Safari Room", 436, "Furniture", 631, [],
                                                  update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Safari Room R Ceiling Light": LMLocationData("Safari Room", 437, "Furniture", 632, [],
                                                  update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Cellar Center Shelf Boxes": LMLocationData("Cellar", 465, "Furniture", 19, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Center High Shelf": LMLocationData("Cellar", 466, "Furniture", 13, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Center Middle Shelf": LMLocationData("Cellar", 467, "Furniture", 15, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Center Low Shelf": LMLocationData("Cellar", 468, "Furniture", 17, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Right Shelf Boxes": LMLocationData("Cellar", 469, "Furniture", 20, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Right High Shelf": LMLocationData("Cellar", 470, "Furniture", 14, [],
                                              update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Right Middle Shelf": LMLocationData("Cellar", 471, "Furniture", 16, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Cellar Right Low Shelf": LMLocationData("Cellar", 472, "Furniture", 18, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=66)]),
    "Telephone Room L Ceiling Light": LMLocationData("Telephone Room", 479, "Furniture", 671, [],
                                                     update_ram_addr=[LMRamData(in_game_room_id=53)]),
    "Telephone Room R Cabinet": LMLocationData("Telephone Room", 482, "Furniture", 679, [],
                                               update_ram_addr=[LMRamData(in_game_room_id=53)], require_poltergust=False),
    "Clockwork Room R Ceiling Light": LMLocationData("Clockwork Room", 486, "Furniture", 596, [],
                                                     update_ram_addr=[LMRamData(in_game_room_id=59)]),
    "Clockwork Room L Ceiling Light": LMLocationData("Clockwork Room", 487, "Furniture", 595, [],
                                                     update_ram_addr=[LMRamData(in_game_room_id=59)]),
    "Ceramics Studio Vase (bottom right)": LMLocationData("Ceramics Studio", 544, "Furniture", 660, [],
                                                          update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (bottom middle)": LMLocationData("Ceramics Studio", 545, "Furniture", 661, [],
                                                           update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (bottom left)": LMLocationData("Ceramics Studio", 546, "Furniture", 662, [],
                                                         update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (bottom more left)": LMLocationData("Ceramics Studio", 547, "Furniture", 663, [],
                                                              update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (top more left)": LMLocationData("Ceramics Studio", 548, "Furniture", 666, [],
                                                           update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (top left)": LMLocationData("Ceramics Studio", 549, "Furniture", 667, [],
                                                      update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (top middle)": LMLocationData("Ceramics Studio", 550, "Furniture", 668, [],
                                                        update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Ceramics Studio Vase (top right)": LMLocationData("Ceramics Studio", 551, "Furniture", 669, [],
                                                       update_ram_addr=[LMRamData(in_game_room_id=58)]),
    "Sealed Room Chandelier": LMLocationData("Sealed Room", 557, "Furniture", 463, [],
                                             update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room L Trophy": LMLocationData("Sealed Room", 567, "Furniture", 456, [],
                                           update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room R Rear Trophy": LMLocationData("Sealed Room", 568, "Furniture", 457, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room R Middle Trophy": LMLocationData("Sealed Room", 569, "Furniture", 458, [],
                                                  update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Sealed Room R Front Trophy": LMLocationData("Sealed Room", 570, "Furniture", 455, [],
                                                 update_ram_addr=[LMRamData(in_game_room_id=37)]),
    "Pipe Room Bucket": LMLocationData("Pipe Room", 594, "Furniture", 22, [],
                                       update_ram_addr=[LMRamData(in_game_room_id=68)]),
    "Guest Room End Table": LMLocationData("Guest Room", 580, "Furniture", 497, ["Water Element Medal"],
                                           update_ram_addr=[LMRamData(in_game_room_id=29)]),
    "Secret Altar Brazier (near front, closest to 4th Wall)": LMLocationData("Secret Altar", 610, "Furniture", 28, [],
                                                                             update_ram_addr=[
                                                                                 LMRamData(in_game_room_id=73)]),
    "Secret Altar Brazier (near front, away from 4th Wall)": LMLocationData("Secret Altar", 611, "Furniture", 29, [],
                                                                            update_ram_addr=[
                                                                                LMRamData(in_game_room_id=73)]),
    "Secret Altar C Chandelier": LMLocationData("Secret Altar", 613, "Furniture", 38, [],
                                                update_ram_addr=[LMRamData(in_game_room_id=73)]),
    "Wardrobe Balcony Plant R (near Door)": LMLocationData("Wardrobe Balcony", 65, "Plant", 473,
                                                           ["Water Element Medal"],
                                                           update_ram_addr=[LMRamData(in_game_room_id=40)]),
    "Wardrobe Balcony Plant L (next to Toad)": LMLocationData("Wardrobe Balcony", 66, "Plant", 474,
                                                              ["Water Element Medal"],
                                                              update_ram_addr=[LMRamData(in_game_room_id=40)]),
    "Wardrobe Balcony Plant C": LMLocationData("Wardrobe Balcony", 67, "Plant", 475, ["Water Element Medal"],
                                               update_ram_addr=[LMRamData(in_game_room_id=40)]),
    "Wardrobe Balcony Plant L (in Corner)": LMLocationData("Wardrobe Balcony", 68, "Plant", 476,
                                                           ["Water Element Medal"],
                                                           update_ram_addr=[LMRamData(in_game_room_id=40)]),
    "Master Bedroom Plant": LMLocationData("Master Bedroom", 69, "Plant", 542, ["Water Element Medal"],
                                           update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard SW Plant": LMLocationData("Courtyard", 71, "Plant", 142, ["Water Element Medal"],
                                         update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard NW Plant": LMLocationData("Courtyard", 72, "Plant", 143, ["Water Element Medal"],
                                         update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard NE Plant": LMLocationData("Courtyard", 73, "Plant", 144, ["Water Element Medal"],
                                         update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Balcony Plant NW Not Near Bench": LMLocationData("Balcony", 75, "Plant", 604, ["Water Element Medal"],
                                                      update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant E Railing": LMLocationData("Balcony", 76, "Plant", 605, ["Water Element Medal"],
                                              update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant NE Near Bench": LMLocationData("Balcony", 77, "Plant", 606, ["Water Element Medal"],
                                                  update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant SW Corner": LMLocationData("Balcony", 81, "Plant", 610, ["Water Element Medal"],
                                              update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant NW Corner": LMLocationData("Balcony", 82, "Plant", 611, ["Water Element Medal"],
                                              update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony West Double Plant": LMLocationData("Balcony", 83, "Plant", 612, ["Water Element Medal"],
                                                update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Center R": LMLocationData("Balcony", 84, "Plant", 613, ["Water Element Medal"],
                                             update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Directly in NE Corner": LMLocationData("Balcony", 872, "Plant", 608, ["Water Element Medal"],
                                      update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant SE Corner": LMLocationData("Balcony", 80, "Plant", 609, ["Water Element Medal"],
                                              update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Second from Right Plant near NW Bench": LMLocationData("Balcony", 873, "Plant", 615, ["Water Element Medal"],
                                       update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Furthest Right Plant near NW Bench": LMLocationData("Balcony", 874, "Plant", 614, ["Water Element Medal"],
                                       update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Second from Right in NE Corner": LMLocationData("Balcony", 875, "Plant", 616, ["Water Element Medal"],
                                       update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Center L": LMLocationData("Balcony", 88, "Plant", 712, ["Water Element Medal"],
                                             update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Balcony Plant Third from Right in NE Corner": LMLocationData("Balcony", 876, "Plant", 713, ["Water Element Medal"],
                                              update_ram_addr=[LMRamData(in_game_room_id=62)]),
    "Sitting Room Plant": LMLocationData("Sitting Room", 91, "Plant", 409, ["Water Element Medal"],
                                         update_ram_addr=[LMRamData(in_game_room_id=28)]),
    "Guest Room Plant": LMLocationData("Guest Room", 92, "Plant", 417, ["Water Element Medal"],
                                       update_ram_addr=[LMRamData(in_game_room_id=29)]),
}


# Map Visits, furniture items in maps, other stuff?
WDYM_LOCATION_TABLE ={
    "Roof Entry Hut": LMLocationData("Roof", 854, "Furniture", 716, [],
                                                              update_ram_addr=[LMRamData(in_game_room_id=63)], require_poltergust=False),
    "Graveyard West Tree": LMLocationData("Graveyard", 855, "Furniture", 184, ["Water Element Medal"],
                                                              update_ram_addr=[LMRamData(in_game_room_id=15)]),
    "Graveyard East Tree": LMLocationData("Graveyard", 856, "Furniture", 185, ["Water Element Medal"],
                                                              update_ram_addr=[LMRamData(in_game_room_id=15)]),
    "Courtyard Far West Tree": LMLocationData("Courtyard", 857, "Plant", 139, ["Water Element Medal"],
                                         update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard Tree by Fountain": LMLocationData("Courtyard", 858, "Plant", 140, ["Water Element Medal"],
                                         update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard Tree by Well": LMLocationData("Courtyard", 859, "Plant", 138, ["Water Element Medal"],
                                         update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard Tree by Woman Statue": LMLocationData("Courtyard", 860, "Plant", 141, ["Water Element Medal"],
                                         update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Courtyard Laundry Bar": LMLocationData("Courtyard", 861, "Furniture", 314, [],
                                                     update_ram_addr=[LMRamData(in_game_room_id=24)]),
    "Cellar Fence": LMLocationData("Cellar", 864, "Furniture", 9, [],
                                     update_ram_addr=[LMRamData(in_game_room_id=66)], require_poltergust=False),
    "Pipe Room Fence": LMLocationData("Pipe Room", 865, "Furniture", 23, [],
                                       update_ram_addr=[LMRamData(in_game_room_id=68)], require_poltergust=False),
    "Safari Room L Cheetah Carpet": LMLocationData("Safari Room", 866, "Furniture", 628, [],
                                                  update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Safari Room R Cheetah Carpet": LMLocationData("Safari Room", 867, "Furniture", 629, [],
                                                   update_ram_addr=[LMRamData(in_game_room_id=55)]),
    "Safari Room C Cheetah Carpet": LMLocationData("Safari Room", 868, "Furniture", 683, [],
                                                   update_ram_addr=[LMRamData(in_game_room_id=55)]),
}


# Adds Portrait Ghosts as locations
# Certain Ghosts such as Extra Clockwork soliders don't have a jmp entry to check against.
# Bosses do NOT have entries in map2, but we can check their bit addresses if they are captured.
PORTRAIT_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Neville, the Bookish Father": LMLocationData("Study", 621, "Portrait", 70, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DA9, bit_position=6, in_game_room_id=35)]),
    "Lydia, the Mirror-Gazing Mother": LMLocationData("Master Bedroom", 622, "Portrait", 71, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DA9, bit_position=7, in_game_room_id=34)]),
    "Chauncey, the Spoiled Baby": LMLocationData("Nursery", 623, "Portrait", 77, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DAC, bit_position=0, in_game_room_id=26)], map_id=[2,10]),
    "Henry and Orville, the Twin Brothers": LMLocationData("Twins' Room", 624, "Portrait", 79, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DAA, bit_position=1, in_game_room_id=27)]),
    "The Floating Whirlindas": LMLocationData("Ballroom", 625, "Portrait", 34, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DBE, bit_position=2, in_game_room_id=9)]),
    "Shivers, the Wandering Butler": LMLocationData("Butler's Room", 626, "Portrait", 29, ["Fire Element Medal", "Shivers Spawn"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DA5, bit_position=6, in_game_room_id=0)]),
    "Madame Clairvoya, the Freaky Fortune-Teller": LMLocationData("Fortune-Teller's Room", 627, "Portrait", 31, [],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DBB, bit_position=2, in_game_room_id=3)]),
    "Melody Pianissima, the Beautiful Pianist": LMLocationData("Conservatory", 628, "Portrait", 24, [],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DA8, bit_position=6, in_game_room_id=22)]),
    "Mr. Luggs, the Glutton": LMLocationData("Dining Room", 629, "Portrait", 27, ["Fire Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DA9, bit_position=4, in_game_room_id=8)]),
    "Spooky, the Guard Dog": LMLocationData("Boneyard", 630, "Portrait", 23, ["Water Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DB3, bit_position=0, in_game_room_id=11)]),
    "Bogmire, the Cemetary Shadow": LMLocationData("Graveyard", 631, "Portrait", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DBE, bit_position=5, in_game_room_id=15)], map_id=[2,13]),
    "Biff Atlas, the Bodybuilder": LMLocationData("Rec Room", 632, "Portrait", 32, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DAA, bit_position=6, in_game_room_id=23)]),
    "Slim Bankshot, the Lonely Poolshark": LMLocationData("Billiards Room", 633, "Portrait", 26, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DAA, bit_position=7, in_game_room_id=12)]),
    "Miss Petunia, the Bathing Beauty": LMLocationData("2F Bathroom", 634, "Portrait", 83, ["Ice Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DA9, bit_position=5, in_game_room_id=48)]),
    "Nana, the Scarf-Knitting Granny": LMLocationData("Nana's Room", 635, "Portrait", 76, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DAA, bit_position=0, in_game_room_id=49)]),
    "Sue Pea, the Dozing Girl": LMLocationData("Guest Room", 636, "Portrait", 78, ["Water Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DB9, bit_position=5, in_game_room_id=29)]),
    "Uncle Grimmly, Hermit of the Darkness": LMLocationData("Wardrobe", 637, "Portrait", 16, ["Blackout"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DBB, bit_position=3, in_game_room_id=41)]),
    "Boolossus, the Jumbo Ghost": LMLocationData("Balcony", 638, "Portrait", 56, ["Ice Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DBF, bit_position=4, in_game_room_id=62)], map_id=[2,11]),
    "Jarvis, the Jar Collector": LMLocationData("Ceramics Studio", 639, "Portrait", 0, ["Ice Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DA8, bit_position=5, in_game_room_id=58)]),
    "Clockwork Soldiers, the Toy Platoon": LMLocationData("Clockwork Room", 640, "Portrait", 115, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DAB, bit_position=1, in_game_room_id=59)]),
    "Vincent van Gore, the Starving Artist": LMLocationData("Artist's Studio", 641, "Portrait", 107, [],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5DAA, bit_position=5, in_game_room_id=60)]),
    "Sir Weston, the Chilly Climber": LMLocationData("Cold Storage", 642, "Portrait", 1, ["Fire Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DBF, bit_position=1, in_game_room_id=64)]),
}

# Adds Blue Speedy Spirits as Loacations
SPEEDY_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Wardrobe Speedy Spirit": LMLocationData("Wardrobe", 643, "BSpeedy", 128, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC3, bit_position=0, in_game_room_id=41)]),
    "Study Speedy Spirit": LMLocationData("Study", 644, "BSpeedy", 113, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC2, bit_position=2, in_game_room_id=35)]),
    "Nursery Speedy Spirit": LMLocationData("Nursery", 645, "BSpeedy", 148, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC3, bit_position=5, in_game_room_id=26)]),
    "Storage Room Speedy Spirit": LMLocationData("Storage Room", 646, "BSpeedy", 58, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC1, bit_position=7, in_game_room_id=14)]),
    "Hidden Room Speedy Spirit": LMLocationData("Hidden Room", 647, "BSpeedy", 72, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC3, bit_position=3, in_game_room_id=1)]),
    "Conservatory Speedy Spirit": LMLocationData("Conservatory", 648, "BSpeedy", 113, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC3, bit_position=4, in_game_room_id=22)]),
    "Dining Room Speedy Spirit": LMLocationData("Dining Room", 649, "BSpeedy", 60, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC2, bit_position=1, in_game_room_id=8)]),
    "Kitchen Speedy Spirit": LMLocationData("Kitchen", 650, "BSpeedy", 66, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC2, bit_position=5, in_game_room_id=7)]),
    "Rec Room Speedy Spirit": LMLocationData("Rec Room", 651, "BSpeedy", 67, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC2, bit_position=7, in_game_room_id=23)]),
    "Nana's Room Speedy Spirit": LMLocationData("Nana's Room", 652, "BSpeedy", 115, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC2, bit_position=4, in_game_room_id=49)]),
    "Billiards Room Speedy Spirit": LMLocationData("Billiards Room", 653, "BSpeedy", 59, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC2, bit_position=0, in_game_room_id=12)]),
    "Twins' Room Speedy Spirit": LMLocationData("Twins' Room", 655, "BSpeedy", 114, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC2, bit_position=3, in_game_room_id=27)]),
    "Breaker Room Speedy Spirit": LMLocationData("Breaker Room", 656, "BSpeedy", 7, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC3, bit_position=2, in_game_room_id=69)]),
    "Cellar Speedy Spirit": LMLocationData("Cellar", 657, "BSpeedy", 6, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC3, bit_position=1, in_game_room_id=66)]),
    "Sealed Room Speedy Spirit": LMLocationData("Sealed Room", 658, "BSpeedy", 125, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC2, bit_position=6, in_game_room_id=37)]),
}

GOLD_MICE_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Study Cheese Gold Mouse":        LMLocationData("Study", 659, "Mouse", 75, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC4, bit_position=4)]),
    "Fortune Teller Cheese Mouse":    LMLocationData("Fortune-Teller's Room", 660, "Mouse", 52, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC4, bit_position=3)]),
    "Dining Room Cheese Mouse":       LMLocationData("Dining Room", 661, "Mouse", 53, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DB1, bit_position=7)]),
    "Tea Room Cheese Mouse":          LMLocationData("Tea Room", 662, "Mouse", 72, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC4, bit_position=5)]),
    "Safari Room Cheese Mouse":       LMLocationData("Safari Room", 663, "Mouse", 82, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC4, bit_position=6)]),
    "1F Hallway Chance Mouse":        LMLocationData("1F Hallway", 664, "Mouse", 49, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DC4, bit_position=7)]),
    "Kitchen Chance Mouse":           LMLocationData("Kitchen", 665, "Mouse", 8, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DB1, bit_position=4)]),
    "Tea Room Chance Mouse":          LMLocationData("Tea Room", 666, "Mouse", 76, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DB1, bit_position=5)]),
    "2F Rear Hallway Chance Mouse":   LMLocationData("2F Rear Hallway", 667, "Mouse", 71, remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DB1, bit_position=3)]),
    "Sealed Room Chance Mouse":       LMLocationData("Sealed Room", 668, "Mouse", 73, remote_only=True,
        update_ram_addr=[LMRamData(0x803D5DB1, bit_position=6)])
}

# Individual Boo room Locations
ROOM_BOO_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Butler's Room Boo": LMLocationData("Butler's Room", 677, "Boo", 0, ["Fire Element Medal", "Shivers Spawn"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5E04, bit_position=0)]),
    "Hidden Room Boo": LMLocationData("Hidden Room", 678, "Boo", 1, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E04, bit_position=1)]),
    "Fortune-Teller's Room Boo": LMLocationData("Fortune-Teller's Room", 679, "Boo", 2, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E04, bit_position=2)]),
    "Mirror Room Boo": LMLocationData("Mirror Room", 680, "Boo", 3, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E04, bit_position=3)]),
    "Laundry Room Boo": LMLocationData("Laundry Room", 676, "Boo", 4, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E04, bit_position=4)]),
    "Kitchen Boo": LMLocationData("Kitchen", 684, "Boo", 5, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E04, bit_position=5)]),
    "Dining Room Boo": LMLocationData("Dining Room", 683, "Boo", 6, ["Fire Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E04, bit_position=6)]),
    "Ballroom Boo": LMLocationData("Ballroom", 681, "Boo", 7, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E04, bit_position=7)]),
    "Billiards Room Boo": LMLocationData("Billiards Room", 687, "Boo", 8, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E05, bit_position=0)]),
    "Projection Room Boo": LMLocationData("Projection Room", 688, "Boo", 9, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E05, bit_position=1)]),
    "Storage Room Boo": LMLocationData("Storage Room", 682, "Boo", 10, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E05, bit_position=2)]),
    "Conservatory Boo": LMLocationData("Conservatory", 685, "Boo", 11, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E05, bit_position=3)]),
    "Rec Room Boo": LMLocationData("Rec Room", 686, "Boo", 12, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E05, bit_position=4)]),
    "Nursery Boo": LMLocationData("Nursery", 674, "Boo", 13, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E05, bit_position=5)]),
    "Twins' Room Boo": LMLocationData("Twins' Room", 675, "Boo", 14, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E05, bit_position=6)]),
    "Sitting Room Boo": LMLocationData("Sitting Room", 691, "Boo", 15, ["Fire Element Medal", "Water Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803D5E05, bit_position=7)]),
    "Guest Room Boo": LMLocationData("Guest Room", 692, "Boo", 16, ["Water Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E06, bit_position=0)]),
    "Master Bedroom Boo": LMLocationData("Master Bedroom", 673, "Boo", 17, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E06, bit_position=1)]),
    "Study Boo": LMLocationData("Study", 672, "Boo", 18, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E06, bit_position=2)]),
    "Parlor Boo": LMLocationData("Parlor", 669, "Boo", 19, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E06, bit_position=3)]),
    "Wardrobe Boo": LMLocationData("Wardrobe", 671, "Boo", 20, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E06, bit_position=4)]),
    "Anteroom Boo": LMLocationData("Anteroom", 670, "Boo", 21, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E06, bit_position=5)]),
    "Astral Hall Boo": LMLocationData("Astral Hall", 699, "Boo", 22, ["Fire Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E06, bit_position=6)]),
    "Nana's Room Boo": LMLocationData("Nana's Room", 690, "Boo", 23, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E06, bit_position=7)]),
    "Tea Room Boo": LMLocationData("Tea Room", 689, "Boo", 24, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E07, bit_position=0)]),
    "Armory Boo": LMLocationData("Armory", 695, "Boo", 25, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E07, bit_position=1)]),
    "Telephone Room Boo": LMLocationData("Telephone Room", 697, "Boo", 26, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E07, bit_position=2)]),
    "Safari Room Boo": LMLocationData("Safari Room", 693, "Boo", 27, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E07, bit_position=3)]),
    "Ceramics Studio Boo": LMLocationData("Ceramics Studio", 696, "Boo", 28, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E07, bit_position=4)]),
    "Clockwork Room Boo": LMLocationData("Clockwork Room", 698, "Boo", 29, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E07, bit_position=5)]),
    "Artist's Studio Boo": LMLocationData("Artist's Studio", 694, "Boo", 30, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E07, bit_position=6)]),
    "Cold Storage Boo": LMLocationData("Cold Storage", 703, "Boo", 31, ["Fire Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E07, bit_position=7)]),
    "Cellar Boo": LMLocationData("Cellar", 701, "Boo", 32, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E08, bit_position=0)]),
    "Pipe Room Boo": LMLocationData("Pipe Room", 702, "Boo", 33, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E08, bit_position=1)]),
    "Breaker Room Boo": LMLocationData("Breaker Room", 700, "Boo", 34, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E08, bit_position=2)]),
}

# Tracking the individual boos of Boolossues
BOOLOSSUS_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Boolossus Fragment 1": LMLocationData("Balcony", 720, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E08, bit_position=3, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 2": LMLocationData("Balcony", 721, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E08, bit_position=4, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 3": LMLocationData("Balcony", 722, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E08, bit_position=5, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 4": LMLocationData("Balcony", 723, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E08, bit_position=6, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 5": LMLocationData("Balcony", 724, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E08, bit_position=7, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 6": LMLocationData("Balcony", 725, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E09, bit_position=0, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 7": LMLocationData("Balcony", 726, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E09, bit_position=1, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 8": LMLocationData("Balcony", 727, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E09, bit_position=2, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 9": LMLocationData("Balcony", 728, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E09, bit_position=3, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 10": LMLocationData("Balcony", 729, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E09, bit_position=4, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 11": LMLocationData("Balcony", 730, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E09, bit_position=5, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 12": LMLocationData("Balcony", 731, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E09, bit_position=6, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 13": LMLocationData("Balcony", 732, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E09, bit_position=7, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 14": LMLocationData("Balcony", 733, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E0A, bit_position=0, in_game_room_id=62)], map_id=[2,11]),
    "Boolossus Fragment 15": LMLocationData("Balcony", 734, "Boolossus Boo", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803D5E0A, bit_position=1, in_game_room_id=62)], map_id=[2,11]),
}

BOO_LOCATION_TABLE = {**ROOM_BOO_LOCATION_TABLE,
                      **BOOLOSSUS_LOCATION_TABLE}

LIGHT_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Butler's Room Light On": LMLocationData("Butler's Room", 743, "KingdomHearts", 0, ["Fire Element Medal", "Shivers Spawn"],
        remote_only=True, update_ram_addr=[LMRamData(0x803CDF50, bit_position=1, in_game_room_id=0, ram_byte_size=2)]),
    "Hidden Room Light On": LMLocationData("Hidden Room", 744, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF52, bit_position=1, in_game_room_id=1, ram_byte_size=2)]),
    "Foyer Light On": LMLocationData("Foyer", 771, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF54, bit_position=1, in_game_room_id=2, ram_byte_size=2),
                         LMRamData(0x803CDF8C, bit_position=1, in_game_room_id=31, ram_byte_size=2)]),
    "Fortune-Teller's Room Light On": LMLocationData("Fortune-Teller's Room", 745, "KingdomHearts", 0, [],
        remote_only=True, update_ram_addr=[LMRamData(0x803CDF56, bit_position=1, in_game_room_id=3, ram_byte_size=2)]),
    "Mirror Room Light On": LMLocationData("Mirror Room", 746, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF58, bit_position=1, in_game_room_id=4, ram_byte_size=2)]),
    "Laundry Room Light On": LMLocationData("Laundry Room", 742, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF5A, bit_position=1, in_game_room_id=5, ram_byte_size=2)]),
    "1F Hallway Light On": LMLocationData("1F Hallway", 773, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF5C, bit_position=1, in_game_room_id=6, ram_byte_size=2),
                         LMRamData(0x803CDF5E, bit_position=1, in_game_room_id=20, ram_byte_size=2),
                         LMRamData(0x803CDFBA, bit_position=1, in_game_room_id=10, ram_byte_size=2)]),
    "Kitchen Light On": LMLocationData("Kitchen", 750, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF60, bit_position=1, in_game_room_id=7, ram_byte_size=2)]),
    "Dining Room Light On": LMLocationData("Dining Room", 749, "KingdomHearts", 0, ["Fire Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803CDF62, bit_position=1, in_game_room_id=8, ram_byte_size=2)]),
    "Ballroom Light On": LMLocationData("Ballroom", 747, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF64, bit_position=1, in_game_room_id=9, ram_byte_size=2)]),
    "Boneyard Light On": LMLocationData("Boneyard", 779, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF66, bit_position=1, in_game_room_id=11, ram_byte_size=2)]),
    "Billiards Room Light On": LMLocationData("Billiards Room", 753, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF68, bit_position=1, in_game_room_id=12, ram_byte_size=2)]),
    "Projection Room Light On": LMLocationData("Projection Room", 754, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF6A, bit_position=1, in_game_room_id=13, ram_byte_size=2)]),
    "Storage Room Light On": LMLocationData("Storage Room", 748, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF6C, bit_position=1, in_game_room_id=14, ram_byte_size=2)]),
    "Graveyard Light On": LMLocationData("Graveyard", 780, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF70, bit_position=1, in_game_room_id=15, ram_byte_size=2)]),
    "1F Washroom Light On": LMLocationData("1F Washroom", 776, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF72, bit_position=1, in_game_room_id=16, ram_byte_size=2)]),
    "2F Stairwell Light On": LMLocationData("2F Stairwell", 778, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF76, bit_position=1, in_game_room_id=18, ram_byte_size=2)]),
    "1F Bathroom Light On": LMLocationData("1F Bathroom", 777, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF78, bit_position=1, in_game_room_id=21, ram_byte_size=2)]),
    "Conservatory Light On": LMLocationData("Conservatory", 751, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF7A, bit_position=1, in_game_room_id=22, ram_byte_size=2)]),
    "Rec Room Light On": LMLocationData("Rec Room", 752, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF7C, bit_position=1, in_game_room_id=23, ram_byte_size=2)]),
    "Courtyard Light On": LMLocationData("Courtyard", 781, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF7E, bit_position=1, in_game_room_id=24, ram_byte_size=2)]),
    "Nursery Light On": LMLocationData("Nursery", 740, "KingdomHearts", 13, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF80, bit_position=1, in_game_room_id=26, ram_byte_size=2)]),
    "Twins' Room Light On": LMLocationData("Twins' Room", 741, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF82, bit_position=1, in_game_room_id=27, ram_byte_size=2)]),
    "Sitting Room Light On": LMLocationData("Sitting Room", 757, "KingdomHearts", 0, ["Fire Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803CDF86, bit_position=1, in_game_room_id=28, ram_byte_size=2)]),
    "Guest Room Light On": LMLocationData("Guest Room", 758, "KingdomHearts", 0, ["Water Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803CDF88, bit_position=1, in_game_room_id=29, ram_byte_size=2)]),
    "Family Hallway Light On": LMLocationData("Family Hallway", 772, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF8A, bit_position=1, in_game_room_id=30, ram_byte_size=2)]),
    "Attic Stairwell Light On": LMLocationData("2F Rear Hallway", 782, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF76, bit_position=1, in_game_room_id=33, ram_byte_size=2),
                         LMRamData(0x803CDF76, bit_position=1, in_game_room_id=56, ram_byte_size=2)]),
    "Master Bedroom Light On": LMLocationData("Master Bedroom", 739, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF92, bit_position=1, in_game_room_id=34, ram_byte_size=2)]),
    "Study Light On": LMLocationData("Study", 738, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF94, bit_position=1, in_game_room_id=35, ram_byte_size=2)]),
    "Parlor Light On": LMLocationData("Parlor", 735, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF96, bit_position=1, in_game_room_id=36, ram_byte_size=2)]),
    "Sealed Room Light On": LMLocationData("Sealed Room", 788, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF98, bit_position=1, in_game_room_id=37, ram_byte_size=2)]),
    "Wardrobe Balcony Light On": LMLocationData("Wardrobe Balcony", 775, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF9A, bit_position=1, in_game_room_id=40, ram_byte_size=2)]),
    "Wardrobe Light On": LMLocationData("Wardrobe", 737, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF9C, bit_position=1, in_game_room_id=41, ram_byte_size=2)]),
    "Anteroom Light On": LMLocationData("Anteroom", 736, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF9E, bit_position=1, in_game_room_id=42, ram_byte_size=2)]),
    "Astral Hall Light On": LMLocationData("Astral Hall", 765, "KingdomHearts", 0, ["Fire Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803CDFA0, bit_position=1, in_game_room_id=43, ram_byte_size=2)]),
    "2F Washroom Light On": LMLocationData("2F Washroom", 785, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFA4, bit_position=1, in_game_room_id=45, ram_byte_size=2)]),
    "2F Rear Hallway Light On": LMLocationData("2F Rear Hallway", 784, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFA6, bit_position=1, in_game_room_id=46, ram_byte_size=2),
                         LMRamData(0x803CDFA8, bit_position=1, in_game_room_id=47, ram_byte_size=2),
                         LMRamData(0x803CDF84, bit_position=1, in_game_room_id=39, ram_byte_size=2),
                         LMRamData(0x803CDFD0, bit_position=1, in_game_room_id=38, ram_byte_size=2),
                         LMRamData(0x803CDF8E, bit_position=1, in_game_room_id=32, ram_byte_size=2),
                         LMRamData(0x803CDF90, bit_position=1, in_game_room_id=32, ram_byte_size=2)]),
    "2F Bathroom Light On": LMLocationData("2F Bathroom", 786, "KingdomHearts", 0, ["Ice Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803CDFAA, bit_position=1, in_game_room_id=48, ram_byte_size=2)]),
    "Nana's Room Light On": LMLocationData("Nana's Room", 756, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFAC, bit_position=1, in_game_room_id=49, ram_byte_size=2)]),
    "Tea Room Light On": LMLocationData("Tea Room", 755, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFAE, bit_position=1, in_game_room_id=50, ram_byte_size=2)]),
    "Armory Light On": LMLocationData("Armory", 761, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFB0, bit_position=1, in_game_room_id=51, ram_byte_size=2)]),
    "Telephone Room Light On": LMLocationData("Telephone Room", 763, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFB4, bit_position=1, in_game_room_id=53, ram_byte_size=2)]),
    "East Attic Hallway Light On": LMLocationData("East Attic Hallway", 789, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFB6, bit_position=1, in_game_room_id=54, ram_byte_size=2),
                         LMRamData(0x803CDFC4, bit_position=1, in_game_room_id=61, ram_byte_size=2)]),
    "Safari Room Light On": LMLocationData("Safari Room", 759, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFB8, bit_position=1, in_game_room_id=55, ram_byte_size=2)]),
    "West Attic Hallway Light On": LMLocationData("West Attic Hallway", 790, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFBC, bit_position=1, in_game_room_id=57, ram_byte_size=2),
                         LMRamData(0x803CDFB2, bit_position=1, in_game_room_id=52, ram_byte_size=2)]),
    "Ceramics Studio Light On": LMLocationData("Ceramics Studio", 762, "KingdomHearts", 0, ["Ice Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803CDFBE, bit_position=1, in_game_room_id=58, ram_byte_size=2)]),
    "Clockwork Room Light On": LMLocationData("Clockwork Room", 764, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFC0, bit_position=1, in_game_room_id=59, ram_byte_size=2)]),
    "Artist's Studio Light On": LMLocationData("Artist's Studio", 760, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFC2, bit_position=1, in_game_room_id=60, ram_byte_size=2)]),
    "Balcony Light On": LMLocationData("Balcony", 770, "KingdomHearts", 0, ["Ice Element Medal"], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFC6, bit_position=1, in_game_room_id=62, ram_byte_size=2)]),
    "Roof Light On": LMLocationData("Roof", 791, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFC8, bit_position=1, in_game_room_id=63, ram_byte_size=2)]),
    "Cold Storage Light On": LMLocationData("Cold Storage", 769, "KingdomHearts", 0, ["Fire Element Medal"],
        remote_only=True, update_ram_addr=[LMRamData(0x803CDFCA, bit_position=1, in_game_room_id=64, ram_byte_size=2)]),
    "Cellar Light On": LMLocationData("Cellar", 767, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFCE, bit_position=1, in_game_room_id=66, ram_byte_size=2)]),
    "Basement Stairwell Light On": LMLocationData("Basement Stairwell", 851, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFD2, bit_position=1, in_game_room_id=67, ram_byte_size=2)]),
    "Pipe Room Light On": LMLocationData("Pipe Room", 768, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFD4, bit_position=1, in_game_room_id=68, ram_byte_size=2)]),
    "Breaker Room Light On": LMLocationData("Breaker Room", 766, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFD6, bit_position=1, in_game_room_id=69, ram_byte_size=2)]),
    "Well Light On": LMLocationData("The Well", 774, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFDA, bit_position=1, in_game_room_id=72, ram_byte_size=2)]),
    "Observatory Light On": LMLocationData("Observatory", 929, "KingdomHearts", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFA2, bit_position=1, in_game_room_id=44, ram_byte_size=2)]),
}

WALK_LOCATION_TABLE: dict[str, LMLocationData] = {
    "Butler's Room Visited": LMLocationData("Butler's Room", 800, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF50, bit_position=0, in_game_room_id=0, ram_byte_size=2)], require_poltergust=False),
    "Hidden Room Visited": LMLocationData("Hidden Room", 801, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF52, bit_position=0, in_game_room_id=1, ram_byte_size=2)], require_poltergust=False),
    "Foyer Visited": LMLocationData("Foyer", 828, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF54, bit_position=0, in_game_room_id=2, ram_byte_size=2),
                         LMRamData(0x803CDF8C, bit_position=0, in_game_room_id=31, ram_byte_size=2)], require_poltergust=False),
    "Fortune-Teller's Room Visited": LMLocationData("Fortune-Teller's Room", 802, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF56, bit_position=0, in_game_room_id=3, ram_byte_size=2)], require_poltergust=False),
    "Mirror Room Visited": LMLocationData("Mirror Room", 803, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF58, bit_position=0, in_game_room_id=4, ram_byte_size=2)], require_poltergust=False),
    "Laundry Room Visited": LMLocationData("Laundry Room", 799, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF5A, bit_position=0, in_game_room_id=5, ram_byte_size=2)], require_poltergust=False),
    "1F Hallway Visited": LMLocationData("1F Hallway", 830, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF5C, bit_position=0, in_game_room_id=6, ram_byte_size=2),
                         LMRamData(0x803CDF5E, bit_position=0, in_game_room_id=20, ram_byte_size=2),
                         LMRamData(0x803CDFBA, bit_position=0, in_game_room_id=10, ram_byte_size=2),
                         LMRamData(0x803CDF74, bit_position=0, in_game_room_id=17, ram_byte_size=2)], require_poltergust=False),
    "Kitchen Visited": LMLocationData("Kitchen", 807, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF60, bit_position=0, in_game_room_id=7, ram_byte_size=2)], require_poltergust=False),
    "Dining Room Visited": LMLocationData("Dining Room", 806, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF62, bit_position=0, in_game_room_id=8, ram_byte_size=2)], require_poltergust=False),
    "Ballroom Visited": LMLocationData("Ballroom", 804, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF64, bit_position=0, in_game_room_id=9, ram_byte_size=2)], require_poltergust=False),
    "Boneyard Visited": LMLocationData("Boneyard", 836, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF66, bit_position=0, in_game_room_id=11, ram_byte_size=2)]),
    "Billiards Room Visited": LMLocationData("Billiards Room", 810, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF68, bit_position=0, in_game_room_id=12, ram_byte_size=2)], require_poltergust=False),
    "Projection Room Visited": LMLocationData("Projection Room", 811, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF6A, bit_position=0, in_game_room_id=13, ram_byte_size=2)], require_poltergust=False),
    "Storage Room Visited": LMLocationData("Storage Room", 805, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF6C, bit_position=0, in_game_room_id=14, ram_byte_size=2)], require_poltergust=False),
    "Graveyard Visited": LMLocationData("Graveyard", 837, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF70, bit_position=0, in_game_room_id=15, ram_byte_size=2)]),
    "1F Washroom Visited": LMLocationData("1F Washroom", 833, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF72, bit_position=0, in_game_room_id=16, ram_byte_size=2)], require_poltergust=False),
    "2F Stairwell Visited": LMLocationData("2F Stairwell", 839, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF76, bit_position=0, in_game_room_id=18, ram_byte_size=2)], require_poltergust=False),
    "1F Bathroom Visited": LMLocationData("1F Bathroom", 834, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF78, bit_position=0, in_game_room_id=21, ram_byte_size=2)], require_poltergust=False),
    "Conservatory Visited": LMLocationData("Conservatory", 808, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF7A, bit_position=0, in_game_room_id=22, ram_byte_size=2)], require_poltergust=False),
    "Rec Room Visited": LMLocationData("Rec Room", 809, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF7C, bit_position=0, in_game_room_id=23, ram_byte_size=2)], require_poltergust=False),
    "Courtyard Visited": LMLocationData("Courtyard", 838, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF7E, bit_position=0, in_game_room_id=24, ram_byte_size=2)], require_poltergust=False),
    "Nursery Visited": LMLocationData("Nursery", 797, "Walk", 13, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF80, bit_position=0, in_game_room_id=26, ram_byte_size=2)], require_poltergust=False),
    "Twins' Room Visited": LMLocationData("Twins' Room", 798, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF82, bit_position=0, in_game_room_id=27, ram_byte_size=2)], require_poltergust=False),
    "Sitting Room Visited": LMLocationData("Sitting Room", 814, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF86, bit_position=0, in_game_room_id=28, ram_byte_size=2)], require_poltergust=False),
    "Guest Room Visited": LMLocationData("Guest Room", 815, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF88, bit_position=0, in_game_room_id=29, ram_byte_size=2)], require_poltergust=False),
    "Family Hallway Visited": LMLocationData("Family Hallway", 829, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF8A, bit_position=0, in_game_room_id=30, ram_byte_size=2)], require_poltergust=False),
    "Attic Stairwell Visited": LMLocationData("2F Rear Hallway", 870, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF76, bit_position=0, in_game_room_id=33, ram_byte_size=2),
                        LMRamData(0x803CDFA6, bit_position=0, in_game_room_id=33, ram_byte_size=2),
                        LMRamData(0x803CDFA8, bit_position=0, in_game_room_id=33, ram_byte_size=2),
                        LMRamData(0x803CDF84, bit_position=0, in_game_room_id=33, ram_byte_size=2),
                        LMRamData(0x803CDFD0, bit_position=0, in_game_room_id=33, ram_byte_size=2),
                        LMRamData(0x803CDF8E, bit_position=0, in_game_room_id=33, ram_byte_size=2),
                        LMRamData(0x803CDF90, bit_position=0, in_game_room_id=33, ram_byte_size=2),
                        LMRamData(0x803CDF76, bit_position=0, in_game_room_id=56, ram_byte_size=2),
                        LMRamData(0x803CDFA6, bit_position=0, in_game_room_id=56, ram_byte_size=2),
                        LMRamData(0x803CDFA8, bit_position=0, in_game_room_id=56, ram_byte_size=2),
                        LMRamData(0x803CDF84, bit_position=0, in_game_room_id=56, ram_byte_size=2),
                        LMRamData(0x803CDFD0, bit_position=0, in_game_room_id=56, ram_byte_size=2),
                        LMRamData(0x803CDF8E, bit_position=0, in_game_room_id=56, ram_byte_size=2),
                        LMRamData(0x803CDF90, bit_position=0, in_game_room_id=56, ram_byte_size=2)], require_poltergust=False),
    "Master Bedroom Visited": LMLocationData("Master Bedroom", 796, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF92, bit_position=0, in_game_room_id=34, ram_byte_size=2)], require_poltergust=False),
    "Study Visited": LMLocationData("Study", 795, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF94, bit_position=0, in_game_room_id=35, ram_byte_size=2)], require_poltergust=False),
    "Parlor Visited": LMLocationData("Parlor", 792, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF96, bit_position=0, in_game_room_id=36, ram_byte_size=2)], require_poltergust=False),
    "Sealed Room Visited": LMLocationData("Sealed Room", 845, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF98, bit_position=0, in_game_room_id=37, ram_byte_size=2)], require_poltergust=False),
    "Wardrobe Balcony Visited": LMLocationData("Wardrobe Balcony", 832, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF9A, bit_position=0, in_game_room_id=40, ram_byte_size=2)], require_poltergust=False),
    "Wardrobe Visited": LMLocationData("Wardrobe", 794, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF9C, bit_position=0, in_game_room_id=41, ram_byte_size=2)], require_poltergust=False),
    "Anteroom Visited": LMLocationData("Anteroom", 793, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF9E, bit_position=0, in_game_room_id=42, ram_byte_size=2)], require_poltergust=False),
    "Astral Hall Visited": LMLocationData("Astral Hall", 822, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFA0, bit_position=0, in_game_room_id=43, ram_byte_size=2)], require_poltergust=False),
    "Observatory Visited": LMLocationData("Observatory", 844, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFA2, bit_position=0, in_game_room_id=44, ram_byte_size=2)]),
    "2F Washroom Visited": LMLocationData("2F Washroom", 842, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFA4, bit_position=0, in_game_room_id=45, ram_byte_size=2)], require_poltergust=False),
    "2F Rear Hallway Visited": LMLocationData("2F Rear Hallway", 841, "Walk", 0, [], remote_only=True,
      update_ram_addr=[LMRamData(0x803CDFA6, bit_position=0, in_game_room_id=46, ram_byte_size=2),
                       LMRamData(0x803CDFA8, bit_position=0, in_game_room_id=47, ram_byte_size=2),
                       LMRamData(0x803CDF84, bit_position=0, in_game_room_id=39, ram_byte_size=2),
                       LMRamData(0x803CDFD0, bit_position=0, in_game_room_id=38, ram_byte_size=2),
                       LMRamData(0x803CDF8E, bit_position=0, in_game_room_id=32, ram_byte_size=2),
                       LMRamData(0x803CDF90, bit_position=0, in_game_room_id=32, ram_byte_size=2)], require_poltergust=False),
    "2F Bathroom Visited": LMLocationData("2F Bathroom", 843, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFAA, bit_position=0, in_game_room_id=48, ram_byte_size=2)], require_poltergust=False),
    "Nana's Room Visited": LMLocationData("Nana's Room", 813, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFAC, bit_position=0, in_game_room_id=49, ram_byte_size=2)], require_poltergust=False),
    "Tea Room Visited": LMLocationData("Tea Room", 812, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFAE, bit_position=0, in_game_room_id=50, ram_byte_size=2)]),
    "Armory Visited": LMLocationData("Armory", 818, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFB0, bit_position=0, in_game_room_id=51, ram_byte_size=2)], require_poltergust=False),
    "Telephone Room Visited": LMLocationData("Telephone Room", 820, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFB4, bit_position=0, in_game_room_id=53, ram_byte_size=2)], require_poltergust=False),
    "East Attic Hallway Visited": LMLocationData("East Attic Hallway", 846, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFB6, bit_position=0, in_game_room_id=54, ram_byte_size=2),
                         LMRamData(0x803CDFC4, bit_position=0, in_game_room_id=61, ram_byte_size=2)], require_poltergust=False),
    "Safari Room Visited": LMLocationData("Safari Room", 816, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFB8, bit_position=0, in_game_room_id=55, ram_byte_size=2)], require_poltergust=False),
    "West Attic Hallway Visited": LMLocationData("West Attic Hallway", 847, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFBC, bit_position=0, in_game_room_id=57, ram_byte_size=2),
                         LMRamData(0x803CDFB2, bit_position=0, in_game_room_id=52, ram_byte_size=2)], require_poltergust=False),
    "Ceramics Studio Visited": LMLocationData("Ceramics Studio", 819, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFBE, bit_position=0, in_game_room_id=58, ram_byte_size=2)], require_poltergust=False),
    "Clockwork Room Visited": LMLocationData("Clockwork Room", 821, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFC0, bit_position=0, in_game_room_id=59, ram_byte_size=2)], require_poltergust=False),
    "Artist's Studio Visited": LMLocationData("Artist's Studio", 817, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFC2, bit_position=0, in_game_room_id=60, ram_byte_size=2)], require_poltergust=False),
    "Balcony Visited": LMLocationData("Balcony", 827, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFC6, bit_position=0, in_game_room_id=62, ram_byte_size=2)], require_poltergust=False),
    "Roof Visited": LMLocationData("Roof", 848, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFC8, bit_position=0, in_game_room_id=63, ram_byte_size=2)], require_poltergust=False),
    "Cold Storage Visited": LMLocationData("Cold Storage", 826, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFCA, bit_position=0, in_game_room_id=64, ram_byte_size=2)], require_poltergust=False),
    "Basement Hallway Visited": LMLocationData("Basement Hallway", 840, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFCC, bit_position=0, in_game_room_id=65, ram_byte_size=2)], require_poltergust=False),
    "Cellar Visited": LMLocationData("Cellar", 824, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFCE, bit_position=0, in_game_room_id=66, ram_byte_size=2)], require_poltergust=False),
    "Basement Stairwell Visited": LMLocationData("Basement Stairwell", 835, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFD2, bit_position=0, in_game_room_id=67, ram_byte_size=2)], require_poltergust=False),
    "Pipe Room Visited": LMLocationData("Pipe Room", 825, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFD4, bit_position=0, in_game_room_id=68, ram_byte_size=2)], require_poltergust=False),
    "Breaker Room Visited": LMLocationData("Breaker Room", 823, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFD6, bit_position=0, in_game_room_id=69, ram_byte_size=2)], require_poltergust=False),
    "Altar Hallway Visited": LMLocationData("Altar Hallway", 849, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFD8, bit_position=0, in_game_room_id=71, ram_byte_size=2)], require_poltergust=False),
    "Well Visited": LMLocationData("The Well", 831, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDFDA, bit_position=0, in_game_room_id=72, ram_byte_size=2)], require_poltergust=False),
}

MEME_LOCATION_TABLE = {
    "Touch Courtyard Grass": LMLocationData("Courtyard", 880, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF7E, bit_position=0, in_game_room_id=24, ram_byte_size=2)], require_poltergust=False),
    "Touch Boneyard Grass":  LMLocationData("Boneyard", 881, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF66, bit_position=0, in_game_room_id=11, ram_byte_size=2)]),
    "Touch Graveyard Grass": LMLocationData("Graveyard", 882, "Walk", 0, [], remote_only=True,
        update_ram_addr=[LMRamData(0x803CDF70, bit_position=0, in_game_room_id=15, ram_byte_size=2)]),
}

ALL_LOCATION_TABLE = {**BASE_LOCATION_TABLE,
                      **CLEAR_LOCATION_TABLE,
                      **ENEMIZER_LOCATION_TABLE,
                      **FURNITURE_LOCATION_TABLE,
                      **GOLD_MICE_LOCATION_TABLE,
                      **SPEEDY_LOCATION_TABLE,
                      **BOO_LOCATION_TABLE,
                      **TOAD_LOCATION_TABLE,
                      **PORTRAIT_LOCATION_TABLE,
                      **WALK_LOCATION_TABLE,
                      **LIGHT_LOCATION_TABLE,
                      **MEME_LOCATION_TABLE}



SELF_LOCATIONS_TO_RECV: list[int] = [
    LMLocation.get_apid(value.code) for value in ALL_LOCATION_TABLE.values() if value.remote_only]

BOOLOSSUS_AP_ID_LIST = [LMLocation.get_apid(value.code) for value in BOOLOSSUS_LOCATION_TABLE.values()]

FLIP_BALCONY_BOO_EVENT_LIST = ["Clockwork Room", "Telephone Room", "Armory", "Ceramics Studio"]