# fmt: off
"""Shufflable exit class and list file."""

from __future__ import annotations

from typing import TYPE_CHECKING, Optional, Set

from randomizer.Enums.ExitCategories import ExitCategories
from randomizer.Enums.Kongs import Kongs
from randomizer.Enums.Regions import Regions
from randomizer.Enums.Transitions import Transitions
from randomizer.Lists.LevelInfo import LevelInfoList
from randomizer.LogicClasses import TransitionBack


class ShufflableExit:
    """Class that stores data about an exit to be shuffled."""

    def __init__(self, name: str, region: Regions, back: TransitionBack, category: Optional[ExitCategories] = None, entryKongs: Optional[Set[Kongs]] = None, regionKongs: Optional[Set[Kongs]] = None, move: bool = False) -> None:
        """Initialize with given parameters."""
        if entryKongs is None:
            entryKongs = {Kongs.donkey, Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky}
        if regionKongs is None:
            regionKongs = {}
        self.name = name
        self.region = region
        self.back = back
        self.category = category  # Indicates the origin of this transition is in an area with multiple loading zones
        self.entryKongs = entryKongs  # Indicates either need to be one of a certain set of kongs to gain access to this transition
        self.regionKongs = regionKongs  # Indicates need to be a certain kong or kongs on the back side of the transition. Should only apply if the destination region doesn't have a tag barrel
        self.move = move  # Indicates this transition needs a kong-specific move to access, so it's even more restrictive
        # shuffledId is pointing to the shuffled destination exit within ShufflableExits
        # Initialized as none until it gets shuffled
        self.shuffledId = None
        self.shuffled = False
        self.toBeShuffled = False


ShufflableExits = {
    # Level Exits
    Transitions.IslesToJapes: ShufflableExit("Japes Lobby to Jungle Japes", Regions.JungleJapesLobby, TransitionBack(Regions.JungleJapesEntryHandler, "From Japes Lobby", "Jungle Japes from Japes Lobby", Transitions.JapesToIsles), ExitCategories.JapesLobby),
    Transitions.JapesToIsles: ShufflableExit("Jungle Japes to Japes Lobby", Regions.JungleJapesEntryHandler, TransitionBack(Regions.JungleJapesLobby, "From Japes", "Japes Lobby from Jungle Japes", Transitions.IslesToJapes), ExitCategories.JapesExterior),
    Transitions.IslesToAztec: ShufflableExit("Aztec Lobby to Angry Aztec", Regions.AngryAztecLobby, TransitionBack(Regions.AngryAztecEntryHandler, "From Aztec Lobby", "Angry Aztec from Aztec Lobby", Transitions.AztecToIsles), ExitCategories.AztecLobby),
    Transitions.AztecToIsles: ShufflableExit("Angry Aztec to Aztec Lobby", Regions.AngryAztecEntryHandler, TransitionBack(Regions.AngryAztecLobby, "From Aztec", "Aztec Lobby from Angry Aztec", Transitions.IslesToAztec), ExitCategories.AztecExterior),
    Transitions.IslesToFactory: ShufflableExit("Factory Lobby to Frantic Factory", Regions.FranticFactoryLobby, TransitionBack(Regions.FranticFactoryEntryHandler, "From Factory Lobby", "Frantic Factory from Factory Lobby", Transitions.FactoryToIsles), ExitCategories.FactoryLobby),
    Transitions.FactoryToIsles: ShufflableExit("Frantic Factory to Factory Lobby", Regions.FranticFactoryEntryHandler, TransitionBack(Regions.FranticFactoryLobby, "From Factory", "Factory Lobby from Frantic Factory", Transitions.IslesToFactory), ExitCategories.FactoryExterior),
    Transitions.IslesToGalleon: ShufflableExit("Galleon Lobby to Gloomy Galleon", Regions.GloomyGalleonLobby, TransitionBack(Regions.GloomyGalleonEntryHandler, "From Galleon Lobby", "Gloomy Galleon from Galleon Lobby", Transitions.GalleonToIsles), ExitCategories.GalleonLobby),
    Transitions.GalleonToIsles: ShufflableExit("Gloomy Galleon to Galleon Lobby", Regions.GloomyGalleonEntryHandler, TransitionBack(Regions.GloomyGalleonLobby, "From Galleon", "Galleon Lobby from Gloomy Galleon", Transitions.IslesToGalleon), ExitCategories.GalleonExterior),
    Transitions.IslesToForest: ShufflableExit("Fungi Lobby to Fungi Forest", Regions.FungiForestLobby, TransitionBack(Regions.FungiForestEntryHandler, "From Fungi Lobby", "Fungi Forest from Fungi Lobby", Transitions.ForestToIsles), ExitCategories.ForestLobby),
    Transitions.ForestToIsles: ShufflableExit("Fungi Forest to Fungi Lobby", Regions.FungiForestEntryHandler, TransitionBack(Regions.FungiForestLobby, "From Fungi", "Fungi Lobby from Fungi Forest", Transitions.IslesToForest), ExitCategories.ForestExterior),
    Transitions.IslesToCaves: ShufflableExit("Caves Lobby to Crystal Caves", Regions.CrystalCavesLobby, TransitionBack(Regions.CrystalCavesEntryHandler, "From Caves Lobby", "Crystal Caves from Caves Lobby", Transitions.CavesToIsles), ExitCategories.CavesLobby),
    Transitions.CavesToIsles: ShufflableExit("Crystal Caves to Caves Lobby", Regions.CrystalCavesEntryHandler, TransitionBack(Regions.CrystalCavesLobby, "From Caves", "Caves Lobby from Crystal Caves", Transitions.IslesToCaves), ExitCategories.CavesExterior),
    Transitions.IslesToCastle: ShufflableExit("Castle Lobby to Creepy Castle", Regions.CreepyCastleLobby, TransitionBack(Regions.CreepyCastleEntryHandler, "From Castle Lobby", "Creepy Castle from Castle Lobby", Transitions.CastleToIsles), ExitCategories.CastleLobby),
    Transitions.CastleToIsles: ShufflableExit("Creepy Castle to Castle Lobby", Regions.CreepyCastleEntryHandler, TransitionBack(Regions.CreepyCastleLobby, "From Castle", "Castle Lobby from Creepy Castle", Transitions.IslesToCastle), ExitCategories.CastleExterior),
    Transitions.IslesToHelm: ShufflableExit("Helm Lobby to Hideout Helm", Regions.HideoutHelmLobbyPastVines, TransitionBack(Regions.HideoutHelmEntry, "From Helm Lobby", "Hideout Helm from Helm Lobby", Transitions.HelmToIsles), ExitCategories.HelmLobby),
    Transitions.HelmToIsles: ShufflableExit("Hideout Helm to Helm Lobby", Regions.HideoutHelmEntry, TransitionBack(Regions.HideoutHelmLobbyPastVines, "From Helm", "Helm Lobby from Hideout Helm", Transitions.IslesToHelm), ExitCategories.HideoutHelm),

    # DK Isles Exits
    Transitions.IslesTreehouseToStart: ShufflableExit("DK's Treehouse to Training Grounds", Regions.Treehouse, TransitionBack(Regions.TrainingGrounds, "From Treehouse", "Training Grounds from DK's Treehouse", Transitions.IslesStartToTreehouse)),
    Transitions.IslesStartToTreehouse: ShufflableExit("Training Grounds to DK's Treehouse", Regions.TrainingGrounds, TransitionBack(Regions.Treehouse, "From Training Grounds", "DK's Treehouse (Training Grounds)", Transitions.IslesTreehouseToStart), ExitCategories.IslesTrainingGrounds),
    Transitions.IslesStartToMain: ShufflableExit("Training Grounds to DK Isles Main", Regions.TrainingGrounds, TransitionBack(Regions.IslesMain, "From Training Grounds", "DK Isles Main from Training Grounds", Transitions.IslesMainToStart), ExitCategories.IslesTrainingGrounds),
    Transitions.IslesMainToStart: ShufflableExit("DK Isles Main to Training Grounds", Regions.IslesMain, TransitionBack(Regions.TrainingGrounds, "From DK Isles", "Training Grounds from DK Isles Main", Transitions.IslesStartToMain), ExitCategories.IslesExterior),
    Transitions.IslesMainToPrison: ShufflableExit("DK Isles Main to K-Lumsy", Regions.KremIsle, TransitionBack(Regions.Prison, "From DK Isles", "K-Lumsy", Transitions.IslesPrisonToMain), ExitCategories.IslesExterior),
    Transitions.IslesPrisonToMain: ShufflableExit("DK Isles K-Lumsy to Main", Regions.Prison, TransitionBack(Regions.KremIsle, "From K-Lumsy", "DK Isles Main from K-Lumsy", Transitions.IslesMainToPrison)),
    Transitions.IslesMainToFairy: ShufflableExit("DK Isles Main to Banana Fairy Isle", Regions.OuterIsles, TransitionBack(Regions.BananaFairyRoom, "From DK Isles", "Banana Fairy Isle", Transitions.IslesFairyToMain), ExitCategories.IslesExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.IslesFairyToMain: ShufflableExit("DK Isles Banana Fairy Isle to Main", Regions.BananaFairyRoom, TransitionBack(Regions.OuterIsles, "From Banana Fairy Isle", "DK Isles Main from Banana Fairy Isle", Transitions.IslesMainToFairy)),
    Transitions.IslesMainToSnideRoom: ShufflableExit("DK Isles Main to Snide's Room", Regions.KremIsleBeyondLift, TransitionBack(Regions.IslesSnideRoom, "From DK Isles", "Snide's Room (DK Isles)", Transitions.IslesSnideRoomToMain), ExitCategories.IslesExterior),
    Transitions.IslesSnideRoomToMain: ShufflableExit("DK Isles Snide's Room to Main", Regions.IslesSnideRoom, TransitionBack(Regions.KremIsleBeyondLift, "From Snide's Room", "DK Isles Main from Snide's Room", Transitions.IslesMainToSnideRoom)),
    Transitions.IslesMainToJapesLobby: ShufflableExit("DK Isles Main to Japes Lobby", Regions.IslesMain, TransitionBack(Regions.JungleJapesLobby, "From DK Isles", "Japes Lobby from DK Isles Main", Transitions.IslesJapesLobbyToMain), ExitCategories.IslesExterior),
    Transitions.IslesJapesLobbyToMain: ShufflableExit("DK Isles Japes Lobby to Main", Regions.JungleJapesLobby, TransitionBack(Regions.IslesMain, "From Japes Lobby", "DK Isles Main from Japes Lobby", Transitions.IslesMainToJapesLobby), ExitCategories.JapesLobby),
    Transitions.IslesMainToAztecLobby: ShufflableExit("DK Isles Main to Aztec Lobby", Regions.IslesMainUpper, TransitionBack(Regions.AngryAztecLobby, "From DK Isles", "Aztec Lobby from DK Isles Main", Transitions.IslesAztecLobbyToMain), ExitCategories.IslesExterior),
    Transitions.IslesAztecLobbyToMain: ShufflableExit("DK Isles Aztec Lobby to Main", Regions.AngryAztecLobby, TransitionBack(Regions.IslesMainUpper, "From Aztec Lobby", "DK Isles Main from Aztec Lobby", Transitions.IslesMainToAztecLobby), ExitCategories.AztecLobby),
    Transitions.IslesMainToFactoryLobby: ShufflableExit("DK Isles Main to Factory Lobby", Regions.KremIsleBeyondLift, TransitionBack(Regions.FranticFactoryLobby, "From DK Isles", "Factory Lobby from DK Isles Main", Transitions.IslesFactoryLobbyToMain), ExitCategories.IslesExterior),
    Transitions.IslesFactoryLobbyToMain: ShufflableExit("DK Isles Factory Lobby to Main", Regions.FranticFactoryLobby, TransitionBack(Regions.KremIsleBeyondLift, "From Factory Lobby", "DK Isles Main from Factory Lobby", Transitions.IslesMainToFactoryLobby), ExitCategories.FactoryLobby),
    Transitions.IslesMainToGalleonLobby: ShufflableExit("DK Isles Main to Galleon Lobby", Regions.IslesMain, TransitionBack(Regions.GloomyGalleonLobbyEntrance, "From DK Isles", "Galleon Lobby from DK Isles Main", Transitions.IslesGalleonLobbyToMain), ExitCategories.IslesExterior),
    Transitions.IslesGalleonLobbyToMain: ShufflableExit("DK Isles Galleon Lobby to Main", Regions.GloomyGalleonLobbyEntrance, TransitionBack(Regions.IslesMain, "From Galleon Lobby", "DK Isles Main from Galleon Lobby", Transitions.IslesMainToGalleonLobby), ExitCategories.GalleonLobby),
    Transitions.IslesMainToForestLobby: ShufflableExit("DK Isles Main to Fungi Lobby", Regions.CabinIsle, TransitionBack(Regions.FungiForestLobby, "From DK Isles", "Fungi Lobby from DK Isles Main", Transitions.IslesForestLobbyToMain), ExitCategories.IslesExterior),
    Transitions.IslesForestLobbyToMain: ShufflableExit("DK Isles Fungi Lobby to Main", Regions.FungiForestLobby, TransitionBack(Regions.CabinIsle, "From Fungi Lobby", "DK Isles Main from Fungi Lobby", Transitions.IslesMainToForestLobby), ExitCategories.ForestLobby),
    Transitions.IslesMainToCavesLobby: ShufflableExit("DK Isles Main to Caves Lobby", Regions.IslesEar, TransitionBack(Regions.CrystalCavesLobby, "From DK Isles", "Caves Lobby from DK Isles Main", Transitions.IslesCavesLobbyToMain), ExitCategories.IslesExterior),
    Transitions.IslesCavesLobbyToMain: ShufflableExit("DK Isles Caves Lobby to Main", Regions.CrystalCavesLobby, TransitionBack(Regions.IslesEar, "From Caves Lobby", "DK Isles Main from Caves Lobby", Transitions.IslesMainToCavesLobby), ExitCategories.CavesLobby),
    Transitions.IslesMainToCastleLobby: ShufflableExit("DK Isles Main to Castle Lobby", Regions.IslesMain, TransitionBack(Regions.CreepyCastleLobby, "From DK Isles", "Castle Lobby from DK Isles Main", Transitions.IslesCastleLobbyToMain), ExitCategories.IslesExterior),
    Transitions.IslesCastleLobbyToMain: ShufflableExit("DK Isles Castle Lobby to Main", Regions.CreepyCastleLobby, TransitionBack(Regions.IslesMain, "From Castle Lobby", "DK Isles Main from Castle Lobby", Transitions.IslesMainToCastleLobby), ExitCategories.CastleLobby),
    Transitions.IslesMainToHelmLobby: ShufflableExit("DK Isles Main to Helm Lobby", Regions.KremIsleMouth, TransitionBack(Regions.HideoutHelmLobby, "From DK Isles", "Helm Lobby from DK Isles Main", Transitions.IslesHelmLobbyToMain), ExitCategories.IslesExterior),
    Transitions.IslesHelmLobbyToMain: ShufflableExit("DK Isles Helm Lobby to Main", Regions.HideoutHelmLobby, TransitionBack(Regions.KremIsleMouth, "From Helm Lobby", "DK Isles Main from Helm Lobby", Transitions.IslesMainToHelmLobby), ExitCategories.HelmLobby),

    # Jungle Japes Exits
    Transitions.JapesMainToMine: ShufflableExit("Jungle Japes Main to Mountain", Regions.JapesHillTop, TransitionBack(Regions.Mine, "From Japes Main", "Jungle Japes Mountain from Main", Transitions.JapesMineToMain), ExitCategories.JapesExterior, entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}, move=True),
    Transitions.JapesMineToMain: ShufflableExit("Jungle Japes Mountain to Main", Regions.Mine, TransitionBack(Regions.JapesHillTop, "From Mountain", "Jungle Japes Main from Mountain", Transitions.JapesMainToMine)),
    Transitions.JapesMainToLankyCave: ShufflableExit("Jungle Japes Main to Painting Room", Regions.JapesPaintingRoomHill, TransitionBack(Regions.JapesLankyCave, "From Japes Main", "Jungle Japes Painting Room", Transitions.JapesLankyCaveToMain), ExitCategories.JapesExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}, move=True),
    Transitions.JapesLankyCaveToMain: ShufflableExit("Jungle Japes Painting Room to Main", Regions.JapesLankyCave, TransitionBack(Regions.JapesPaintingRoomHill, "From Painting Room", "Jungle Japes Main from Painting Room", Transitions.JapesMainToLankyCave)),
    Transitions.JapesMainToCatacomb: ShufflableExit("Jungle Japes Main to Underground", Regions.JungleJapesStart, TransitionBack(Regions.JapesCatacomb, "From Japes Main", "Jungle Japes Underground", Transitions.JapesCatacombToMain), ExitCategories.JapesExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}),
    Transitions.JapesCatacombToMain: ShufflableExit("Jungle Japes Underground to Main", Regions.JapesCatacomb, TransitionBack(Regions.JungleJapesStart, "From Underground", "Jungle Japes Main from Underground", Transitions.JapesMainToCatacomb)),
    Transitions.JapesMainToTinyHive: ShufflableExit("Jungle Japes Main to Beehive", Regions.JapesBeyondFeatherGate, TransitionBack(Regions.TinyHive, "From Japes Main", "Jungle Japes Beehive", Transitions.JapesTinyHiveToMain), ExitCategories.JapesExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.JapesTinyHiveToMain: ShufflableExit("Jungle Japes Beehive to Main", Regions.TinyHive, TransitionBack(Regions.JapesBeyondFeatherGate, "From Beehive", "Jungle Japes Main from Beehive", Transitions.JapesMainToTinyHive)),
    # Transitions.JapesMainToBBlast: ShufflableExit("Jungle Japes Main to Baboon Blast", Regions.JapesBlastPadPlatform, TransitionBack(Regions.JapesBaboonBlast, "From Japes Main", "Jungle Japes Baboon Blast"), ExitCategories.JapesExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),
    Transitions.JapesMineToCarts: ShufflableExit("Jungle Japes Mountain to Minecart", Regions.Mine, TransitionBack(Regions.JapesMinecarts, "From Mountain", "Jungle Japes Minecart"), entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}),
    Transitions.JapesCartsToMain: ShufflableExit("Jungle Japes Minecart to Main", Regions.JapesMinecarts, TransitionBack(Regions.JungleJapesMain, "From Minecart", "Jungle Japes Main from Minecart")),

    # Angry Aztec Exits
    Transitions.AztecStartToTemple: ShufflableExit("Angry Aztec Main to Tiny Temple", Regions.AngryAztecOasis, TransitionBack(Regions.TempleStart, "From Aztec Main", "Angry Aztec Tiny Temple", Transitions.AztecTempleToStart), ExitCategories.AztecExterior, entryKongs={Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky}, regionKongs={Kongs.diddy, Kongs.lanky, Kongs.tiny, Kongs.chunky}, move=True),
    Transitions.AztecTempleToStart: ShufflableExit("Angry Aztec Tiny Temple to Main", Regions.TempleStart, TransitionBack(Regions.AngryAztecOasis, "From Tiny Temple", "Angry Aztec Main from Tiny Temple", Transitions.AztecStartToTemple)),
    Transitions.AztecMainToDonkey: ShufflableExit("Angry Aztec Main to Donkey 5DTemple", Regions.AngryAztecMain, TransitionBack(Regions.DonkeyTemple, "From Aztec Main", "Angry Aztec DK 5DTemple", Transitions.AztecDonkeyToMain), ExitCategories.AztecExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),
    Transitions.AztecDonkeyToMain: ShufflableExit("Angry Aztec Donkey 5DTemple to Main", Regions.DonkeyTemple, TransitionBack(Regions.AngryAztecMain, "From DK 5DTemple", "Angry Aztec Main from DK 5DTemple", Transitions.AztecMainToDonkey)),
    Transitions.AztecMainToDiddy: ShufflableExit("Angry Aztec Main to Diddy 5DTemple", Regions.AngryAztecMain, TransitionBack(Regions.DiddyTemple, "From Aztec Main", "Angry Aztec Diddy 5DTemple", Transitions.AztecDiddyToMain), ExitCategories.AztecExterior, entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}, move=True),
    Transitions.AztecDiddyToMain: ShufflableExit("Angry Aztec Diddy 5DTemple to Main", Regions.DiddyTemple, TransitionBack(Regions.AngryAztecMain, "From Diddy 5DTemple", "Angry Aztec Main from Diddy 5DTemple", Transitions.AztecMainToDiddy)),
    Transitions.AztecMainToLanky: ShufflableExit("Angry Aztec Main to Lanky 5DTemple", Regions.AngryAztecMain, TransitionBack(Regions.LankyTempleEntrance, "From Aztec Main", "Angry Aztec Lanky 5DTemple", Transitions.AztecLankyToMain), ExitCategories.AztecExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}, move=True),
    Transitions.AztecLankyToMain: ShufflableExit("Angry Aztec Lanky 5DTemple to Main", Regions.LankyTempleEntrance, TransitionBack(Regions.AngryAztecMain, "From Lanky 5DTemple", "Angry Aztec Main from Lanky 5DTemple", Transitions.AztecMainToLanky)),
    Transitions.AztecMainToTiny: ShufflableExit("Angry Aztec Main to Tiny 5DTemple", Regions.AngryAztecMain, TransitionBack(Regions.TinyTempleEntrance, "From Aztec Main", "Angry Aztec Tiny 5DTemple", Transitions.AztecTinyToMain), ExitCategories.AztecExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.AztecTinyToMain: ShufflableExit("Angry Aztec Tiny 5DTemple To Main", Regions.TinyTempleEntrance, TransitionBack(Regions.AngryAztecMain, "From Tiny 5DTemple", "Angry Aztec Main from Tiny 5DTemple", Transitions.AztecMainToTiny)),
    Transitions.AztecMainToChunky: ShufflableExit("Angry Aztec Main to Chunky 5DTemple", Regions.AngryAztecMain, TransitionBack(Regions.ChunkyTempleEntrance, "From Aztec Main", "Angry Aztec Chunky 5DTemple", Transitions.AztecChunkyToMain), ExitCategories.AztecExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}, move=True),
    Transitions.AztecChunkyToMain: ShufflableExit("Angry Aztec Chunky 5DTemple to Main", Regions.ChunkyTempleEntrance, TransitionBack(Regions.AngryAztecMain, "From Chunky 5DTemple", "Angry Aztec Main from Chunky 5DTemple", Transitions.AztecMainToChunky)),
    Transitions.AztecMainToRace: ShufflableExit("Angry Aztec Main to Beetle Race", Regions.AngryAztecMain, TransitionBack(Regions.AztecTinyRace, "From Aztec Main", "Angry Aztec Beetle Race", Transitions.AztecRaceToMain), ExitCategories.AztecExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.AztecRaceToMain: ShufflableExit("Angry Aztec Beetle Race to Main", Regions.AztecTinyRace, TransitionBack(Regions.AngryAztecMain, "From Beetle Race", "Angry Aztec Main from Beetle Race", Transitions.AztecMainToRace)),
    Transitions.AztecLlamaToMain: ShufflableExit("Angry Aztec Llama Temple to Main", Regions.LlamaTemple, TransitionBack(Regions.AngryAztecMain, "From Llama Temple", "Angry Aztec Main from Llama Temple", Transitions.AztecMainToLlama)),
    Transitions.AztecMainToLlama: ShufflableExit("Angry Aztec Main to Llama Temple", Regions.AngryAztecMain, TransitionBack(Regions.LlamaTemple, "From Aztec Main", "Angry Aztec Llama Temple", Transitions.AztecLlamaToMain), ExitCategories.AztecExterior, entryKongs={Kongs.donkey, Kongs.lanky, Kongs.tiny}, move=True),  # entryKongs can be affected by switchsanity!
    # Transitions.AztecMainToBBlast: ShufflableExit("Angry Aztec Main to Baboon Blast", Regions.AngryAztecMain, TransitionBack(Regions.AztecBaboonBlast, "From Aztec Main", "Angry Aztec Baboon Blast"), ExitCategories.AztecExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),

    # Frantic Factory Exits
    Transitions.FactoryRandDToRace: ShufflableExit("Frantic Factory R&D to Car Race", Regions.FactoryTinyRaceLobby, TransitionBack(Regions.FactoryTinyRace, "From Factory Main", "Frantic Factory Car Race", Transitions.FactoryRaceToRandD), ExitCategories.FactoryExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.FactoryRaceToRandD: ShufflableExit("Frantic Factory Car Race to R&D", Regions.FactoryTinyRace, TransitionBack(Regions.FactoryTinyRaceLobby, "From Car Race", "Frantic Factory Main from Car Race", Transitions.FactoryRandDToRace)),
    Transitions.FactoryChunkyRoomToPower: ShufflableExit("Frantic Factory Main to Power Shed", Regions.ChunkyRoomPlatform, TransitionBack(Regions.PowerHut, "From Factory Main", "Frantic Factory Power Shed", Transitions.FactoryPowerToChunkyRoom), ExitCategories.FactoryExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}),
    Transitions.FactoryPowerToChunkyRoom: ShufflableExit("Frantic Factory Power Room to Chunky Room", Regions.PowerHut, TransitionBack(Regions.ChunkyRoomPlatform, "From Power Shed", "Frantic Factory Main from Power Shed", Transitions.FactoryChunkyRoomToPower)),
    Transitions.FactoryLowerCoreToInsideCore: ShufflableExit("Frantic Factory Main to Crusher Room", Regions.LowerCore, TransitionBack(Regions.InsideCore, "From Factory Main", "Frantic Factory Crusher Room", Transitions.FactoryInsideCoreToLowerCore), ExitCategories.FactoryExterior, regionKongs={Kongs.donkey, Kongs.lanky}),
    Transitions.FactoryInsideCoreToLowerCore: ShufflableExit("Frantic Factory Crusher Room to Main", Regions.InsideCore, TransitionBack(Regions.LowerCore, "From Crusher Room", "Frantic Factory Main from Crusher Room", Transitions.FactoryLowerCoreToInsideCore)),
    Transitions.FactoryStorageToArcade: ShufflableExit("Frantic Factory Storage to Arcade", Regions.BeyondHatch, TransitionBack(Regions.FactoryArcadeTunnel, "From Tiny BP Area (To Arcade Area)", "Frantic Factory Arcade from Storage", Transitions.FactoryArcadeToStorage), ExitCategories.FactoryExterior, move=True),
    Transitions.FactoryArcadeToStorage: ShufflableExit("Frantic Factory Arcade to Storage", Regions.FactoryArcadeTunnel, TransitionBack(Regions.BeyondHatch, "From Arcade Area (near Tiny BP)", "Frantic Factory Storage from Arcade", Transitions.FactoryStorageToArcade), ExitCategories.FactoryExterior),
    # Transitions.FactoryMainToBBlast: ShufflableExit("Frantic Factory Main to BBlast", Regions.BeyondHatch, TransitionBack(Regions.FactoryBaboonBlast, "From Factory Main", "Frantic Factory Baboon Blast"), ExitCategories.FactoryExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),

    # Gloomy Galleon Exits
    Transitions.GalleonLighthouseAreaToLighthouse: ShufflableExit("Gloomy Galleon Main to Lighthouse", Regions.LighthousePlatform, TransitionBack(Regions.Lighthouse, "From Galleon Main", "Gloomy Galleon Lighthouse", Transitions.GalleonLighthouseToLighthouseArea), ExitCategories.GalleonExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}),
    Transitions.GalleonLighthouseToLighthouseArea: ShufflableExit("Gloomy Galleon Lighthouse to Main", Regions.Lighthouse, TransitionBack(Regions.LighthousePlatform, "From Lighthouse", "Gloomy Galleon Main from Lighthouse", Transitions.GalleonLighthouseAreaToLighthouse)),
    Transitions.GalleonLighthouseAreaToMermaid: ShufflableExit("Gloomy Galleon Main to Mermaid", Regions.LighthouseUnderwater, TransitionBack(Regions.MermaidRoom, "From Galleon Main", "Gloomy Galleon Mermaid", Transitions.GalleonMermaidToLighthouseArea), ExitCategories.GalleonExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.GalleonMermaidToLighthouseArea: ShufflableExit("Gloomy Galleon Mermaid to Main", Regions.MermaidRoom, TransitionBack(Regions.LighthouseUnderwater, "From Mermaid", "Gloomy Galleon Main from Mermaid", Transitions.GalleonLighthouseAreaToMermaid)),
    Transitions.GalleonLighthouseAreaToSickBay: ShufflableExit("Gloomy Galleon Main to Seasick Ship", Regions.LighthousePlatform, TransitionBack(Regions.SickBay, "From Galleon Main", "Gloomy Galleon Seasick Ship", Transitions.GalleonSickBayToLighthouseArea), ExitCategories.GalleonExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}, move=True),
    Transitions.GalleonSickBayToLighthouseArea: ShufflableExit("Gloomy Galleon Seasick Ship to Main", Regions.SickBay, TransitionBack(Regions.LighthousePlatform, "From Seasick Ship", "Gloomy Galleon Main from Seasick Ship", Transitions.GalleonLighthouseAreaToSickBay)),
    Transitions.GalleonShipyardToSeal: ShufflableExit("Gloomy Galleon Main to Seal Race", Regions.Shipyard, TransitionBack(Regions.SealRace, "From Galleon Main", "Gloomy Galleon Seal Race", Transitions.GalleonSealToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),
    Transitions.GalleonSealToShipyard: ShufflableExit("Gloomy Galleon Seal Race to Main", Regions.SealRace, TransitionBack(Regions.Shipyard, "From Seal Race", "Gloomy Galleon Main from Seal Race", Transitions.GalleonShipyardToSeal)),
    Transitions.GalleonShipyardToSubmarine: ShufflableExit("Gloomy Galleon Main to Submarine", Regions.ShipyardUnderwater, TransitionBack(Regions.Submarine, "From Galleon Main", "Gloomy Galleon Submarine", Transitions.GalleonSubmarineToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.GalleonSubmarineToShipyard: ShufflableExit("Gloomy Galleon Submarine to Main", Regions.Submarine, TransitionBack(Regions.ShipyardUnderwater, "From Submarine", "Gloomy Galleon Main from Submarine", Transitions.GalleonShipyardToSubmarine)),
    Transitions.GalleonShipyardToLanky: ShufflableExit("Gloomy Galleon Main to Lanky 2DShip", Regions.ShipyardUnderwater, TransitionBack(Regions.LankyShip, "From Galleon (Lanky Entrance)", "Gloomy Galleon Lanky 2DShip", Transitions.GalleonLankyToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}),
    Transitions.GalleonLankyToShipyard: ShufflableExit("Gloomy Galleon Lanky 2DShip to Main", Regions.LankyShip, TransitionBack(Regions.ShipyardUnderwater, "From Lanky 2DShip", "Gloomy Galleon Main from Lanky 2DShip", Transitions.GalleonShipyardToLanky)),
    Transitions.GalleonShipyardToTiny: ShufflableExit("Gloomy Galleon Main to Tiny 2DShip", Regions.ShipyardUnderwater, TransitionBack(Regions.TinyShip, "From Galleon (Tiny Entrance)", "Gloomy Galleon Tiny 2DShip", Transitions.GalleonTinyToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}),
    Transitions.GalleonTinyToShipyard: ShufflableExit("Gloomy Galleon Tiny 2DShip to Main", Regions.TinyShip, TransitionBack(Regions.ShipyardUnderwater, "From Tiny 2DShip", "Gloomy Galleon Main from Tiny 2DShip", Transitions.GalleonShipyardToTiny)),
    Transitions.GalleonShipyardToBongos: ShufflableExit("Gloomy Galleon Main to DK 5DShip", Regions.ShipyardUnderwater, TransitionBack(Regions.BongosShip, "From Galleon (DK Entrance)", "Gloomy Galleon DK 5DShip", Transitions.GalleonBongosToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),
    Transitions.GalleonBongosToShipyard: ShufflableExit("Gloomy Galleon DK 5DShip to Main", Regions.BongosShip, TransitionBack(Regions.ShipyardUnderwater, "From Donkey 5DShip", "Gloomy Galleon Main from DK 5DShip", Transitions.GalleonShipyardToBongos)),
    Transitions.GalleonShipyardToGuitar: ShufflableExit("Gloomy Galleon Main to Diddy 5DShip", Regions.ShipyardUnderwater, TransitionBack(Regions.GuitarShip, "From Galleon (Diddy Entrance)", "Gloomy Galleon Diddy 5DShip", Transitions.GalleonGuitarToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}, move=True),
    Transitions.GalleonGuitarToShipyard: ShufflableExit("Gloomy Galleon Diddy 5DShip to Main", Regions.GuitarShip, TransitionBack(Regions.ShipyardUnderwater, "From Diddy 5DShip", "Gloomy Galleon Main from Diddy 5DShip", Transitions.GalleonShipyardToGuitar)),
    Transitions.GalleonShipyardToTrombone: ShufflableExit("Gloomy Galleon Main to Lanky 5DShip", Regions.ShipyardUnderwater, TransitionBack(Regions.TromboneShip, "From Galleon (Lanky Entrance)", "Gloomy Galleon Lanky 5DShip", Transitions.GalleonTromboneToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}, move=True),
    Transitions.GalleonTromboneToShipyard: ShufflableExit("Gloomy Galleon Lanky 5DShip to Main", Regions.TromboneShip, TransitionBack(Regions.ShipyardUnderwater, "From Lanky 5DShip", "Gloomy Galleon Main from Lanky 5DShip", Transitions.GalleonShipyardToTrombone)),
    Transitions.GalleonShipyardToSaxophone: ShufflableExit("Gloomy Galleon Main to Tiny 5DShip", Regions.ShipyardUnderwater, TransitionBack(Regions.SaxophoneShip, "From Galleon (Tiny Entrance)", "Gloomy Galleon Tiny 5DShip", Transitions.GalleonSaxophoneToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.GalleonSaxophoneToShipyard: ShufflableExit("Gloomy Galleon Tiny 5DShip to Main", Regions.SaxophoneShip, TransitionBack(Regions.ShipyardUnderwater, "From Tiny 5DShip", "Gloomy Galleon Main from Tiny 5DShip", Transitions.GalleonShipyardToSaxophone)),
    Transitions.GalleonShipyardToTriangle: ShufflableExit("Gloomy Galleon Main to Chunky 5DShip", Regions.ShipyardUnderwater, TransitionBack(Regions.TriangleShip, "From Galleon (Chunky Entrance)", "Gloomy Galleon Chunky 5DShip", Transitions.GalleonTriangleToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}, move=True),
    Transitions.GalleonTriangleToShipyard: ShufflableExit("Gloomy Galleon Chunky 5DShip to Main", Regions.TriangleShip, TransitionBack(Regions.ShipyardUnderwater, "From Chunky 5DShip", "Gloomy Galleon Main from Chunky 5DShip", Transitions.GalleonShipyardToTriangle)),
    Transitions.GalleonTreasureToChest: ShufflableExit("Gloomy Galleon Main to Treasure Chest", Regions.TreasureRoom, TransitionBack(Regions.TinyChest, "From Galleon Main", "Gloomy Galleon Treasure Chest", Transitions.GalleonChestToTreasure), ExitCategories.GalleonExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.GalleonChestToTreasure: ShufflableExit("Gloomy Galleon Treasure Chest to Main", Regions.TinyChest, TransitionBack(Regions.TreasureRoom, "From Treasure Chest", "Gloomy Galleon Main from Treasure Chest", Transitions.GalleonTreasureToChest)),
    Transitions.GalleonShipyardToMechFish: ShufflableExit("Gloomy Galleon Main to Mech Fish", Regions.ShipyardUnderwater, TransitionBack(Regions.Mechafish, "From Galleon Main", "Gloomy Galleon Mech Fish", Transitions.GalleonMechFishToShipyard), ExitCategories.GalleonExterior, entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}),
    Transitions.GalleonMechFishToShipyard: ShufflableExit("Gloomy Galleon Mech Fish to Main", Regions.Mechafish, TransitionBack(Regions.ShipyardUnderwater, "From Mech Fish", "Gloomy Galleon Main from Mech Fish", Transitions.GalleonShipyardToMechFish)),
    # Transitions.GalleonMainToBBlast: ShufflableExit("Gloomy Galleon Main to Baboon Blast", Regions.LighthousePlatform, TransitionBack(Regions.GalleonBaboonBlast, "From Galleon Main", "Gloomy Galleon Baboon Blast"), ExitCategories.GalleonExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),

    # Fungi Forest Exits
    Transitions.ForestMainToCarts: ShufflableExit("Fungi Forest Main To Minecart", Regions.FungiForestStart, TransitionBack(Regions.ForestMinecarts, "From Fungi Main", "Fungi Forest Minecart"), ExitCategories.ForestExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}),
    Transitions.ForestCartsToMain: ShufflableExit("Fungi Forest Minecart To Main", Regions.ForestMinecarts, TransitionBack(Regions.MillArea, "From Minecart", "Fungi Forest Main from Minecart")),
    Transitions.ForestMainToLowerMushroom: ShufflableExit("Fungi Forest Main to Giant Mushroom (Lower)", Regions.GiantMushroomArea, TransitionBack(Regions.MushroomLower, "From Fungi (Low)", "Fungi Forest Giant Mushroom (Lower) from Low door", Transitions.ForestLowerMushroomToMain), ExitCategories.ForestExterior),
    Transitions.ForestLowerMushroomToMain: ShufflableExit("Fungi Forest Giant Mushroom (Lower) to Main", Regions.MushroomLower, TransitionBack(Regions.GiantMushroomArea, "From G. Mush (Low)", "Fungi Forest Main from Giant Mushroom (Bottom)", Transitions.ForestMainToLowerMushroom), ExitCategories.ForestMushroom),
    Transitions.ForestLowerExteriorToLowerMushroom: ShufflableExit("Fungi Forest Main to Giant Mushroom (Middle Low)", Regions.MushroomLowerExterior, TransitionBack(Regions.MushroomLowerMid, "From Fungi (Low Middle)", "Fungi Forest Giant Mushroom (Middle Low)", Transitions.ForestLowerMushroomToLowerExterior), ExitCategories.ForestExterior),
    Transitions.ForestLowerMushroomToLowerExterior: ShufflableExit("Fungi Forest Giant Mushroom (Middle Low) to Main", Regions.MushroomLowerMid, TransitionBack(Regions.MushroomLowerExterior, "From G. Mush (Low Middle)", "Fungi Forest Main from Giant Mushroom (Middle low)", Transitions.ForestLowerExteriorToLowerMushroom), ExitCategories.ForestMushroom),
    Transitions.ForestLowerExteriorToUpperMushroom: ShufflableExit("Fungi Forest Main to Giant Mushroom (Middle High)", Regions.MushroomUpperMidExterior, TransitionBack(Regions.MushroomMiddle, "From Fungi (Middle)", "Fungi Forest Giant Mushroom (Middle High)", Transitions.ForestUpperMushroomToLowerExterior), ExitCategories.ForestExterior),
    Transitions.ForestUpperMushroomToLowerExterior: ShufflableExit("Fungi Forest Giant Mushroom (Middle High) to Main", Regions.MushroomMiddle, TransitionBack(Regions.MushroomUpperMidExterior, "From G. Mush (Middle)", "Fungi Forest Main from Giant Mushroom (Middle high)", Transitions.ForestLowerExteriorToUpperMushroom), ExitCategories.ForestMushroom),
    Transitions.ForestUpperExteriorToUpperMushroom: ShufflableExit("Fungi Forest Main to Giant Mushroom (Top)", Regions.MushroomUpperExterior, TransitionBack(Regions.MushroomUpper, "From Fungi (High)", "Fungi Forest Giant Mushroom (Top)", Transitions.ForestUpperMushroomToUpperExterior), ExitCategories.ForestExterior),
    Transitions.ForestUpperMushroomToUpperExterior: ShufflableExit("Fungi Forest Giant Mushroom (Top) to Main", Regions.MushroomUpper, TransitionBack(Regions.MushroomUpperExterior, "From G. Mush (High)", "Fungi Forest Main from Giant Mushroom (Top)", Transitions.ForestUpperExteriorToUpperMushroom), ExitCategories.ForestMushroom),
    # Transitions.ForestMainToBBlast: ShufflableExit("Fungi Forest Main to Baboon Blast", Regions.MushroomLowerExterior, TransitionBack(Regions.ForestBaboonBlast, "From Fungi Main", "Fungi Forest Baboon Blast"), ExitCategories.ForestExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),
    Transitions.ForestExteriorToNight: ShufflableExit("Fungi Forest Main to Giant Mushroom (Night Door)", Regions.MushroomNightExterior, TransitionBack(Regions.MushroomNightDoor, "From Fungi (High Middle)", "Fungi Forest Giant Mushroom from Night door", Transitions.ForestNightToExterior), ExitCategories.ForestExterior),
    Transitions.ForestNightToExterior: ShufflableExit("Fungi Forest Giant Mushroom (Night Door) to Main", Regions.MushroomNightDoor, TransitionBack(Regions.MushroomNightExterior, "From G. Mush (High Middle)", "Fungi Forest Main from Giant Mushroom (Night door)", Transitions.ForestExteriorToNight), ExitCategories.ForestMushroom),
    Transitions.ForestExteriorToChunky: ShufflableExit("Fungi Forest Main to Face Puzzle (Chunky)", Regions.MushroomUpperExterior, TransitionBack(Regions.MushroomChunkyRoom, "From Fungi Main", "Fungi Forest Face Puzzle (Chunky)", Transitions.ForestChunkyToExterior), ExitCategories.ForestExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}),
    Transitions.ForestChunkyToExterior: ShufflableExit("Fungi Forest Face Puzzle (Chunky) to Main", Regions.MushroomChunkyRoom, TransitionBack(Regions.MushroomUpperExterior, "From Face Puzzle (Chunky)", "Fungi Forest Main from Face Puzzle (Chunky)", Transitions.ForestExteriorToChunky)),
    Transitions.ForestExteriorToZingers: ShufflableExit("Fungi Forest Main to Bouncy Shrooms (Lanky)", Regions.MushroomUpperExterior, TransitionBack(Regions.MushroomLankyZingersRoom, "From Fungi Main", "Fungi Forest Bouncy Shrooms (Lanky)", Transitions.ForestZingersToExterior), ExitCategories.ForestExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}),
    Transitions.ForestZingersToExterior: ShufflableExit("Fungi Forest Bouncy Shrooms (Lanky) to Main", Regions.MushroomLankyZingersRoom, TransitionBack(Regions.MushroomUpperExterior, "From Zingers Room (Lanky)", "Fungi Forest Main from Bouncy Shrooms (Lanky)", Transitions.ForestExteriorToZingers)),
    Transitions.ForestExteriorToMushrooms: ShufflableExit("Fungi Forest Main to Mushroom Puzzle (Lanky)", Regions.MushroomUpperExterior, TransitionBack(Regions.MushroomLankyMushroomsRoom, "From Fungi Main", "Fungi Forest Mushroom Puzzle (Lanky)", Transitions.ForestMushroomsToExterior), ExitCategories.ForestExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}),
    Transitions.ForestMushroomsToExterior: ShufflableExit("Fungi Forest Mushroom Puzzle (Lanky) to Main", Regions.MushroomLankyMushroomsRoom, TransitionBack(Regions.MushroomUpperExterior, "From Mushrooms Room (Lanky)", "Fungi Forest Main from Mushroom Puzzle (Lanky)", Transitions.ForestExteriorToMushrooms)),
    Transitions.ForestTreeToAnthill: ShufflableExit("Fungi Forest Main to Anthill", Regions.HollowTreeArea, TransitionBack(Regions.Anthill, "From Fungi Main", "Fungi Forest Anthill", Transitions.ForestAnthillToTree), ExitCategories.ForestExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.ForestAnthillToTree: ShufflableExit("Fungi Forest Anthill to Main", Regions.Anthill, TransitionBack(Regions.HollowTreeArea, "From Anthill", "Fungi Forest Main from Anthill", Transitions.ForestTreeToAnthill)),
    Transitions.ForestMainToChunkyMill: ShufflableExit("Fungi Forest Main to Mill Back Door", Regions.MillArea, TransitionBack(Regions.MillChunkyTinyArea, "From Fungi (PPunch Door)", "Fungi Forest Back Mill from Back Door", Transitions.ForestChunkyMillToMain), ExitCategories.ForestExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}, move=True),
    Transitions.ForestChunkyMillToMain: ShufflableExit("Fungi Forest Mill Back Door to Main", Regions.MillChunkyTinyArea, TransitionBack(Regions.MillArea, "From Mill (PPunch Door)", "Fungi Forest Main from Mill (Back Door)", Transitions.ForestMainToChunkyMill), ExitCategories.ForestMill, entryKongs={Kongs.chunky}),
    Transitions.ForestMainToTinyMill: ShufflableExit("Fungi Forest Main to Mill Tiny Hole", Regions.MillArea, TransitionBack(Regions.MillChunkyTinyArea, "From Fungi (Tiny Hole)", "Fungi Forest Back Mill from Tiny Hole", Transitions.ForestTinyMillToMain), ExitCategories.ForestExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.ForestTinyMillToMain: ShufflableExit("Fungi Forest Mill Tiny Hole to Main", Regions.MillChunkyTinyArea, TransitionBack(Regions.MillArea, "From Mill (Tiny Hole)", "Fungi Forest Main from Mill (Tiny Hole)", Transitions.ForestMainToTinyMill), ExitCategories.ForestMill, entryKongs={Kongs.tiny}),
    Transitions.ForestMainToGrinder: ShufflableExit("Fungi Forest Main to Mill Front Door", Regions.MillArea, TransitionBack(Regions.GrinderRoom, "From Fungi (Front)", "Fungi Forest Front Mill from Main", Transitions.ForestGrinderToMain), ExitCategories.ForestExterior),
    Transitions.ForestGrinderToMain: ShufflableExit("Fungi Forest Mill Front Door to Main", Regions.GrinderRoom, TransitionBack(Regions.MillArea, "From Mill (Front)", "Fungi Forest Main from Mill (Front)", Transitions.ForestMainToGrinder), ExitCategories.ForestGrinder),
    Transitions.ForestMainToRafters: ShufflableExit("Fungi Forest Main to Dark Rafters", Regions.MillArea, TransitionBack(Regions.MillRafters, "From Fungi Main", "Fungi Forest Dark Rafters", Transitions.ForestRaftersToMain), ExitCategories.ForestExterior, entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}, move=True),
    Transitions.ForestRaftersToMain: ShufflableExit("Fungi Forest Dark Rafters to Main", Regions.MillRafters, TransitionBack(Regions.MillArea, "From Rafters", "Fungi Forest Main from Dark Rafters", Transitions.ForestMainToRafters)),
    Transitions.ForestMainToWinch: ShufflableExit("Fungi Forest Main to Winch Room", Regions.ForestVeryTopOfMill, TransitionBack(Regions.WinchRoom, "From Fungi Main", "Fungi Forest Winch Room", Transitions.ForestWinchToMain), ExitCategories.ForestExterior, entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}),
    Transitions.ForestWinchToMain: ShufflableExit("Fungi Forest Winch Room to Main", Regions.WinchRoom, TransitionBack(Regions.ForestVeryTopOfMill, "From Winch", "Fungi Forest Main from Winch Room", Transitions.ForestMainToWinch)),
    Transitions.ForestMainToAttic: ShufflableExit("Fungi Forest Main to Mill Attic (Lanky)", Regions.ForestTopOfMill, TransitionBack(Regions.MillAttic, "From Fungi Main", "Fungi Forest Mill Attic (Lanky)", Transitions.ForestAtticToMain), ExitCategories.ForestExterior, regionKongs={Kongs.lanky}),
    Transitions.ForestAtticToMain: ShufflableExit("Fungi Forest Mill Attic (Lanky) to Main", Regions.MillAttic, TransitionBack(Regions.ForestTopOfMill, "From Mill Attic", "Fungi Forest Main from Mill Attic (Lanky)", Transitions.ForestMainToAttic)),
    Transitions.ForestTinyMillToSpider: ShufflableExit("Fungi Forest Back Mill to Spider", Regions.MillChunkyTinyArea, TransitionBack(Regions.SpiderRoom, "From Mill (Rear)", "Fungi Forest Spider", Transitions.ForestSpiderToTinyMill), ExitCategories.ForestMill, regionKongs={Kongs.tiny}),
    Transitions.ForestSpiderToTinyMill: ShufflableExit("Fungi Forest Spider to Back Mill", Regions.SpiderRoom, TransitionBack(Regions.MillChunkyTinyArea, "From Spider Boss", "Fungi Forest Back Mill from Spider", Transitions.ForestTinyMillToSpider), regionKongs={Kongs.tiny}),
    Transitions.ForestTinyMillToGrinder: ShufflableExit("Fungi Forest Back Mill to Front Mill", Regions.MillChunkyTinyArea, TransitionBack(Regions.GrinderRoom, "From Mill (Rear)", "Fungi Forest Front Mill from Back Mill", Transitions.ForestGrinderToTinyMill), ExitCategories.ForestMill, entryKongs={Kongs.tiny}, move=True),
    Transitions.ForestGrinderToTinyMill: ShufflableExit("Fungi Forest Front Mill to Back Mill", Regions.GrinderRoom, TransitionBack(Regions.MillChunkyTinyArea, "From Mill (Front)", "Fungi Forest Back Mill from Front Mill", Transitions.ForestTinyMillToGrinder), ExitCategories.ForestGrinder, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.ForestMainToBarn: ShufflableExit("Fungi Forest Main to Thornvine Barn", Regions.ThornvineArea, TransitionBack(Regions.ThornvineBarn, "From Fungi Main", "Fungi Forest Thornvine Barn", Transitions.ForestBarnToMain), ExitCategories.ForestExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}),
    Transitions.ForestBarnToMain: ShufflableExit("Fungi Forest Thornvine Barn to Main", Regions.ThornvineBarn, TransitionBack(Regions.ThornvineArea, "From Thornvine Barn", "Fungi Forest Main from Thornvine Barn", Transitions.ForestMainToBarn)),

    # Crystal Caves Exits
    Transitions.CavesMainToRace: ShufflableExit("Crystal Caves Main to Beetle Race", Regions.CrystalCavesMain, TransitionBack(Regions.CavesLankyRace, "From Caves Main", "Crystal Caves Beetle Race", Transitions.CavesRaceToMain), ExitCategories.CavesExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}, move=True),
    Transitions.CavesRaceToMain: ShufflableExit("Crystal Caves Beetle Race to Main", Regions.CavesLankyRace, TransitionBack(Regions.CrystalCavesMain, "From Beetle Race", "Crystal Caves Main from Beetle Race", Transitions.CavesMainToRace)),
    Transitions.CavesMainToCastle: ShufflableExit("Crystal Caves Main to Ice Castle", Regions.CrystalCavesMain, TransitionBack(Regions.FrozenCastle, "From Caves Main", "Crystal Caves Ice Castle", Transitions.CavesCastleToMain), ExitCategories.CavesExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}),
    Transitions.CavesCastleToMain: ShufflableExit("Crystal Caves Ice Castle to Main", Regions.FrozenCastle, TransitionBack(Regions.CrystalCavesMain, "From Tile Flipping", "Crystal Caves Main from Ice Castle", Transitions.CavesMainToCastle)),
    Transitions.CavesIglooToDonkey: ShufflableExit("Crystal Caves Main to DK 5DIgloo", Regions.IglooArea, TransitionBack(Regions.DonkeyIgloo, "From Caves Main", "Crystal Caves DK 5DIgloo", Transitions.CavesDonkeyToIgloo), ExitCategories.CavesExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),
    Transitions.CavesDonkeyToIgloo: ShufflableExit("Crystal Caves DK 5DIgloo to Main", Regions.DonkeyIgloo, TransitionBack(Regions.IglooArea, "From DK 5DIgloo", "Crystal Caves Main from DK5DIgloo", Transitions.CavesIglooToDonkey)),
    Transitions.CavesIglooToDiddy: ShufflableExit("Crystal Caves Main to Diddy 5DIgloo", Regions.IglooArea, TransitionBack(Regions.DiddyIgloo, "From Caves Main", "Crystal Caves Diddy 5DIgloo", Transitions.CavesDiddyToIgloo), ExitCategories.CavesExterior, entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}, move=True),
    Transitions.CavesDiddyToIgloo: ShufflableExit("Crystal Caves Diddy 5DIgloo to Main", Regions.DiddyIgloo, TransitionBack(Regions.IglooArea, "From Diddy 5DIgloo", "Crystal Caves Main from Diddy 5DIgloo", Transitions.CavesIglooToDiddy)),
    Transitions.CavesIglooToLanky: ShufflableExit("Crystal Caves Main to Lanky 5DIgloo", Regions.IglooArea, TransitionBack(Regions.LankyIgloo, "From Caves Main", "Crystal Caves Lanky 5DIgloo", Transitions.CavesLankyToIgloo), ExitCategories.CavesExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}, move=True),
    Transitions.CavesLankyToIgloo: ShufflableExit("Crystal Caves Lanky 5DIgloo to Main", Regions.LankyIgloo, TransitionBack(Regions.IglooArea, "From Lanky 5DIgloo", "Crystal Caves Main from Lanky 5DIgloo", Transitions.CavesIglooToLanky)),
    Transitions.CavesIglooToTiny: ShufflableExit("Crystal Caves Main to Tiny 5DIgloo", Regions.IglooArea, TransitionBack(Regions.TinyIgloo, "From Caves Main", "Crystal Caves Tiny 5DIgloo", Transitions.CavesTinyToIgloo), ExitCategories.CavesExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.CavesTinyToIgloo: ShufflableExit("Crystal Caves Tiny 5DIgloo to Main", Regions.TinyIgloo, TransitionBack(Regions.IglooArea, "From Tiny 5DIgloo", "Crystal Caves Main from Tiny 5DIgloo", Transitions.CavesIglooToTiny)),
    Transitions.CavesIglooToChunky: ShufflableExit("Crystal Caves Main to Chunky 5DIgloo", Regions.IglooArea, TransitionBack(Regions.ChunkyIgloo, "From Caves Main", "Crystal Caves Chunky 5DIgloo", Transitions.CavesChunkyToIgloo), ExitCategories.CavesExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}, move=True),
    Transitions.CavesChunkyToIgloo: ShufflableExit("Crystal Caves Chunky 5DIgloo to Main", Regions.ChunkyIgloo, TransitionBack(Regions.IglooArea, "From Chunky 5DIgloo", "Crystal Caves Main from Chunky 5DIgloo", Transitions.CavesIglooToChunky)),
    Transitions.CavesCabinToRotating: ShufflableExit("Crystal Caves Main to Rotating Room", Regions.CabinArea, TransitionBack(Regions.RotatingCabin, "From Caves Main", "Crystal Caves Rotating Room", Transitions.CavesRotatingToCabin), ExitCategories.CavesExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),
    Transitions.CavesRotatingToCabin: ShufflableExit("Crystal Caves Rotating Room to Main", Regions.RotatingCabin, TransitionBack(Regions.CabinArea, "From Rotating Cabin", "Crystal Caves Main from Rotating Room", Transitions.CavesCabinToRotating)),
    Transitions.CavesCabinToDonkey: ShufflableExit("Crystal Caves Main to DK 5DCabin", Regions.CabinArea, TransitionBack(Regions.DonkeyCabin, "From Caves Main", "Crystal Caves DK 5DCabin", Transitions.CavesDonkeyToCabin), ExitCategories.CavesExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),
    Transitions.CavesDonkeyToCabin: ShufflableExit("Crystal Caves DK 5DCabin to Main", Regions.DonkeyCabin, TransitionBack(Regions.CabinArea, "From DK 5DCabin", "Crystal Caves Main from DK 5DCabin", Transitions.CavesCabinToDonkey)),
    Transitions.CavesCabinToDiddyLower: ShufflableExit("Crystal Caves Main to Diddy Lower 5DCabin", Regions.CabinArea, TransitionBack(Regions.DiddyLowerCabin, "From Caves Main", "Crystal Caves Diddy Lower 5DCabin", Transitions.CavesDiddyLowerToCabin), ExitCategories.CavesExterior, entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}, move=True),
    Transitions.CavesDiddyLowerToCabin: ShufflableExit("Crystal Caves Diddy Lower 5DCabin to Main", Regions.DiddyLowerCabin, TransitionBack(Regions.CabinArea, "From Diddy Lower 5DCabin", "Crystal Caves Main from Diddy Lower 5DCabin", Transitions.CavesCabinToDiddyLower)),
    Transitions.CavesCabinToDiddyUpper: ShufflableExit("Crystal Caves Main to Diddy Upper 5DCabin", Regions.CabinArea, TransitionBack(Regions.DiddyUpperCabin, "From Caves Main", "Crystal Caves Diddy Upper 5DCabin", Transitions.CavesDiddyUpperToCabin), ExitCategories.CavesExterior, entryKongs={Kongs.diddy}, regionKongs={Kongs.diddy}, move=True),
    Transitions.CavesDiddyUpperToCabin: ShufflableExit("Crystal Caves Diddy Upper 5DCabin to Main", Regions.DiddyUpperCabin, TransitionBack(Regions.CabinArea, "From Diddy Upper 5DCabin", "Crystal Caves Main from Diddy Upper 5DCabin", Transitions.CavesCabinToDiddyUpper)),
    Transitions.CavesCabinToLanky: ShufflableExit("Crystal Caves Main to Lanky 1DCabin", Regions.CabinArea, TransitionBack(Regions.LankyCabin, "From Caves Main", "Crystal Caves Lanky 1DCabin", Transitions.CavesLankyToCabin), ExitCategories.CavesExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}, move=True),
    Transitions.CavesLankyToCabin: ShufflableExit("Crystal Caves Lanky 1DCabin to Main", Regions.LankyCabin, TransitionBack(Regions.CabinArea, "From Lanky Cabin", "Crystal Caves Main from Lanky 1DCabin", Transitions.CavesCabinToLanky)),
    Transitions.CavesCabinToTiny: ShufflableExit("Crystal Caves Main to Tiny 5DCabin", Regions.CabinArea, TransitionBack(Regions.TinyCabin, "From Caves Main", "Crystal Caves Tiny 5DCabin", Transitions.CavesTinyToCabin), ExitCategories.CavesExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.CavesTinyToCabin: ShufflableExit("Crystal Caves Tiny 5DCabin to Main", Regions.TinyCabin, TransitionBack(Regions.CabinArea, "From Tiny 5DCabin", "Crystal Caves Main from Tiny 5DCabin", Transitions.CavesCabinToTiny)),
    Transitions.CavesCabinToChunky: ShufflableExit("Crystal Caves Main to Chunky 5DCabin", Regions.CabinArea, TransitionBack(Regions.ChunkyCabin, "From Caves Main", "Crystal Caves Chunky 5DCabin", Transitions.CavesChunkyToCabin), ExitCategories.CavesExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}, move=True),
    Transitions.CavesChunkyToCabin: ShufflableExit("Crystal Caves Chunky 5DCabin to Main", Regions.ChunkyCabin, TransitionBack(Regions.CabinArea, "From Chunky 5DCabin", "Crystal Caves Main from Chunky 5DCabin", Transitions.CavesCabinToChunky)),
    # Transitions.CavesMainToBBlast: ShufflableExit("Crystal Caves Main to Baboon Blast", Regions.CrystalCavesMain, TransitionBack(Regions.CavesBaboonBlast, "From Caves Main", "Crystal Caves Baboon Blast"), ExitCategories.CavesExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),

    # Creepy Castle Exits
    # Transitions.CastleMainToBBlast: ShufflableExit("Creepy Castle Main to Baboon Blast", Regions.CreepyCastleMain, TransitionBack(Regions.CastleBaboonBlast, "From Castle Main", "Creepy Castle Baboon Blast"), ExitCategories.CastleExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}, move=True),
    Transitions.CastleMainToTree: ShufflableExit("Creepy Castle Main to Tree", Regions.CreepyCastleMain, TransitionBack(Regions.CastleTree, "From Castle Main", "Creepy Castle Tree (Entrance)", Transitions.CastleTreeToMain), ExitCategories.CastleExterior, regionKongs={Kongs.donkey, Kongs.chunky}),
    Transitions.CastleTreeToMain: ShufflableExit("Creepy Castle Tree to Main", Regions.CastleTree, TransitionBack(Regions.CreepyCastleMain, "From Tree (Entrance)", "Creepy Castle Main from Tree (Entrance)", Transitions.CastleMainToTree)),
    Transitions.CastleTreeDrainToMain: ShufflableExit("Creepy Castle Tree Drain to Main", Regions.CastleTree, TransitionBack(Regions.CreepyCastleMain, "From Tree (Drain)", "Creepy Castle Main from Tree (Drain)"), entryKongs={Kongs.donkey}),
    Transitions.CastleMainToLibraryStart: ShufflableExit("Creepy Castle Main to Library (Entrance)", Regions.CreepyCastleMain, TransitionBack(Regions.Library, "From Castle Main (Start)", "Creepy Castle Library (Entrance)", Transitions.CastleLibraryStartToMain), ExitCategories.CastleExterior, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}),
    Transitions.CastleLibraryStartToMain: ShufflableExit("Creepy Castle Library (Entrance) to Main", Regions.Library, TransitionBack(Regions.CreepyCastleMain, "From Library (Start)", "Creepy Castle Main from Library (Entrance)", Transitions.CastleMainToLibraryStart)),
    Transitions.CastleMainToLibraryEnd: ShufflableExit("Creepy Castle Main to Library (Exit)", Regions.CreepyCastleMain, TransitionBack(Regions.LibraryPastBooks, "From Castle Main (End)", "Creepy Castle Library (Exit)"), ExitCategories.CastleExterior),
    Transitions.CastleLibraryEndToMain: ShufflableExit("Creepy Castle Library (Exit) to Main", Regions.LibraryPastBooks, TransitionBack(Regions.CreepyCastleMain, "From Library (End)", "Creepy Castle Main from Library (Exit)"), entryKongs={Kongs.donkey}),
    Transitions.CastleMainToBallroom: ShufflableExit("Creepy Castle Main to Ballroom", Regions.CreepyCastleMain, TransitionBack(Regions.Ballroom, "From Castle Main", "Creepy Castle Ballroom (Entrance)", Transitions.CastleBallroomToMain), ExitCategories.CastleExterior, regionKongs={Kongs.diddy, Kongs.tiny}),
    Transitions.CastleBallroomToMain: ShufflableExit("Creepy Castle Ballroom to Main", Regions.Ballroom, TransitionBack(Regions.CreepyCastleMain, "From Ballroom", "Creepy Castle Main from Ballroom", Transitions.CastleMainToBallroom), ExitCategories.CastleBallroom),
    Transitions.CastleMainToTower: ShufflableExit("Creepy Castle Main to Tower", Regions.CreepyCastleMain, TransitionBack(Regions.Tower, "From Castle Main", "Creepy Castle Tower", Transitions.CastleTowerToMain), ExitCategories.CastleExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}),
    Transitions.CastleTowerToMain: ShufflableExit("Creepy Castle Tower to Main", Regions.Tower, TransitionBack(Regions.CreepyCastleMain, "From Tower", "Creepy Castle Main from Tower", Transitions.CastleMainToTower)),
    Transitions.CastleMainToGreenhouse: ShufflableExit("Creepy Castle Main to Greenhouse", Regions.CreepyCastleMain, TransitionBack(Regions.Greenhouse, "From Castle Main", "Creepy Castle Greenhouse (Entrance)", Transitions.CastleGreenhouseStartToMain), ExitCategories.CastleExterior, entryKongs={Kongs.lanky}, regionKongs={Kongs.lanky}),
    Transitions.CastleGreenhouseStartToMain: ShufflableExit("Creepy Castle Greenhouse (Entrance) to Main", Regions.Greenhouse, TransitionBack(Regions.CreepyCastleMain, "From Greenhouse (Start)", "Creepy Castle Main from Greenhouse (Entrance)", Transitions.CastleMainToGreenhouse)),
    Transitions.CastleGreenhouseEndToMain: ShufflableExit("Creepy Castle Greenhouse (Exit) to Main", Regions.Greenhouse, TransitionBack(Regions.CreepyCastleMain, "From Greenhouse (End)", "Creepy Castle Main from Greenhouse (Exit)"), entryKongs={Kongs.lanky}),
    Transitions.CastleMainToTrash: ShufflableExit("Creepy Castle Main to Trash Can", Regions.CreepyCastleMain, TransitionBack(Regions.TrashCan, "From Castle Main", "Creepy Castle Trash Can", Transitions.CastleTrashToMain), ExitCategories.CastleExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}, move=True),
    Transitions.CastleTrashToMain: ShufflableExit("Creepy Castle Trash Can to Main", Regions.TrashCan, TransitionBack(Regions.CreepyCastleMain, "From Trash Can", "Creepy Castle Main from Trash Can", Transitions.CastleMainToTrash)),
    Transitions.CastleMainToShed: ShufflableExit("Creepy Castle Main to Shed", Regions.CreepyCastleMain, TransitionBack(Regions.Shed, "From Castle Main", "Creepy Castle Shed", Transitions.CastleShedToMain), ExitCategories.CastleExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}, move=True),
    Transitions.CastleShedToMain: ShufflableExit("Creepy Castle Shed to Main", Regions.Shed, TransitionBack(Regions.CreepyCastleMain, "From Shed", "Creepy Castle Main from Shed", Transitions.CastleMainToShed)),
    Transitions.CastleMainToMuseum: ShufflableExit("Creepy Castle Main to Museum", Regions.CreepyCastleMain, TransitionBack(Regions.Museum, "From Castle Main", "Creepy Castle Museum (Entrance)", Transitions.CastleMuseumToMain), ExitCategories.CastleExterior, entryKongs={Kongs.chunky}, regionKongs={Kongs.chunky}),
    Transitions.CastleMuseumToMain: ShufflableExit("Creepy Castle Museum to Main", Regions.Museum, TransitionBack(Regions.CreepyCastleMain, "From Museum", "Creepy Castle Main from Museum", Transitions.CastleMainToMuseum)),
    Transitions.CastleMainToLower: ShufflableExit("Creepy Castle Main to Lower Cave", Regions.CastleVeryBottom, TransitionBack(Regions.LowerCave, "From Castle Main", "Creepy Castle Lower Cave from Main", Transitions.CastleLowerToMain), ExitCategories.CastleExterior),
    Transitions.CastleLowerToMain: ShufflableExit("Creepy Castle Lower Cave to Main", Regions.LowerCave, TransitionBack(Regions.CastleVeryBottom, "From Lower Cave", "Creepy Castle Main from Lower Cave", Transitions.CastleMainToLower), ExitCategories.CastleLower),
    Transitions.CastleMainToUpper: ShufflableExit("Creepy Castle Main to Upper Cave (Back door)", Regions.CreepyCastleMain, TransitionBack(Regions.UpperCave, "From Castle (Rear)", "Creepy Castle Upper Cave from Main (Back door)", Transitions.CastleUpperToMain), ExitCategories.CastleExterior),
    Transitions.CastleUpperToMain: ShufflableExit("Creepy Castle Upper Cave to Main (Back door)", Regions.UpperCave, TransitionBack(Regions.CreepyCastleMain, "From Tunnel (Rear)", "Creepy Castle Main from Upper Cave (Back door)", Transitions.CastleMainToUpper), ExitCategories.CastleUpper),
    Transitions.CastleWaterfallToUpper: ShufflableExit("Creepy Castle Main to Upper Cave (Moat door)", Regions.CastleWaterfall, TransitionBack(Regions.UpperCave, "From Castle (Front)", "Creepy Castle Upper Cave from Main (Moat door)", Transitions.CastleUpperToWaterfall), ExitCategories.CastleExterior),
    Transitions.CastleUpperToWaterfall: ShufflableExit("Creepy Castle Upper Cave to Main (Moat door)", Regions.UpperCave, TransitionBack(Regions.CreepyCastleMain, "From Tunnel (Front)", "Creepy Castle Main from Upper Cave (Moat door)", Transitions.CastleWaterfallToUpper), ExitCategories.CastleUpper),
    Transitions.CastleBallroomToMuseum: ShufflableExit("Creepy Castle Ballroom to Museum", Regions.Ballroom, TransitionBack(Regions.MuseumBehindGlass, "From Ballroom (Monkeyport)", "Creepy Castle Museum (Monkeyport)", Transitions.CastleMuseumToBallroom), ExitCategories.CastleBallroom, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}),
    Transitions.CastleMuseumToBallroom: ShufflableExit("Creepy Castle Museum to Ballroom", Regions.MuseumBehindGlass, TransitionBack(Regions.Ballroom, "From Museum (Monkeyport)", "Creepy Castle Ballroom (Monkeyport)", Transitions.CastleBallroomToMuseum), ExitCategories.CastleCarRaceExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}),
    Transitions.CastleMuseumToCarRace: ShufflableExit("Creepy Castle Museum to Car Race", Regions.MuseumBehindGlass, TransitionBack(Regions.CastleTinyRace, "From Museum", "Creepy Castle Car Race", Transitions.CastleRaceToMuseum), ExitCategories.CastleCarRaceExterior, entryKongs={Kongs.tiny}, regionKongs={Kongs.tiny}),
    Transitions.CastleRaceToMuseum: ShufflableExit("Creepy Castle Car Race to Museum", Regions.CastleTinyRace, TransitionBack(Regions.MuseumBehindGlass, "From Car Race", "Creepy Castle Museum from Car Race", Transitions.CastleMuseumToCarRace), regionKongs={Kongs.tiny}),
    Transitions.CastleLowerToCrypt: ShufflableExit("Creepy Castle Lower Cave to Crypt", Regions.LowerCave, TransitionBack(Regions.Crypt, "From Lower Cave", "Creepy Castle Crypt (Entrance)", Transitions.CastleCryptToLower), ExitCategories.CastleLower, entryKongs={Kongs.donkey, Kongs.diddy, Kongs.chunky}, regionKongs={Kongs.donkey, Kongs.diddy, Kongs.chunky}),
    Transitions.CastleCryptToLower: ShufflableExit("Creepy Castle Crypt to Lower Cave", Regions.Crypt, TransitionBack(Regions.LowerCave, "From Crypt (DK/Diddy/Chunky)", "Creepy Castle Lower Cave from Crypt", Transitions.CastleLowerToCrypt), ExitCategories.CastleCrypt),
    Transitions.CastleLowerToMausoleum: ShufflableExit("Creepy Castle Lower Cave to Mausoleum", Regions.LowerCave, TransitionBack(Regions.Mausoleum, "From Lower Cave", "Creepy Castle Mausoleum", Transitions.CastleMausoleumToLower), ExitCategories.CastleLower, entryKongs={Kongs.lanky, Kongs.tiny}, regionKongs={Kongs.lanky, Kongs.tiny}),
    Transitions.CastleMausoleumToLower: ShufflableExit("Creepy Castle Mausoleum to Lower cave", Regions.Mausoleum, TransitionBack(Regions.LowerCave, "From Mausoleum (Lanky/Tiny)", "Creepy Castle Lower Cave from Mausoleum", Transitions.CastleLowerToMausoleum)),
    Transitions.CastleCryptToCarts: ShufflableExit("Creepy Castle Crypt to Minecart", Regions.CryptDonkeyRoom, TransitionBack(Regions.CastleMinecarts, "From Crypt", "Creepy Castle Minecart", Transitions.CastleCartsToCrypt), ExitCategories.CastleCrypt, entryKongs={Kongs.donkey}, regionKongs={Kongs.donkey}),
    Transitions.CastleCartsToCrypt: ShufflableExit("Creepy Castle Minecart to Crypt", Regions.CastleMinecarts, TransitionBack(Regions.CryptDonkeyRoom, "From Minecart", "Creepy Castle Crypt from Minecart", Transitions.CastleCryptToCarts)),
    Transitions.CastleUpperToDungeon: ShufflableExit("Creepy Castle Upper Cave to Dungeon", Regions.UpperCave, TransitionBack(Regions.Dungeon, "From Tunnel", "Creepy Castle Dungeon", Transitions.CastleDungeonToUpper), ExitCategories.CastleUpper),
    Transitions.CastleDungeonToUpper: ShufflableExit("Creepy Castle Dungeon to Upper Cave", Regions.Dungeon, TransitionBack(Regions.UpperCave, "From Dungeon", "Creepy Castle Upper Cave from Dungeon", Transitions.CastleUpperToDungeon)),
}


def GetShuffledLevelIndex(level):
    """Get index of where the given level fits in the level order. Ex: If Caves is the 1st level, passing 5 will return 0."""
    lobbyExit = ShufflableExits[LevelInfoList[level].TransitionsFrom].shuffledId
    if lobbyExit is not None:
        levelIndex = [key for key, item in LevelInfoList.items() if item.TransitionsFrom == lobbyExit][0]
    else:
        levelIndex = level
    return levelIndex


def GetLevelShuffledToIndex(levelIndex):
    """Get level located at the given level index. Ex: If Caves is the 1st level, passing 0 will return 5."""
    lobbyEntrance = ShufflableExits[LevelInfoList[levelIndex].TransitionTo].shuffledId
    if lobbyEntrance is not None:
        level = [key for key, item in LevelInfoList.items() if item.TransitionTo == lobbyEntrance][0]
    else:
        level = levelIndex
    return level
