import struct
from .options import KirbyFlavorPreset
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from . import K64World
    from .rom import K64ProcedurePatch

kirby_flavor_presets = {
    1: {
        "1": "080000",
        "2": "FE67AC",
        "3": "962656",
        "4": "FCFFFF",
        "5": "FF99C1",
        "6": "DD2638",
        "7": "0A1437",
        "8": "D55E86",
        "9": "530011",
        "10": "D8E6FF",
        "11": "0D111D",
        "12": "F2C6D7",
        "13": "1A44D6",
        "14": "9D4B7C",
        "15": "C66A99",
        "16": "5D0F1C",
        "17": "F12153"
    },
    2: {
        "1": "080000",  
        "2": "FF00A5",  
        "3": "6B0808",  
        "4": "F6F6F6",  
        "5": "FF0039",  
        "6": "6B0808",  
        "7": "00085A",  
        "8": "DE0029",  
        "9": "B50029",  
        "10": "D5D5D5",  
        "11": "4A0000",  
        "12": "888888",  
        "13": "0000D5",  
        "14": "940808",  
        "15": "B50818",  
        "16": "B50029",  
        "17": "730000",  
    },
    3: {
        "0": "FFB5B5",
        "1": "000800",
        "2": "FF7384",
        "3": "213A84",
        "4": "F7F7F7",
        "5": "94B5FF",
        "6": "EF6B7B",
        "7": "00085A",
        "8": "7B94DE",
        "9": "294A94",
        "10": "D6D6D6",
        "11": "312129",
        "12": "8C949C",
        "13": "0000D6",
        "14": "6384CE",
        "15": "4A6BCE",
        "16": "D9CDFF",
        "17": "3333B3"
    },
    4: {
        "0": "FFB5B5",
        "1": "000800",
        "2": "FF734A",
        "3": "0945A0",
        "4": "F7F7F7",
        "5": "FFD673",
        "6": "FF6342",
        "7": "00085A",
        "8": "F7AD4A",
        "9": "423173",
        "10": "D6D6D6",
        "11": "312129",
        "12": "8C949C",
        "13": "0000D6",
        "14": "BD7321",
        "15": "DE9C31",
        "16": "FFF4C2",
        "17": "FF8000"
    },
    5: {
        "0": "FFB5B5",
        "1": "000800",
        "2": "FF8473",
        "3": "3A6321",
        "4": "F7F7F7",
        "5": "94D66B",
        "6": "EF7B5A",
        "7": "00085A",
        "8": "84C55A",
        "9": "423173",
        "10": "D6D6D6",
        "11": "312129",
        "12": "8C949C",
        "13": "0000D6",
        "14": "63A53A",
        "15": "5A8C42",
        "16": "B6FCBE",
        "17": "665C33"
    },
    6: {
        "1": "080000",
        "2": "B54184",
        "3": "63289C",
        "4": "F6F6F6",
        "5": "B59AFF",
        "6": "A3417A",
        "7": "271438",
        "8": "A571FF",
        "9": "600052",
        "10": "D5D5D5",
        "11": "4A206B",
        "12": "8B949C",
        "13": "522E6B",
        "14": "7308D6",
        "15": "8C51FF",
        "16": "600052",
        "17": "94206B"
    },
    7: {
        "1": "573822",
        "2": "4A3120",
        "3": "7A5030",
        "4": "FFFFFF",
        "5": "BD835A",
        "6": "382518",
        "7": "7A5030",
        "8": "AB7145",
        "9": "1E130D",
        "10": "694529",
        "11": "694529",
        "12": "573822",
        "13": "8D5C38",
        "14": "8D5C38",
        "15": "080000",
        "16": "1E130D",
        "17": "4A3120"
    },
    8: {
        "1": "525152",
        "2": "DEDBDE",
        "3": "949294",
        "4": "FFFFFF",
        "5": "FFFBFF",
        "6": "BDBABD",
        "7": "949294",
        "8": "DEDBDE",
        "9": "949294",
        "10": "737173",
        "11": "737173",
        "12": "525152",
        "13": "BDBABD",
        "14": "BDBABD",
        "15": "000000",
        "16": "949294",
        "17": "DEDBDE"
    },
    9: {
        "1": "080000",  
        "2": "930000",  
        "3": "464142",  
        "4": "F6F6F6",  
        "5": "353535",  
        "6": "5D0000",  
        "7": "930000",  
        "8": "212121",  
        "9": "6D0000",  
        "10": "D5D5D5",  
        "11": "292829",  
        "12": "888888",  
        "13": "D50000",  
        "14": "2A2A2A",  
        "15": "828282",  
        "16": "6D0000",  
        "17": "800000",  
    },
    10: {
        "1": "080000",  
        "2": "126018",  
        "3": "C84F6B",  
        "4": "F6F6F6",  
        "5": "EF8A9D",  
        "6": "126018",  
        "7": "00085A",  
        "8": "E77B8D",  
        "9": "48953F",  
        "10": "D5D5D5",  
        "11": "B74F54",  
        "12": "888888",  
        "13": "0000D5",  
        "14": "DF5C68",  
        "15": "D85F6F",  
        "16": "48953F",  
        "17": "168025",  
    },
    11: {
        "1": "842810",
        "2": "FF4900",
        "3": "FF5929",
        "4": "FFFFFF",
        "5": "FFB218",
        "6": "C63000",
        "7": "FF5929",
        "8": "FF9A18",
        "9": "842810",
        "10": "C64118",
        "11": "C64118",
        "12": "842810",
        "13": "FF7921",
        "14": "FF7921",
        "15": "000000",
        "16": "842810",
        "17": "FF4900"
    },
    12: {
        "1": "080000",
        "2": "949400",
        "3": "3e6806",
        "4": "F6F6F6",
        "5": "96f11d",
        "6": "b8b800",
        "7": "2d2d2d",
        "8": "86e10e",
        "9": "525252",
        "10": "D5D5D5",
        "11": "1e3203",
        "12": "949494",
        "13": "6b6b6b",
        "14": "599509",
        "15": "77c70c",
        "16": "ffff00",
        "17": "747400",
    },
    13: {
        "1": "000008",
        "2": "5A4AF6",
        "3": "52416A",
        "4": "F6F6F6",
        "5": "B4A4F6",
        "6": "626AE6",
        "7": "5A0800",
        "8": "A48BEE",
        "9": "733141",
        "10": "D5D5D5",
        "11": "292031",
        "12": "9C948B",
        "13": "D50000",
        "14": "6A628B",
        "15": "8B83BD",
        "16": "4669FF",
        "17": "781CFF",
    },
    14: {
        "1": "000008",
        "2": "E6298A",
        "3": "008B8F",
        "4": "C4CED4",
        "5": "75C7C1",
        "6": "E84053",
        "7": "04292B",
        "8": "3AA6A4",
        "9": "430B1D",
        "10": "BEC9D0",
        "11": "121415",
        "12": "2F3436",
        "13": "5DCCD9",
        "14": "00ADC9",
        "15": "2EB8C6",
        "16": "FBDDCD",
        "17": "494E59",
    },
    15: {
        "1": "234331",
        "10": "D3E29A",
        "11": "234331",
        "12": "537A3E",
        "13": "234331",
        "14": "537A3E",
        "15": "537A3E",
        "16": "A7BA4A",
        "17": "234331",
        "2": "537A3E",
        "3": "234331",
        "4": "D3E29A",
        "5": "A7BA4A",
        "6": "537A3E",
        "7": "234331",
        "8": "A7BA4A",
        "9": "234331"
    },
    16: {
        "1": "080000",
        "2": "F64A5A",
        "3": "005B18",
        "4": "F6F6F6",
        "5": "45EF64",
        "6": "E66A62",
        "7": "00085A",
        "8": "45EF64",
        "9": "640A00",
        "10": "D5D5D5",
        "11": "311F29",
        "12": "8B949C",
        "13": "0000D5",
        "14": "8B626A",
        "15": "45EF64",
        "16": "FFAC8F",
        "17": "005B18",
    },
    17: {
        "1": "080000",
        "2": "ff9a08",
        "3": "464142",
        "4": "F6F6F6",
        "5": "636163",
        "6": "ff7900",
        "7": "232323",
        "8": "726350",
        "9": "603f29",
        "10": "D5D5D5",
        "11": "292829",
        "12": "8B949C",
        "13": "2f2f2f",
        "14": "603f29",
        "15": "886068",
        "16": "ff6946",
        "17": "FF7900"
    },
    18: {
        "1": "383738",
        "2": "FF4963",
        "3": "737173",
        "4": "FFFBFF",
        "5": "FFFBFF",
        "6": "D62042",
        "7": "949294",
        "8": "BDBABD",
        "9": "B50029",
        "10": "737173",
        "11": "525152",
        "12": "525152",
        "13": "BDBABD",
        "14": "949294",
        "15": "080000",
        "16": "B50029",
        "17": "FF3056"
    },
    19: {
        "1": "00207B",
        "2": "5200BD",
        "3": "0051B5",
        "4": "FFFFFF",
        "5": "7B9AEF",
        "6": "420094",
        "7": "0051B5",
        "8": "6382DE",
        "9": "39006B",
        "10": "00389C",
        "11": "00389C",
        "12": "00207B",
        "13": "5271CE",
        "14": "5271CE",
        "15": "080000",
        "16": "39006B",
        "17": "5200BD"
    },
    20: {
        "1": "080000",
        "2": "FFA24A",
        "3": "006952",
        "4": "F6F6F6",
        "5": "73FBBD",
        "6": "E59B42",
        "7": "003533",
        "8": "5EDBA5",
        "9": "004C4C",
        "10": "D5D5D5",
        "11": "005139",
        "12": "8B949C",
        "13": "004C4C",
        "14": "008263",
        "15": "00A26B",
        "16": "004C4C",
        "17": "DE7921"
    },
    21: {
        "1": "080000",
        "2": "C87018",
        "3": "684008",
        "4": "F6F6F6",
        "5": "F8E8B8",
        "6": "B56F29",
        "7": "492D22",
        "8": "E0C8A0",
        "9": "4C291D",
        "10": "D5D5D5",
        "11": "602810",
        "12": "8B949C",
        "13": "916F47",
        "14": "B08860",
        "15": "C8A880",
        "16": "4C291D",
        "17": "C87018"
    },
    22: {
        "1": "3B4551",
        "2": "FAB0B0",
        "3": "745D51",
        "4": "F6F6F6",
        "5": "FBEFAF",
        "6": "EAA49F",
        "7": "5786A0",
        "8": "E5D885",
        "9": "774847",
        "10": "D5D5D5",
        "11": "595D51",
        "12": "8B949C",
        "13": "85C3D2",
        "14": "8C7C70",
        "15": "B78F80",
        "16": "774847",
        "17": "96BECE"
    },
    23: {
        "1": "002020",
        "2": "68C020",
        "3": "389040",
        "4": "F6F6F6",
        "5": "F8F850",
        "6": "78E810",
        "7": "1A3F3B",
        "8": "D0F038",
        "9": "3E5B3E",
        "10": "D5D5D5",
        "11": "006860",
        "12": "8B949C",
        "13": "56AF5D",
        "14": "68C020",
        "15": "A0E800",
        "16": "3E5B3E",
        "17": "10C010"
    },
    24: {
        "1": "393839",
        "2": "FF5050",
        "3": "996666",
        "4": "FFFFFF",
        "5": "EFD6C6",
        "6": "CC3333",
        "7": "996666",
        "8": "CC9999",
        "9": "993333",
        "10": "663333",
        "11": "663333",
        "12": "393839",
        "13": "CC9999",
        "14": "CC9999",
        "15": "080000",
        "16": "993333",
        "17": "FF5050"
    },
    25: {
        "1": "3333FF",
        "2": "CC33FF",
        "3": "3333FF",
        "4": "CCFFFF",
        "5": "CCFFFF",
        "6": "9933FF",
        "7": "3366FF",
        "8": "99CCFF",
        "9": "6E24B7",
        "10": "737173",
        "11": "2020A0",
        "12": "525152",
        "13": "6699FF",
        "14": "3366FF",
        "15": "080000",
        "16": "6E24B7",
        "17": "CC33FF"
    },
    26: {
        "1": "1A4267",
        "2": "FFC859",
        "3": "1A4267",
        "4": "FFFFFF",
        "5": "F6E2C0",
        "6": "CC9F47",
        "7": "1F68A8",
        "8": "D9C8A9",
        "9": "CC8A10",
        "10": "737173",
        "11": "000000",
        "12": "525152",
        "13": "3E9CDA",
        "14": "1F68A8",
        "15": "080000",
        "16": "CC8A10",
        "17": "EF74AE"
    },
    27: {
        "1": "760A0F",
        "2": "3EBD37",
        "3": "9D0D14",
        "4": "FFFFFF",
        "5": "EF3A44",
        "6": "2F8F29",
        "7": "9D0D14",
        "8": "ED1C24",
        "9": "277823",
        "10": "760A0F",
        "11": "760A0F",
        "12": "5A070C",
        "13": "ED1C24",
        "14": "ED1C24",
        "15": "080000",
        "16": "277823",
        "17": "3EBD37"
    },
    28: {
        "1": "525152",
        "2": "24DB28",
        "3": "949294",
        "4": "FFFFFF",
        "5": "FFFFFF",
        "6": "1CA61F",
        "7": "949294",
        "8": "BDBABD",
        "9": "157B17",
        "10": "737173",
        "11": "737173",
        "12": "525152",
        "13": "DEDBDE",
        "14": "DEDBDE",
        "15": "080000",
        "16": "157B17",
        "17": "24DB28"
    },
    29: {
        "1": "431760",
        "2": "82E12B",
        "3": "792BAE",
        "4": "FFFFFF",
        "5": "B77BDF",
        "6": "6BC41C",
        "7": "792BAE",
        "8": "A65BD7",
        "9": "4D8B14",
        "10": "62228C",
        "11": "62228C",
        "12": "431760",
        "13": "9B48D2",
        "14": "9B48D2",
        "15": "000000",
        "16": "4D8B14",
        "17": "82E12B"
    },
    30: {
        "1": "904800",
        "2": "00E6E6",
        "3": "D19000",
        "4": "FFFFFF",
        "5": "F7E84B",
        "6": "00B7B7",
        "7": "D19000",
        "8": "F5E623",
        "9": "D19000",
        "10": "9F7000",
        "11": "9F7000",
        "12": "904800",
        "13": "F7C412",
        "14": "F7C412",
        "15": "080000",
        "16": "D19000",
        "17": "00E6E6"
    },
    31: {
        "1": "590009",
        "2": "1562FF",
        "3": "AA0011",
        "4": "FFFFFF",
        "5": "FF3549",
        "6": "0048DF",
        "7": "AA0011",
        "8": "FF0921",
        "9": "0037AA",
        "10": "8A000E",
        "11": "8A000E",
        "12": "573822",
        "13": "D90015",
        "14": "D90015",
        "15": "080000",
        "16": "0037AA",
        "17": "1562FF"
    },
    32: {
        "1": "002C3C",
        "2": "FFD900",
        "3": "00668C",
        "4": "FFFFFF",
        "5": "22C2FF",
        "6": "D7B700",
        "7": "00668C",
        "8": "00B8FD",
        "9": "B09600",
        "10": "00506F",
        "11": "00506F",
        "12": "002C3C",
        "13": "00A0DD",
        "14": "00A0DD",
        "15": "080000",
        "16": "B09600",
        "17": "FFD900"
    },
    33: {
        "1": "091433",
        "2": "F87010",
        "3": "1A40A6",
        "4": "FFFFFF",
        "5": "28A0F8",
        "6": "E66506",
        "7": "1A40A6",
        "8": "078AEB",
        "9": "B86000",
        "10": "102765",
        "11": "102765",
        "12": "091433",
        "13": "2050D0",
        "14": "2050D0",
        "15": "000000",
        "16": "B86000",
        "17": "F87010"
    },
    34: {
        "1": "6B1502",
        "2": "F2CB6F",
        "3": "DF1D15",
        "4": "FFFFFF",
        "5": "FE5104",
        "6": "EA9557",
        "7": "DF1D15",
        "8": "FE5104",
        "9": "DD6645",
        "10": "9E2002",
        "11": "9E2002",
        "12": "6B1502",
        "13": "DF1D15",
        "14": "DF1D15",
        "15": "000000",
        "16": "DD6645",
        "17": "F2CB6F"
    },
    35: {
        "1": "3897D4",
        "2": "FF92B7",
        "3": "3897D4",
        "4": "8BFEFF",
        "5": "FDF9DD",
        "6": "FFC8CA",
        "7": "1DA4FF",
        "8": "D9C8A9",
        "9": "FC679B",
        "10": "737173",
        "11": "3F93C1",
        "12": "525152",
        "13": "2FE2F7",
        "14": "1DA4FF",
        "15": "080000",
        "16": "FC679B",
        "17": "EF74AE"
    },
    36: {
        "1": "615749",
        "2": "CB714D",
        "3": "C4A275",
        "4": "FFFFFF",
        "5": "FFFF94",
        "6": "AC5D29",
        "7": "C4A275",
        "8": "EFE68C",
        "9": "753F1B",
        "10": "7D705F",
        "11": "7D705F",
        "12": "615749",
        "13": "E0CF84",
        "14": "E0CF84",
        "15": "000000",
        "16": "753F1B",
        "17": "CB714D"
    },
    37: {
        "1": "0B450F",
        "2": "573502",
        "3": "0F6615",
        "4": "FFFFFF",
        "5": "1DB828",
        "6": "3F2601",
        "7": "0F6615",
        "8": "15911F",
        "9": "291A01",
        "10": "0E5613",
        "11": "0E5613",
        "12": "0B450F",
        "13": "117118",
        "14": "117118",
        "15": "000000",
        "16": "291A01",
        "17": "573502"
    },
    38: {
        "1": "E60EFE",
        "2": "CF01E7",
        "3": "ED54FE",
        "4": "FFFFFF",
        "5": "F295FF",
        "6": "B101C5",
        "7": "ED54FE",
        "8": "EE75FF",
        "9": "9801A9",
        "10": "EA38FE",
        "11": "EA38FE",
        "12": "E60EFE",
        "13": "EE63FE",
        "14": "EE63FE",
        "15": "000000",
        "16": "9801A9",
        "17": "CF01E7"
    },
    39: {
        "1": "50A800",
        "2": "469300",
        "3": "63D300",
        "4": "FFFFFF",
        "5": "A9FD71",
        "6": "3C7D00",
        "7": "63D300",
        "8": "7BFB29",
        "9": "346F00",
        "10": "5BBF00",
        "11": "5BBF00",
        "12": "50A800",
        "13": "6FE914",
        "14": "6FE914",
        "15": "000000",
        "16": "346F00",
        "17": "469300"
    },
    40: {
        "1": "A18595",
        "2": "FF53B6",
        "3": "CDA9BE",
        "4": "FFFFFF",
        "5": "FFD2EC",
        "6": "FF0D98",
        "7": "CDA9BE",
        "8": "EFC5DD",
        "9": "CE0077",
        "10": "BA99AC",
        "11": "BA99AC",
        "12": "A18595",
        "13": "E4BCD3",
        "14": "E4BCD3",
        "15": "080000",
        "16": "CE0077",
        "17": "FF53B6"
    },
    41: {
        "1": "72948B",
        "2": "09FFC1",
        "3": "A3D4C8",
        "4": "FFFFFF",
        "5": "C4FFF0",
        "6": "07CD9B",
        "7": "A3D4C8",
        "8": "B6EDDF",
        "9": "009D76",
        "10": "8EB9AE",
        "11": "8EB9AE",
        "12": "72948B",
        "13": "B0E5D8",
        "14": "B0E5D8",
        "15": "080000",
        "16": "009D76",
        "17": "09FFC1"
    },
    42: {
        "1": "29307B",
        "2": "6359FF",
        "3": "2979FF",
        "4": "FFFFFF",
        "5": "8CDBFF",
        "6": "4A41DE",
        "7": "2979FF",
        "8": "6BBAFF",
        "9": "3118B5",
        "10": "1061C6",
        "11": "1061C6",
        "12": "29307B",
        "13": "529AFF",
        "14": "529AFF",
        "15": "080000",
        "16": "3118B5",
        "17": "6359FF"
    },
    43: {
        "1": "592400",
        "2": "444444",
        "3": "A84300",
        "4": "FFFFFF",
        "5": "FF7F27",
        "6": "363636",
        "7": "A84300",
        "8": "FB6400",
        "9": "242424",
        "10": "803300",
        "11": "803300",
        "12": "592400",
        "13": "CC5200",
        "14": "CC5200",
        "15": "080000",
        "16": "242424",
        "17": "444444"
    },
    44: {
        "1": "313031",
        "2": "7B797B",
        "3": "737173",
        "4": "FFFFFF",
        "5": "DEDBDE",
        "6": "5A595A",
        "7": "737173",
        "8": "BDBABD",
        "9": "424142",
        "10": "525152",
        "11": "525152",
        "12": "313031",
        "13": "9C9A9C",
        "14": "9C9A9C",
        "15": "000000",
        "16": "424142",
        "17": "7B797B"
    },
    45: {
        "1": "525152",
        "2": "525152",
        "3": "949294",
        "4": "FFFFFF",
        "5": "FFFBFF",
        "6": "393839",
        "7": "949294",
        "8": "DEDBDE",
        "9": "292829",
        "10": "737173",
        "11": "737173",
        "12": "525152",
        "13": "BDBABD",
        "14": "BDBABD",
        "15": "000000",
        "16": "292829",
        "17": "525152"
    },
    46: {
        "1": "313031",
        "2": "525152",
        "3": "525152",
        "4": "FFFFFF",
        "5": "848284",
        "6": "393839",
        "7": "525152",
        "8": "737173",
        "9": "292829",
        "10": "424142",
        "11": "424142",
        "12": "313031",
        "13": "636163",
        "14": "636163",
        "15": "000000",
        "16": "292829",
        "17": "525152"
    },
    47: {
        "1": "4B0352",
        "2": "33035F",
        "3": "83058D",
        "4": "FFFFFF",
        "5": "CF09DF",
        "6": "270248",
        "7": "83058D",
        "8": "BB09CA",
        "9": "160129",
        "10": "6C0475",
        "11": "6C0475",
        "12": "4B0352",
        "13": "9B07A7",
        "14": "9B07A7",
        "15": "000000",
        "16": "160129",
        "17": "33035F"
    },
    48: {
        "1": "880044",
        "2": "00ACFF",
        "3": "F9007C",
        "4": "FFFFFF",
        "5": "FF93C9",
        "6": "008BCE",
        "7": "F9007C",
        "8": "FF80C0",
        "9": "0070A6",
        "10": "BF0060",
        "11": "BF0060",
        "12": "880044",
        "13": "FF3C9D",
        "14": "FF3C9D",
        "15": "000000",
        "16": "0070A6",
        "17": "00ACFF"
    },
    49: {
        "1": "633A21",
        "2": "98684D",
        "3": "593A25",
        "4": "F6F6F6",
        "5": "5FE4B5",
        "6": "5DC890",
        "7": "6F4B2C",
        "8": "56C699",
        "9": "633A21",
        "10": "D5D5D5",
        "11": "312029",
        "12": "8B949C",
        "13": "986B5D",
        "14": "8B626A",
        "15": "BD838B",
        "16": "FF6946",
        "17": "98684D",
    },
    50: {
        "1": "760A0F",
        "2":  "a1d501",
        "3":  "9D0D14",
        "4":  "FFFFFF",
        "5":  "faab3e",
        "6":  "336a06",
        "7":  "cf1704",
        "8":  "d58a21",
        "9":  "4ba205",
        "10": "760A0F",
        "11": "760A0F",
        "12": "5A070C",
        "13": "e81802",
        "14": "ED1C24",
        "15": "080000",
        "16": "4ba205",
        "17": "913b20",
    },
    51: {
        "1": "080000",
        "2": "FA004F",  
        "3": "DB9A02",  
        "4": "F6F6F6",
        "5": "FFCC11",  
        "6": "CE0041",  
        "7": "13464E",  
        "8": "EEAA00",  
        "9": "A50034",  
        "10": "D5D5D5",  
        "11": "C48A01",  
        "12": "888888",
        "13": "2D8182",  
        "14": "F0A902",  
        "15": "FFD708",  
        "16": "A50034",  
        "17": "B9B8BF",  
    },
    52: {
        "1": "080000",
        "2": "FA004F",  
        "3": "DB9A02",  
        "4": "F6F6F6",
        "5": "FFEE11",  
        "6": "CE0041",  
        "7": "13464E",  
        "8": "EEAA00",  
        "9": "A50034",  
        "10": "D5D5D5",  
        "11": "C48A01",  
        "12": "888888",
        "13": "2D8182",  
        "14": "F0A902",  
        "15": "FFD708",  
        "16": "A50034",  
        "17": "636267",  
    },
    53: {
        "1": "080000",
        "2": "E2C95B",         
        "3": "E99EC5",       
        "4": "F6F6F6",
        "5": "FFBBCC",       
        "6": "CFAD39",         
        "7": "1E3466",       
        "8": "A40020",       
        "9": "906C08",         
        "10": "D5D5D5",
        "11": "A37498",      
        "12": "888888",
        "13": "6E8BD1",        
        "14": "F0B2D0",        
        "15": "FED9EF",      
        "16": "906C08",        
        "17": "4A4D4C",        
    },
    54: {
        "1": "080000",
        "2": "B19369",  
        "3": "670007",  
        "4": "F6F6F6",
        "5": "DE4444",  
        "6": "7B5B42",  
        "7": "4C2D25",  
        "8": "E69ABD",  
        "9": "3F2819",  
        "10": "D5D5D5",
        "11": "5F0009",  
        "12": "888888",
        "13": "6B4F39",  
        "14": "71000B",  
        "15": "930014",  
        "16": "3F2819",  
        "17": "7B543B",  
    },
    55: {
        "1": "080000",
        "2": "FEE95E",  
        "3": "16359D",  
        "4": "F6F6F6",
        "5": "3366CC",  
        "6": "FCC950",  
        "7": "15357C",  
        "8": "1E40A2",  
        "9": "E9A04E",  
        "10": "D5D5D5",
        "11": "083097",  
        "12": "888888",
        "13": "3670C0",  
        "14": "2F58BE",  
        "15": "3D6FD2",  
        "16": "E9A04E",  
        "17": "FFFFFF",  
    },

}

kirby_target_palettes = [
    0x614818,
    0x859A06,
    0xA80832,
    0xB6B674,
    0xB6D8F4,
    0xDC7344,
    0xDD03A4,
    0xED0D14,
]

kirby_24bit_targets = [
    (0x5AF9C4, "5"),
    (0x5AFAD4, "5"),
    (0x5B3AF4, "5"),
    (0x5B3C0C, "5"),
    (0x7E285C, "5"),
    (0x7E29CC, "5"),
    (0x7E2B14, "5"),
    (0x7E2C8C, "5"),
    (0x7E5854, "5"),
    (0x7E5BB4, "5"),
    (0x7E5F94, "5"),
    (0x7E620C, "5"),
    (0x7E634C, "5"),
    (0x7E9C1C, "5"),
    (0x7E9D94, "5"),
    (0x7E9F04, "5"),
    (0x7EA074, "5"),
    (0x7F4E6C, "5"),
    (0x7F6A8C, "5"),
    (0x7F973C, "5"),
    (0x7F9904, "5"),
    (0x80410C, "5"),
    (0x80422C, "5"),
    (0x80AC8C, "5"),
    (0x80B19C, "5"),
    (0x80B2BC, "5"),
    (0x81519C, "5"),
    (0x8152BC, "5"),
    (0x815E2C, "5"),
    (0x8171BC, "5"),
    (0x828ED4, "5"),
    (0x828FF4, "5"),
    (0x848E04, "5"),
    (0x9C9BFC, "5"),
    (0x9CC898, "5"),
    (0x9CC910, "5"),
    (0xD27A54, "5"),
    (0xD27E2C, "5"),
    (0xD27FEC, "5"),
    (0xD28134, "5"),
    (0x7E5F4C, "16"),
    (0x7E6124, "16"),
    (0x817174, "16"),
    (0xD27DD4, "16"),
    (0x5AEE74, "17"),
    (0x5AEF84, "17"),
    (0x5B34B4, "17"),
    (0x5B35CC, "17"),
    (0x7E1A84, "17"),
    (0x7E1C44, "17"),
    (0x7E1D8C, "17"),
    (0x7E1F4C, "17"),
    (0x7E5374, "17"),
    (0x7E550C, "17"),
    (0x7E8E54, "17"),
    (0x7E9014, "17"),
    (0x7E9144, "17"),
    (0x7E9304, "17"),
    (0x7F520C, "17"),
    (0x7F5414, "17"),
    (0x7F6734, "17"),
    (0x7F68E4, "17"),
    (0x7F9884, "17"),
    (0x80374C, "17"),
    (0x803864, "17"),
    (0x80AA5C, "17"),
    (0x80AB74, "17"),
    (0x80DA9C, "17"),
    (0x8111EC, "17"),
    (0x811394, "17"),
    (0x812B3C, "17"),
    (0x814B34, "17"),
    (0x814C4C, "17"),
    (0x816CB4, "17"),
    (0x8287E4, "17"),
    (0x828984, "17"),
    (0x82AAA4, "17"),
    (0x82AC54, "17"),
    (0x848994, "17"),
    (0x97681C, "17"),
    (0x9769CC, "17"),
    (0x9C959C, "17"),
    (0x9CC6C8, "17"),
    (0x9CC740, "17"),
    (0xBE0B94, "17"),
    (0xBE0F74, "17"),
    (0xBE3DB4, "17"),
    (0xBE3FB4, "17"),
    (0xBE6004, "17"),
    (0xBE6164, "17"),
    (0xBE7A44, "17"),
    (0xBE7B34, "17"),
    (0xD27744, "17"),
    (0xD278CC, "17"),
]


def get_kirby_palette(world):
    palette = world.options.kirby_flavor_preset.value
    if palette == KirbyFlavorPreset.option_custom:
        return world.options.kirby_flavor.value
    return kirby_flavor_presets.get(palette, None)


def rgb888_to_rgba5551(red, green, blue) -> bytes:
    red = red >> 3
    green = green >> 3
    blue = blue >> 3
    outcol = (red << 11) + (green << 6) + (blue << 1) + 1
    return struct.pack(">H", outcol)


def get_palette_bytes(palette, target):
    output_data = bytearray()
    for color in target:
        hexcol = palette[color]
        if hexcol.startswith("#"):
            hexcol = hexcol.replace("#", "")
        colint = int(hexcol, 16)
        col = ((colint & 0xFF0000) >> 16, (colint & 0xFF00) >> 8, colint & 0xFF)
        byte_data = rgb888_to_rgba5551(col[0], col[1], col[2])
        output_data.extend(bytearray(byte_data))
    return output_data


def write_aesthetics(world: "K64World", patch: "K64ProcedurePatch"):
    if world.options.kirby_flavor_preset != world.options.kirby_flavor_preset.default:
        str_pal = get_kirby_palette(world)
        palette = get_palette_bytes(str_pal, [f"{i}" for i in range(1, 16)])
        for target in kirby_target_palettes:
            patch.write_bytes(target, palette)

        for addr, color in kirby_24bit_targets:
            hexcol = str_pal[color]
            if hexcol.startswith("#"):
                hexcol = hexcol.replace("#", "")
            colint = int(hexcol, 16)
            patch.write_bytes(addr, colint.to_bytes(3, "big"))
