from typing import Dict, List, Optional, Tuple
from enum import Enum
from BaseClasses import Location
from .Items import ItemData, Items

class ALBWLocation(Location):
    game = "A Link Between Worlds"

class LocationType(Enum):
    Normal = 0
    Maiamai = 1
    Ravio = 2
    Upgrade = 3
    Minigame = 4
    Prize = 5
    Shop = 6
    Event = 7
    Vane = 8

class LocationData:
    name: str
    code: Optional[int]
    course: Optional[int]
    flag: Optional[int]
    loctype: LocationType
    default_item: Optional[ItemData]

    def __init__(
        self,
        code: Optional[int],
        course: Optional[int],
        flag: Optional[int],
        name: str,
        loctype: LocationType = LocationType.Normal,
        default_item: Optional[ItemData] = None
    ):
        self.name = name
        self.code = code
        self.course = course
        self.flag = flag
        self.loctype = loctype
        self.default_item = default_item

Maiamai = LocationType.Maiamai
Ravio = LocationType.Ravio
Upgrade = LocationType.Upgrade
Minigame = LocationType.Minigame
Prize = LocationType.Prize
Shop = LocationType.Shop
Event = LocationType.Event
Vane = LocationType.Vane

hyrule_locations: List[LocationData] = [
    LocationData(0, None, 85, "Ravio's Gift"),
    LocationData(1, 2, 249, "Ravio's Shop (1)", Ravio),
    LocationData(2, 2, 251, "Ravio's Shop (2)", Ravio),
    LocationData(3, 2, 250, "Ravio's Shop (3)", Ravio),
    LocationData(4, 2, 242, "Ravio's Shop (4)", Ravio),
    LocationData(5, 2, 244, "Ravio's Shop (5)", Ravio),
    LocationData(6, 2, 247, "Ravio's Shop (6)", Ravio),
    LocationData(7, 2, 246, "Ravio's Shop (7)", Ravio),
    LocationData(8, 2, 243, "Ravio's Shop (8)", Ravio),
    LocationData(9, 2, 248, "Ravio's Shop (9)", Ravio),
    LocationData(None, None, None, "Your House Weather Vane", Vane, Items.YourHouseWV),
    LocationData(None, None, None, "Kakariko Village Weather Vane", Vane, Items.KakarikoVillageWV),
    LocationData(None, None, None, "Sanctuary Weather Vane", Vane, Items.SanctuaryWV),
    LocationData(None, None, None, "Witch's House Weather Vane", Vane, Items.WitchsHouseWV),
    LocationData(10, None, 57, "Dampe"),
    LocationData(11, None, 181, "Irene"),
    LocationData(12, 0, 63, "Sanctuary Pegs"),
    LocationData(13, None, 971, "Blacksmith Ledge"),
    LocationData(14, None, 992, "Hyrule Castle Rocks"),
    LocationData(15, None, 913, "Haunted Grove Stump"),
    LocationData(16, 0, 189, "Southern Ruins Ledge"),
    LocationData(17, 0, 159, "Lake Hylia Ledge Chest"),
    LocationData(18, None, 916, "Lake Hylia Eastern Shore"),
    LocationData(19, None, None, "Hyrule Hotfoot 75s", Minigame),
    LocationData(20, None, 969, "Hyrule Hotfoot 65s", Minigame),
    LocationData(21, None, 946, "Bird Lover"),
    LocationData(22, None, 945, "Street Merchant (Left)"),
    LocationData(23, None, 209, "Street Merchant (Right)"),
    LocationData(24, None, 963, "Shady Guy"),
    LocationData(25, None, 972, "Dodge the Cuccos", Minigame),
    LocationData(26, None, 980, "Rupee Rush (Hyrule)", Minigame),
    LocationData(27, 0, 306, "[Mai] Kakariko Bush", Maiamai),
    LocationData(28, 0, 267, "[Mai] Lost Woods Path Rock", Maiamai),
    LocationData(29, 0, 268, "[Mai] Fortune-Teller Tent", Maiamai),
    LocationData(30, 0, 275, "[Mai] Woman's Roof", Maiamai),
    LocationData(None, None, None, "Woman Roof Maiamai", Event, Items.WomanRoofMaiamai),
    LocationData(31, None, 970, "Eastern Ruins Peg Circle"),
    LocationData(32, 0, 259, "[Mai] Rosso Wall", Maiamai),
    LocationData(33, 0, 269, "[Mai] Small Pond", Maiamai),
    LocationData(34, 0, 270, "[Mai] Sanctuary Wall", Maiamai),
    LocationData(35, 0, 277, "[Mai] Blacksmith Tree", Maiamai),
    LocationData(36, 0, 305, "[Mai] Lost Woods Tree", Maiamai),
    LocationData(37, 0, 278, "[Mai] Hyrule Castle Tree", Maiamai),
    LocationData(38, 0, 279, "[Mai] Hyrule Castle Tiles", Maiamai),
    LocationData(39, 0, 280, "[Mai] Wooden Bridge", Maiamai),
    LocationData(40, 0, 281, "[Mai] Eastern Ruins Wall", Maiamai),
    LocationData(41, 0, 282, "[Mai] Eastern Ruins Yellow Tree", Maiamai),
    LocationData(42, 0, 291, "[Mai] Eastern Ruins Green Tree", Maiamai),
    LocationData(43, 0, 292, "[Mai] Eastern Ruins Rock", Maiamai),
    LocationData(44, 0, 283, "[Mai] Blacksmith Tiles", Maiamai),
    LocationData(45, 0, 284, "[Mai] Eastern Ruins Bonk Rocks", Maiamai),
    LocationData(46, 0, 285, "[Mai] Hyrule Rupee Rush Wall", Maiamai),
    LocationData(47, 0, 286, "[Mai] Cucco Ranch Tree", Maiamai),
    LocationData(48, 0, 287, "[Mai] Haunted Grove Tree", Maiamai),
    LocationData(49, 0, 288, "[Mai] Your House Tree", Maiamai),
    LocationData(50, 0, 289, "[Mai] Behind Your House", Maiamai),
    LocationData(51, 0, 290, "[Mai] Eastern Ruins River", Maiamai),
    LocationData(52, 0, 297, "[Mai] Southern Ruins Pillars", Maiamai),
    LocationData(53, 0, 296, "[Mai] Outside Flippers Mini-Dungeon", Maiamai),
    LocationData(54, 0, 298, "[Mai] Outside Maiamai Cave", Maiamai),
    LocationData(55, 0, 304, "[Mai] Lake Hylia East River", Maiamai),
    LocationData(56, 0, 300, "[Mai] Hyrule Hotfoot Rock", Maiamai),
    LocationData(57, 0, 301, "[Mai] Southern Ruins Big Rock", Maiamai),
    LocationData(58, 0, 302, "[Mai] Lake Hylia Shallow Ring", Maiamai),
    LocationData(None, None, None, "Ravio's Signs", Event, Items.RavioSigns),
    LocationData(59, None, None, "Maiamai Bow Upgrade", Upgrade, Items.Bow),
    LocationData(60, None, None, "Maiamai Boomerang Upgrade", Upgrade, Items.Boomerang),
    LocationData(61, None, None, "Maiamai Hookshot Upgrade", Upgrade, Items.Hookshot),
    LocationData(62, None, None, "Maiamai Hammer Upgrade", Upgrade, Items.Hammer),
    LocationData(63, None, None, "Maiamai Bombs Upgrade", Upgrade, Items.Bombs),
    LocationData(64, None, None, "Maiamai Fire Rod Upgrade", Upgrade, Items.FireRod),
    LocationData(65, None, None, "Maiamai Ice Rod Upgrade", Upgrade, Items.IceRod),
    LocationData(66, None, None, "Maiamai Tornado Rod Upgrade", Upgrade, Items.TornadoRod),
    LocationData(67, None, None, "Maiamai Sand Rod Upgrade", Upgrade, Items.SandRod),
    LocationData(68, None, None, "100 Maiamai", Upgrade, Items.GreatSpin),
    LocationData(69, 2, 5, "Woman"),
    LocationData(70, 0, 276, "[Mai] Kakariko Sand", Maiamai),
    LocationData(71, 0, 272, "[Mai] Waterfall Ledge", Maiamai),
    LocationData(72, 0, 295, "[Mai] Outside Cucco Mini-Dungeon", Maiamai),
    LocationData(73, 22, 2, "Cucco Mini-Dungeon"),
    LocationData(None, None, None, "Access Potion Shop", Event, Items.AccessPotionShop),
    LocationData(74, 0, 273, "[Mai] Witch's House", Maiamai),
    LocationData(None, None, None, "Eastern Palace Weather Vane", Vane, Items.EasternPalaceWV),
    LocationData(75, 0, 2, "Eastern Ruins Armos Chest"),
    LocationData(76, 0, 5, "Eastern Ruins Hookshot Chest"),
    LocationData(77, 0, 3, "Eastern Ruins Merge Chest"),
    LocationData(78, None, 213, "Queen Oren"),
    LocationData(None, None, None, "Give Oren Smooth Gem", Event, Items.RavioShopOpen),
    LocationData(None, None, None, "Shady Guy Trigger", Event, Items.ShadyGuyTrigger),
    LocationData(79, 0, 145, "Zora's Domain Ledge"),
    LocationData(80, 0, 266, "[Mai] Zora's Domain", Maiamai),
    LocationData(81, 0, 274, "[Mai] South of Zora's Domain", Maiamai),
    LocationData(82, None, 978, "Waterfall Cave"),
    LocationData(83, 22, 1, "Merge Mini-Dungeon"),
    LocationData(84, None, 979, "Eastern Ruins Cave"),
    LocationData(None, None, None, "House of Gales Weather Vane", Vane, Items.HouseOfGalesWV),
    LocationData(85, 0, 299, "[Mai] Lake Hylia Island Tile", Maiamai),
    LocationData(None, None, None, "Reach House of Gales Island", Event, Items.RavioShopOpen),
    LocationData(86, None, 261, "Rosso (1)"),
    LocationData(87, 2, 142, "Rosso (2)"),
    LocationData(88, 4, 134, "Rosso Cave"),
    LocationData(89, 22, 132, "River Mini-Dungeon"),
    LocationData(90, 0, 271, "[Mai] Hyrule Graveyard Wall", Maiamai),
    LocationData(91, None, 990, "Graveyard Ledge Cave"),
    LocationData(92, None, 37, "Blacksmith Table"),
    LocationData(93, None, 885, "Blacksmith"),
    LocationData(None, None, None, "Access Hyrule Blacksmith", Event, Items.AccessHyruleBlacksmith),
    LocationData(94, None, 981, "Blacksmith Cave"),
    LocationData(95, None, 101, "[HC] Throne"),
    LocationData(96, 2, 192, "[HC] West Wing"),
    LocationData(97, 0, 51, "[HC] Battlement"),
    LocationData(98, None, 976, "Lost Woods Alcove"),
    LocationData(99, 0, 133, "Lost Woods Chest"),
    LocationData(100, 0, 258, "[Mai] Lost Woods Bush", Maiamai),
    LocationData(101, 0, 257, "[Mai] Lost Woods Rock", Maiamai),
    LocationData(102, 0, 150, "Master Sword Pedestal"),
    LocationData(103, None, 889, "Fortune-Teller"),
    LocationData(104, 2, 2, "Kakariko Jail"),
    LocationData(105, None, 977, "Kakariko Well (Top)"),
    LocationData(106, 4, 138, "Kakariko Well (Bottom)"),
    LocationData(107, None, 974, "Stylish Woman"),
    LocationData(None, None, None, "Open Stylish Woman's House", Event, Items.StylishWomansHouseOpen),
    LocationData(None, None, None, "Access Milk Bar", Event, Items.AccessMilkBar),
    LocationData(108, 2, 48, "Bee Guy (1)"),
    LocationData(109, None, 912, "Bee Guy (2)"),
    LocationData(None, None, None, "Kakariko Item Shop (1)", Shop, Items.ScootFruit),
    LocationData(None, None, None, "Kakariko Item Shop (2)", Shop, Items.FoulFruit),
    LocationData(None, None, None, "Kakariko Item Shop (3)", Shop, Items.Shield),
    LocationData(None, None, None, "Lakeside Item Shop (1)", Shop, Items.ScootFruit),
    LocationData(None, None, None, "Lakeside Item Shop (2)", Shop, Items.FoulFruit),
    LocationData(None, None, None, "Lakeside Item Shop (3)", Shop, Items.Shield),
    LocationData(110, 4, 168, "Runaway Item Seller"),
    LocationData(111, 22, 128, "Flippers Mini-Dungeon"),
    LocationData(112, 0, 303, "[Mai] Southern Ruins Bomb Cave", Maiamai),
    LocationData(113, None, 983, "Southern Ruins Pillar Cave"),
    LocationData(114, 4, 143, "Lake Hylia Dark Cave"),
    LocationData(115, 4, 141, "Ice Rod Cave"),
    LocationData(None, None, None, "Open Sanctuary Doors", Event, Items.OpenSanctuaryDoors),
    LocationData(116, 0, 265, "[Mai] Moldorm Ledge", Maiamai),
    LocationData(None, None, None, "Death Mountain (Hyrule) Weather Vane", Vane, Items.DeathMountainHyruleWV),
    LocationData(117, 0, 261, "[Mai] Death Mountain Base Rock", Maiamai),
    LocationData(None, None, None, "Eruption Cutscene", Event, Items.RavioShopOpen),
    LocationData(118, 4, 131, "Death Mountain Blocked Cave"),
    LocationData(119, 4, 130, "Death Mountain Open Cave"),
    LocationData(120, 4, 133, "Death Mountain Fairy Cave"),
    LocationData(121, 4, 128, "Donkey Cave"),
    LocationData(122, 0, 128, "Donkey Cave Ledge"),
    LocationData(123, 0, 260, "[Mai] Death Mountain West Ledge", Maiamai),
    LocationData(124, 4, 129, "Death Mountain West Highest Cave"),
    LocationData(125, None, 975, "Spectacle Rock"),
    LocationData(None, None, None, "Tower of Hera Weather Vane", Vane, Items.TowerOfHeraWV),
    LocationData(126, 0, 264, "[Mai] Outside Hookshot Mini-Dungeon", Maiamai),
    LocationData(127, 22, 130, "Hookshot Mini-Dungeon"),
    LocationData(128, None, 995, "Fire Cave Pillar"),
    LocationData(129, 0, 262, "[Mai] Fire Cave Ledge", Maiamai),
    LocationData(130, None, 947, "Bouldering Guy"),
    LocationData(131, None, 947, "Bouldering Guy's Trash", default_item=Items.Bottle),
    LocationData(132, 0, 263, "[Mai] Rosso's Ore Mine", Maiamai),
    LocationData(133, None, 982, "Floating Island"),
]

hyrule_sanctuary_locations: List[LocationData] = [
    LocationData(134, 4, 10, "[HS] Entrance"),
    LocationData(135, 4, 12, "[HS] Lower Chest"),
    LocationData(136, 4, 11, "[HS] Upper Chest"),
    LocationData(137, None, 973, "[HS] Ledge"),
]

lorule_locations: List[LocationData] = [
    LocationData(None, None, None, "Vacant House Weather Vane", Vane, Items.VacantHouseWV),
    LocationData(None, None, None, "Blacksmith Weather Vane", Vane, Items.BlacksmithWV),
    LocationData(None, None, None, "Lorule Castle Weather Vane", Vane, Items.LoruleCastleWV),
    LocationData(None, None, None, "Thieves' Town Weather Vane", Vane, Items.ThievesTownWV),
    LocationData(138, None, 987, "Rupee Rush (Lorule)", Minigame),
    LocationData(139, 1, 255, "Octoball Derby", Minigame),
    LocationData(None, None, None, "Access Hilda Barrier", Event, Items.AccessLoruleCastleField),
    LocationData(140, 1, 279, "[Mai] Lorule Castle Wall", Maiamai),
    LocationData(141, 1, 278, "[Mai] Lorule Castle Tree", Maiamai),
    LocationData(142, 1, 276, "[Mai] Thieves' Town Wall", Maiamai),
    LocationData(143, 1, 277, "[Mai] Lorule Fortune-Teller Rock", Maiamai),
    LocationData(144, 1, 283, "[Mai] Lorule Blacksmith Wall", Maiamai),
    LocationData(145, 1, 285, "[Mai] Lorule Rupee Rush Wall", Maiamai),
    LocationData(146, 1, 287, "[Mai] Octoball Derby Skull", Maiamai),
    LocationData(147, 1, 288, "[Mai] Vacant House Rock", Maiamai),
    LocationData(148, 1, 289, "[Mai] Behind Vacant House", Maiamai),
    LocationData(149, 1, 296, "[Mai] Lorule S Ruins Pillars", Maiamai),
    LocationData(150, 1, 297, "[Mai] Lorule S Ruins Wall", Maiamai),
    LocationData(151, 1, 303, "[Mai] Lorule S Ruins Water", Maiamai),
    LocationData(152, 1, 306, "[Mai] Thieves' Town Tree", Maiamai),
    LocationData(None, None, None, "Fortune's Choice", Event, Items.HeartPiece), # currently unrandomized
    LocationData(None, None, None, "Thieves' Town Item Shop (1)", Shop, Items.Bee),
    # LocationData(None, None, None, "Thieves' Town Item Shop (2)", Shop, Items.GoldBee),
    LocationData(None, None, None, "Thieves' Town Item Shop (3)", Shop, Items.Fairy),
    LocationData(None, None, None, "Thieves' Town Item Shop (4)", Shop, Items.Shield),
    LocationData(None, None, None, "Obtain Big Bomb Flower", Event, Items.BigBombFlower),
    LocationData(153, 1, 286, "[Mai] Big Bomb Flower Grass", Maiamai),
    LocationData(None, None, None, "Graveyard Weather Vane", Vane, Items.GraveyardWV),
    LocationData(154, 1, 205, "Graveyard Peninsula"),
    LocationData(155, 1, 270, "[Mai] Lorule Graveyard Big Rock", Maiamai),
    LocationData(156, 1, 271, "[Mai] Lorule Graveyard Wall", Maiamai),
    LocationData(157, 1, 280, "[Mai] Lorule Graveyard Tree", Maiamai),
    LocationData(158, 5, 198, "Philosopher's Cave"),
    LocationData(159, None, 948, "Great Rupee Fairy"),
    LocationData(160, 3, 206, "Blacksmith (Lorule)"),
    LocationData(161, 23, 90, "Pegasus Boots Pyramid"),
    LocationData(162, None, 949, "Vacant House"),
    LocationData(163, 5, 188, "Thief Girl"),
    LocationData(164, 5, 129, "Swamp Cave (Left)"),
    LocationData(165, None, 985, "Swamp Cave (Middle)"),
    LocationData(166, 5, 130, "Swamp Cave (Right)"),
    LocationData(167, 5, 128, "Big Bomb Flower Cave"),
    LocationData(None, None, None, "Swamp Palace Weather Vane", Vane, Items.SwampPalaceWV),
    LocationData(168, 1, 64, "Lorule Field Hookshot Chest"),
    LocationData(169, 1, 295, "[Mai] Lorule Haunted Grove Wall", Maiamai),
    LocationData(170, 0, 294, "[Mai] Buried in the Desert", Maiamai),
    LocationData(None, None, None, "Desert Palace Weather Vane", Vane, Items.DesertPalaceWV),
    LocationData(171, 0, 293, "[Mai] Buried near Desert Palace", Maiamai),
    LocationData(None, None, None, "Misery Mire Weather Vane", Vane, Items.MiseryMireWV),
    LocationData(172, 1, 294, "[Mai] Misery Mire Wall", Maiamai),
    LocationData(173, 1, 293, "[Mai] Misery Mire Water", Maiamai),
    LocationData(174, 1, 301, "[Mai] Misery Mire Rock", Maiamai),
    LocationData(175, 23, 171, "Sand Mini-Dungeon"),
    LocationData(176, None, 968, "Misery Mire Ledge"),
    LocationData(177, 1, 304, "[Mai] Lorule Lake SE Wall", Maiamai),
    LocationData(178, 1, 300, "[Mai] Lorule Lake Skull", Maiamai),
    LocationData(None, None, None, "Turtle (wall)", Event, Items.TurtleWall),
    LocationData(179, 1, 236, "Lorule Lake Chest"),
    LocationData(180, 1, 298, "[Mai] Lorule Lake West Wall", Maiamai),
    LocationData(None, None, None, "Turtle (flipped)", Event, Items.TurtleFlipped),
    LocationData(181, 1, 302, "[Mai] Lorule Lake Rock", Maiamai),
    LocationData(None, None, None, "Lorule Lakeside Item Shop (1)", Shop, Items.Bee),
    # LocationData(None, None, None, "Lorule Lakeside Item Shop (2)", Shop, Items.GoldBee),
    LocationData(None, None, None, "Lorule Lakeside Item Shop (3)", Shop, Items.Fairy),
    LocationData(None, None, None, "Lorule Lakeside Item Shop (4)", Shop, Items.Shield),
    LocationData(None, None, None, "Turtle (bullied)", Event, Items.TurtleAttacked),
    LocationData(182, 1, 299, "[Mai] Lorule Lake Water", Maiamai),
    LocationData(None, None, None, "Turtle Rock Weather Vane", Vane, Items.TurtleRockWV),
    LocationData(183, 1, 209, "Dark/Turtle Chest"),
    LocationData(184, 1, 272, "[Mai] Dark Ruins Waterfall", Maiamai),
    LocationData(185, 1, 281, "[Mai] Dark Maze Entrance Wall", Maiamai),
    LocationData(186, 1, 284, "[Mai] Dark Ruins Bonk Rocks", Maiamai),
    LocationData(187, 1, 290, "[Mai] Dark Ruins West Tree", Maiamai),
    LocationData(188, 1, 291, "[Mai] Dark Ruins East Tree", Maiamai),
    LocationData(189, 1, 292, "[Mai] Dark Ruins South Wall", Maiamai),
    LocationData(190, 1, 177, "Dark Maze Chest"),
    LocationData(191, None, 986, "Dark Maze Ledge"),
    LocationData(192, 1, 282, "[Mai] Dark Maze Center Wall", Maiamai),
    LocationData(None, None, None, "Dark Palace Weather Vane", Vane, Items.DarkPalaceWV),
    LocationData(193, 1, 274, "[Mai] Ku's Domain Grass", Maiamai),
    LocationData(194, 1, 237, "Ku's Domain Fight"),
    LocationData(195, 1, 266, "[Mai] Ku's Domain Water", Maiamai),
    LocationData(196, 1, 273, "[Mai] Outside Hinox Cave", Maiamai),
    LocationData(197, None, 861, "Hinox (1)"),
    LocationData(198, None, 861, "Hinox (2)"),
    LocationData(199, None, 861, "Hinox (3)"),
    LocationData(200, None, 861, "Hinox (4)"),
    LocationData(201, None, 861, "Hinox (5)"),
    LocationData(202, None, 861, "Hinox (6)"),
    LocationData(None, None, None, "Skull Woods Weather Vane", Vane, Items.SkullWoodsWV),
    LocationData(203, None, 989, "n-Shaped House"),
    LocationData(204, None, 988, "Destroyed House"),
    LocationData(205, 1, 257, "[Mai] Skull Woods Grass", Maiamai),
    LocationData(206, 1, 258, "[Mai] Skull Woods Skull", Maiamai),
    LocationData(207, 1, 259, "[Mai] Destroyed House Tree", Maiamai),
    LocationData(208, 1, 265, "[Mai] Skull Woods Bush", Maiamai),
    LocationData(209, 1, 267, "[Mai] Skull Woods Rock", Maiamai),
    LocationData(210, 1, 268, "[Mai] Skull Woods Entrance Wall", Maiamai),
    LocationData(211, 1, 269, "[Mai] Skull Woods Dry Pond", Maiamai),
    LocationData(212, 1, 275, "[Mai] n-Shaped House Wall", Maiamai),
    LocationData(None, None, None, "Mysterious Man", Shop, Items.GoldBee),
    LocationData(None, None, None, "Treacherous Tower Weather Vane", Vane, Items.TreacherousTowerWV),
    LocationData(213, 1, 8, "Ice Gimos Fight"),
    LocationData(214, 1, 6, "Lorule Mountain W Ledge"),
    LocationData(215, 1, 5, "Treacherous Tower", Minigame),
    LocationData(None, None, None, "Clear Treacherous Tower", Event, Items.ClearTreacherousTower),
    LocationData(216, 1, 260, "[Mai] Lorule Mountain W Skull", Maiamai),
    LocationData(217, 1, 261, "[Mai] Lorule Mountain W Big Rock", Maiamai),
    LocationData(218, 1, 305, "[Mai] Lorule Mountain E Big Rock", Maiamai),
    LocationData(None, None, None, "Death Mountain (Lorule) Weather Vane", Vane, Items.DeathMountainLoruleWV),
    LocationData(219, 1, 263, "[Mai] Lorule Mountain E Wall", Maiamai),
    LocationData(220, 1, 12, "Ice Cave Ledge"),
    LocationData(221, 1, 262, "[Mai] Ice Cave Ledge", Maiamai),
    LocationData(None, None, None, "Ice Ruins Weather Vane", Vane, Items.IceRuinsWV),
    LocationData(222, 1, 11, "Behind Ice Gimos"),
    LocationData(223, 1, 264, "[Mai] Outside Ice Ruins", Maiamai),
]

lorule_sanctuary_locations: List[LocationData] = [
    LocationData(224, 23, 10, "[LS] Entrance Chest"),
    LocationData(225, 23, 12, "[LS] Lower Chest"),
    LocationData(226, 23, 11, "[LS] Upper Chest"),
    LocationData(227, 23, 16, "[LS] Ledge"),
]

eastern_locations: List[LocationData] = [
    LocationData(228, 9, 26, "[EP] (1F) Merge Chest"),
    LocationData(229, 9, 5, "[EP] (1F) Left Door Chest"),
    LocationData(230, 9, 2, "[EP] (1F) Popo Room"),
    LocationData(231, 9, 7, "[EP] (1F) Secret Room"),
    LocationData(232, 9, 3, "[EP] (1F) Switch Room"),
    LocationData(233, 9, 75, "[EP] (2F) Defeat Popos"),
    LocationData(234, 9, 73, "[EP] (2F) Ball Room"),
    LocationData(235, 9, 71, "[EP] (2F) Switch Room"),
    LocationData(236, 9, 69, "[EP] (2F) Big Chest"),
    LocationData(237, 9, 22, "[EP] Yuga (1)"),
    LocationData(238, 9, 27, "[EP] Yuga (2)"),
    LocationData(None, None, None, "[EP] Prize", Prize, Items.PendantOfCourage),
    LocationData(None, None, None, "Eastern Palace Complete", Event, Items.Yuga),
    LocationData(239, 9, 24, "[EP] (3F) Escape Chest"),
    LocationData(240, 9, 25, "[EP] (1F) Escape Chest"),
]

gales_locations: List[LocationData] = [
    LocationData(241, 10, 1, "[HG] (1F) Torches"),
    LocationData(242, 10, 2, "[HG] (1F) Switch Room"),
    LocationData(243, 10, 5, "[HG] (1F) Fire Bubbles"),
    LocationData(244, 10, 3, "[HG] (1F) West Room"),
    LocationData(245, 10, 9, "[HG] (1F) West Room Secret"),
    LocationData(246, 10, 15, "[HG] (2F) Narrow Ledge"),
    LocationData(247, 10, 16, "[HG] (2F) Big Chest"),
    LocationData(248, 10, 17, "[HG] (2F) Fire Ring"),
    LocationData(249, 10, 34, "[HG] (3F) Fire Bubbles"),
    LocationData(250, 10, 26, "[HG] (3F) Rat Room"),
    LocationData(251, 10, 30, "[HG] Margomill"),
    LocationData(None, None, None, "[HG] Prize", Prize, Items.PendantOfWisdom),
    LocationData(None, None, None, "Margomill Defeated", Event, Items.Margomill),
]

hera_locations: List[LocationData] = [
    LocationData(252, 11, 2, "[TH] (1F) Outside"),
    LocationData(253, 11, 1, "[TH] (1F) Center"),
    LocationData(254, 11, 9, "[TH] (3F) Platform"),
    LocationData(255, 11, 3, "[TH] (5F) Red/Blue Switches"),
    LocationData(256, 11, 4, "[TH] (6F) Right Mole"),
    LocationData(257, 11, 11, "[TH] (6F) Left Mole"),
    LocationData(258, 11, 8, "[TH] (7F) Outside (Ledge)"),
    LocationData(259, 11, 14, "[TH] (8F) Fairy Room"),
    LocationData(260, 11, 5, "[TH] (11F) Big Chest"),
    LocationData(261, 11, 6, "[TH] Moldorm"),
    LocationData(None, None, None, "[TH] Prize", Prize, Items.PendantOfPower),
    LocationData(None, None, None, "Moldorm", Event, Items.Moldorm),
]

dark_locations: List[LocationData] = [
    LocationData(262, 13, 34, "[PD] (1F) Right Pit"),
    LocationData(263, 13, 36, "[PD] (1F) Left Pit"),
    LocationData(264, 13, 43, "[PD] (1F) Switch Puzzle"),
    LocationData(265, 13, 46, "[PD] (1F) Hidden Room (Upper)"),
    LocationData(266, 13, 54, "[PD] (1F) Hidden Room (Lower)"),
    LocationData(267, 13, 1, "[PD] (B1) Fall From 1F"),
    LocationData(268, 13, 24, "[PD] (B1) Helmasaur Room"),
    LocationData(269, 13, 2, "[PD] (B1) Helmasaur Room (Fall)"),
    LocationData(270, 13, 3, "[PD] (B1) Glow-in-the-Dark Maze"),
    LocationData(271, 13, 39, "[PD] (1F) Fall From 2F"),
    LocationData(272, 13, 67, "[PD] (2F) Big Chest (Hidden)"),
    LocationData(None, None, None, "[PD] (2F) South Hidden Room", Event, Items.RupeeGold), # currently unrandomized
    LocationData(273, 13, 68, "[PD] (2F) Alcove"),
    LocationData(274, 13, 9, "[PD] (B1) Bomb Bowling"),
    LocationData(275, 13, 22, "[PD] Gemesaur King"),
    LocationData(None, None, None, "[PD] Prize", Prize, Items.Gulley),
    LocationData(None, None, None, "Gemesaur King", Event, Items.GemesaurKing),
]

swamp_locations: List[LocationData] = [
    LocationData(276, 14, 64, "[SP] (B1) Center"),
    LocationData(277, 14, 58, "[SP] (B1) Waterfall Room"),
    LocationData(278, 14, 57, "[SP] (B1) Raft Room (Pillar)"),
    LocationData(279, 14, 56, "[SP] (B1) Raft Room (Right)"),
    LocationData(280, 14, 55, "[SP] (B1) Raft Room (Left)"),
    LocationData(281, 14, 14, "[SP] (B1) Gyorm"),
    LocationData(282, 14, 59, "[SP] (B1) Big Chest (Secret)"),
    LocationData(283, 14, 16, "[SP] (1F) West Room"),
    LocationData(284, 14, 13, "[SP] (1F) East Room"),
    LocationData(285, 14, 15, "[SP] (1F) Water Puzzle"),
    LocationData(286, 14, 28, "[SP] (1F) Big Chest (Fire)"),
    LocationData(287, 14, 31, "[SP] Arrghus"),
    LocationData(None, None, None, "[SP] Prize", Prize, Items.Oren),
    LocationData(None, None, None, "Arrghus", Event, Items.Arrghus),
]

skull_locations: List[LocationData] = [
    LocationData(288, 15, 4, "[SW] (B1) South Chest"),
    LocationData(289, 15, 3, "[SW] (B1) Gibdo Room (Lower)"),
    LocationData(290, 15, 7, "[SW] (B1) Gibdo Room (Hole)"),
    LocationData(291, 15, 5, "[SW] (B1) Grate Room"),
    LocationData(292, 15, 26, "[SW] (B2) Moving Platform Room"),
    LocationData(293, 15, 31, "[SW] Knucklemaster"),
    LocationData(None, None, None, "[SW] Prize", Prize, Items.Seres),
    LocationData(None, None, None, "Knucklemaster", Event, Items.Knucklemaster),
    LocationData(294, 15, 8, "[SW] (B1) Big Chest (Eyes)"),
    LocationData(None, None, None, "Skull Eye Right", Event, Items.SkullEyeRight),
    LocationData(295, 15, 64, "[SW] (B1) Big Chest (Upper)"),
    LocationData(None, None, None, "Skull Eye Left", Event, Items.SkullEyeLeft),
    LocationData(296, 1, 238, "[SW] Outdoor Chest"),
]

thieves_locations: List[LocationData] = [
    LocationData(297, 16, 2, "[TT] (B1) Grate Chest"),
    LocationData(298, 16, 13, "[TT] (B1) Jail Cell"),
    LocationData(None, None, None, "Thieves' Hideout B1 Door Open", Event, Items.ThievesB1DoorOpen),
    LocationData(299, 16, 84, "[TT] (B2) Grate Chest (Fall)"),
    LocationData(None, None, None, "Thieves' Hideout B2 Door Open", Event, Items.ThievesB2DoorOpen),
    LocationData(300, 16, 106, "[TT] (B2) Jail Cell"),
    LocationData(301, 16, 78, "[TT] (B2) Switch Puzzle Room"),
    LocationData(302, 16, 64, "[TT] (B2) Eyegores"),
    LocationData(None, None, None, "Thieves' Hideout B3 Water Drained", Event, Items.ThievesB3WaterDrained),
    LocationData(303, 16, 132, "[TT] (B3) Underwater"),
    LocationData(304, 16, 130, "[TT] (B3) Big Chest (Hidden)"),
    LocationData(305, 16, 16, "[TT] (B1) Behind Wall"),
    LocationData(306, 16, 5, "[TT] (B1) Big Chest (Entrance)"),
    LocationData(307, 3, 124, "[TT] Stalblind"),
    LocationData(None, None, None, "[TT] Prize", Prize, Items.Osfala),
    LocationData(None, None, None, "Stalblind Defeated", Event, Items.Stalblind),
]

ice_locations: List[LocationData] = [
    LocationData(308, 17, 8, "[IR] (1F) Hidden Chest"),
    LocationData(309, 17, 11, "[IR] (B2) Ice Pillar"),
    LocationData(310, 17, 37, "[IR] (B3) Grate Chest (Left)"),
    LocationData(311, 17, 39, "[IR] (B3) Grate Chest (Right)"),
    LocationData(312, 17, 50, "[IR] (B5) Big Chest"),
    LocationData(313, 17, 16, "[IR] (B1) Narrow Ledge"),
    LocationData(314, 17, 17, "[IR] (B1) East Chest"),
    LocationData(315, 17, 10, "[IR] (B1) Upper Chest"),
    LocationData(316, 17, 36, "[IR] (B2) Long Merge Chest"),
    LocationData(317, 17, 38, "[IR] (B3) Big Chest (Puzzle)"),
    LocationData(318, 17, 40, "[IR] (B4) Switches"),
    LocationData(319, 17, 51, "[IR] (B4) Southwest Chest (Fall)"),
    LocationData(320, 17, 47, "[IR] (B4) Narrow Platform"),
    LocationData(321, 17, 49, "[IR] (B4) Southeast Chest (Fall)"),
    LocationData(322, 17, 61, "[IR] Dharkstare"),
    LocationData(None, None, None, "[IR] Prize", Prize, Items.Impa),
    LocationData(None, None, None, "Dharkstare", Event, Items.Dharkstare),
]

desert_locations: List[LocationData] = [
    LocationData(323, 18, 4, "[DP] (1F) Entrance"),
    LocationData(324, 18, 1, "[DP] (1F) Sand Switch Room"),
    LocationData(325, 18, 12, "[DP] (1F) Sand Room (North)"),
    LocationData(326, 18, 157, "[DP] (1F) Sand Room (South)"),
    LocationData(327, 18, 16, "[DP] (1F) Behind Rocks"),
    LocationData(328, 18, 6, "[DP] (1F) Big Chest (Behind Wall)"),
    LocationData(329, 18, 76, "[DP] (2F) Under Rock (Left)"),
    LocationData(330, 18, 73, "[DP] (2F) Under Rock (Right)"),
    LocationData(331, 18, 71, "[DP] (2F) Under Rock (Ball Room)"),
    LocationData(332, 18, 22, "[DP] (2F) Beamos Room"),
    LocationData(333, 18, 161, "[DP] (2F) Red/Blue Switches"),
    LocationData(334, 18, 26, "[DP] (2F) Big Chest (Puzzle)"),
    LocationData(335, 18, 24, "[DP] (2F) Leever Room"),
    LocationData(336, 18, 70, "[DP] (3F) Behind Falling Sand"),
    LocationData(337, 18, 30, "[DP] (3F) Armos Room"),
    LocationData(338, 1, 251, "[DP] Zaganaga"),
    LocationData(None, None, None, "[DP] Prize", Prize, Items.Irene),
    LocationData(None, None, None, "Zaganaga Defeated", Event, Items.Zaganaga),
]

turtle_locations: List[LocationData] = [
    LocationData(339, 19, 5, "[TR] (1F) Center"),
    LocationData(340, 19, 17, "[TR] (1F) Northeast Ledge"),
    LocationData(341, 19, 8, "[TR] (1F) Southeast Chest"),
    LocationData(342, 19, 12, "[TR] (1F) Defeat Flamolas"),
    LocationData(343, 19, 7, "[TR] (1F) Northwest Room"),
    LocationData(344, 19, 1, "[TR] (1F) Grate Chest"),
    LocationData(345, 19, 69, "[TR] (B1) Northeast Room"),
    LocationData(346, 19, 66, "[TR] (B1) Grate Chest (Small)"),
    LocationData(347, 19, 65, "[TR] (B1) Big Chest (Top)"),
    LocationData(348, 19, 81, "[TR] (B1) Big Chest (Center)"),
    LocationData(349, 19, 82, "[TR] (B1) Platform"),
    LocationData(None, None, None, "[TR] (1F) Under Center", Event, Items.RupeeSilver), # currently unrandomized
    LocationData(None, None, None, "[TR] (B1) Under Center", Event, Items.RupeeGold), # currently unrandomized
    LocationData(350, None, 984, "[TR] Left Balcony"),
    LocationData(351, 19, 129, "[TR] Grinexx"),
    LocationData(None, None, None, "[TR] Prize", Prize, Items.Rosso),
    LocationData(None, None, None, "Grinexx", Event, Items.Grinexx),
]

lorule_castle_locations: List[LocationData] = [
    LocationData(352, 20, 169, "[LC] (1F) Ledge"),
    LocationData(353, 20, 171, "[LC] (1F) Center"),
    LocationData(354, 20, 212, "[LC] (2F) Near Torches"),
    LocationData(355, 20, 213, "[LC] (2F) Hidden Path"),
    LocationData(356, 20, 170, "[LC] (2F) Ledge"),
    LocationData(357, 20, 224, "[LC] Bomb Trial (1)"),
    LocationData(358, 20, 222, "[LC] Bomb Trial (2)"),
    LocationData(None, None, None, "Bomb Trial", Event, Items.BombTrial),
    LocationData(359, 20, 200, "[LC] Tile Trial (1)"),
    LocationData(360, 20, 202, "[LC] Tile Trial (2)"),
    LocationData(None, None, None, "Tile Trial", Event, Items.TileTrial),
    LocationData(361, 20, 231, "[LC] Lamp Trial"),
    LocationData(None, None, None, "Lamp Trial", Event, Items.LampTrial),
    LocationData(362, 20, 80, "[LC] Hook Trial (1)"),
    LocationData(363, 20, 239, "[LC] Hook Trial (2)"),
    LocationData(None, None, None, "Hookshot Trial", Event, Items.HookTrial),
    LocationData(364, 20, 215, "[LC] (4F) Center"),
    LocationData(365, 20, 172, "[LC] (4F) Hidden Path"),
    LocationData(366, None, 862, "[LC] Zelda"),
    LocationData(None, None, None, "Sacred Realm", Event, Items.Triforce),
]

location_lists: List[List[LocationData]] = [
    hyrule_locations,
    hyrule_sanctuary_locations,
    lorule_locations,
    lorule_sanctuary_locations,
    eastern_locations,
    gales_locations,
    hera_locations,
    dark_locations,
    swamp_locations,
    skull_locations,
    thieves_locations,
    ice_locations,
    desert_locations,
    turtle_locations,
    lorule_castle_locations,
]

all_locations: List[LocationData] = [loc for loc_list in location_lists for loc in loc_list]
location_table: Dict[str, LocationData] = {loc.name: loc for loc in all_locations}

class Dungeon:
    name: str
    locations: List[LocationData]
    items: List[ItemData]

    def __init__(self, name: str, locations: List[LocationData], items: List[ItemData]):
        self.name = name
        self.locations = locations
        self.items = items

dungeon_table: List[Dungeon] = [
    Dungeon("Hyrule Sanctuary", hyrule_sanctuary_locations, [Items.KeyHyruleSanctuary]),
    Dungeon("Lorule Sanctuary", lorule_sanctuary_locations, [Items.KeyLoruleSanctuary]),
    Dungeon("Eastern Palace", eastern_locations, [Items.KeyEastern, Items.BigKeyEastern, Items.CompassEastern]),
    Dungeon("House of Gales", gales_locations, [Items.KeyGales, Items.BigKeyGales, Items.CompassGales]),
    Dungeon("Tower of Hera", hera_locations, [Items.KeyHera, Items.BigKeyHera, Items.CompassHera]),
    Dungeon("Dark Palace", dark_locations, [Items.KeyDark, Items.BigKeyDark, Items.CompassDark]),
    Dungeon("Swamp Palace", swamp_locations, [Items.KeySwamp, Items.BigKeySwamp, Items.CompassSwamp]),
    Dungeon("Skull Woods", skull_locations, [Items.KeySkull, Items.BigKeySkull, Items.CompassSkull]),
    Dungeon("Thieves' Hideout", thieves_locations, [Items.KeyThieves, Items.BigKeyThieves, Items.CompassThieves]),
    Dungeon("Ice Ruins", ice_locations, [Items.KeyIce, Items.BigKeyIce, Items.CompassIce]),
    Dungeon("Desert Palace", desert_locations, [Items.KeyDesert, Items.BigKeyDesert, Items.CompassDesert]),
    Dungeon("Turtle Rock", turtle_locations, [Items.KeyTurtle, Items.BigKeyTurtle, Items.CompassTurtle]),
    Dungeon("Lorule Castle", lorule_castle_locations, [Items.KeyCastle, Items.CompassCastle])
]

# Locations that cannot have dungeon items because they are outside the dungeon they are associated with
dungeon_item_excludes: List[str] = [
    "[SW] Outdoor Chest",
    "[TT] Stalblind",
    "[DP] Zaganaga",
    "[TR] Left Balcony",
    "[LC] Zelda",
]

starting_weapon_locations: List[str] = [
    "Ravio's Gift",
    "Ravio's Shop (1)",
    "Ravio's Shop (2)",
    "Ravio's Shop (3)",
    "Ravio's Shop (4)",
    "Ravio's Shop (5)",
    "Ravio's Shop (7)",
    "Ravio's Shop (8)",
    "Ravio's Shop (9)",
]
