from BaseClasses import Location
from .data import lname, iname
from .options import CV64Options, SubWeaponShuffle, DraculasCondition, RenonFightCondition, VincentFightCondition

from typing import Dict, Optional, Union, List, Tuple

base_id = 0xC64000


class CV64Location(Location):
    game: str = "Castlevania 64"


# # #    KEY    # # #
# "code" = The unique part of the Location's AP code attribute, as well as the in-game bitflag index starting from
#          0x80389BE4 that indicates the Location has been checked. Add this + base_id to get the actual AP code.
# "offset" = The offset in the ROM to overwrite to change the Item on that Location.
# "normal item" = The Item normally there in vanilla on most difficulties in most versions of the game. Used to
#                 determine the World's Item counts by checking what Locations are active.
# "hard item" = The Item normally there in Hard Mode in the PAL version of CV64 specifically. Used instead of the
#               normal Item when the hard Item pool is enabled if it's in the Location's data dict.
# "add conds" = A list of player options conditions that must be satisfied for the Location to be added. Can be of
#               varying length depending on how many conditions need to be satisfied. In the add_conds dict's tuples,
#               the first element is the name of the option, the second is the option value to check for, and the third
#               is a boolean for whether we are evaluating for the option value or not.
# "event" = What event Item to place on that Location, for Locations that are events specifically.
# "countdown" = What Countdown number in the array of Countdown numbers that Location contributes to. For the most part,
#               this is figured out by taking that Location's corresponding stage's postion in the vanilla stage order,
#               but there are some exceptions made for Locations in parts of Villa and Castle Center that split off into
#               their own numbers.
# "type" = Anything special about this Location in-game, whether it be NPC-given, invisible, etc.
location_info = {
    # Forest of Silence
    lname.forest_pillars_right:    {"code": 0x1C,  "offset": 0x10C67B, "normal item": iname.red_jewel_l,
                                    "hard item": iname.red_jewel_s},
    lname.forest_pillars_left:     {"code": 0x46,  "offset": 0x10C6EB, "normal item": iname.knife,
                                    "add conds": ["sub"]},
    lname.forest_pillars_top:      {"code": 0x13,  "offset": 0x10C71B, "normal item": iname.roast_beef,
                                    "hard item": iname.red_jewel_l},
    lname.forest_boss_one:         {"event": iname.trophy, "add conds": ["boss"]},
    lname.forest_king_skeleton:    {"code": 0xC,   "offset": 0x10C6BB, "normal item": iname.five_hundred_gold},
    lname.forest_lgaz_in:          {"code": 0x1A,  "offset": 0x10C68B, "normal item": iname.moon_card},
    lname.forest_lgaz_top:         {"code": 0x19,  "offset": 0x10C693, "normal item": iname.red_jewel_l,
                                    "hard item": iname.red_jewel_s},
    lname.forest_hgaz_in:          {"code": 0xB,   "offset": 0x10C6C3, "normal item": iname.sun_card},
    lname.forest_hgaz_top:         {"code": 0x3,   "offset": 0x10C6E3, "normal item": iname.roast_chicken,
                                    "hard item": iname.five_hundred_gold},
    lname.forest_weretiger_sw:     {"code": 0xA,   "offset": 0x10C6CB, "normal item": iname.five_hundred_gold},
    lname.forest_boss_two:         {"event": iname.trophy, "add conds": ["boss"]},
    lname.forest_weretiger_gate:   {"code": 0x7,   "offset": 0x10C683, "normal item": iname.powerup},
    lname.forest_dirge_tomb_l:     {"code": 0x59,  "offset": 0x10C74B, "normal item": iname.one_hundred_gold,
                                    "add conds": ["empty"]},
    lname.forest_dirge_tomb_u:     {"code": 0x8,   "offset": 0x10C743, "normal item": iname.one_hundred_gold},
    lname.forest_dirge_plaque:     {"code": 0x6,   "offset": 0x7C7F9D, "normal item": iname.roast_chicken,
                                    "hard item": iname.one_hundred_gold, "type": "inv"},
    lname.forest_dirge_ped:        {"code": 0x45,  "offset": 0x10C6FB, "normal item": iname.cross,
                                    "add conds": ["sub"]},
    lname.forest_dirge_rock1:      {"code": 0x221, "offset": 0x10C791, "normal item": iname.five_hundred_gold,
                                    "add conds": ["3hb"]},
    lname.forest_dirge_rock2:      {"code": 0x222, "offset": 0x10C793, "normal item": iname.five_hundred_gold,
                                    "add conds": ["3hb"]},
    lname.forest_dirge_rock3:      {"code": 0x223, "offset": 0x10C795, "normal item": iname.five_hundred_gold,
                                    "add conds": ["3hb"]},
    lname.forest_dirge_rock4:      {"code": 0x224, "offset": 0x10C797, "normal item": iname.five_hundred_gold,
                                    "add conds": ["3hb"]},
    lname.forest_dirge_rock5:      {"code": 0x225, "offset": 0x10C799, "normal item": iname.five_hundred_gold,
                                    "add conds": ["3hb"]},
    lname.forest_corpse_save:      {"code": 0xF,   "offset": 0x10C6A3, "normal item": iname.red_jewel_s},
    lname.forest_dbridge_wall:     {"code": 0x18,  "offset": 0x10C69B, "normal item": iname.red_jewel_s},
    lname.forest_dbridge_sw:       {"code": 0x9,   "offset": 0x10C6D3, "normal item": iname.roast_beef,
                                    "hard item": iname.one_hundred_gold},
    lname.forest_dbridge_gate_l:   {"code": 0x44,  "offset": 0x10C6F3, "normal item": iname.axe, "add conds": ["sub"]},
    lname.forest_dbridge_gate_r:   {"code": 0xE,   "offset": 0x10C6AB, "normal item": iname.red_jewel_l,
                                    "hard item": iname.red_jewel_s},
    lname.forest_dbridge_tomb_l:   {"code": 0xEA,  "offset": 0x10C763, "normal item": iname.three_hundred_gold,
                                    "add conds": ["empty"]},
    lname.forest_dbridge_tomb_ur:  {"code": 0xE4,  "offset": 0x10C773, "normal item": iname.three_hundred_gold,
                                    "add conds": ["empty"]},
    lname.forest_dbridge_tomb_uf:  {"code": 0x1B,  "offset": 0x10C76B, "normal item": iname.red_jewel_s},
    lname.forest_bface_tomb_lf:    {"code": 0x10,  "offset": 0x10C75B, "normal item": iname.roast_chicken},
    lname.forest_bface_tomb_lr:    {"code": 0x58,  "offset": 0x10C753, "normal item": iname.three_hundred_gold,
                                    "add conds": ["empty"]},
    lname.forest_bface_tomb_u:     {"code": 0x1E,  "offset": 0x10C77B, "normal item": iname.one_hundred_gold},
    lname.forest_ibridge:          {"code": 0x2,   "offset": 0x10C713, "normal item": iname.one_hundred_gold},
    lname.forest_bridge_rock1:     {"code": 0x227, "offset": 0x10C79D, "normal item": iname.red_jewel_l,
                                    "add conds": ["3hb"]},
    lname.forest_bridge_rock2:     {"code": 0x228, "offset": 0x10C79F, "normal item": iname.five_hundred_gold,
                                    "hard item": iname.three_hundred_gold, "add conds": ["3hb"]},
    lname.forest_bridge_rock3:     {"code": 0x229, "offset": 0x10C7A1, "normal item": iname.powerup,
                                    "hard item": iname.three_hundred_gold, "add conds": ["3hb"]},
    lname.forest_bridge_rock4:     {"code": 0x22A, "offset": 0x10C7A3, "normal item": iname.roast_chicken,
                                    "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.forest_werewolf_tomb_lf: {"code": 0xE7,  "offset": 0x10C783, "normal item": iname.one_hundred_gold,
                                    "add conds": ["empty"]},
    lname.forest_werewolf_tomb_lr: {"code": 0xE6,  "offset": 0x10C73B, "normal item": iname.three_hundred_gold,
                                    "add conds": ["empty"]},
    lname.forest_werewolf_tomb_r:  {"code": 0x4,   "offset": 0x10C733, "normal item": iname.sun_card},
    lname.forest_werewolf_plaque:  {"code": 0x1,   "offset": 0xBFC8AF, "normal item": iname.roast_chicken,
                                    "type": "inv"},
    lname.forest_werewolf_tree:    {"code": 0xD,   "offset": 0x10C6B3, "normal item": iname.red_jewel_s},
    lname.forest_werewolf_island:  {"code": 0x41,  "offset": 0x10C703, "normal item": iname.holy_water,
                                    "add conds": ["sub"]},
    lname.forest_final_sw:         {"code": 0x12,  "offset": 0x10C72B, "normal item": iname.roast_beef},
    lname.forest_boss_three:       {"event": iname.trophy, "add conds": ["boss"]},

    # Castle Wall
    lname.cwr_bottom:        {"code": 0x1DD, "offset": 0x10C7E7, "normal item": iname.sun_card,
                              "hard item": iname.one_hundred_gold},
    lname.cw_dragon_sw:      {"code": 0x153, "offset": 0x10C817, "normal item": iname.roast_chicken},
    lname.cw_boss:           {"event": iname.trophy, "add conds": ["boss"]},
    lname.cw_save_slab1:     {"code": 0x22C, "offset": 0x10C84D, "normal item": iname.red_jewel_l,
                              "add conds": ["3hb"]},
    lname.cw_save_slab2:     {"code": 0x22D, "offset": 0x10C84F, "normal item": iname.red_jewel_l,
                              "hard item": iname.red_jewel_s, "add conds": ["3hb"]},
    lname.cw_save_slab3:     {"code": 0x22E, "offset": 0x10C851, "normal item": iname.red_jewel_l,
                              "hard item": iname.red_jewel_s, "add conds": ["3hb"]},
    lname.cw_save_slab4:     {"code": 0x22F, "offset": 0x10C853, "normal item": iname.red_jewel_l,
                              "hard item": iname.red_jewel_s, "add conds": ["3hb"]},
    lname.cw_save_slab5:     {"code": 0x230, "offset": 0x10C855, "normal item": iname.red_jewel_l,
                              "hard item": iname.red_jewel_s, "add conds": ["3hb"]},
    lname.cw_rrampart:       {"code": 0x156, "offset": 0x10C7FF, "normal item": iname.five_hundred_gold},
    lname.cw_lrampart:       {"code": 0x155, "offset": 0x10C807, "normal item": iname.moon_card,
                              "hard item": iname.one_hundred_gold},
    lname.cw_pillar:         {"code": 0x14D, "offset": 0x7F9A0F, "normal item": iname.holy_water, "add conds": ["sub"]},
    lname.cw_shelf_visible:  {"code": 0x158, "offset": 0x7F99A9, "normal item": iname.powerup},
    lname.cw_shelf_sandbags: {"code": 0x14E, "offset": 0x7F9A3E, "normal item": iname.five_hundred_gold, "type": "inv"},
    lname.cw_shelf_torch:    {"code": 0x14C, "offset": 0x10C82F, "normal item": iname.cross, "add conds": ["sub"]},
    lname.cw_ground_left:    {"code": 0x14B, "offset": 0x10C827, "normal item": iname.knife, "add conds": ["sub"]},
    lname.cw_ground_middle:  {"code": 0x159, "offset": 0x10C7F7, "normal item": iname.left_tower_key},
    lname.cw_ground_right:   {"code": 0x14A, "offset": 0x10C81F, "normal item": iname.axe, "add conds": ["sub"]},
    lname.cwl_bottom:        {"code": 0x1DE, "offset": 0x10C7DF, "normal item": iname.moon_card},
    lname.cwl_bridge:        {"code": 0x1DC, "offset": 0x10C7EF, "normal item": iname.roast_beef},
    lname.cw_drac_sw:        {"code": 0x154, "offset": 0x10C80F, "normal item": iname.roast_chicken,
                              "hard item": iname.one_hundred_gold},
    lname.cw_drac_slab1:     {"code": 0x232, "offset": 0x10C859, "normal item": iname.five_hundred_gold,
                              "add conds": ["3hb"]},
    lname.cw_drac_slab2:     {"code": 0x233, "offset": 0x10C85B, "normal item": iname.five_hundred_gold,
                              "add conds": ["3hb"]},
    lname.cw_drac_slab3:     {"code": 0x234, "offset": 0x10C85D, "normal item": iname.five_hundred_gold,
                              "add conds": ["3hb"]},
    lname.cw_drac_slab4:     {"code": 0x235, "offset": 0x10C85F, "normal item": iname.five_hundred_gold,
                              "add conds": ["3hb"]},
    lname.cw_drac_slab5:     {"code": 0x236, "offset": 0x10C861, "normal item": iname.five_hundred_gold,
                              "add conds": ["3hb"]},
    # Villa
    lname.villafy_outer_gate_l:         {"code": 0x133, "offset": 0x10C87F, "normal item": iname.red_jewel_l},
    lname.villafy_outer_gate_r:         {"code": 0x132, "offset": 0x10C887, "normal item": iname.red_jewel_l},
    lname.villafy_dog_platform:         {"code": 0x134, "offset": 0x10C89F, "normal item": iname.red_jewel_l},
    lname.villafy_inner_gate:           {"code": 0x138, "offset": 0xBFC8D7, "normal item": iname.roast_beef},
    lname.villafy_gate_marker:          {"code": 0x131, "offset": 0x10C8A7, "normal item": iname.powerup,
                                         "hard item": iname.one_hundred_gold},
    lname.villafy_villa_marker:         {"code": 0x13E, "offset": 0x10C897, "normal item": iname.roast_beef,
                                         "hard item": iname.one_hundred_gold},
    lname.villafy_tombstone:            {"code": 0x12F, "offset": 0x8099CC, "normal item": iname.moon_card,
                                         "type": "inv"},
    lname.villafy_fountain_fl:          {"code": 0x139, "offset": 0xBFC8CF, "normal item": iname.five_hundred_gold},
    lname.villafy_fountain_fr:          {"code": 0x130, "offset": 0x80997D, "normal item": iname.purifying},
    lname.villafy_fountain_ml:          {"code": 0x13A, "offset": 0x809956, "normal item": iname.sun_card},
    lname.villafy_fountain_mr:          {"code": 0x13D, "offset": 0x80992D, "normal item": iname.moon_card},
    lname.villafy_fountain_rl:          {"code": 0x13B, "offset": 0xBFC8D3, "normal item": iname.roast_beef,
                                         "hard item": iname.five_hundred_gold},
    lname.villafy_fountain_rr:          {"code": 0x13C, "offset": 0x80993C, "normal item": iname.five_hundred_gold},
    lname.villafo_front_r:              {"code": 0x3D,  "offset": 0x10C8E7, "normal item": iname.red_jewel_l,
                                         "hard item": iname.five_hundred_gold},
    lname.villafo_front_l:              {"code": 0x3B,  "offset": 0x10C8DF, "normal item": iname.red_jewel_s},
    lname.villafo_mid_l:                {"code": 0x3C,  "offset": 0x10C8D7, "normal item": iname.red_jewel_s},
    lname.villafo_mid_r:                {"code": 0xE5,  "offset": 0x10C8CF, "normal item": iname.three_hundred_gold,
                                         "add conds": ["empty"]},
    lname.villafo_rear_r:               {"code": 0x38,  "offset": 0x10C8C7, "normal item": iname.red_jewel_s},
    lname.villafo_rear_l:               {"code": 0x39,  "offset": 0x10C8BF, "normal item": iname.red_jewel_l,
                                         "hard item": iname.red_jewel_s},
    lname.villafo_pot_r:                {"code": 0x2E,  "offset": 0x10C8AF, "normal item": iname.red_jewel_l,
                                         "hard item": iname.red_jewel_s},
    lname.villafo_pot_l:                {"code": 0x2F,  "offset": 0x10C8B7, "normal item": iname.red_jewel_s},
    lname.villafo_sofa:                 {"code": 0x2D,  "offset": 0x81F07C, "normal item": iname.purifying,
                                         "type": "inv"},
    lname.villafo_chandelier1:          {"code": 0x27D, "offset": 0x10C8F5, "normal item": iname.red_jewel_l,
                                         "add conds": ["3hb"]},
    lname.villafo_chandelier2:          {"code": 0x27E, "offset": 0x10C8F7, "normal item": iname.purifying,
                                         "add conds": ["3hb"]},
    lname.villafo_chandelier3:          {"code": 0x27F, "offset": 0x10C8F9, "normal item": iname.five_hundred_gold,
                                         "add conds": ["3hb"]},
    lname.villafo_chandelier4:          {"code": 0x280, "offset": 0x10C8FB, "normal item": iname.cure_ampoule,
                                         "add conds": ["3hb"]},
    lname.villafo_chandelier5:          {"code": 0x281, "offset": 0x10C8FD, "normal item": iname.roast_chicken,
                                         "add conds": ["3hb"]},
    lname.villala_hallway_stairs:       {"code": 0x34,  "offset": 0x10C927, "normal item": iname.red_jewel_l},
    lname.villala_hallway_l:            {"code": 0x40,  "offset": 0xBFC903, "normal item": iname.knife,
                                         "add conds": ["sub"]},
    lname.villala_hallway_r:            {"code": 0x4F,  "offset": 0xBFC8F7, "normal item": iname.axe,
                                         "add conds": ["sub"]},
    lname.villala_bedroom_chairs:       {"code": 0x33,  "offset": 0x83A588, "normal item": iname.purifying,
                                         "hard item": iname.three_hundred_gold},
    lname.villala_bedroom_bed:          {"code": 0x32,  "offset": 0xBFC95B, "normal item": iname.red_jewel_l,
                                         "hard item": iname.three_hundred_gold},
    lname.villala_vincent:              {"code": 0x23,  "offset": 0xBFE42F, "normal item": iname.archives_key,
                                         "type": "npc"},
    lname.villala_slivingroom_table:    {"code": 0x2B,  "offset": 0xBFC96B, "normal item": iname.five_hundred_gold,
                                         "type": "inv"},
    lname.villala_slivingroom_mirror:   {"code": 0x49,  "offset": 0x83A5D9, "normal item": iname.cross,
                                         "add conds": ["sub"]},
    lname.villala_diningroom_roses:     {"code": 0x2A,  "offset": 0xBFC90B, "normal item": iname.purifying,
                                         "hard item": iname.three_hundred_gold, "type": "inv"},
    lname.villala_llivingroom_pot_r:    {"code": 0x26,  "offset": 0x10C90F, "normal item": iname.storeroom_key},
    lname.villala_llivingroom_pot_l:    {"code": 0x25,  "offset": 0x10C917, "normal item": iname.roast_chicken},
    lname.villala_llivingroom_painting: {"code": 0x2C,  "offset": 0xBFC907, "normal item": iname.purifying,
                                         "hard item": iname.one_hundred_gold, "type": "inv"},
    lname.villala_llivingroom_light:    {"code": 0x28,  "offset": 0x10C91F, "normal item": iname.purifying},
    lname.villala_llivingroom_lion:     {"code": 0x30,  "offset": 0x83A610, "normal item": iname.roast_chicken,
                                         "hard item": iname.five_hundred_gold, "type": "inv"},
    lname.villala_exit_knight:          {"code": 0x27,  "offset": 0xBFC967, "normal item": iname.purifying,
                                         "type": "inv"},
    lname.villala_storeroom_l:          {"code": 0x36,  "offset": 0xBFC95F, "normal item": iname.roast_beef},
    lname.villala_storeroom_r:          {"code": 0x37,  "offset": 0xBFC8FF, "normal item": iname.roast_chicken,
                                         "hard item": iname.five_hundred_gold},
    lname.villala_storeroom_s:          {"code": 0x31,  "offset": 0xBFC963, "normal item": iname.purifying,
                                         "hard item": iname.one_hundred_gold, "type": "inv"},
    lname.villala_archives_entrance:    {"code": 0x48,  "offset": 0x83A5E5, "normal item": iname.holy_water,
                                         "add conds": ["sub"]},
    lname.villala_archives_table:       {"code": 0x29,  "offset": 0xBFC90F, "normal item": iname.purifying,
                                         "type": "inv"},
    lname.villala_archives_rear:        {"code": 0x24,  "offset": 0x83A5B1, "normal item": iname.garden_key},
    lname.villam_malus_torch:           {"code": 0x173, "offset": 0x10C967, "normal item": iname.red_jewel_s,
                                         "countdown": 13},
    lname.villam_malus_bush:            {"code": 0x16C, "offset": 0x850FEC, "normal item": iname.roast_chicken,
                                         "type": "inv", "countdown": 13},
    lname.villam_fplatform:             {"code": 0x16B, "offset": 0x10C987, "normal item": iname.knife,
                                         "add conds": ["sub"], "countdown": 13},
    lname.villam_frankieturf_l:         {"code": 0x177, "offset": 0x10C947, "normal item": iname.three_hundred_gold,
                                         "countdown": 13},
    lname.villam_frankieturf_r:         {"code": 0x16A, "offset": 0x10C98F, "normal item": iname.holy_water,
                                         "add conds": ["sub"], "countdown": 13},
    lname.villam_frankieturf_ru:        {"code": 0x16E, "offset": 0x10C9A7, "normal item": iname.red_jewel_s,
                                         "countdown": 13},
    lname.villam_fgarden_f:             {"code": 0x172, "offset": 0x10C96F, "normal item": iname.red_jewel_s,
                                         "countdown": 13},
    lname.villam_fgarden_mf:            {"code": 0x171, "offset": 0x10C977, "normal item": iname.red_jewel_s,
                                         "countdown": 13},
    lname.villam_fgarden_mr:            {"code": 0x174, "offset": 0x10C95F, "normal item": iname.roast_chicken,
                                         "countdown": 13},
    lname.villam_fgarden_r:             {"code": 0x170, "offset": 0x10C97F, "normal item": iname.red_jewel_l,
                                         "countdown": 13},
    lname.villam_rplatform:             {"code": 0x169, "offset": 0x10C997, "normal item": iname.axe,
                                         "add conds": ["sub"], "countdown": 13},
    lname.villam_rplatform_de:          {"code": 0x176, "offset": 0x10C94F, "normal item": iname.five_hundred_gold,
                                         "countdown": 13},
    lname.villam_exit_de:               {"code": 0x175, "offset": 0x10C957, "normal item": iname.three_hundred_gold,
                                         "countdown": 13},
    lname.villam_serv_path:             {"code": 0x17A, "offset": 0x10C92F, "normal item": iname.copper_key,
                                         "countdown": 13},
    lname.villafo_serv_ent:             {"code": 0x3E,  "offset": 0x10C8EF, "normal item": iname.roast_chicken},
    lname.villam_crypt_ent:             {"code": 0x178, "offset": 0x10C93F, "normal item": iname.purifying,
                                         "countdown": 13},
    lname.villam_crypt_upstream:        {"code": 0x179, "offset": 0x10C937, "normal item": iname.roast_beef,
                                         "countdown": 13},
    lname.villac_ent_l:                 {"code": 0xC9,  "offset": 0x10CF4B, "normal item": iname.red_jewel_s,
                                         "countdown": 13},
    lname.villac_ent_r:                 {"code": 0xC0,  "offset": 0x10CF63, "normal item": iname.five_hundred_gold,
                                         "countdown": 13},
    lname.villac_wall_l:                {"code": 0xC2,  "offset": 0x10CF6B, "normal item": iname.roast_chicken,
                                         "countdown": 13},
    lname.villac_wall_r:                {"code": 0xC1,  "offset": 0x10CF5B, "normal item": iname.red_jewel_l,
                                         "countdown": 13},
    lname.villac_coffin_l:              {"code": 0xD8,  "offset": 0x10CF73, "normal item": iname.knife,
                                         "add conds": ["sub"], "countdown": 13},
    lname.villac_coffin_r:              {"code": 0xC8,  "offset": 0x10CF53, "normal item": iname.red_jewel_s,
                                         "countdown": 13},
    lname.villa_boss_one:               {"event": iname.trophy, "add conds": ["boss"]},
    lname.villa_boss_two:               {"event": iname.trophy, "add conds": ["boss"]},
    # Tunnel
    lname.tunnel_landing:               {"code": 0x197, "offset": 0x10C9AF, "normal item": iname.red_jewel_l,
                                         "hard item": iname.one_hundred_gold},
    lname.tunnel_landing_rc:            {"code": 0x196, "offset": 0x10C9B7, "normal item": iname.red_jewel_s,
                                         "hard item": iname.one_hundred_gold},
    lname.tunnel_stone_alcove_r:        {"code": 0xE1,  "offset": 0x10CA57, "normal item": iname.holy_water,
                                         "add conds": ["sub"]},
    lname.tunnel_stone_alcove_l:        {"code": 0x187, "offset": 0x10CA9F, "normal item": iname.roast_beef,
                                         "hard item": iname.roast_chicken},
    lname.tunnel_twin_arrows:           {"code": 0x195, "offset": 0xBFC993, "normal item": iname.cure_ampoule,
                                         "type": "inv"},
    lname.tunnel_arrows_rock1:          {"code": 0x238, "offset": 0x10CABD, "normal item": iname.purifying,
                                         "add conds": ["3hb"]},
    lname.tunnel_arrows_rock2:          {"code": 0x239, "offset": 0x10CABF, "normal item": iname.purifying,
                                         "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.tunnel_arrows_rock3:          {"code": 0x23A, "offset": 0x10CAC1, "normal item": iname.cure_ampoule,
                                         "add conds": ["3hb"]},
    lname.tunnel_arrows_rock4:          {"code": 0x23B, "offset": 0x10CAC3, "normal item": iname.cure_ampoule,
                                         "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.tunnel_arrows_rock5:          {"code": 0x23C, "offset": 0x10CAC5, "normal item": iname.roast_chicken,
                                         "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.tunnel_lonesome_bucket:       {"code": 0x189, "offset": 0xBFC99B, "normal item": iname.cure_ampoule,
                                         "type": "inv"},
    lname.tunnel_lbucket_mdoor_l:       {"code": 0x198, "offset": 0x10CA67, "normal item": iname.knife,
                                         "add conds": ["sub"]},
    lname.tunnel_lbucket_quag:          {"code": 0x191, "offset": 0x10C9DF, "normal item": iname.red_jewel_l},
    lname.tunnel_bucket_quag_rock1:     {"code": 0x23E, "offset": 0x10CAC9, "normal item": iname.roast_beef,
                                         "hard item": iname.roast_chicken, "add conds": ["3hb"]},
    lname.tunnel_bucket_quag_rock2:     {"code": 0x23F, "offset": 0x10CACB, "normal item": iname.roast_beef,
                                         "hard item": iname.roast_chicken, "add conds": ["3hb"]},
    lname.tunnel_bucket_quag_rock3:     {"code": 0x240, "offset": 0x10CACD, "normal item": iname.roast_beef,
                                         "hard item": iname.roast_chicken, "add conds": ["3hb"]},
    lname.tunnel_lbucket_albert:        {"code": 0x190, "offset": 0x10C9E7, "normal item": iname.red_jewel_s},
    lname.tunnel_albert_camp:           {"code": 0x192, "offset": 0x10C9D7, "normal item": iname.red_jewel_s},
    lname.tunnel_albert_quag:           {"code": 0x193, "offset": 0x10C9CF, "normal item": iname.red_jewel_l},
    lname.tunnel_gondola_rc_sdoor_l:    {"code": 0x53,  "offset": 0x10CA5F, "normal item": iname.cross,
                                         "add conds": ["sub"]},
    lname.tunnel_gondola_rc_sdoor_m:    {"code": 0x19E, "offset": 0x10CAA7, "normal item": iname.roast_beef,
                                         "hard item": iname.one_hundred_gold},
    lname.tunnel_gondola_rc_sdoor_r:    {"code": 0x188, "offset": 0x10CA27, "normal item": iname.roast_beef,
                                         "hard item": iname.one_hundred_gold},
    lname.tunnel_gondola_rc:            {"code": 0x19C, "offset": 0x10CAB7, "normal item": iname.powerup},
    lname.tunnel_rgondola_station:      {"code": 0x194, "offset": 0x10C9C7, "normal item": iname.red_jewel_s},
    lname.tunnel_gondola_transfer:      {"code": 0x186, "offset": 0x10CA2F, "normal item": iname.five_hundred_gold},
    lname.tunnel_corpse_bucket_quag:    {"code": 0x18E, "offset": 0x10C9F7, "normal item": iname.red_jewel_s},
    lname.tunnel_corpse_bucket_mdoor_l: {"code": 0x52,  "offset": 0x10CA6F, "normal item": iname.holy_water,
                                         "add conds": ["sub"]},
    lname.tunnel_corpse_bucket_mdoor_r: {"code": 0x185, "offset": 0x10CA37, "normal item": iname.sun_card,
                                         "hard item": iname.one_hundred_gold},
    lname.tunnel_shovel_quag_start:     {"code": 0x18D, "offset": 0x10C9FF, "normal item": iname.red_jewel_l},
    lname.tunnel_exit_quag_start:       {"code": 0x18C, "offset": 0x10CA07, "normal item": iname.red_jewel_l},
    lname.tunnel_shovel_quag_end:       {"code": 0x18B, "offset": 0x10CA0F, "normal item": iname.red_jewel_l},
    lname.tunnel_exit_quag_end:         {"code": 0x184, "offset": 0x10CA3F, "normal item": iname.five_hundred_gold},
    lname.tunnel_shovel:                {"code": 0x18F, "offset": 0x86D8FC, "normal item": iname.roast_beef,
                                         "type": "inv"},
    lname.tunnel_shovel_save:           {"code": 0x18A, "offset": 0x10CA17, "normal item": iname.red_jewel_l},
    lname.tunnel_shovel_mdoor_l:        {"code": 0x183, "offset": 0x10CA47, "normal item": iname.sun_card,
                                         "hard item": iname.one_hundred_gold},
    lname.tunnel_shovel_mdoor_r:        {"code": 0x51,  "offset": 0x10CA77, "normal item": iname.axe,
                                         "add conds": ["sub"]},
    lname.tunnel_shovel_sdoor_l:        {"code": 0x182, "offset": 0x10CA4F, "normal item": iname.moon_card},
    lname.tunnel_shovel_sdoor_m:        {"code": 0x19D, "offset": 0x10CAAF, "normal item": iname.roast_chicken},
    lname.tunnel_shovel_sdoor_r:        {"code": 0x50,  "offset": 0x10CA7F, "normal item": iname.cross,
                                         "add conds": ["sub"]},
    # Underground Waterway
    lname.uw_near_ent:         {"code": 0x4C,  "offset": 0x10CB03, "normal item": iname.three_hundred_gold},
    lname.uw_across_ent:       {"code": 0x4E,  "offset": 0x10CAF3, "normal item": iname.five_hundred_gold},
    lname.uw_first_ledge1:     {"code": 0x242, "offset": 0x10CB39, "normal item": iname.five_hundred_gold,
                                "add conds": ["3hb"]},
    lname.uw_first_ledge2:     {"code": 0x243, "offset": 0x10CB3B, "normal item": iname.five_hundred_gold,
                                "add conds": ["3hb"]},
    lname.uw_first_ledge3:     {"code": 0x244, "offset": 0x10CB3D, "normal item": iname.purifying,
                                "hard item": iname.five_hundred_gold, "add conds": ["3hb"]},
    lname.uw_first_ledge4:     {"code": 0x245, "offset": 0x10CB3F, "normal item": iname.cure_ampoule,
                                "hard item": iname.five_hundred_gold, "add conds": ["3hb"]},
    lname.uw_first_ledge5:     {"code": 0x246, "offset": 0x10CB41, "normal item": iname.purifying,
                                "hard item": iname.three_hundred_gold, "add conds": ["3hb"]},
    lname.uw_first_ledge6:     {"code": 0x247, "offset": 0x10CB43, "normal item": iname.cure_ampoule,
                                "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.uw_poison_parkour:   {"code": 0x4D,  "offset": 0x10CAFB, "normal item": iname.cure_ampoule},
    lname.uw_boss:             {"event": iname.trophy, "add conds": ["boss"]},
    lname.uw_waterfall_alcove: {"code": 0x57,  "offset": 0x10CB23, "normal item": iname.five_hundred_gold},
    lname.uw_carrie1:          {"code": 0x4B,  "offset": 0x10CB0B, "normal item": iname.moon_card,
                                "hard item": iname.five_hundred_gold, "add conds": ["carrie"]},
    lname.uw_carrie2:          {"code": 0x4A,  "offset": 0x10CB13, "normal item": iname.roast_beef,
                                "hard item": iname.five_hundred_gold, "add conds": ["carrie"]},
    lname.uw_bricks_save:      {"code": 0x5A,  "offset": 0x10CB33, "normal item": iname.powerup,
                                "hard item": iname.one_hundred_gold},
    lname.uw_above_skel_ledge: {"code": 0x56,  "offset": 0x10CB2B, "normal item": iname.roast_chicken},
    lname.uw_in_skel_ledge1:   {"code": 0x249, "offset": 0x10CB45, "normal item": iname.roast_chicken,
                                "add conds": ["3hb"]},
    lname.uw_in_skel_ledge2:   {"code": 0x24A, "offset": 0x10CB47, "normal item": iname.roast_chicken,
                                "add conds": ["3hb"]},
    lname.uw_in_skel_ledge3:   {"code": 0x24B, "offset": 0x10CB49, "normal item": iname.roast_chicken,
                                "add conds": ["3hb"]},
    # Castle Center
    lname.ccb_skel_hallway_ent:          {"code": 0x1AF, "offset": 0x10CB67, "normal item": iname.red_jewel_s},
    lname.ccb_skel_hallway_jun:          {"code": 0x1A8, "offset": 0x10CBD7, "normal item": iname.powerup},
    lname.ccb_skel_hallway_tc:           {"code": 0x1AE, "offset": 0x10CB6F, "normal item": iname.red_jewel_l},
    lname.ccb_skel_hallway_ba:           {"code": 0x1B6, "offset": 0x10CBC7, "normal item": iname.cross,
                                          "add conds": ["sub"]},
    lname.ccb_behemoth_l_ff:             {"code": 0x1AD, "offset": 0x10CB77, "normal item": iname.red_jewel_s},
    lname.ccb_behemoth_l_mf:             {"code": 0x1B3, "offset": 0x10CBA7, "normal item": iname.three_hundred_gold,
                                          "hard item": iname.one_hundred_gold},
    lname.ccb_behemoth_l_mr:             {"code": 0x1AC, "offset": 0x10CB7F, "normal item": iname.red_jewel_l},
    lname.ccb_behemoth_l_fr:             {"code": 0x1B2, "offset": 0x10CBAF, "normal item": iname.three_hundred_gold,
                                          "hard item": iname.one_hundred_gold},
    lname.ccb_behemoth_r_ff:             {"code": 0x1B1, "offset": 0x10CBB7, "normal item": iname.three_hundred_gold,
                                          "hard item": iname.one_hundred_gold},
    lname.ccb_behemoth_r_mf:             {"code": 0x1AB, "offset": 0x10CB87, "normal item": iname.red_jewel_s},
    lname.ccb_behemoth_r_mr:             {"code": 0x1B0, "offset": 0x10CBBF, "normal item": iname.three_hundred_gold,
                                          "hard item": iname.one_hundred_gold},
    lname.ccb_behemoth_r_fr:             {"code": 0x1AA, "offset": 0x10CB8F, "normal item": iname.red_jewel_l},
    lname.ccb_behemoth_crate1:           {"code": 0x24D, "offset": 0x10CBDD, "normal item": iname.five_hundred_gold,
                                          "add conds": ["3hb"]},
    lname.ccb_behemoth_crate2:           {"code": 0x24E, "offset": 0x10CBDF, "normal item": iname.five_hundred_gold,
                                          "add conds": ["3hb"]},
    lname.ccb_behemoth_crate3:           {"code": 0x24F, "offset": 0x10CBE1, "normal item": iname.five_hundred_gold,
                                          "add conds": ["3hb"]},
    lname.ccb_behemoth_crate4:           {"code": 0x250, "offset": 0x10CBE3, "normal item": iname.five_hundred_gold,
                                          "add conds": ["3hb"]},
    lname.ccb_behemoth_crate5:           {"code": 0x251, "offset": 0x10CBE5, "normal item": iname.five_hundred_gold,
                                          "add conds": ["3hb"]},
    lname.ccelv_near_machine:            {"code": 0x11A, "offset": 0x10CBF7, "normal item": iname.red_jewel_s},
    lname.ccelv_atop_machine:            {"code": 0x118, "offset": 0x10CC17, "normal item": iname.powerup,
                                          "hard item": iname.three_hundred_gold},
    lname.ccelv_stand1:                  {"code": 0x253, "offset": 0x10CC1D, "normal item": iname.roast_beef,
                                          "add conds": ["3hb"]},
    lname.ccelv_stand2:                  {"code": 0x254, "offset": 0x10CC1F, "normal item": iname.roast_beef,
                                          "hard item": iname.three_hundred_gold, "add conds": ["3hb"]},
    lname.ccelv_stand3:                  {"code": 0x255, "offset": 0x10CC21, "normal item": iname.roast_beef,
                                          "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.ccelv_pipes:                   {"code": 0x11B, "offset": 0x10CC07, "normal item": iname.one_hundred_gold},
    lname.ccelv_switch:                  {"code": 0x100, "offset": 0x10CC0F, "normal item": iname.holy_water,
                                          "add conds": ["sub"]},
    lname.ccelv_staircase:               {"code": 0x119, "offset": 0x10CBFF, "normal item": iname.red_jewel_l,
                                          "hard item": iname.five_hundred_gold},
    lname.ccff_redcarpet_knight:         {"code": 0x10A, "offset": 0x8C44D9, "normal item": iname.red_jewel_l,
                                          "hard item": iname.red_jewel_s, "type": "inv"},
    lname.ccff_gears_side:               {"code": 0x10F, "offset": 0x10CC33, "normal item": iname.red_jewel_s},
    lname.ccff_gears_mid:                {"code": 0x10E, "offset": 0x10CC3B, "normal item": iname.purifying,
                                          "hard item": iname.one_hundred_gold},
    lname.ccff_gears_corner:             {"code": 0x10D, "offset": 0x10CC43, "normal item": iname.roast_chicken,
                                          "hard item": iname.one_hundred_gold},
    lname.ccff_lizard_knight:            {"code": 0x109, "offset": 0x8C44E7, "normal item": iname.roast_chicken,
                                          "hard item": iname.three_hundred_gold, "type": "inv"},
    lname.ccff_lizard_near_knight:       {"code": 0x101, "offset": 0x10CC5B, "normal item": iname.axe,
                                          "add conds": ["sub"]},
    lname.ccff_lizard_pit:               {"code": 0x10C, "offset": 0x10CC4B, "normal item": iname.sun_card,
                                          "hard item": iname.five_hundred_gold},
    lname.ccff_lizard_corner:            {"code": 0x10B, "offset": 0x10CC53, "normal item": iname.moon_card,
                                          "hard item": iname.five_hundred_gold},
    lname.ccff_lizard_locker_nfr:        {"code": 0x104, "offset": 0x8C450A, "normal item": iname.red_jewel_l,
                                          "add conds": ["liz"]},
    lname.ccff_lizard_locker_nmr:        {"code": 0x105, "offset": 0xBFC9C3, "normal item": iname.five_hundred_gold,
                                          "add conds": ["liz"]},
    lname.ccff_lizard_locker_nml:        {"code": 0x106, "offset": 0xBFC9C7, "normal item": iname.red_jewel_l,
                                          "hard item": iname.cure_ampoule, "add conds": ["liz"]},
    lname.ccff_lizard_locker_nfl:        {"code": 0x107, "offset": 0xBFCA07, "normal item": iname.powerup,
                                          "add conds": ["liz"]},
    lname.ccff_lizard_locker_fl:         {"code": 0x102, "offset": 0xBFCA03, "normal item": iname.five_hundred_gold,
                                          "add conds": ["liz"]},
    lname.ccff_lizard_locker_fr:         {"code": 0x103, "offset": 0x8C44F5, "normal item": iname.sun_card,
                                          "hard item": iname.three_hundred_gold, "add conds": ["liz"]},
    lname.ccff_lizard_slab1:             {"code": 0x257, "offset": 0x10CC61, "normal item": iname.purifying,
                                          "hard item": iname.roast_chicken, "add conds": ["3hb"]},
    lname.ccff_lizard_slab2:             {"code": 0x258, "offset": 0x10CC63, "normal item": iname.purifying,
                                          "hard item": iname.powerup, "add conds": ["3hb"]},
    lname.ccff_lizard_slab3:             {"code": 0x259, "offset": 0x10CC65, "normal item": iname.cure_ampoule,
                                          "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.ccff_lizard_slab4:             {"code": 0x25A, "offset": 0x10CC67, "normal item": iname.cure_ampoule,
                                          "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.ccb_mandrag_shelf_l:           {"code": 0x1A0, "offset": 0xBFCBB3, "normal item": iname.mandragora},
    lname.ccb_mandrag_shelf_r:           {"code": 0x1A1, "offset": 0xBFCBAF, "normal item": iname.mandragora},
    lname.ccb_torture_rack:              {"code": 0x1A9, "offset": 0x8985E5, "normal item": iname.purifying,
                                          "type": "inv"},
    lname.ccb_torture_rafters:           {"code": 0x1A2, "offset": 0x8985D6, "normal item": iname.roast_beef},
    lname.cc_behind_the_seal:            {"event": iname.crystal, "add conds": ["crystal"]},
    lname.cc_boss_one:                   {"event": iname.trophy, "add conds": ["boss"]},
    lname.cc_boss_two:                   {"event": iname.trophy, "add conds": ["boss"]},
    lname.ccll_brokenstairs_floor:       {"code": 0x7B,  "offset": 0x10CC8F, "normal item": iname.red_jewel_l,
                                          "countdown": 14},
    lname.ccll_brokenstairs_knight:      {"code": 0x74,  "offset": 0x8DF782, "normal item": iname.roast_beef,
                                          "hard item": iname.one_hundred_gold, "type": "inv", "countdown": 14},
    lname.ccll_brokenstairs_save:        {"code": 0x7C,  "offset": 0x10CC87, "normal item": iname.red_jewel_l,
                                          "countdown": 14},
    lname.ccll_glassknight_l:            {"code": 0x7A,  "offset": 0x10CC97, "normal item": iname.red_jewel_s,
                                          "hard item": iname.five_hundred_gold, "countdown": 14},
    lname.ccll_glassknight_r:            {"code": 0x7E,  "offset": 0x10CC77, "normal item": iname.red_jewel_s,
                                          "hard item": iname.five_hundred_gold, "countdown": 14},
    lname.ccll_butlers_door:             {"code": 0x7D,  "offset": 0x10CC7F, "normal item": iname.red_jewel_s,
                                          "countdown": 14},
    lname.ccll_butlers_side:             {"code": 0x79,  "offset": 0x10CC9F, "normal item": iname.purifying,
                                          "hard item": iname.one_hundred_gold, "countdown": 14},
    lname.ccll_cwhall_butlerflames_past: {"code": 0x78,  "offset": 0x10CCA7, "normal item": iname.cure_ampoule,
                                          "hard item": iname.red_jewel_l, "countdown": 14},
    lname.ccll_cwhall_flamethrower:      {"code": 0x73,  "offset": 0x8DF580, "normal item": iname.five_hundred_gold,
                                          "type": "inv", "countdown": 14},
    lname.ccll_cwhall_cwflames:          {"code": 0x77,  "offset": 0x10CCAF, "normal item": iname.roast_chicken,
                                          "hard item": iname.red_jewel_l, "countdown": 14},
    lname.ccll_heinrich:                 {"code": 0x69,  "offset": 0xBFE443, "normal item": iname.chamber_key,
                                          "type": "npc", "countdown": 14},
    lname.ccia_nitro_crates:             {"code": 0x66,  "offset": 0x90FCE9, "normal item": iname.healing_kit,
                                          "hard item": iname.one_hundred_gold, "type": "inv", "countdown": 14},
    lname.ccia_nitro_shelf_h:            {"code": 0x55,  "offset": 0xBFCC03, "normal item": iname.magical_nitro,
                                          "countdown": 14},
    lname.ccia_stairs_knight:            {"code": 0x61,  "offset": 0x90FE5C, "normal item": iname.five_hundred_gold,
                                          "type": "inv", "countdown": 14},
    lname.ccia_maids_vase:               {"code": 0x63,  "offset": 0x90FF1D, "normal item": iname.red_jewel_l,
                                          "type": "inv", "countdown": 14},
    lname.ccia_maids_outer:              {"code": 0x6B,  "offset": 0x10CCFF, "normal item": iname.purifying,
                                          "hard item": iname.three_hundred_gold, "countdown": 14},
    lname.ccia_maids_inner:              {"code": 0x6A,  "offset": 0x10CD07, "normal item": iname.cure_ampoule,
                                          "hard item": iname.three_hundred_gold, "countdown": 14},
    lname.ccia_inventions_maids:         {"code": 0x6C,  "offset": 0x10CCE7, "normal item": iname.moon_card,
                                          "hard item": iname.one_hundred_gold, "countdown": 14},
    lname.ccia_inventions_crusher:       {"code": 0x6E,  "offset": 0x10CCDF, "normal item": iname.sun_card,
                                          "hard item": iname.one_hundred_gold, "countdown": 14},
    lname.ccia_inventions_famicart:      {"code": 0x64,  "offset": 0x90FBB3, "normal item": iname.five_hundred_gold,
                                          "type": "inv", "countdown": 14},
    lname.ccia_inventions_zeppelin:      {"code": 0x6D,  "offset": 0x90FBC0, "normal item": iname.roast_beef,
                                          "countdown": 14},
    lname.ccia_inventions_round:         {"code": 0x65,  "offset": 0x90FBA7, "normal item": iname.roast_beef,
                                          "hard item": iname.five_hundred_gold, "type": "inv", "countdown": 14},
    lname.ccia_nitrohall_flamethrower:   {"code": 0x62,  "offset": 0x90FCDA, "normal item": iname.red_jewel_l,
                                          "type": "inv", "countdown": 14},
    lname.ccia_nitrohall_torch:          {"code": 0x6F,  "offset": 0x10CCD7, "normal item": iname.roast_chicken,
                                          "hard item": iname.red_jewel_s, "countdown": 14},
    lname.ccia_nitro_shelf_i:            {"code": 0x60,  "offset": 0xBFCBFF, "normal item": iname.magical_nitro,
                                          "countdown": 14},
    lname.ccll_cwhall_wall:              {"code": 0x76,  "offset": 0x10CCB7, "normal item": iname.roast_beef,
                                          "hard item": iname.one_hundred_gold, "countdown": 14},
    lname.ccl_bookcase:                  {"code": 0x166, "offset": 0x8F1197, "normal item": iname.sun_card,
                                          "countdown": 14},
    # Duel Tower
    lname.dt_boss_one:       {"event": iname.trophy, "add conds": ["boss"]},
    lname.dt_boss_two:       {"event": iname.trophy, "add conds": ["boss"]},
    lname.dt_ibridge_l:      {"code": 0x81, "offset": 0x10CE8B, "normal item": iname.roast_beef,
                              "hard item": iname.five_hundred_gold},
    lname.dt_ibridge_r:      {"code": 0x80, "offset": 0x10CE93, "normal item": iname.powerup},
    lname.dt_stones_start:   {"code": 0x83, "offset": 0x10CE73, "normal item": iname.roast_chicken,
                              "hard item": iname.five_hundred_gold},
    lname.dt_stones_end:     {"code": 0x97, "offset": 0x10CE83, "normal item": iname.knife, "add conds": ["sub"]},
    lname.dt_werebull_arena: {"code": 0x82, "offset": 0x10CE7B, "normal item": iname.roast_beef},
    lname.dt_boss_three:     {"event": iname.trophy, "add conds": ["boss"]},
    lname.dt_boss_four:      {"event": iname.trophy, "add conds": ["boss"]},
    # Tower of Execution
    lname.toe_ledge1:          {"code": 0x25C, "offset": 0x10CD5D, "normal item": iname.red_jewel_l,
                                "add conds": ["3hb"]},
    lname.toe_ledge2:          {"code": 0x25D, "offset": 0x10CD5F, "normal item": iname.purifying,
                                "hard item": iname.red_jewel_s, "add conds": ["3hb"]},
    lname.toe_ledge3:          {"code": 0x25E, "offset": 0x10CD61, "normal item": iname.five_hundred_gold,
                                "hard item": iname.red_jewel_s, "add conds": ["3hb"]},
    lname.toe_ledge4:          {"code": 0x25F, "offset": 0x10CD63, "normal item": iname.cure_ampoule,
                                "hard item": iname.five_hundred_gold, "add conds": ["3hb"]},
    lname.toe_ledge5:          {"code": 0x260, "offset": 0x10CD65, "normal item": iname.holy_water,
                                "add conds": ["3hb", "sub"]},
    lname.toe_midsavespikes_r: {"code": 0x9C,  "offset": 0x10CD1F, "normal item": iname.five_hundred_gold},
    lname.toe_midsavespikes_l: {"code": 0x9B,  "offset": 0x10CD27, "normal item": iname.roast_chicken,
                                "hard item": iname.five_hundred_gold},
    lname.toe_elec_grate:      {"code": 0x99,  "offset": 0x10CD17, "normal item": iname.execution_key},
    lname.toe_ibridge:         {"code": 0x98,  "offset": 0x10CD47, "normal item": iname.one_hundred_gold},
    lname.toe_top:             {"code": 0x9D,  "offset": 0x10CD4F, "normal item": iname.red_jewel_l},
    lname.toe_keygate_l:       {"code": 0x9A,  "offset": 0x10CD37, "normal item": iname.roast_beef,
                                "hard item": iname.one_hundred_gold},
    lname.toe_keygate_r:       {"code": 0x9E,  "offset": 0x10CD3F, "normal item": iname.cross, "add conds": ["sub"]},
    # Tower of Science
    lname.tosci_elevator:        {"code": 0x1FC, "offset": 0x10CE0B, "normal item": iname.three_hundred_gold},
    lname.tosci_plain_sr:        {"code": 0x1FF, "offset": 0x10CDF3, "normal item": iname.science_key1},
    lname.tosci_stairs_sr:       {"code": 0x1FB, "offset": 0x10CE13, "normal item": iname.three_hundred_gold},
    lname.tosci_three_door_hall: {"code": 0x1FE, "offset": 0x10CDFB, "normal item": iname.science_key2},
    lname.tosci_ibridge_t:       {"code": 0x1F3, "offset": 0x10CE3B, "normal item": iname.roast_beef,
                                  "hard item": iname.red_jewel_l},
    lname.tosci_ibridge_b1:      {"code": 0x262, "offset": 0x10CE59, "normal item": iname.red_jewel_l,
                                  "add conds": ["3hb"]},
    lname.tosci_ibridge_b2:      {"code": 0x263, "offset": 0x10CE5B, "normal item": iname.red_jewel_l,
                                  "add conds": ["3hb"]},
    lname.tosci_ibridge_b3:      {"code": 0x264, "offset": 0x10CE5D, "normal item": iname.five_hundred_gold,
                                  "add conds": ["3hb"]},
    lname.tosci_ibridge_b4:      {"code": 0x265, "offset": 0x10CE5F, "normal item": iname.five_hundred_gold,
                                  "add conds": ["3hb"]},
    lname.tosci_ibridge_b5:      {"code": 0x266, "offset": 0x10CE61, "normal item": iname.roast_chicken,
                                  "add conds": ["3hb"]},
    lname.tosci_ibridge_b6:      {"code": 0x267, "offset": 0x10CE63, "normal item": iname.roast_chicken,
                                  "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.tosci_conveyor_sr:     {"code": 0x1F7, "offset": 0x10CE33, "normal item": iname.red_jewel_l,
                                  "hard item": iname.red_jewel_s},
    lname.tosci_exit:            {"code": 0x1FD, "offset": 0x10CE03, "normal item": iname.science_key3},
    lname.tosci_key3_r:          {"code": 0x1FA, "offset": 0x10CE1B, "normal item": iname.five_hundred_gold},
    lname.tosci_key3_m:          {"code": 0x1F2, "offset": 0x10CE2B, "normal item": iname.cross, "add conds": ["sub"]},
    lname.tosci_key3_l:          {"code": 0x1F9, "offset": 0x10CE23, "normal item": iname.five_hundred_gold},
    # Tower of Sorcery
    lname.tosor_stained_tower:  {"code": 0x96, "offset": 0x10CDB3, "normal item": iname.red_jewel_l},
    lname.tosor_savepoint:      {"code": 0x95, "offset": 0x10CDBB, "normal item": iname.red_jewel_l},
    lname.tosor_trickshot:      {"code": 0x92, "offset": 0x10CDD3, "normal item": iname.roast_beef},
    lname.tosor_yellow_bubble:  {"code": 0x91, "offset": 0x10CDDB, "normal item": iname.five_hundred_gold},
    lname.tosor_blue_platforms: {"code": 0x94, "offset": 0x10CDC3, "normal item": iname.red_jewel_s},
    lname.tosor_side_isle:      {"code": 0x93, "offset": 0x10CDCB, "normal item": iname.red_jewel_s},
    lname.tosor_ibridge:        {"code": 0x90, "offset": 0x10CDE3, "normal item": iname.three_hundred_gold},
    # Room of Clocks
    lname.roc_ent_l:  {"code": 0xC6, "offset": 0x10CF7B, "normal item": iname.roast_beef,
                       "hard item": iname.red_jewel_l},
    lname.roc_ent_r:  {"code": 0xC3, "offset": 0x10CFBB, "normal item": iname.powerup,
                       "hard item": iname.five_hundred_gold},
    lname.roc_elev_r: {"code": 0xD4, "offset": 0x10CF93, "normal item": iname.holy_water, "add conds": ["sub"]},
    lname.roc_elev_l: {"code": 0xD5, "offset": 0x10CF8B, "normal item": iname.axe, "add conds": ["sub"]},
    lname.roc_cont_r: {"code": 0xC5, "offset": 0x10CFB3, "normal item": iname.powerup,
                       "hard item": iname.one_hundred_gold},
    lname.roc_cont_l: {"code": 0xDF, "offset": 0x10CFA3, "normal item": iname.three_hundred_gold,
                       "add conds": ["empty"]},
    lname.roc_exit:   {"code": 0xDC, "offset": 0x10CF9B, "normal item": iname.three_hundred_gold,
                       "add conds": ["empty"]},
    lname.roc_boss:   {"event": iname.trophy, "add conds": ["boss"]},
    # Clock Tower
    lname.ct_gearclimb_battery_slab1: {"code": 0x269, "offset": 0x10CEF9, "normal item": iname.roast_chicken,
                                       "add conds": ["3hb"]},
    lname.ct_gearclimb_battery_slab2: {"code": 0x26A, "offset": 0x10CEFB, "normal item": iname.roast_chicken,
                                       "hard item": iname.red_jewel_s, "add conds": ["3hb"]},
    lname.ct_gearclimb_battery_slab3: {"code": 0x26B, "offset": 0x10CEFD, "normal item": iname.roast_chicken,
                                       "hard item": iname.red_jewel_s, "add conds": ["3hb"]},
    lname.ct_gearclimb_corner:        {"code": 0xA7,  "offset": 0x10CEB3, "normal item": iname.red_jewel_s},
    lname.ct_gearclimb_side:          {"code": 0xAD,  "offset": 0x10CEC3, "normal item": iname.clocktower_key1},
    lname.ct_gearclimb_door_slab1:    {"code": 0x26D, "offset": 0x10CF01, "normal item": iname.roast_beef,
                                       "add conds": ["3hb"]},
    lname.ct_gearclimb_door_slab2:    {"code": 0x26E, "offset": 0x10CF03, "normal item": iname.roast_beef,
                                       "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.ct_gearclimb_door_slab3:    {"code": 0x26F, "offset": 0x10CF05, "normal item": iname.roast_beef,
                                       "hard item": iname.one_hundred_gold, "add conds": ["3hb"]},
    lname.ct_bp_chasm_fl:             {"code": 0xA5,  "offset": 0x99BC4D, "normal item": iname.five_hundred_gold},
    lname.ct_bp_chasm_fr:             {"code": 0xA6,  "offset": 0x99BC3E, "normal item": iname.red_jewel_l},
    lname.ct_bp_chasm_rl:             {"code": 0xA4,  "offset": 0x99BC5A, "normal item": iname.holy_water,
                                       "add conds": ["sub"]},
    lname.ct_bp_chasm_k:              {"code": 0xAC,  "offset": 0x99BC30, "normal item": iname.clocktower_key2},
    lname.ct_finalroom_door_slab1:    {"code": 0x271, "offset": 0x10CEF5, "normal item": iname.five_hundred_gold,
                                       "add conds": ["3hb"]},
    lname.ct_finalroom_door_slab2:    {"code": 0x272, "offset": 0x10CEF7, "normal item": iname.five_hundred_gold,
                                       "add conds": ["3hb"]},
    lname.ct_finalroom_fl:            {"code": 0xB3,  "offset": 0x10CED3, "normal item": iname.axe,
                                       "add conds": ["sub"]},
    lname.ct_finalroom_fr:            {"code": 0xB4,  "offset": 0x10CECB, "normal item": iname.knife,
                                       "add conds": ["sub"]},
    lname.ct_finalroom_rl:            {"code": 0xB2,  "offset": 0x10CEE3, "normal item": iname.holy_water,
                                       "add conds": ["sub"]},
    lname.ct_finalroom_rr:            {"code": 0xB0,  "offset": 0x10CEDB, "normal item": iname.cross,
                                       "add conds": ["sub"]},
    lname.ct_finalroom_platform:      {"code": 0xAB,  "offset": 0x10CEBB, "normal item": iname.clocktower_key3},
    lname.ct_finalroom_renon_slab1:   {"code": 0x274, "offset": 0x10CF09, "normal item": iname.five_hundred_gold,
                                       "add conds": ["3hb"]},
    lname.ct_finalroom_renon_slab2:   {"code": 0x275, "offset": 0x10CF0B, "normal item": iname.five_hundred_gold,
                                       "add conds": ["3hb"]},
    lname.ct_finalroom_renon_slab3:   {"code": 0x276, "offset": 0x10CF0D, "normal item": iname.five_hundred_gold,
                                       "add conds": ["3hb"]},
    lname.ct_finalroom_renon_slab4:   {"code": 0x277, "offset": 0x10CF0F, "normal item": iname.five_hundred_gold,
                                       "add conds": ["3hb"]},
    lname.ct_finalroom_renon_slab5:   {"code": 0x278, "offset": 0x10CF11, "normal item": iname.five_hundred_gold,
                                       "add conds": ["3hb"]},
    lname.ct_finalroom_renon_slab6:   {"code": 0x279, "offset": 0x10CF13, "normal item": iname.five_hundred_gold,
                                       "add conds": ["3hb"]},
    lname.ct_finalroom_renon_slab7:   {"code": 0x27A, "offset": 0x10CF15, "normal item": iname.red_jewel_l,
                                       "add conds": ["3hb"]},
    lname.ct_finalroom_renon_slab8:   {"code": 0x27B, "offset": 0x10CF17, "normal item": iname.red_jewel_l,
                                       "add conds": ["3hb"]},
    # Castle Keep
    lname.ck_boss_one:    {"event": iname.trophy, "add conds": ["boss", "renon"]},
    lname.ck_boss_two:    {"event": iname.trophy, "add conds": ["boss", "vincent"]},
    lname.ck_flame_l:     {"code": 0xAF,  "offset": 0x9778C8, "normal item": iname.healing_kit, "type": "inv"},
    lname.ck_flame_r:     {"code": 0xAE,  "offset": 0xBFCA67, "normal item": iname.healing_kit, "type": "inv"},
    lname.ck_behind_drac: {"code": 0xBF,  "offset": 0x10CE9B, "normal item": iname.red_jewel_l},
    lname.ck_cube:        {"code": 0xB5,  "offset": 0x10CEA3, "normal item": iname.healing_kit},
    lname.renon1:         {"code": 0x1C8, "offset": 0xBFD8E5, "normal item": iname.roast_chicken, "type": "shop"},
    lname.renon2:         {"code": 0x1C9, "offset": 0xBFD8E7, "normal item": iname.roast_beef, "type": "shop"},
    lname.renon3:         {"code": 0x1CA, "offset": 0xBFD8E9, "normal item": iname.healing_kit, "type": "shop"},
    lname.renon4:         {"code": 0x1CB, "offset": 0xBFD8EB, "normal item": iname.purifying, "type": "shop"},
    lname.renon5:         {"code": 0x1CC, "offset": 0xBFD8ED, "normal item": iname.cure_ampoule, "type": "shop"},
    lname.renon6:         {"code": 0x1CD, "offset": 0xBFD907, "normal item": iname.sun_card, "type": "shop"},
    lname.renon7:         {"code": 0x1CE, "offset": 0xBFD909, "normal item": iname.moon_card, "type": "shop"},
    lname.the_end:        {"event": iname.victory},
}


add_conds = {"carrie":  ("carrie_logic", True, True),
             "liz":     ("lizard_locker_items", True, True),
             "sub":     ("sub_weapon_shuffle", SubWeaponShuffle.option_anywhere, True),
             "3hb":     ("multi_hit_breakables", True, True),
             "empty":   ("empty_breakables", True, True),
             "shop":    ("shopsanity", True, True),
             "crystal": ("draculas_condition", DraculasCondition.option_crystal, True),
             "boss":    ("draculas_condition", DraculasCondition.option_bosses, True),
             "renon":   ("renon_fight_condition", RenonFightCondition.option_never, False),
             "vincent": ("vincent_fight_condition", VincentFightCondition.option_never, False)}


def get_location_info(location: str, info: str) -> Union[int, str, List[str], None]:
    return location_info[location].get(info, None)


def get_location_names_to_ids() -> Dict[str, int]:
    return {name: get_location_info(name, "code")+base_id for name in location_info if get_location_info(name, "code")
            is not None}


def verify_locations(options: CV64Options, locations: List[str]) -> Tuple[Dict[str, Optional[int]], Dict[str, str]]:

    verified_locations = {}
    events = {}

    for loc in locations:
        loc_add_conds = get_location_info(loc, "add conds")
        loc_code = get_location_info(loc, "code")

        # Check any options that might be associated with the Location before adding it.
        add_it = True
        if isinstance(loc_add_conds, list):
            for cond in loc_add_conds:
                if not ((getattr(options, add_conds[cond][0]).value == add_conds[cond][1]) == add_conds[cond][2]):
                    add_it = False

        if not add_it:
            continue

        # Add the location to the verified Locations if the above check passes.
        # If we are looking at an event Location, add its associated event Item to the events' dict.
        # Otherwise, add the base_id to the Location's code.
        if loc_code is None:
            events[loc] = get_location_info(loc, "event")
        else:
            loc_code += base_id
        verified_locations.update({loc: loc_code})

    return verified_locations, events
