from typing import Optional, Dict
from enum import Enum

class LocationGroup(Enum):
    """
    Used to group locations
    """
    Boss      = 0
    SubSpirit = 1
    Tablet    = 2
    Finding   = 3

class LocationData():
    address: Optional[int]
    key: Optional[str]
    content: Optional[str]
    location_group: Optional[LocationGroup]

    def __init__(self, address: Optional[int] = None, key: Optional[str] = None, content: Optional[str] = None, group: Optional[LocationGroup] = None):
        self.address = address
        self.key = key
        self.content = content
        self.location_group = group

locations : Dict[str, LocationData]= {
	"The Abyss 01 - Silva's Blight-Stained Note 1"         : LocationData(address=0x7171e5, key='Abyss_01_GAMEPLAY.BP_Interactable_Item_Tip3',                             group=LocationGroup.Finding,     content='Tip.tip_066'),
	"The Abyss 01 - Furious Blight x100"                   : LocationData(address=0x7171e6, key='Abyss_01_GAMEPLAY.BP_SCR_LV2M_2171_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv2_L'),
	"The Abyss 02 - Amulet Fragment"                       : LocationData(address=0x7171e7, key='Abyss_02_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                     group=None,                      content='Parameter.i_maxHPUp_01'),
	"The Abyss 02 - Silva's Blight-Stained Note 2"         : LocationData(address=0x7171e8, key='Abyss_02_GAMEPLAY.BP_Interactable_Item_Tip3',                             group=LocationGroup.Finding,     content='Tip.tip_067'),
	"The Abyss 02 - Fretia's Memoirs 5"                    : LocationData(address=0x7171e9, key='Abyss_02_GAMEPLAY.BP_Interactable_Item_Tip4',                             group=LocationGroup.Finding,     content='Tip.tip_028'),
	"The Abyss 02 - White Priestess' Earrings"             : LocationData(address=0x7171ea, key='Abyss_02_GAMEPLAY.BP_Interactable_Passive_healcountup_2',                 group=None,                      content='Passive.i_passive_heal_count_up_2'),
	"The Abyss 04 - Upper Stagnant Blight x100"            : LocationData(address=0x7171eb, key='Abyss_04_GAMEPLAY.BP_SCR_LV1L_2170_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"The Abyss 04 - Lower Stagnant Blight x100"            : LocationData(address=0x7171ec, key='Abyss_04_GAMEPLAY.BP_SCR_LV1L_2170_3',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"The Abyss 05 - Stone Tablet Fragment"                 : LocationData(address=0x7171ed, key='Abyss_05_GAMEPLAY.BP_Interactable_Item_FinalPassivePart_2',               group=LocationGroup.Tablet,      content='Generic.i_FinalPassivePart_Up'),
	"The Abyss 05 - The Deathless Pact"                    : LocationData(address=0x7171ee, key='Abyss_05_GAMEPLAY.BP_Interactable_Item_Tip4',                             group=LocationGroup.Finding,     content='Tip.tip_065'),
	"The Abyss 05 - Ancient Soul x2"                       : LocationData(address=0x7171ef, key='Abyss_05_GAMEPLAY.BP_SCR_LV3S_5000_1',                                    group=None,                      content='Generic.i_SpiritCurrencyLv3_2'),
	"Ruined Castle 01 - Castle Town Maiden"                : LocationData(address=0x7171f0, key='Castle_01_GAMEPLAY.BP_e2082_Dog',                                         group=LocationGroup.SubSpirit,   content='Spirit.s2082'),
	"Ruined Castle 01 - Priestess' Doll"                   : LocationData(address=0x7171f1, key='Castle_01_GAMEPLAY.BP_Interactable_Passive_healcountup_4',                group=None,                      content='Passive.i_passive_heal_count_up_3'),
	"Ruined Castle 01 - Decayed Crown"                     : LocationData(address=0x7171f2, key='Castle_01_GAMEPLAY.BP_Interactable_Passive_Spirit_StunStaminaDamageUp_2', group=None,                      content='Passive.i_passive_stunstamina_damage_up'),
	"Ruined Castle 01 - Stagnant Blight x100"              : LocationData(address=0x7171f3, key='Castle_01_GAMEPLAY.BP_SCR_LV1L_2031_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Ruined Castle 01 - Furious Blight x100"               : LocationData(address=0x7171f4, key='Castle_01_GAMEPLAY.BP_SCR_LV2M_2001_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv2_L'),
	"Ruined Castle 02 - Amulet Fragment"                   : LocationData(address=0x7171f5, key='Castle_02_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                    group=None,                      content='Parameter.i_maxHPUp_01'),
	"Ruined Castle 02 - Stagnant Blight x30"               : LocationData(address=0x7171f6, key='Castle_02_GAMEPLAY.BP_SCR_LV1M_2000_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Ruined Castle 03 - Amulet Fragment"                   : LocationData(address=0x7171f7, key='Castle_03_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                    group=None,                      content='Parameter.i_maxHPUp_01'),
	"Ruined Castle 03 - Stagnant Blight x30"               : LocationData(address=0x7171f8, key='Castle_03_GAMEPLAY.BP_SCR_LV1M_2001_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Ruined Castle 04 - Report from a Verboten Mage"       : LocationData(address=0x7171f9, key='Castle_04_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_035'),
	"Ruined Castle 06 - Royal Aegis Crest"                 : LocationData(address=0x7171fa, key='Castle_06_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                  group=None,                      content='Passive.i_passive_maxhpup_LV2'),
	"Ruined Castle 06 - Stagnant Blight x30"               : LocationData(address=0x7171fb, key='Castle_06_GAMEPLAY.BP_SCR_LV1L_2030_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Ruined Castle 07 - Proof of Founding"                 : LocationData(address=0x7171fc, key='Castle_07_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_019'),
	"Ruined Castle 08 - Chain of Sorcery"                  : LocationData(address=0x7171fd, key='Castle_08_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',                   group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Ruined Castle 08 - Blightwreathed Blade"              : LocationData(address=0x7171fe, key='Castle_08_GAMEPLAY.BP_Interactable_Passive_expup_5',                      group=None,                      content='Passive.i_passive_expup_2'),
	"Ruined Castle 08 - Stagnant Blight x10"               : LocationData(address=0x7171ff, key='Castle_08_GAMEPLAY.BP_SCR_LV1M_2000_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 08 - Upper Stagnant Blight x10"         : LocationData(address=0x717200, key='Castle_08_GAMEPLAY.BP_SCR_LV1M_2000_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 08 - Lower Stagnant Blight x10"         : LocationData(address=0x717201, key='Castle_08_GAMEPLAY.BP_SCR_LV1S_2010_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 09 - Furious Blight x30"                : LocationData(address=0x717202, key='Castle_09_GAMEPLAY.BP_SCR_LV2M_2000_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Ruined Castle 10 - Julius' Book"                      : LocationData(address=0x717203, key='Castle_10_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_064'),
	"Ruined Castle 10 - Stagnant Blight x30"               : LocationData(address=0x717204, key='Castle_10_GAMEPLAY.BP_SCR_LV1M_2000_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Ruined Castle 12 - Chain of Sorcery"                  : LocationData(address=0x717205, key='Castle_12_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',                   group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Ruined Castle 12 - King of the First Age's Diary 1"   : LocationData(address=0x717206, key='Castle_12_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_042'),
	"Ruined Castle 12 - King of the First Age's Diary 2"   : LocationData(address=0x717207, key='Castle_12_GAMEPLAY.BP_Interactable_Item_Tip4',                            group=LocationGroup.Finding,     content='Tip.tip_043'),
	"Ruined Castle 12 - Stagnant Blight x30"               : LocationData(address=0x717208, key='Castle_12_GAMEPLAY.BP_SCR_LV1M_2001_1',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Ruined Castle 13 - Amulet Gem"                        : LocationData(address=0x717209, key='Castle_13_GAMEPLAY.BP_Interactable_Item_MaxHPUp_02_2',                    group=None,                      content='Parameter.i_maxHPUp_02'),
	"Ruined Castle 14 - Amulet Fragment"                   : LocationData(address=0x71720a, key='Castle_14_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                    group=None,                      content='Parameter.i_maxHPUp_01'),
	"Ruined Castle 14 - Stagnant Blight x100"              : LocationData(address=0x71720b, key='Castle_14_GAMEPLAY.BP_SCR_LV1L_2220_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Ruined Castle 14 - Furious Blight x800"               : LocationData(address=0x71720c, key='Castle_14_GAMEPLAY.BP_SCR_LV2LL_0000_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv2_LL'),
	"Ruined Castle 15 - Priestess' Wish"                   : LocationData(address=0x71720d, key='Castle_15_GAMEPLAY.BP_Interactable_Item_HealPower_Up_2',                  group=None,                      content='Generic.i_HealPower_Up'),
	"Ruined Castle 15 - Priestess' Castle Memo"            : LocationData(address=0x71720e, key='Castle_15_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_069'),
	"Ruined Castle 16 - One-Eyed Royal Aegis"              : LocationData(address=0x71720f, key='Castle_16_GAMEPLAY.BP_e2032_BigKnight',                                   group=LocationGroup.SubSpirit,   content='Spirit.s2032'),
	"Ruined Castle 16 - 2nd Stagnant Blight x10"           : LocationData(address=0x717210, key='Castle_16_GAMEPLAY.BP_SCR_LV1M_2190_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 16 - 5th Stagnant Blight x10"           : LocationData(address=0x717211, key='Castle_16_GAMEPLAY.BP_SCR_LV1M_2190_4',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 16 - 9th Stagnant Blight x10"           : LocationData(address=0x717212, key='Castle_16_GAMEPLAY.BP_SCR_LV1M_2190_6',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 16 - 1st Stagnant Blight x10"           : LocationData(address=0x717213, key='Castle_16_GAMEPLAY.BP_SCR_LV1M_2190_7',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 16 - 8th Stagnant Blight x10"           : LocationData(address=0x717214, key='Castle_16_GAMEPLAY.BP_SCR_LV1M_2191_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 16 - 3rd Stagnant Blight x10"           : LocationData(address=0x717215, key='Castle_16_GAMEPLAY.BP_SCR_LV1S_2100_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 16 - 7th Stagnant Blight x10"           : LocationData(address=0x717216, key='Castle_16_GAMEPLAY.BP_SCR_LV1S_2100_4',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 16 - 6th Stagnant Blight x10"           : LocationData(address=0x717217, key='Castle_16_GAMEPLAY.BP_SCR_LV1S_2100_6',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 16 - 4th Stagnant Blight x10"           : LocationData(address=0x717218, key='Castle_16_GAMEPLAY.BP_SCR_LV1S_2101_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 16 - 10th Stagnant Blight x10"          : LocationData(address=0x717219, key='Castle_16_GAMEPLAY.BP_SCR_LV1S_2101_4',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Ruined Castle 17 - Stagnant Blight x30"               : LocationData(address=0x71721a, key='Castle_17_GAMEPLAY.BP_SCR_LV1M_2000_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Ruined Castle 18 - Amulet Fragment"                   : LocationData(address=0x71721b, key='Castle_18_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                    group=None,                      content='Parameter.i_maxHPUp_01'),
	"Ruined Castle 18 - King's Note 2"                     : LocationData(address=0x71721c, key='Castle_18_GAMEPLAY.BP_Interactable_Item_Tip4',                            group=LocationGroup.Finding,     content='Tip.tip_063'),
	"Ruined Castle 18 - Eldred's Ring"                     : LocationData(address=0x71721d, key='Castle_18_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                  group=None,                      content='Passive.i_passive_mp_restore_up_LV1'),
	"Ruined Castle 18 - Furious Blight x100"               : LocationData(address=0x71721e, key='Castle_18_GAMEPLAY.BP_SCR_LV2L_2221_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv2_L'),
	"Ruined Castle 19 - King's Note 1"                     : LocationData(address=0x71721f, key='Castle_19_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_062'),
	"Ruined Castle 20 - Knight Captain Julius"             : LocationData(address=0x717220, key='Castle_20_GAMEPLAY.BP_e5030_Leader',                                      group=LocationGroup.Boss,        content='Spirit.s5030'),
	"Ruined Castle 21 - King of the First Age's Diary 3"   : LocationData(address=0x717221, key='Castle_21_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_044'),
	"Ruined Castle 21 - Stone Tablet Fragment"             : LocationData(address=0x717222, key='Castle_21_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                  group=LocationGroup.Tablet,      content='Generic.i_FinalPassivePart_Up'),
	"Ruined Castle 21 - Stagnant Blight x100"              : LocationData(address=0x717223, key='Castle_21_GEO.BP_SCR_LV1L_2120_2',                                        group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Catacombs 01 - Fretia's Ring"                         : LocationData(address=0x717224, key='Cave_01_GAMEPLAY.BP_Interactable_Passive_Parry_2',                        group=None,                      content='Passive.i_passive_parry'),
	"Catacombs 01 - Furious Blight x800"                   : LocationData(address=0x717225, key='Cave_01_GAMEPLAY.BP_SCR_LV2LL_0000_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv2_LL'),
	"Catacombs 02 - Stagnant Blight x30"                   : LocationData(address=0x717226, key='Cave_02_GAMEPLAY.BP_SCR_LV1M_2160_1',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Catacombs 02 - Stagnant Blight x10"                   : LocationData(address=0x717227, key='Cave_02_GAMEPLAY.BP_SCR_LV1S_2020_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Catacombs 03 - Defense of the Twin Spires 2"          : LocationData(address=0x717228, key='Cave_03_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_022'),
	"Catacombs 03 - Fretia's Memoirs 4"                    : LocationData(address=0x717229, key='Cave_03_GAMEPLAY.BP_Interactable_Item_Tip4',                              group=LocationGroup.Finding,     content='Tip.tip_027'),
	"Catacombs 04 - Amulet Fragment"                       : LocationData(address=0x71722a, key='Cave_04_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                      group=None,                      content='Parameter.i_maxHPUp_01'),
	"Catacombs 05 - Furious Blight x10"                    : LocationData(address=0x71722b, key='Cave_05_GAMEPLAY.BP_SCR_LV1M_2161_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Catacombs 06 - Fretia's Memoirs 2"                    : LocationData(address=0x71722c, key='Cave_06_GAMEPLAY.BP_Interactable_Item_Tip4',                              group=LocationGroup.Finding,     content='Tip.tip_025'),
	"Catacombs 06 - Fretia's Memoirs 1"                    : LocationData(address=0x71722d, key='Cave_06_GAMEPLAY.BP_Interactable_Item_Tip5',                              group=LocationGroup.Finding,     content='Tip.tip_024'),
	"Catacombs 06 - Kilteus' Ring"                         : LocationData(address=0x71722e, key='Cave_06_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                    group=None,                      content='Passive.i_passive_recast_time_cut_LV1'),
	"Catacombs 07 - Elder Crypt Keeper"                    : LocationData(address=0x71722f, key='Cave_07_GAMEPLAY.BP_e2162_Spider',                                        group=LocationGroup.SubSpirit,   content='Spirit.s2162'),
	"Catacombs 07 - Stagnant Blight x10"                   : LocationData(address=0x717230, key='Cave_07_GAMEPLAY.BP_SCR_LV1M_2161_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Catacombs 08 - Amulet Fragment"                       : LocationData(address=0x717231, key='Cave_08_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                      group=None,                      content='Parameter.i_maxHPUp_01'),
	"Catacombs 09 - Amulet Fragment"                       : LocationData(address=0x717232, key='Cave_09_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                   group=None,                      content='Parameter.i_maxHPUp_01'),
	"Catacombs 09 - Chain of Sorcery"                      : LocationData(address=0x717233, key='Cave_09_GAMEPLAY.BP_Interactable_Item_PassiveSlot_Drop',                  group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Catacombs 10 - Amulet Fragment"                       : LocationData(address=0x717234, key='Cave_10_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                   group=None,                      content='Parameter.i_maxHPUp_01'),
	"Catacombs 10 - Stagnant Blight x10"                   : LocationData(address=0x717235, key='Cave_10_GAMEPLAY.BP_SCR_LV1S_2011_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Catacombs 11 - Silva's Note 2"                        : LocationData(address=0x717236, key='Cave_11_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_030'),
	"Catacombs 11 - Stagnant Blight x10"                   : LocationData(address=0x717237, key='Cave_11_GAMEPLAY.BP_SCR_LV1M_2161_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Catacombs 11 - Furious Blight x10"                    : LocationData(address=0x717238, key='Cave_11_GAMEPLAY.BP_SCR_LV2M_2050_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Catacombs 12 - Stone Tablet Fragment"                 : LocationData(address=0x717239, key='Cave_12_GAMEPLAY.BP_Interactable_Item_FinalPassivePart_2',                group=LocationGroup.Tablet,      content='Generic.i_FinalPassivePart_Up'),
	"Catacombs 13 - Silva's Note 1"                        : LocationData(address=0x71723a, key='Cave_13_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_029'),
	"Catacombs 13 - Furious Blight x30"                    : LocationData(address=0x71723b, key='Cave_13_GAMEPLAY.BP_SCR_LV2L_2091_4',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Catacombs 13 - Furious Blight x10"                    : LocationData(address=0x71723c, key='Cave_13_GAMEPLAY.BP_SCR_LV2M_2161_5',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Catacombs 14 - Stagnant Blight x10"                   : LocationData(address=0x71723d, key='Cave_14_GAMEPLAY.BP_SCR_LV1S_2020_7',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Catacombs 15 - Defense of the Twin Spires 1"          : LocationData(address=0x71723e, key='Cave_15_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_021'),
	"Catacombs 15 - Right Stagnant Blight x10"             : LocationData(address=0x71723f, key='Cave_15_GAMEPLAY.BP_SCR_LV1L_2070_3',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Catacombs 15 - Left Stagnant Blight x10"              : LocationData(address=0x717240, key='Cave_15_GAMEPLAY.BP_SCR_LV1S_2020_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Catacombs 16 - The Next White Priestess"              : LocationData(address=0x717241, key='Cave_16_GAMEPLAY.BP_Interactable_Item_Tip4',                              group=LocationGroup.Finding,     content='Tip.tip_020'),
	"Catacombs 16 - Stagnant Blight x30"                   : LocationData(address=0x717242, key='Cave_16_GAMEPLAY.BP_SCR_LV1L_2091_5',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Catacombs 17 - Ancient Soul x2"                       : LocationData(address=0x717243, key='Cave_17_GAMEPLAY.BP_SCR_LV3M_5000_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv3_2'),
	"Catacombs 18 - Holy Spring Water"                     : LocationData(address=0x717244, key='Cave_18_GAMEPLAY.BP_Interactable_Passive_healpowerup_2',                  group=None,                      content='Passive.i_passive_heal_power_up'),
	"Catacombs 19 - Fallen Archer"                         : LocationData(address=0x717245, key='Cave_19_GAMEPLAY.BP_e2022_Soldier',                                       group=LocationGroup.SubSpirit,   content='Spirit.s2022'),
	"Catacombs 19 - Priestess' Wish"                       : LocationData(address=0x717246, key='Cave_19_GAMEPLAY.BP_Interactable_Item_HealPower_Up_2',                    group=None,                      content='Generic.i_HealPower_Up'),
	"Catacombs 19 - Chain of Sorcery"                      : LocationData(address=0x717247, key='Cave_19_GAMEPLAY.BP_Interactable_Item_PassiveSlot_Drop',                  group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Catacombs 20 - Amulet Fragment"                       : LocationData(address=0x717248, key='Cave_20_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                   group=None,                      content='Parameter.i_maxHPUp_01'),
	"Catacombs 20 - The Heirloom of Land's End"            : LocationData(address=0x717249, key='Cave_20_GAMEPLAY.BP_Interactable_Item_Tip4',                              group=LocationGroup.Finding,     content='Tip.tip_023'),
	"Catacombs 20 - Stagnant Blight x10"                   : LocationData(address=0x71724a, key='Cave_20_GAMEPLAY.BP_SCR_LV1S_2011_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Catacombs 21 - White Priestess Statue"                : LocationData(address=0x71724b, key='Cave_21_GAMEPLAY.BP_Interactable_Passive_healcountup_2',                  group=None,                      content='Passive.i_passive_heal_count_up_1'),
	"Catacombs 22 - Fretia's Memoirs 3"                    : LocationData(address=0x71724c, key='Cave_22_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_026'),
	"Catacombs 23 - Guardian Silva"                        : LocationData(address=0x71724d, key='Cave_23_GAMEPLAY.BP_e5021_OlderSister',                                   group=LocationGroup.Boss,        content='Spirit.s5020'),
	"Catacombs 23 - Unfinished Note"                       : LocationData(address=0x71724e, key='Cave_23_GAMEPLAY.BP_Interactable_Item_Tip4',                              group=LocationGroup.Finding,     content='Tip.tip_031'),
	"White Parish 02 - Amulet Fragment"                    : LocationData(address=0x71724f, key='Church_02_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                 group=None,                      content='Parameter.i_maxHPUp_01'),
	"White Parish 02 - Eleine's Letter"                    : LocationData(address=0x717250, key='Church_02_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_002'),
	"White Parish 03 - Guardian Siegrid"                   : LocationData(address=0x717251, key='Church_03_GAMEPLAY.BP_e5011_YoungerSister',                               group=LocationGroup.Boss,        content='Spirit.s5010'),
	"White Parish 04 - Groa's Letter"                      : LocationData(address=0x717252, key='Church_04_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_005'),
	"White Parish 05 - Soiled Prayer Beads"                : LocationData(address=0x717253, key='Church_05_GAMEPLAY.BP_Interactable_Passive_MaxHPUp_Lv1_2',                group=None,                      content='Passive.i_passive_maxhpup_LV1'),
	"White Parish 07 - Cliffside Hamlet Youth"             : LocationData(address=0x717254, key='Church_07_GAMEPLAY.BP_e2012_Slime_Unique',                                group=LocationGroup.SubSpirit,   content='Spirit.s2012'),
	"White Parish 07 - Chain of Sorcery"                   : LocationData(address=0x717255, key='Church_07_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',                   group=None,                      content='Generic.i_PassiveSlot_Up'),
	"White Parish 08 - Amulet Fragment"                    : LocationData(address=0x717256, key='Church_08_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                    group=None,                      content='Parameter.i_maxHPUp_01'),
	"White Parish 08 - The Parish Way 2"                   : LocationData(address=0x717257, key='Church_08_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_016'),
	"White Parish 09 - Chief Guardian"                     : LocationData(address=0x717258, key='Church_09_GAMEPLAY.BP_e2092_Priest',                                      group=LocationGroup.SubSpirit,   content='Spirit.s2092'),
	"White Parish 09 - Vibrant Plume"                      : LocationData(address=0x717259, key='Church_09_GAMEPLAY.BP_Interactable_Passive_JumpHeightUp_2',               group=None,                      content='Passive.i_passive_jump_height_up'),
	"White Parish 09 - Furious Blight x30"                 : LocationData(address=0x71725a, key='Church_09_GAMEPLAY.BP_SCR_LV2M_2001_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"White Parish 10 - On the Blighted 1"                  : LocationData(address=0x71725b, key='Church_10_GAMEPLAY.BP_Interactable_Item_Tip_2',                           group=LocationGroup.Finding,     content='Tip.tip_003'),
	"White Parish 10 - On the Blighted 2"                  : LocationData(address=0x71725c, key='Church_10_GAMEPLAY.BP_Interactable_Item_Tip2',                            group=LocationGroup.Finding,     content='Tip.tip_004'),
	"White Parish 11 - Amulet Fragment"                    : LocationData(address=0x71725d, key='Church_11_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                    group=None,                      content='Parameter.i_maxHPUp_01'),
	"White Parish 11 - The Parish Way 1"                   : LocationData(address=0x71725e, key='Church_11_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_015'),
	"White Parish 12 - Statue Inscription"                 : LocationData(address=0x71725f, key='Church_12_GAMEPLAY.BP_Interactable_Item_Tip1',                            group=LocationGroup.Finding,     content='Tip.tip_001'),
	"White Parish 13 - Restoring the Aegis Curio"          : LocationData(address=0x717260, key='Church_13_GAMEPLAY.BP_Interactable_Item_Tip4',                            group=LocationGroup.Finding,     content='Tip.tip_068'),
	"White Parish 14 - Priestess' Wish"                    : LocationData(address=0x717261, key='Church_14_GAMEPLAY.BP_Interactable_Item_HealPower_Up_2',                  group=None,                      content='Generic.i_HealPower_Up'),
	"White Parish 14 - Lily's Note"                        : LocationData(address=0x717262, key='Church_14_GAMEPLAY.BP_Interactable_Item_Tip5',                            group=LocationGroup.Finding,     content='Tip.tip_070'),
	"Witch's Thicket 02 - Stagnant Blight x10"             : LocationData(address=0x717263, key='Forest_02_GAMEPLAY.BP_SCR_LV1S_2130_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Witch's Thicket 03 - Amulet Fragment"                 : LocationData(address=0x717264, key='Forest_03_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                    group=None,                      content='Parameter.i_maxHPUp_01'),
	"Witch's Thicket 03 - Chain of Sorcery"                : LocationData(address=0x717265, key='Forest_03_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',                   group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Witch's Thicket 04 - Amulet Fragment"                 : LocationData(address=0x717266, key='Forest_04_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                 group=None,                      content='Parameter.i_maxHPUp_01'),
	"Witch's Thicket 04 - Rusted Blue Ornament"            : LocationData(address=0x717267, key='Forest_04_GAMEPLAY.BP_Interactable_Passive_dmgup_swimming_2',             group=None,                      content='Passive.i_passive_dmgup_swimming'),
	"Witch's Thicket 04 - Stagnant Blight x10"             : LocationData(address=0x717268, key='Forest_04_GAMEPLAY.BP_SCR_LV1S_2100_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Witch's Thicket 05 - Coven Handbook"                  : LocationData(address=0x717269, key='Forest_05_GAMEPLAY.BP_Interactable_Item_Tip4',                            group=LocationGroup.Finding,     content='Tip.tip_010'),
	"Witch's Thicket 06 - Fungal Sorcerer"                 : LocationData(address=0x71726a, key='Forest_06_GAMEPLAY.BP_e2122_Fungus',                                      group=LocationGroup.SubSpirit,   content='Spirit.s2122'),
	"Witch's Thicket 06 - Lover's Letter"                  : LocationData(address=0x71726b, key='Forest_06_GAMEPLAY.BP_Interactable_Item_Tip4',                            group=LocationGroup.Finding,     content='Tip.tip_011'),
	"Witch's Thicket 06 - Furious Blight x10"              : LocationData(address=0x71726c, key='Forest_06_GAMEPLAY.BP_SCR_LV2M_2120_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Witch's Thicket 07 - Amulet Fragment"                 : LocationData(address=0x71726d, key='Forest_07_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                 group=None,                      content='Parameter.i_maxHPUp_01'),
	"Witch's Thicket 07 - Stagnant Blight x30"             : LocationData(address=0x71726e, key='Forest_07_GAMEPLAY.BP_SCR_LV1S_2121_1',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Witch's Thicket 07 - Stagnant Blight x10"             : LocationData(address=0x71726f, key='Forest_07_GAMEPLAY.BP_SCR_LV1S_2130_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Witch's Thicket 08 - Manisa's Ring"                   : LocationData(address=0x717270, key='Forest_08_GAMEPLAY.BP_Interactable_Passive_Treasure_2',                   group=None,                      content='Passive.i_passive_spirit_maxcast_count_up_LV1'),
	"Witch's Thicket 08 - Cracked Familiar Stone"          : LocationData(address=0x717271, key='Forest_08_GAMEPLAY.BP_Interactable_Passive_Treasure2',                    group=None,                      content='Passive.i_passive_dmgcut_LV2'),
	"Witch's Thicket 08 - Furious Blight x10"              : LocationData(address=0x717272, key='Forest_08_GAMEPLAY.BP_SCR_LV2M_2120_1',                                   group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Witch's Thicket 09 - Priestess' Wish"                 : LocationData(address=0x717273, key='Forest_09_GAMEPLAY.BP_Interactable_Item_HealPower_Up_2',                  group=None,                      content='Generic.i_HealPower_Up'),
	"Witch's Thicket 09 - Tarnished Picture"               : LocationData(address=0x717274, key='Forest_09_GAMEPLAY.BP_Interactable_Item_Tip1',                            group=LocationGroup.Finding,     content='Tip.tip_012'),
	"Witch's Thicket 09 - Furious Blight x10"              : LocationData(address=0x717275, key='Forest_09_GAMEPLAY.BP_SCR_LV2S_2130_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Witch's Thicket 10 - Sorcerer's Notes"                : LocationData(address=0x717276, key='Forest_10_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_013'),
	"Witch's Thicket 11 - Floral Sorceress"                : LocationData(address=0x717277, key='Forest_11_GAMEPLAY.BP_e2132_Mandrake',                                    group=LocationGroup.SubSpirit,   content='Spirit.s2132'),
	"Witch's Thicket 11 - Amulet Fragment"                 : LocationData(address=0x717278, key='Forest_11_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                    group=None,                      content='Parameter.i_maxHPUp_01'),
	"Witch's Thicket 11 - Chain of Sorcery"                : LocationData(address=0x717279, key='Forest_11_GAMEPLAY.BP_Interactable_Item_PassiveSlot_Drop',                group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Witch's Thicket 11 - Ruined Witch's Book"             : LocationData(address=0x71727a, key='Forest_11_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                  group=None,                      content='Passive.i_passive_swim_fast'),
	"Witch's Thicket 11 - Stagnant Blight x30"             : LocationData(address=0x71727b, key='Forest_11_GAMEPLAY.BP_SCR_LV1M_2120_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Witch's Thicket 11 - Stagnant Blight x10"             : LocationData(address=0x71727c, key='Forest_11_GAMEPLAY.BP_SCR_LV1S_2130_2',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Witch's Thicket 12 - Amulet Fragment"                 : LocationData(address=0x71727d, key='Forest_12_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                 group=None,                      content='Parameter.i_maxHPUp_01'),
	"Witch's Thicket 12 - Stagnant Blight x10"             : LocationData(address=0x71727e, key='Forest_12_GAMEPLAY.BP_SCR_LV1M_2131_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Witch's Thicket 13 - Stagnant Blight x10"             : LocationData(address=0x71727f, key='Forest_13_GAMEPLAY.BP_SCR_LV1S_2130_3',                                   group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Witch's Thicket 14 - The Parish Way 3"                : LocationData(address=0x717280, key='Forest_14_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_017'),
	"Witch's Thicket 15 - Dark Witch Eleine"               : LocationData(address=0x717281, key='Forest_15_GAMEPLAY.BP_e5040_Witch',                                       group=LocationGroup.Boss,        content='Spirit.s5040'),
	"Witch's Thicket 16 - Eleine's Diary 3"                : LocationData(address=0x717282, key='Forest_16_GAMEPLAY.BP_Interactable_Item_Tip3',                            group=LocationGroup.Finding,     content='Tip.tip_057'),
	"Witch's Thicket 16 - Stone Tablet Fragment"           : LocationData(address=0x717283, key='Forest_16_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                  group=LocationGroup.Tablet,      content='Generic.i_FinalPassivePart_Up'),
	"Witch's Thicket 16 - Stagnant Blight x800"            : LocationData(address=0x717284, key='Forest_16_GAMEPLAY.BP_SCR_LV1LL_0000_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_LL'),
	"Witch's Thicket 17 - Amulet Gem"                      : LocationData(address=0x717285, key='Forest_17_GAMEPLAY.BP_Interactable_Item_MaxHPUp_02_Treasure_2',           group=None,                      content='Parameter.i_maxHPUp_02'),
	"Witch's Thicket 17 - Eleine's Diary 2"                : LocationData(address=0x717286, key='Forest_17_GAMEPLAY.BP_Interactable_Item_Tip4',                            group=LocationGroup.Finding,     content='Tip.tip_056'),
	"Witch's Thicket 17 - Spellbound Anklet"               : LocationData(address=0x717287, key='Forest_17_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                  group=None,                      content='Passive.i_passive_move_speed_up'),
	"Witch's Thicket 17 - Eleine's Diary 1"                : LocationData(address=0x717288, key='Forest_17_Map.BP_Interactable_Item_Tip5',                                 group=LocationGroup.Finding,     content='Tip.tip_055'),
	"Twin Spires 01 - Fallen Sentinel"                     : LocationData(address=0x717289, key='Fort_01_GAMEPLAY.BP_e2192_Gargoyle',                                      group=LocationGroup.SubSpirit,   content='Spirit.s2192'),
	"Twin Spires 01 - Furious Blight x30"                  : LocationData(address=0x71728a, key='Fort_01_GAMEPLAY.BP_SCR_LV1M_2190_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Twin Spires 02 - Stagnant Blight x10"                 : LocationData(address=0x71728b, key='Fort_02_GAMEPLAY.BP_SCR_LV1S_2021_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Twin Spires 02 - Furious Blight x10"                  : LocationData(address=0x71728c, key='Fort_02_GAMEPLAY.BP_SCR_LV2S_2021_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Twin Spires 03 - Bloodied Note 1"                     : LocationData(address=0x71728d, key='Fort_03_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_037'),
	"Twin Spires 04 - Stagnant Blight x100"                : LocationData(address=0x71728e, key='Fort_04_GAMEPLAY.BP_SCR_LV1L_2230_8',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Twin Spires 05 - Amulet Fragment"                     : LocationData(address=0x71728f, key='Fort_05_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                   group=None,                      content='Parameter.i_maxHPUp_01'),
	"Twin Spires 05 - Chain of Sorcery"                    : LocationData(address=0x717290, key='Fort_05_GAMEPLAY.BP_Interactable_Item_PassiveSlot_Drop',                  group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Twin Spires 05 - Stagnant Blight x10"                 : LocationData(address=0x717291, key='Fort_05_GAMEPLAY.BP_SCR_LV1S_2020_6',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Twin Spires 06 - Stagnant Blight x30"                 : LocationData(address=0x717292, key='Fort_06_GAMEPLAY.BP_SCR_LV1M_2190_4',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Twin Spires 07 - Stagnant Blight x10"                 : LocationData(address=0x717293, key='Fort_07_GAMEPLAY.BP_SCR_LV1S_2021_1',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Twin Spires 08 - Hoenir's Diary 3"                    : LocationData(address=0x717294, key='Fort_08_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_060'),
	"Twin Spires 09 - Note on the Castle Wall"             : LocationData(address=0x717295, key='Fort_09_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_041'),
	"Twin Spires 09 - Snowdrop Bracelet"                   : LocationData(address=0x717296, key='Fort_09_GAMEPLAY.BP_Interactable_Passive_dmgcut_Lv3_2',                   group=None,                      content='Passive.i_passive_dmgcut_LV3'),
	"Twin Spires 09 - Stagnant Blight x10"                 : LocationData(address=0x717297, key='Fort_09_GAMEPLAY.BP_SCR_LV1S_2020_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Twin Spires 09 - Furious Blight x30"                  : LocationData(address=0x717298, key='Fort_09_GAMEPLAY.BP_SCR_LV2M_2190_3',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Twin Spires 10 - Amulet Fragment"                     : LocationData(address=0x717299, key='Fort_10_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                      group=None,                      content='Parameter.i_maxHPUp_01'),
	"Twin Spires 10 - Bloodied Note 2"                     : LocationData(address=0x71729a, key='Fort_10_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_038'),
	"Twin Spires 11 - Ricorus' Ring"                       : LocationData(address=0x71729b, key='Fort_11_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                    group=None,                      content='Passive.i_passive_maxmpup'),
	"Twin Spires 11 - Stagnant Blight x30"                 : LocationData(address=0x71729c, key='Fort_11_GAMEPLAY.BP_SCR_LV1M_2190_4',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Twin Spires 12 - Forsaken Fellwyrm"                   : LocationData(address=0x71729d, key='Fort_12_GAMEPLAY.BP_e2232_Dragon',                                        group=LocationGroup.SubSpirit,   content='Spirit.s2232'),
	"Twin Spires 12 - Amulet Fragment"                     : LocationData(address=0x71729e, key='Fort_12_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_7',                      group=None,                      content='Parameter.i_maxHPUp_01'),
	"Twin Spires 13 - Amulet Fragment"                     : LocationData(address=0x71729f, key='Fort_13_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                      group=None,                      content='Parameter.i_maxHPUp_01'),
	"Twin Spires 13 - Chain of Sorcery"                    : LocationData(address=0x7172a0, key='Fort_13_GAMEPLAY.BP_Interactable_Item_PassiveSlot_Drop',                  group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Twin Spires 14 - Writing on the Wall"                 : LocationData(address=0x7172a1, key='Fort_14_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_061'),
	"Twin Spires 14 - Stone Tablet Fragment"               : LocationData(address=0x7172a2, key='Fort_14_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                    group=LocationGroup.Tablet,      content='Generic.i_FinalPassivePart_Up'),
	"Twin Spires 14 - Stagnant Blight x100"                : LocationData(address=0x7172a3, key='Fort_14_GAMEPLAY.BP_SCR_LV1L_2231_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Twin Spires 14 - Stagnant Blight x800"                : LocationData(address=0x7172a4, key='Fort_14_GAMEPLAY.BP_SCR_LV1LL_0000_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_LL'),
	"Twin Spires 14 - Furious Blight x10"                  : LocationData(address=0x7172a5, key='Fort_14_GAMEPLAY.BP_SCR_LV2S_2020_3',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Twin Spires 15 - Bloodied Note 3"                     : LocationData(address=0x7172a6, key='Fort_15_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_039'),
	"Twin Spires 15 - Immortal's Crest"                    : LocationData(address=0x7172a7, key='Fort_15_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                    group=None,                      content='Passive.i_passive_regenHP_attack'),
	"Twin Spires 15 - Stagnant Blight x30"                 : LocationData(address=0x7172a8, key='Fort_15_GAMEPLAY.BP_SCR_LV1M_2190_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Twin Spires 15 - Stagnant Blight x10"                 : LocationData(address=0x7172a9, key='Fort_15_GAMEPLAY.BP_SCR_LV1S_2021_1',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Twin Spires 16 - Monument Engraving"                  : LocationData(address=0x7172aa, key='Fort_16_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_040'),
	"Twin Spires 17 - Amulet Fragment"                     : LocationData(address=0x7172ab, key='Fort_17_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                      group=None,                      content='Parameter.i_maxHPUp_01'),
	"Twin Spires 18 - Amulet Gem"                          : LocationData(address=0x7172ac, key='Fort_18_GAMEPLAY.BP_Interactable_Item_MaxHPUp_02_Treasure_2',             group=None,                      content='Parameter.i_maxHPUp_02'),
	"Twin Spires 18 - Chain of Sorcery"                    : LocationData(address=0x7172ad, key='Fort_18_GAMEPLAY.BP_Interactable_Item_PassiveSlot_Drop',                  group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Twin Spires 19 - Ancient Dragon Claw"                 : LocationData(address=0x7172ae, key='Fort_19_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                    group=None,                      content='Passive.i_passive_dmgup_airborne'),
	"Twin Spires 19 - Furious Blight x30"                  : LocationData(address=0x7172af, key='Fort_19_GAMEPLAY.BP_SCR_LV2M_2190_4',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Twin Spires 19 - Furious Blight x10"                  : LocationData(address=0x7172b0, key='Fort_19_GAMEPLAY.BP_SCR_LV2S_2021_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Twin Spires 21 - Ulv, the Mad Knight"                 : LocationData(address=0x7172b1, key='Fort_21_GAMEPLAY.BP_e5070_Killer',                                        group=LocationGroup.Boss,        content='Spirit.s5070'),
	"Twin Spires 21 - Priestess' Wish"                     : LocationData(address=0x7172b2, key='Fort_21_GAMEPLAY.BP_Interactable_Item_HealPower_Up_2',                    group=None,                      content='Generic.i_HealPower_Up'),
	"Twin Spires 21 - White Priestess' Bastion Letter"     : LocationData(address=0x7172b3, key='Fort_21_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_036'),
	"Stockade 01 - Amulet Fragment"                        : LocationData(address=0x7172b4, key='Oubliette_01_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_5',                 group=None,                      content='Parameter.i_maxHPUp_01'),
	"Stockade 01 - Stagnant Blight x10"                    : LocationData(address=0x7172b5, key='Oubliette_01_GAMEPLAY.BP_SCR_LV1S_2010_3',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Stockade 02 - Executioner's Vow"                      : LocationData(address=0x7172b6, key='Oubliette_02_GAMEPLAY.BP_Interactable_Item_Tip3',                         group=LocationGroup.Finding,     content='Tip.tip_032'),
	"Stockade 03 - Right Stagnant Blight x10"              : LocationData(address=0x7172b7, key='Oubliette_03_GAMEPLAY.BP_SCR_LV1M_2050_2',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Stockade 03 - Left Stagnant Blight x10"               : LocationData(address=0x7172b8, key='Oubliette_03_GAMEPLAY.BP_SCR_LV1S_2081_2',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Stockade 03 - Stagnant Blight x800"                   : LocationData(address=0x7172b9, key='Oubliette_03_GEO.BP_SCR_LV1LL_0000_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_LL'),
	"Stockade 04 - Stagnant Blight x30"                    : LocationData(address=0x7172ba, key='Oubliette_04_GAMEPLAY.BP_SCR_LV1S_2020_2',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Stockade 04 - Furious Blight x10"                     : LocationData(address=0x7172bb, key='Oubliette_04_GAMEPLAY.BP_SCR_LV2S_2010_3',                                group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Stockade 05_1 - Furious Blight x10"                   : LocationData(address=0x7172bc, key='Oubliette_05_1_GAMEPLAY.BP_SCR_LV2S_2021_2',                              group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Stockade 05_2 - Stagnant Blight x10"                  : LocationData(address=0x7172bd, key='Oubliette_05_2_GAMEPLAY.BP_SCR_LV1S_2010_3',                              group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Stockade 05_3 - Chain of Sorcery"                     : LocationData(address=0x7172be, key='Oubliette_05_3_GAMEPLAY.BP_Interactable_Item_PassiveSlot_Drop',           group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Stockade 05 - Amulet Gem"                             : LocationData(address=0x7172bf, key='Oubliette_05_GAMEPLAY.BP_Interactable_Item_MaxHPUp_02_2',                 group=None,                      content='Parameter.i_maxHPUp_02'),
	"Stockade 05 - Stagnant Blight x100"                   : LocationData(address=0x7172c0, key='Oubliette_05_GAMEPLAY.BP_SCR_LV1L_2220_3',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Stockade 05 - Stagnant Blight x30"                    : LocationData(address=0x7172c1, key='Oubliette_05_GAMEPLAY.BP_SCR_LV1S_2021_2',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Stockade 06_1 - Hoenir's Diary 1"                     : LocationData(address=0x7172c2, key='Oubliette_06_1_GAMEPLAY.BP_Interactable_Item_Tip3',                       group=LocationGroup.Finding,     content='Tip.tip_058'),
	"Stockade 06_2 - Executioner's Gloves"                 : LocationData(address=0x7172c3, key='Oubliette_06_2_GAMEPLAY.BP_Interactable_Passive_dmgup_maxHP_2',           group=None,                      content='Passive.i_passive_dmgup_maxhp'),
	"Stockade 06_3 - Chain of Sorcery"                     : LocationData(address=0x7172c4, key='Oubliette_06_3_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',              group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Stockade 06_4 - Stone Tablet Fragment"                : LocationData(address=0x7172c5, key='Oubliette_06_4_GAMEPLAY.BP_Interactable_Item_FinalPassivePart_2',         group=LocationGroup.Tablet,      content='Generic.i_FinalPassivePart_Up'),
	"Stockade 06 - Amulet Fragment"                        : LocationData(address=0x7172c6, key='Oubliette_06_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',              group=None,                      content='Parameter.i_maxHPUp_01'),
	"Stockade 06 - Chain of Sorcery"                       : LocationData(address=0x7172c7, key='Oubliette_06_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',                group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Stockade 06 - Stagnant Blight x10"                    : LocationData(address=0x7172c8, key='Oubliette_06_GAMEPLAY.BP_SCR_LV1S_2021_2',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Stockade 07_1 - Forbidden Text Scrap"                 : LocationData(address=0x7172c9, key='Oubliette_07_1_GAMEPLAY.BP_Interactable_Item_Tip3',                       group=LocationGroup.Finding,     content='Tip.tip_014'),
	"Stockade 07_2 - Hidden Test Subject"                  : LocationData(address=0x7172ca, key='Oubliette_07_2_GAMEPLAY.BP_e2072_Mimic',                                  group=LocationGroup.SubSpirit,   content='Spirit.s2072'),
	"Stockade 08 - Lower Furious Blight x30"               : LocationData(address=0x7172cb, key='Oubliette_08_GAMEPLAY.BP_SCR_LV2M_2050_3',                                group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Stockade 08 - Upper Furious Blight x30"               : LocationData(address=0x7172cc, key='Oubliette_08_GAMEPLAY.BP_SCR_LV2S_2081_2',                                group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Stockade 09 - Right Amulet Fragment"                  : LocationData(address=0x7172cd, key='Oubliette_09_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                 group=None,                      content='Parameter.i_maxHPUp_01'),
	"Stockade 09 - Left Amulet Fragment"                   : LocationData(address=0x7172ce, key='Oubliette_09_GAMEPLAY.BP_Interactable_Item_MaxHPUp_5',                    group=None,                      content='Parameter.i_maxHPUp_01'),
	"Stockade 09 - Stagnant Blight x30"                    : LocationData(address=0x7172cf, key='Oubliette_09_GAMEPLAY.BP_SCR_LV1M_2050_2',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Stockade 10 - Executioner's Missive"                  : LocationData(address=0x7172d0, key='Oubliette_10_GAMEPLAY.BP_Interactable_Item_Tip3',                         group=LocationGroup.Finding,     content='Tip.tip_034'),
	"Stockade 11 - Furious Blight x10"                     : LocationData(address=0x7172d1, key='Oubliette_11_GAMEPLAY.BP_SCR_LV2S_2020_2',                                group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Stockade 12 - Dark Executioner"                       : LocationData(address=0x7172d2, key='Oubliette_12_GAMEPLAY.BP_e2182_Shadow',                                   group=LocationGroup.SubSpirit,   content='Spirit.s2182'),
	"Stockade 13_1 - Aura's Ring"                          : LocationData(address=0x7172d3, key='Oubliette_13_1_GAMEPLAY.BP_Interactable_Passives_Treasure_2',             group=None,                      content='Passive.i_passive_spirit_maxcast_count_up_LV2'),
	"Stockade 13_2 - Ancient Soul x1"                      : LocationData(address=0x7172d4, key='Oubliette_13_2_GAMEPLAY.BP_SCR_LV3S_5000_1',                              group=None,                      content='Generic.i_SpiritCurrencyLv3_1'),
	"Stockade 13 - Chain of Sorcery"                       : LocationData(address=0x7172d5, key='Oubliette_13_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',                group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Stockade 13 - Slip of Paper"                          : LocationData(address=0x7172d6, key='Oubliette_13_GAMEPLAY.BP_Interactable_Item_Tip3',                         group=LocationGroup.Finding,     content='Tip.tip_033'),
	"Stockade 13 - Stagnant Blight x30"                    : LocationData(address=0x7172d7, key='Oubliette_13_GAMEPLAY.BP_SCR_LV1M_2001_3',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Stockade 13 - Stagnant Blight x100"                   : LocationData(address=0x7172d8, key='Oubliette_13_GAMEPLAY.BP_SCR_LV1M_2180_2',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Stockade 14 - Hoenir's Diary 2"                       : LocationData(address=0x7172d9, key='Oubliette_14_GAMEPLAY.BP_Interactable_Item_Tip3',                         group=LocationGroup.Finding,     content='Tip.tip_059'),
	"Stockade 15 - Hoenir, Keeper of the Abyss"            : LocationData(address=0x7172da, key='Oubliette_15_GAMEPLAY.BP_e5060_Assassin',                                 group=LocationGroup.Boss,        content='Spirit.s5060'),
	"Stockade 16 - Priestess' Wish"                        : LocationData(address=0x7172db, key='Oubliette_16_GAMEPLAY.BP_Interactable_Item_HealPower_Up_2',               group=None,                      content='Generic.i_HealPower_Up'),
	"Stockade 16 - Stagnant Blight x10"                    : LocationData(address=0x7172dc, key='Oubliette_16_GAMEPLAY.BP_SCR_LV1L_2180_3',                                group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Hinterlands 03 - King of the First Age's Torn Note 1" : LocationData(address=0x7172dd, key='Outside_03_GAMEPLAY.BP_Interactable_Item_Tip3',                           group=LocationGroup.Finding,     content='Tip.tip_045'),
	"Hinterlands 03 - King of the First Age's Torn Note 2" : LocationData(address=0x7172de, key='Outside_03_GAMEPLAY.BP_Interactable_Item_Tip4',                           group=LocationGroup.Finding,     content='Tip.tip_046'),
	"Hinterlands 03 - Nymphilia's Ring"                    : LocationData(address=0x7172df, key='Outside_03_GAMEPLAY.BP_Interactable_Passive_ShortHeal_2',                 group=None,                      content='Passive.i_passive_shortheal'),
	"Hinterlands 03 - Stagnant Blight x100"                : LocationData(address=0x7172e0, key='Outside_03_GAMEPLAY.BP_SCR_LV2L_2230_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Verboten Domain 01 - Faden's Archives 3"              : LocationData(address=0x7172ef, key='Swamp_1_GAMEPLAY.BP_Interactable_Item_Tip3',                              group=LocationGroup.Finding,     content='Tip.tip_051'),
	"Verboten Domain 02 - Furious Blight x10"              : LocationData(address=0x717302, key='Swamp_2_GAMEPLAY.BP_SCR_LV2S_2010_2',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Verboten Domain 02 - Furious Blight x30"              : LocationData(address=0x717303, key='Swamp_2_GAMEPLAY.BP_SCR_LV2S_2121_1',                                     group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Verboten Domain 03 - Amulet Fragment"                 : LocationData(address=0x717304, key='Swamp_3_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                   group=None,                      content='Parameter.i_maxHPUp_01'),
	"Verboten Domain 03 - Stagnant Blight x30"             : LocationData(address=0x717305, key='Swamp_3_GAMEPLAY.BP_SCR_LV1L_2170_3',                                     group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Verboten Domain 04 - Verboten Champion"               : LocationData(address=0x7172e1, key='Swamp_04_GAMEPLAY.BP_e2172_Inferior',                                     group=LocationGroup.SubSpirit,   content='Spirit.s2172'),
	"Verboten Domain 04 - Chain of Sorcery"                : LocationData(address=0x7172e2, key='Swamp_04_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',                    group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Verboten Domain 05 - Furious Blight x100"             : LocationData(address=0x7172e3, key='Swamp_05_GAMEPLAY.BP_SCR_LV2M_2171_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv2_L'),
	"Verboten Domain 05 - Stagnant Blight x30"             : LocationData(address=0x7172e4, key='Swamp_05_GEO.BP_SCR_LV1M_2051_2',                                         group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Verboten Domain 06 - Verboten Domain Notice"          : LocationData(address=0x7172e5, key='Swamp_06_GAMEPLAY.BP_Interactable_Item_Tip3',                             group=LocationGroup.Finding,     content='Tip.tip_048'),
	"Verboten Domain 06 - Heretic's Mask"                  : LocationData(address=0x7172e6, key='Swamp_06_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                   group=None,                      content='Passive.i_passive_ignore_damage_area'),
	"Verboten Domain 06 - Stagnant Blight x30"             : LocationData(address=0x7172e7, key='Swamp_06_GAMEPLAY.BP_SCR_LV1L_2180_3',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Verboten Domain 07 - Amulet Fragment"                 : LocationData(address=0x7172e8, key='Swamp_07_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                     group=None,                      content='Parameter.i_maxHPUp_01'),
	"Verboten Domain 07 - Calivia's Ring"                  : LocationData(address=0x7172e9, key='Swamp_07_GAMEPLAY.BP_Interactable_Passive_RecastTimeCut_Lv2_2',           group=None,                      content='Passive.i_passive_recast_time_cut_LV2'),
	"Verboten Domain 08 - Amulet Gem"                      : LocationData(address=0x7172ea, key='Swamp_08_GAMEPLAY.BP_Interactable_Item_MaxHPUp_02_2',                     group=None,                      content='Parameter.i_maxHPUp_02'),
	"Verboten Domain 08 - Stagnant Blight x100"            : LocationData(address=0x7172eb, key='Swamp_08_GAMEPLAY.BP_SCR_LV1M_2051_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Verboten Domain 08 - Furious Blight x100"             : LocationData(address=0x7172ec, key='Swamp_08_GAMEPLAY.BP_SCR_LV2M_2051_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv2_L'),
	"Verboten Domain 09 - Weathered Necklace"              : LocationData(address=0x7172ed, key='Swamp_09_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                   group=None,                      content='Passive.i_passive_regenHP_kill'),
	"Verboten Domain 09 - Stagnant Blight x30"             : LocationData(address=0x7172ee, key='Swamp_09_GAMEPLAY.BP_SCR_LV2M_2170_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Verboten Domain 10 - Faden's Archives 4"              : LocationData(address=0x7172f0, key='Swamp_10_GAMEPLAY.BP_Interactable_Item_Tip3',                             group=LocationGroup.Finding,     content='Tip.tip_052'),
	"Verboten Domain 10 - Stagnant Blight x30"             : LocationData(address=0x7172f1, key='Swamp_10_GAMEPLAY.BP_SCR_LV1M_2051_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Verboten Domain 11 - Amulet Fragment"                 : LocationData(address=0x7172f2, key='Swamp_11_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                     group=None,                      content='Parameter.i_maxHPUp_01'),
	"Verboten Domain 11 - Blighted Appendage"              : LocationData(address=0x7172f3, key='Swamp_11_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                   group=None,                      content='Passive.i_passive_dmgup'),
	"Verboten Domain 12 - Priestess' Wish"                 : LocationData(address=0x7172f4, key='Swamp_12_GAMEPLAY.BP_Interactable_Item_HealPower_Up_2',                   group=None,                      content='Generic.i_HealPower_Up'),
	"Verboten Domain 12 - Faden's Archives 2"              : LocationData(address=0x7172f5, key='Swamp_12_GAMEPLAY.BP_Interactable_Item_Tip3',                             group=LocationGroup.Finding,     content='Tip.tip_050'),
	"Verboten Domain 13 - Amulet Fragment"                 : LocationData(address=0x7172f6, key='Swamp_13_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                     group=None,                      content='Parameter.i_maxHPUp_01'),
	"Verboten Domain 13 - Stagnant Blight x30"             : LocationData(address=0x7172f7, key='Swamp_13_GAMEPLAY.BP_SCR_LV1L_2170_3',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Verboten Domain 13 - Furious Blight x30"              : LocationData(address=0x7172f8, key='Swamp_13_GAMEPLAY.BP_SCR_LV2L_2170_3',                                    group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Verboten Domain 14 - Chain of Sorcery"                : LocationData(address=0x7172f9, key='Swamp_14_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',                    group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Verboten Domain 14 - Stagnant Blight x100"            : LocationData(address=0x7172fa, key='Swamp_14_GAMEPLAY.BP_SCR_LV1L_2170_3',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_L'),
	"Verboten Domain 15 - Incompetent Sinner"              : LocationData(address=0x7172fb, key='Swamp_15_GAMEPLAY.BP_e2052_Toad',                                         group=LocationGroup.SubSpirit,   content='Spirit.s2052'),
	"Verboten Domain 15 - Stagnant Blight x30"             : LocationData(address=0x7172fc, key='Swamp_15_GAMEPLAY.BP_SCR_LV1M_2171_2',                                    group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Verboten Domain 16 - Faden's Archives 1"              : LocationData(address=0x7172fd, key='Swamp_16_GAMEPLAY.BP_Interactable_Item_Tip3',                             group=LocationGroup.Finding,     content='Tip.tip_049'),
	"Verboten Domain 18 - Miriel's Blighted Letter"        : LocationData(address=0x7172fe, key='Swamp_18_GAMEPLAY.BP_Interactable_Item_Tip3',                             group=LocationGroup.Finding,     content='Tip.tip_047'),
	"Verboten Domain 18 - Faden's Archives 5"              : LocationData(address=0x7172ff, key='Swamp_18_GAMEPLAY.BP_Interactable_Item_Tip4',                             group=LocationGroup.Finding,     content='Tip.tip_053'),
	"Verboten Domain 18 - Faden, the Heretic"              : LocationData(address=0x717300, key='Swamp_18_GAMEPLAY.BP_Interactable_Spirit_s5080_2',                        group=LocationGroup.Boss,        content='Spirit.s5080'),
	"Verboten Domain 18 - Stagnant Blight x800"            : LocationData(address=0x717301, key='Swamp_18_GAMEPLAY.BP_SCR_LV1LL_miliel_2',                                 group=None,                      content='Generic.i_SpiritCurrencyLv1_LL'),
	"Cliffside Hamlet 02 - Amulet Fragment"                : LocationData(address=0x717306, key='Village_02_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_Drop',                group=None,                      content='Parameter.i_maxHPUp_01'),
	"Cliffside Hamlet 02 - Left Stagnant Blight x10"       : LocationData(address=0x717307, key='Village_02_GAMEPLAY.BP_SCR_LV1S_2020_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 02 - Right Stagnant Blight x10"      : LocationData(address=0x717308, key='Village_02_GAMEPLAY.BP_SCR_LV1S_2100_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 03 - Amulet Fragment"                : LocationData(address=0x717309, key='Village_03_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                   group=None,                      content='Parameter.i_maxHPUp_01'),
	"Cliffside Hamlet 03 - Broken Music Box"               : LocationData(address=0x71730a, key='Village_03_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                 group=None,                      content='Passive.i_passive_dmgcut_LV1'),
	"Cliffside Hamlet 03 - Stagnant Blight x10"            : LocationData(address=0x71730b, key='Village_03_GAMEPLAY.BP_SCR_LV1S_2021_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 03 - Furious Blight x10"             : LocationData(address=0x71730c, key='Village_03_GAMEPLAY.BP_SCR_LV2S_2020_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Cliffside Hamlet 04_1 - Western Merchant"             : LocationData(address=0x71730d, key='Village_04_1_GAMEPLAY.BP_e2102_Crow',                                     group=LocationGroup.SubSpirit,   content='Spirit.s2102'),
	"Cliffside Hamlet 04 - Amulet Fragment"                : LocationData(address=0x71730e, key='Village_04_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                   group=None,                      content='Parameter.i_maxHPUp_01'),
	"Cliffside Hamlet 04 - Bloodstained Ribbon"            : LocationData(address=0x71730f, key='Village_04_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                 group=None,                      content='Passive.i_passive_expup_1'),
	"Cliffside Hamlet 04 - Stagnant Blight x10"            : LocationData(address=0x717310, key='Village_04_GAMEPLAY.BP_SCR_LV1S_2010_1',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 05 - True Believer's Note"           : LocationData(address=0x717311, key='Village_05_GAMEPLAY.BP_Interactable_Item_Tip4',                           group=LocationGroup.Finding,     content='Tip.tip_009'),
	"Cliffside Hamlet 06 - Amulet Fragment"                : LocationData(address=0x717312, key='Village_06_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                   group=None,                      content='Parameter.i_maxHPUp_01'),
	"Cliffside Hamlet 06 - Upper Stagnant Blight x10"      : LocationData(address=0x717313, key='Village_06_GAMEPLAY.BP_SCR_LV1M_2000_3',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 06 - Left Stagnant Blight x10"       : LocationData(address=0x717314, key='Village_06_GAMEPLAY.BP_SCR_LV1S_2020_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 06 - Right Stagnant Blight x10"      : LocationData(address=0x717315, key='Village_06_GAMEPLAY.BP_SCR_LV1S_2021_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 06 - Upper Furious Blight x10"       : LocationData(address=0x717316, key='Village_06_GAMEPLAY.BP_SCR_LV2S_2010_3',                                  group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Cliffside Hamlet 06 - Lower Furious Blight x10"       : LocationData(address=0x717317, key='Village_06_GAMEPLAY.BP_SCR_LV2S_2021_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Cliffside Hamlet 07 - Chain of Sorcery"               : LocationData(address=0x717318, key='Village_07_GAMEPLAY.BP_Interactable_Item_PassiveSlot_2',                  group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Cliffside Hamlet 07 - Stagnant Blight x10"            : LocationData(address=0x717319, key='Village_07_GAMEPLAY.BP_SCR_LV1S_2100_3',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 07 - Furious Blight x10"             : LocationData(address=0x71731a, key='Village_07_GAMEPLAY.BP_SCR_LV2S_2080_3',                                  group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Cliffside Hamlet 08 - Headless Defender"              : LocationData(address=0x71731b, key='Village_08_GAMEPLAY.BP_e2002_Knight',                                     group=LocationGroup.SubSpirit,   content='Spirit.s2002'),
	"Cliffside Hamlet 08 - Amulet Fragment"                : LocationData(address=0x71731c, key='Village_08_GAMEPLAY.BP_Interactable_Item_MaxHPUp_01_2',                   group=None,                      content='Parameter.i_maxHPUp_01'),
	"Cliffside Hamlet 08 - Stagnant Blight x10"            : LocationData(address=0x71731d, key='Village_08_GAMEPLAY.BP_SCR_LV1S_2021_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 09 - Adherent's Letter"              : LocationData(address=0x71731e, key='Village_09_GAMEPLAY.BP_Interactable_Item_Tip3',                           group=LocationGroup.Finding,     content='Tip.tip_018'),
	"Cliffside Hamlet 09 - Hamlet Request 1"               : LocationData(address=0x71731f, key='Village_09_GAMEPLAY.BP_Interactable_Item_Tip4',                           group=LocationGroup.Finding,     content='Tip.tip_007'),
	"Cliffside Hamlet 10 - Gerrod, the Elder Warrior"      : LocationData(address=0x717320, key='Village_10_GAMEPLAY.BP_e5050_Giant',                                      group=LocationGroup.Boss,        content='Spirit.s5050'),
	"Cliffside Hamlet 11_1 - Priestess' Wish"              : LocationData(address=0x717321, key='Village_11_1_GAMEPLAY.BP_Interactable_Item_HealPower_Up_5',               group=None,                      content='Generic.i_HealPower_Up'),
	"Cliffside Hamlet 11 - Hamlet Scrawl"                  : LocationData(address=0x717322, key='Village_11_GAMEPLAY.BP_Interactable_Item_Tip1',                           group=LocationGroup.Finding,     content='Tip.tip_006'),
	"Cliffside Hamlet 11 - Hamlet Request 2"               : LocationData(address=0x717323, key='Village_11_GAMEPLAY.BP_Interactable_Item_Tip4',                           group=LocationGroup.Finding,     content='Tip.tip_008'),
	"Cliffside Hamlet 11 - Furious Blight x10"             : LocationData(address=0x717324, key='Village_11_GAMEPLAY.BP_SCR_LV2S_2020_3',                                  group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Cliffside Hamlet 12 - Stagnant Blight x10"            : LocationData(address=0x717325, key='Village_12_GAMEPLAY.BP_SCR_LV1S_2021_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 13 - Amulet Gem"                     : LocationData(address=0x717326, key='Village_13_GAMEPLAY.BP_Interactable_Item_MaxHPUp_02_Treasure_2',          group=None,                      content='Parameter.i_maxHPUp_02'),
	"Cliffside Hamlet 13 - Chain of Sorcery"               : LocationData(address=0x717327, key='Village_13_GAMEPLAY.BP_Interactable_Item_PassiveSlot_Drop',               group=None,                      content='Generic.i_PassiveSlot_Up'),
	"Cliffside Hamlet 13 - Giant's Ring"                   : LocationData(address=0x717328, key='Village_13_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                 group=None,                      content='Passive.i_passive_dmgup_grounded'),
	"Cliffside Hamlet 13 - Stagnant Blight x10"            : LocationData(address=0x717329, key='Village_13_GAMEPLAY.BP_SCR_LV1S_2021_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_S'),
	"Cliffside Hamlet 14 - Cliffside Hamlet Elder"         : LocationData(address=0x71732a, key='Village_14_GAMEPLAY.BP_e2112_Ork',                                        group=LocationGroup.SubSpirit,   content='Spirit.s2112'),
	"Cliffside Hamlet 14 - Stagnant Blight x30"            : LocationData(address=0x71732b, key='Village_14_GAMEPLAY.BP_SCR_LV1L_2111_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Cliffside Hamlet 14 - Furious Blight x30"             : LocationData(address=0x71732c, key='Village_14_GAMEPLAY.BP_SCR_LV2L_2110_3',                                  group=None,                      content='Generic.i_SpiritCurrencyLv2_M'),
	"Cliffside Hamlet 15 - Stagnant Blight x800"           : LocationData(address=0x71732d, key='Village_15_GAMEPLAY.BP_SCR_LV1LL_0000_2',                                 group=None,                      content='Generic.i_SpiritCurrencyLv1_LL'),
	"Cliffside Hamlet 15 - Stagnant Blight x30"            : LocationData(address=0x71732e, key='Village_15_GAMEPLAY.BP_SCR_LV1M_2050_3',                                  group=None,                      content='Generic.i_SpiritCurrencyLv1_M'),
	"Cliffside Hamlet 15 - Furious Blight x10"             : LocationData(address=0x71732f, key='Village_15_GAMEPLAY.BP_SCR_LV2M_2050_2',                                  group=None,                      content='Generic.i_SpiritCurrencyLv2_S'),
	"Cliffside Hamlet 16 - Faden's Letter"                 : LocationData(address=0x717330, key='Village_16_GAMEPLAY.BP_Interactable_Item_Tip4',                           group=LocationGroup.Finding,     content='Tip.tip_054'),
	"Cliffside Hamlet 16 - Stone Tablet Fragment"          : LocationData(address=0x717331, key='Village_16_GAMEPLAY.BP_Interactable_Passives_Treasure_2',                 group=LocationGroup.Tablet,      content='Generic.i_FinalPassivePart_Up'),
	'Starting Spirit'                                      : LocationData(address=0x717332, key='starting_weapon',                                                         group=None,                      content='Spirit.s5000'),

# entrances and events
	'CathedralCloister'                                    : LocationData(content='CathedralCloister'),
	'MourningHall'                                         : LocationData(content='MourningHall'),
	'MonumentOfTheWindStart'                               : LocationData(content='MonumentOfTheWindStart'),
    
    "The Abyss 01 To Verboten Domain 18"                   : LocationData(key='Abyss_01_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_18.A1'),
    "The Abyss 01 To The Abyss 02"                         : LocationData(key='Abyss_01_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_abyss_02.1'),
    "The Abyss 02 To The Abyss 03"                         : LocationData(key='Abyss_02_GAMEPLAY.BP_WorldTravelVolume',                        content='Map.map_abyss_03.2'),
    "The Abyss 02 To The Abyss 01"                         : LocationData(key='Abyss_02_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_abyss_01.2'),
    "The Abyss 03 To The Abyss 02"                         : LocationData(key='Abyss_03_GAMEPLAY.BP_WorldTravelVolume3_8',                     content='Map.map_abyss_02.3'),
    "The Abyss 04 To Verboten Domain 12"                   : LocationData(key='Abyss_04_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_12.4'),
    "The Abyss 04 To The Abyss 05"                         : LocationData(key='Abyss_04_GAMEPLAY.BP_WorldTravelVolume4',                       content='Map.map_abyss_05.4'),
    "The Abyss 05 To Verboten Domain 12"                   : LocationData(key='Abyss_05_GAMEPLAY.BP_Interactable_WorldTravel_2',               content='Map.map_swamp_12.A5'),
    "The Abyss 05 To The Abyss 04"                         : LocationData(key='Abyss_05_GAMEPLAY.BP_WorldTravelVolume',                        content='Map.map_abyss_04.5'),
    "Ruined Castle 01 To Cliffside Hamlet 11"              : LocationData(key='Castle_01_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_village_11.1'),
    "Ruined Castle 01 To Ruined Castle 02 Lower"           : LocationData(key='Castle_01_GAMEPLAY.BP_WorldTravelVolume4',                      content='Map.map_castle_02.1'),
    "Ruined Castle 01 To Ruined Castle 02 Upper"           : LocationData(key='Castle_01_GAMEPLAY.BP_WorldTravelVolume5',                      content='Map.map_castle_02.1B'),
    "Ruined Castle 01 To Ruined Castle 10"                 : LocationData(key='Castle_01_GAMEPLAY.BP_WorldTravelVolume6',                      content='Map.map_castle_10.1'),
    "Ruined Castle 02 To Ruined Castle 05"                 : LocationData(key='Castle_02_GAMEPLAY.BP_Interactable_WorldTravel_2',              content='Map.map_castle_05.2'),
    "Ruined Castle 02 To Ruined Castle 01 Lower"           : LocationData(key='Castle_02_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_01.2'),
    "Ruined Castle 02 To Ruined Castle 04"                 : LocationData(key='Castle_02_GAMEPLAY.BP_WorldTravelVolume2_5',                    content='Map.map_castle_04.2'),
    "Ruined Castle 02 To Ruined Castle 01 Upper"           : LocationData(key='Castle_02_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_castle_01.2B'),
    "Ruined Castle 03 To Ruined Castle 05"                 : LocationData(key='Castle_03_GAMEPLAY.BP_WorldTravelVolume4',                      content='Map.map_castle_05.3'),
    "Ruined Castle 03 To Ruined Castle 11 Left"            : LocationData(key='Castle_03_GAMEPLAY.BP_WorldTravelVolume5',                      content='Map.map_castle_11.3A'),
    "Ruined Castle 03 To Ruined Castle 11 Right"           : LocationData(key='Castle_03_GAMEPLAY.BP_WorldTravelVolume6',                      content='Map.map_castle_11.3B'),
    "Ruined Castle 04 To Ruined Castle 02"                 : LocationData(key='Castle_04_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_02.4'),
    "Ruined Castle 05 To Ruined Castle 02"                 : LocationData(key='Castle_05_GAMEPLAY.BP_Interactable_WorldTravel_2',              content='Map.map_castle_02.5'),
    "Ruined Castle 05 To Ruined Castle 06"                 : LocationData(key='Castle_05_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_06.5'),
    "Ruined Castle 05 To Ruined Castle 08"                 : LocationData(key='Castle_05_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_08.5'),
    "Ruined Castle 05 To Ruined Castle 03"                 : LocationData(key='Castle_05_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_castle_03.5'),
    "Ruined Castle 06 To Ruined Castle 12"                 : LocationData(key='Castle_06_GAMEPLAY.BP_Interactable_WorldTravel_2',              content='Map.map_castle_12.6'),
    "Ruined Castle 06 To Ruined Castle 07"                 : LocationData(key='Castle_06_GAMEPLAY.BP_Interactable_WorldTravel2',               content='Map.map_castle_07.6'),
    "Ruined Castle 06 To Ruined Castle 05"                 : LocationData(key='Castle_06_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_05.6'),
    "Ruined Castle 07 To Twin Spires 01"                   : LocationData(key='Castle_07_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_fort_01.C7'),
    "Ruined Castle 07 To Ruined Castle 06"                 : LocationData(key='Castle_07_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_06.7'),
    "Ruined Castle 08 To Ruined Castle 05"                 : LocationData(key='Castle_08_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_05.8'),
    "Ruined Castle 08 To Ruined Castle 09"                 : LocationData(key='Castle_08_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_09.8'),
    "Ruined Castle 09 To Ruined Castle 11"                 : LocationData(key='Castle_09_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_11.9'),
    "Ruined Castle 09 To Ruined Castle 08"                 : LocationData(key='Castle_09_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_08.9'),
    "Ruined Castle 09 To Ruined Castle 10"                 : LocationData(key='Castle_09_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_castle_10.9'),
    "Ruined Castle 10 To Ruined Castle 09"                 : LocationData(key='Castle_10_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_09.10'),
    "Ruined Castle 10 To Ruined Castle 01"                 : LocationData(key='Castle_10_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_01.10'),
    "Ruined Castle 11 To Ruined Castle 12"                 : LocationData(key='Castle_11_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_12.11'),
    "Ruined Castle 11 To Ruined Castle 09"                 : LocationData(key='Castle_11_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_09.11'),
    "Ruined Castle 11 To Ruined Castle 13"                 : LocationData(key='Castle_11_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_castle_13.11'),
    "Ruined Castle 11 To Ruined Castle 03 Left"            : LocationData(key='Castle_11_GAMEPLAY.BP_WorldTravelVolume4',                      content='Map.map_castle_03.11A'),
    "Ruined Castle 11 To Ruined Castle 03 Right"           : LocationData(key='Castle_11_GAMEPLAY.BP_WorldTravelVolume5',                      content='Map.map_castle_03.11B'),
    "Ruined Castle 12 To Ruined Castle 06"                 : LocationData(key='Castle_12_GAMEPLAY.BP_Interactable_WorldTravel_2',              content='Map.map_castle_06.12'),
    "Ruined Castle 12 To Ruined Castle 11"                 : LocationData(key='Castle_12_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_11.12'),
    "Ruined Castle 12 To Ruined Castle 21"                 : LocationData(key='Castle_12_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_castle_21.12'),
    "Ruined Castle 13 To Ruined Castle 17"                 : LocationData(key='Castle_13_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_17.13'),
    "Ruined Castle 13 To Ruined Castle 11"                 : LocationData(key='Castle_13_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_castle_11.13'),
    "Ruined Castle 13 To Ruined Castle 14"                 : LocationData(key='Castle_13_GAMEPLAY.BP_WorldTravelVolume4',                      content='Map.map_castle_14.13'),
    "Ruined Castle 14 To Ruined Castle 13"                 : LocationData(key='Castle_14_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_13.14'),
    "Ruined Castle 14 To Ruined Castle 15"                 : LocationData(key='Castle_14_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_castle_15.14'),
    "Ruined Castle 15 To Ruined Castle 16"                 : LocationData(key='Castle_15_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_16.15'),
    "Ruined Castle 15 To Ruined Castle 14"                 : LocationData(key='Castle_15_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_14.15'),
    "Ruined Castle 16 To Ruined Castle 18"                 : LocationData(key='Castle_16_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_18.16'),
    "Ruined Castle 16 To Ruined Castle 15"                 : LocationData(key='Castle_16_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_15.16'),
    "Ruined Castle 17 To Ruined Castle 13"                 : LocationData(key='Castle_17_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_13.17'),
    "Ruined Castle 17 To Ruined Castle 18"                 : LocationData(key='Castle_17_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_castle_18.17'),
    "Ruined Castle 18 To Ruined Castle 19"                 : LocationData(key='Castle_18_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_19.18'),
    "Ruined Castle 18 To Ruined Castle 17"                 : LocationData(key='Castle_18_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_17.18'),
    "Ruined Castle 19 To Ruined Castle 20"                 : LocationData(key='Castle_19_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_20.19'),
    "Ruined Castle 19 To Ruined Castle 18"                 : LocationData(key='Castle_19_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_18.19'),
    "Ruined Castle 20 To Ruined Castle 19"                 : LocationData(key='Castle_20_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_castle_19.20'),
    "Ruined Castle 21 To Ruined Castle 12"                 : LocationData(key='Castle_21_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_castle_12.21'),
    "Catacombs 01 To Catacombs 02"                         : LocationData(key='Cave_01_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_02.1'),
    "Catacombs 01 To Cliffside Hamlet 12"                  : LocationData(key='Cave_01_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_village_12.C1'),
    "Catacombs 02 To Catacombs 05"                         : LocationData(key='Cave_02_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_05.2'),
    "Catacombs 02 To Catacombs 07"                         : LocationData(key='Cave_02_GAMEPLAY.BP_WorldTravelVolume4_1',                      content='Map.map_cave_07.2'),
    "Catacombs 02 To Catacombs 01"                         : LocationData(key='Cave_02_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_cave_01.2'),
    "Catacombs 03 To Catacombs 07"                         : LocationData(key='Cave_03_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_07.3'),
    "Catacombs 03 To Catacombs 08"                         : LocationData(key='Cave_03_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_08.3'),
    "Catacombs 03 To Catacombs 06"                         : LocationData(key='Cave_03_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_cave_06.3'),
    "Catacombs 04 To Catacombs 16"                         : LocationData(key='Cave_04_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_16.4'),
    "Catacombs 04 To Catacombs 05"                         : LocationData(key='Cave_04_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_05.4'),
    "Catacombs 04 To Catacombs 12"                         : LocationData(key='Cave_04_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_12.4'),
    "Catacombs 05 To Catacombs 02"                         : LocationData(key='Cave_05_GAMEPLAY.BP_WorldTravelVolume_0',                       content='Map.map_cave_02.5'),
    "Catacombs 05 To Catacombs 04"                         : LocationData(key='Cave_05_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_04.5'),
    "Catacombs 05 To Catacombs 06"                         : LocationData(key='Cave_05_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_06.5'),
    "Catacombs 05 To Catacombs 10"                         : LocationData(key='Cave_05_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_10.5'),
    "Catacombs 06 To Catacombs 05"                         : LocationData(key='Cave_06_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_05.6'),
    "Catacombs 06 To Catacombs 03"                         : LocationData(key='Cave_06_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_03.6'),
    "Catacombs 07 To Catacombs 02"                         : LocationData(key='Cave_07_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_02.7'),
    "Catacombs 07 To Catacombs 03"                         : LocationData(key='Cave_07_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_03.7'),
    "Catacombs 08 To Catacombs 09"                         : LocationData(key='Cave_08_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_09.8'),
    "Catacombs 08 To Catacombs 03"                         : LocationData(key='Cave_08_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_03.8'),
    "Catacombs 08 To Catacombs 17"                         : LocationData(key='Cave_08_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_17.8'),
    "Catacombs 08 To Catacombs 11"                         : LocationData(key='Cave_08_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_cave_11.8'),
    "Catacombs 09 To Catacombs 10"                         : LocationData(key='Cave_09_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_10.9'),
    "Catacombs 09 To Catacombs 08"                         : LocationData(key='Cave_09_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_cave_08.9'),
    "Catacombs 09 To Catacombs 21"                         : LocationData(key='Cave_09_GAMEPLAY.BP_WorldTravelVolume6',                        content='Map.map_cave_21.9'),
    "Catacombs 10 To Catacombs 23"                         : LocationData(key='Cave_10_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_23.10'),
    "Catacombs 10 To Catacombs 05"                         : LocationData(key='Cave_10_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_05.10'),
    "Catacombs 10 To Catacombs 09"                         : LocationData(key='Cave_10_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_cave_09.10'),
    "Catacombs 10 To Catacombs 16"                         : LocationData(key='Cave_10_GAMEPLAY.BP_WorldTravelVolume6',                        content='Map.map_cave_16.10'),
    "Catacombs 11 To Catacombs 08"                         : LocationData(key='Cave_11_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_08.11'),
    "Catacombs 11 To Catacombs 13"                         : LocationData(key='Cave_11_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_13.11'),
    "Catacombs 11 To Catacombs 18"                         : LocationData(key='Cave_11_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_cave_18.11'),
    "Catacombs 12 To Catacombs 04"                         : LocationData(key='Cave_12_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_04.12'),
    "Catacombs 13 To Catacombs 11"                         : LocationData(key='Cave_13_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_11.13'),
    "Catacombs 13 To Catacombs 14"                         : LocationData(key='Cave_13_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_14.13'),
    "Catacombs 13 To Catacombs 23"                         : LocationData(key='Cave_13_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_23.13'),
    "Catacombs 13 To Catacombs 20"                         : LocationData(key='Cave_13_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_cave_20.13'),
    "Catacombs 14 To Catacombs 13"                         : LocationData(key='Cave_14_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_13.14'),
    "Catacombs 14 To Catacombs 15"                         : LocationData(key='Cave_14_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_15.14'),
    "Catacombs 14 To Catacombs 22"                         : LocationData(key='Cave_14_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_cave_22.14'),
    "Catacombs 15 To Catacombs 14"                         : LocationData(key='Cave_15_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_14.15'),
    "Catacombs 15 To Catacombs 16"                         : LocationData(key='Cave_15_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_16.15'),
    "Catacombs 16 To Catacombs 15"                         : LocationData(key='Cave_16_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_15.16'),
    "Catacombs 16 To Catacombs 10"                         : LocationData(key='Cave_16_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_10.16'),
    "Catacombs 16 To Catacombs 04"                         : LocationData(key='Cave_16_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_04.16'),
    "Catacombs 17 To Catacombs 08"                         : LocationData(key='Cave_17_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_08.17'),
    "Catacombs 18 To Catacombs 11 Upper"                   : LocationData(key='Cave_18_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_11.18'),
    "Catacombs 18 To Catacombs 11 Lower"                   : LocationData(key='Cave_18_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_11.18B'),
    "Catacombs 19 To Catacombs 20"                         : LocationData(key='Cave_19_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_20.19'),
    "Catacombs 19 To Catacombs 21"                         : LocationData(key='Cave_19_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_21.19'),
    "Catacombs 20 To Catacombs 22"                         : LocationData(key='Cave_20_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_22.20'),
    "Catacombs 20 To Catacombs 19"                         : LocationData(key='Cave_20_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_19.20'),
    "Catacombs 20 To Catacombs 13"                         : LocationData(key='Cave_20_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_cave_13.20'),
    "Catacombs 21 To Catacombs 09"                         : LocationData(key='Cave_21_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_09.21'),
    "Catacombs 21 To Catacombs 19"                         : LocationData(key='Cave_21_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_19.21'),
    "Catacombs 22 To Twin Spires 02"                       : LocationData(key='Cave_22_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_fort_02.C22'),
    "Catacombs 22 To Catacombs 14"                         : LocationData(key='Cave_22_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_cave_14.22'),
    "Catacombs 22 To Catacombs 20"                         : LocationData(key='Cave_22_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_20.22'),
    "Catacombs 23 To Catacombs 13"                         : LocationData(key='Cave_23_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_cave_13.23'),
    "Catacombs 23 To Catacombs 10"                         : LocationData(key='Cave_23_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_cave_10.23'),
    "White Parish 01 To White Parish 02"                   : LocationData(key='Church_01_GAMEPLAY.BP_Interactable_WorldTravel',                content='Map.map_church_02.0'),
    "White Parish 01 To White Parish 12"                   : LocationData(key='Church_01_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_church_12.1'),
    "White Parish 02 To White Parish 01"                   : LocationData(key='Church_02_GAMEPLAY.BP_Interactable_WorldTravel',                content='Map.map_church_01.1'),
    "White Parish 02 To White Parish 10"                   : LocationData(key='Church_02_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_church_10.2'),
    "White Parish 03 To White Parish 05"                   : LocationData(key='Church_03_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_church_05.1'),
    "White Parish 03 To White Parish 04"                   : LocationData(key='Church_03_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_church_04.0'),
    "White Parish 04 To White Parish 03"                   : LocationData(key='Church_04_GAMEPLAY.BP_WorldTravelVolume',                       content='Map.map_church_03.1'),
    "White Parish 04 To White Parish 06"                   : LocationData(key='Church_04_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_church_06.0'),
    "White Parish 05 To White Parish 11"                   : LocationData(key='Church_05_GAMEPLAY.BP_Interactable_WorldTravel',                content='Map.map_church_11.5'),
    "White Parish 05 To White Parish 03"                   : LocationData(key='Church_05_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_church_03.0'),
    "White Parish 05 To White Parish 09"                   : LocationData(key='Church_05_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_church_09.5'),
    "White Parish 06 To White Parish 04"                   : LocationData(key='Church_06_GAMEPLAY.BP_WorldTravelVolume',                       content='Map.map_church_04.1'),
    "White Parish 06 To White Parish 07"                   : LocationData(key='Church_06_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_church_07.0'),
    "White Parish 07 To White Parish 06"                   : LocationData(key='Church_07_GAMEPLAY.BP_WorldTravelVolume_3',                     content='Map.map_church_06.1'),
    "White Parish 07 To White Parish 08"                   : LocationData(key='Church_07_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_church_08.0'),
    "White Parish 08 To Cliffside Hamlet 01"               : LocationData(key='Church_08_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_village_01.0'),
    "White Parish 08 To Witch's Thicket 01"                : LocationData(key='Church_08_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_forest_01.C8'),
    "White Parish 08 To White Parish 07"                   : LocationData(key='Church_08_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_church_07.1'),
    "White Parish 09 To White Parish 05"                   : LocationData(key='Church_09_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_church_05.9'),
    "White Parish 09 To White Parish 01"                   : LocationData(key='Church_09_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_church_01.2'),
    "White Parish 09 To White Parish 14"                   : LocationData(key='Church_09_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_church_14.9'),
    "White Parish 10 To White Parish 02"                   : LocationData(key='Church_10_GAMEPLAY.BP_WorldTravelVolume_4',                     content='Map.map_church_02.10'),
    "White Parish 10 To White Parish 11"                   : LocationData(key='Church_10_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_church_11.10'),
    "White Parish 11 To White Parish 05"                   : LocationData(key='Church_11_GAMEPLAY.BP_Interactable_WorldTravel_2',              content='Map.map_church_05.11'),
    "White Parish 11 To White Parish 10"                   : LocationData(key='Church_11_GAMEPLAY.BP_WorldTravelVolume_4',                     content='Map.map_church_10.11'),
    "White Parish 12 To White Parish 01"                   : LocationData(key='Church_12_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_church_01.12'),
    "White Parish 12 To White Parish 13"                   : LocationData(key='Church_12_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_church_13.6'),
    "White Parish 13 To White Parish 12"                   : LocationData(key='Church_13_GAMEPLAY.BP_WorldTravelVolume5',                      content='Map.map_church_12.3'),
    "White Parish 14 To White Parish 09"                   : LocationData(key='Church_14_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_church_09.14'),
    "Witch's Thicket 01 To White Parish 08"                : LocationData(key='Forest_01_GAMEPLAY.BP_WorldTravelVolume4',                      content='Map.map_church_08.2'),
    "Witch's Thicket 01 To Witch's Thicket 02"             : LocationData(key='Forest_01_GAMEPLAY.BP_WorldTravelVolume5',                      content='Map.map_forest_02.1'),
    "Witch's Thicket 02 To Witch's Thicket 01"             : LocationData(key='Forest_02_GAMEPLAY.BP_WorldTravelVolume5',                      content='Map.map_forest_01.2'),
    "Witch's Thicket 02 To Witch's Thicket 04"             : LocationData(key='Forest_02_GAMEPLAY.BP_WorldTravelVolume6',                      content='Map.map_forest_04.2'),
    "Witch's Thicket 02 To Witch's Thicket 03"             : LocationData(key='Forest_02_GAMEPLAY.BP_WorldTravelVolume7',                      content='Map.map_forest_03.2'),
    "Witch's Thicket 03 To Witch's Thicket 05"             : LocationData(key='Forest_03_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_forest_05.3'),
    "Witch's Thicket 03 To Witch's Thicket 02"             : LocationData(key='Forest_03_GAMEPLAY.BP_WorldTravelVolume4',                      content='Map.map_forest_02.3'),
    "Witch's Thicket 04 To Witch's Thicket 05"             : LocationData(key='Forest_04_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_forest_05.4'),
    "Witch's Thicket 04 To Witch's Thicket 02"             : LocationData(key='Forest_04_GAMEPLAY.BP_WorldTravelVolume4',                      content='Map.map_forest_02.4'),
    "Witch's Thicket 05 To Witch's Thicket 07"             : LocationData(key='Forest_05_GAMEPLAY.BP_WorldTravelVolume5',                      content='Map.map_forest_07.5'),
    "Witch's Thicket 05 To Witch's Thicket 03"             : LocationData(key='Forest_05_GAMEPLAY.BP_WorldTravelVolume6',                      content='Map.map_forest_03.5'),
    "Witch's Thicket 05 To Witch's Thicket 04"             : LocationData(key='Forest_05_GAMEPLAY.BP_WorldTravelVolume7',                      content='Map.map_forest_04.6'),
    "Witch's Thicket 06 To Witch's Thicket 07"             : LocationData(key='Forest_06_GAMEPLAY.BP_WorldTravelVolume8',                      content='Map.map_forest_07.6'),
    "Witch's Thicket 07 To Stockade 01"                    : LocationData(key='Forest_07_GAMEPLAY.BP_WorldTravelVolume10_2',                   content='Map.map_oubliette_01.F7'),
    "Witch's Thicket 07 To Witch's Thicket 08"             : LocationData(key='Forest_07_GAMEPLAY.BP_WorldTravelVolume12',                     content='Map.map_forest_08.7'),
    "Witch's Thicket 07 To Witch's Thicket 05"             : LocationData(key='Forest_07_GAMEPLAY.BP_WorldTravelVolume9_1',                    content='Map.map_forest_05.7'),
    "Witch's Thicket 07 To Witch's Thicket 06"             : LocationData(key='Forest_07_GEO.BP_WorldTravelVolume11',                          content='Map.map_forest_06.7'),
    "Witch's Thicket 08 To Witch's Thicket 07"             : LocationData(key='Forest_08_GAMEPLAY.BP_WorldTravelVolume12',                     content='Map.map_forest_07.8'),
    "Witch's Thicket 08 To Witch's Thicket 10"             : LocationData(key='Forest_08_GAMEPLAY.BP_WorldTravelVolume13',                     content='Map.map_forest_10.8'),
    "Witch's Thicket 09 To Witch's Thicket 10"             : LocationData(key='Forest_09_GAMEPLAY.BP_Interactable_WorldTravel_2',              content='Map.map_forest_10.9'),
    "Witch's Thicket 09 To Verboten Domain 02"             : LocationData(key='Forest_09_GAMEPLAY.BP_WorldTravelVolume13',                     content='Map.map_swamp_02.F9'),
    "Witch's Thicket 10 To Witch's Thicket 11"             : LocationData(key='Forest_10_DESIGN.BP_Interactable_WorldTravel10',                content='Map.map_forest_11.10'),
    "Witch's Thicket 10 To Witch's Thicket 09"             : LocationData(key='Forest_10_DESIGN.BP_Interactable_WorldTravel9',                 content='Map.map_forest_09.10'),
    "Witch's Thicket 10 To Witch's Thicket 08"             : LocationData(key='Forest_10_GAMEPLAY.BP_WorldTravelVolume12',                     content='Map.map_forest_08.10'),
    "Witch's Thicket 10 To Witch's Thicket 12"             : LocationData(key='Forest_10_GAMEPLAY.BP_WorldTravelVolume13',                     content='Map.map_forest_12.10'),
    "Witch's Thicket 11 To Witch's Thicket 10"             : LocationData(key='Forest_11_GAMEPLAY.BP_Interactable_WorldTravel11',              content='Map.map_forest_10.11'),
    "Witch's Thicket 11 To Witch's Thicket 14"             : LocationData(key='Forest_11_GAMEPLAY.BP_WorldTravelVolume12',                     content='Map.map_forest_14.11'),
    "Witch's Thicket 12 To Witch's Thicket 10"             : LocationData(key='Forest_12_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_forest_10.12'),
    "Witch's Thicket 12 To Witch's Thicket 13"             : LocationData(key='Forest_12_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_forest_13.12'),
    "Witch's Thicket 12 To Witch's Thicket 17"             : LocationData(key='Forest_12_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_forest_17.12'),
    "Witch's Thicket 13 To Witch's Thicket 12"             : LocationData(key='Forest_13_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_forest_12.13'),
    "Witch's Thicket 13 To Witch's Thicket 14"             : LocationData(key='Forest_13_GAMEPLAY.BP_WorldTravelVolume2',                      content='Map.map_forest_14.13'),
    "Witch's Thicket 13 To Witch's Thicket 16"             : LocationData(key='Forest_13_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_forest_16.13'),
    "Witch's Thicket 14 To Witch's Thicket 15"             : LocationData(key='Forest_14_GAMEPLAY.BP_WorldTravelVolume10',                     content='Map.map_forest_15.14'),
    "Witch's Thicket 14 To Witch's Thicket 13"             : LocationData(key='Forest_14_GAMEPLAY.BP_WorldTravelVolume11',                     content='Map.map_forest_13.14'),
    "Witch's Thicket 14 To Witch's Thicket 11"             : LocationData(key='Forest_14_GAMEPLAY.BP_WorldTravelVolume12',                     content='Map.map_forest_11.14'),
    "Witch's Thicket 15 To Witch's Thicket 14"             : LocationData(key='Forest_15_GAMEPLAY.BP_WorldTravelVolume_2',                     content='Map.map_forest_14.15'),
    "Witch's Thicket 16 To Witch's Thicket 13"             : LocationData(key='Forest_16_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_forest_13.16'),
    "Witch's Thicket 17 To Witch's Thicket 12"             : LocationData(key='Forest_17_GAMEPLAY.BP_WorldTravelVolume3',                      content='Map.map_forest_12.17'),
    "Twin Spires 01 To Twin Spires 03"                     : LocationData(key='Fort_01_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_03.1'),
    "Twin Spires 01 To Cliffside Hamlet 15"                : LocationData(key='Fort_01_GAMEPLAY.BP_WorldTravelVolume2',                        content='Map.map_village_15.F1'),
    "Twin Spires 01 To Ruined Castle 07"                   : LocationData(key='Fort_01_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_castle_07.F1'),
    "Twin Spires 02 To Twin Spires 03"                     : LocationData(key='Fort_02_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_03.2'),
    "Twin Spires 02 To Catacombs 22"                       : LocationData(key='Fort_02_GAMEPLAY.BP_WorldTravelVolume3_8',                      content='Map.map_cave_22.F2'),
    "Twin Spires 03 To Twin Spires 04"                     : LocationData(key='Fort_03_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_04.3'),
    "Twin Spires 03 To Twin Spires 01"                     : LocationData(key='Fort_03_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_01.3'),
    "Twin Spires 03 To Twin Spires 02"                     : LocationData(key='Fort_03_GAMEPLAY.BP_WorldTravelVolume3_8',                      content='Map.map_fort_02.3'),
    "Twin Spires 03 To Twin Spires 05"                     : LocationData(key='Fort_03_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_05.3'),
    "Twin Spires 04 To Twin Spires 05"                     : LocationData(key='Fort_04_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_05.4'),
    "Twin Spires 04 To Twin Spires 03"                     : LocationData(key='Fort_04_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_03.4'),
    "Twin Spires 05 To Twin Spires 04"                     : LocationData(key='Fort_05_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_04.5'),
    "Twin Spires 05 To Twin Spires 06"                     : LocationData(key='Fort_05_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_06.5'),
    "Twin Spires 05 To Twin Spires 03"                     : LocationData(key='Fort_05_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_03.5'),
    "Twin Spires 05 To Twin Spires 15"                     : LocationData(key='Fort_05_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_15.5'),
    "Twin Spires 06 To Twin Spires 05"                     : LocationData(key='Fort_06_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_05.6'),
    "Twin Spires 06 To Twin Spires 07"                     : LocationData(key='Fort_06_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_07.6'),
    "Twin Spires 06 To Twin Spires 10"                     : LocationData(key='Fort_06_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_10.6'),
    "Twin Spires 07 To Twin Spires 09 Left"                : LocationData(key='Fort_07_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_09.7'),
    "Twin Spires 07 To Twin Spires 08"                     : LocationData(key='Fort_07_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_08.7'),
    "Twin Spires 07 To Twin Spires 06"                     : LocationData(key='Fort_07_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_06.7'),
    "Twin Spires 07 To Twin Spires 11"                     : LocationData(key='Fort_07_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_11.7'),
    "Twin Spires 07 To Twin Spires 09 Right"               : LocationData(key='Fort_07_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_fort_09.7B'),
    "Twin Spires 08 To Twin Spires 07"                     : LocationData(key='Fort_08_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_07.8'),
    "Twin Spires 09 To Hinterlands 03"                     : LocationData(key='Fort_09_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_outside_03.F9'),
    "Twin Spires 09 To Twin Spires 10"                     : LocationData(key='Fort_09_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_10.9'),
    "Twin Spires 09 To Twin Spires 07 Left"                : LocationData(key='Fort_09_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_07.9'),
    "Twin Spires 09 To Twin Spires 07 Right"               : LocationData(key='Fort_09_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_fort_07.9B'),
    "Twin Spires 10 To Twin Spires 09"                     : LocationData(key='Fort_10_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_09.10'),
    "Twin Spires 10 To Twin Spires 06"                     : LocationData(key='Fort_10_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_06.10'),
    "Twin Spires 11 To Twin Spires 07"                     : LocationData(key='Fort_11_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_07.11'),
    "Twin Spires 11 To Twin Spires 12"                     : LocationData(key='Fort_11_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_12.11'),
    "Twin Spires 11 To Twin Spires 13 Left"                : LocationData(key='Fort_11_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_13.11'),
    "Twin Spires 11 To Twin Spires 13 Right"               : LocationData(key='Fort_11_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_fort_13.11B'),
    "Twin Spires 12 To Twin Spires 11"                     : LocationData(key='Fort_12_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_11.12'),
    "Twin Spires 12 To Twin Spires 14"                     : LocationData(key='Fort_12_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_14.12'),
    "Twin Spires 12 To Twin Spires 16"                     : LocationData(key='Fort_12_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_16.12'),
    "Twin Spires 13 To Twin Spires 11 Left"                : LocationData(key='Fort_13_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_11.13'),
    "Twin Spires 13 To Twin Spires 11 Right"               : LocationData(key='Fort_13_GAMEPLAY.BP_WorldTravelVolume2',                        content='Map.map_fort_11.53'),
    "Twin Spires 13 To Twin Spires 14"                     : LocationData(key='Fort_13_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_14.13'),
    "Twin Spires 13 To Twin Spires 19"                     : LocationData(key='Fort_13_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_19.13'),
    "Twin Spires 14 To Twin Spires 15"                     : LocationData(key='Fort_14_GAMEPLAY.BP_WorldTravelVolume_0',                       content='Map.map_fort_15.14'),
    "Twin Spires 14 To Twin Spires 13"                     : LocationData(key='Fort_14_GAMEPLAY.BP_WorldTravelVolume2_1',                      content='Map.map_fort_13.14'),
    "Twin Spires 14 To Twin Spires 12"                     : LocationData(key='Fort_14_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_12.14'),
    "Twin Spires 15 To Twin Spires 16 Upper"               : LocationData(key='Fort_15_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_16.15'),
    "Twin Spires 15 To Twin Spires 14"                     : LocationData(key='Fort_15_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_14.15'),
    "Twin Spires 15 To Twin Spires 05"                     : LocationData(key='Fort_15_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_05.15'),
    "Twin Spires 15 To Twin Spires 17"                     : LocationData(key='Fort_15_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_17.15'),
    "Twin Spires 15 To Twin Spires 16 Lower"               : LocationData(key='Fort_15_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_fort_16.15B'),
    "Twin Spires 16 To Twin Spires 15 Upper"               : LocationData(key='Fort_16_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_15.16'),
    "Twin Spires 16 To Twin Spires 18"                     : LocationData(key='Fort_16_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_fort_18.16'),
    "Twin Spires 16 To Twin Spires 15 Lower"               : LocationData(key='Fort_16_GAMEPLAY.BP_WorldTravelVolume6',                        content='Map.map_fort_15.16B'),
    "Twin Spires 16 To Twin Spires 12"                     : LocationData(key='Fort_16_GAMEPLAY.BP_WorldTravelVolume7',                        content='Map.map_fort_12.16'),
    "Twin Spires 17 To Twin Spires 15"                     : LocationData(key='Fort_17_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_15.17'),
    "Twin Spires 17 To Twin Spires 18"                     : LocationData(key='Fort_17_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_18.17'),
    "Twin Spires 18 To Twin Spires 17"                     : LocationData(key='Fort_18_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_17.18'),
    "Twin Spires 18 To Twin Spires 19"                     : LocationData(key='Fort_18_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_19.18'),
    "Twin Spires 18 To Twin Spires 16"                     : LocationData(key='Fort_18_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_16.18'),
    "Twin Spires 19 To Twin Spires 18"                     : LocationData(key='Fort_19_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_fort_18.19'),
    "Twin Spires 19 To Twin Spires 20"                     : LocationData(key='Fort_19_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_fort_20.19'),
    "Twin Spires 19 To Twin Spires 13"                     : LocationData(key='Fort_19_GAMEPLAY.BP_WorldTravelVolume5',                        content='Map.map_fort_13.19'),
    "Twin Spires 20 To Twin Spires 19"                     : LocationData(key='Fort_20_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_fort_19.20'),
    "Twin Spires 20 To Twin Spires 21"                     : LocationData(key='Fort_20_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_21.20'),
    "Twin Spires 21 To Twin Spires 20"                     : LocationData(key='Fort_21_GAMEPLAY.BP_WorldTravelVolume2_5',                      content='Map.map_fort_20.21'),
    "Stockade 01 To Witch's Thicket 07"                    : LocationData(key='Oubliette_01_GAMEPLAY.BP_WorldTravelVolume11',                  content='Map.map_forest_07.O1'),
    "Stockade 01 To Stockade 02"                           : LocationData(key='Oubliette_01_GAMEPLAY.BP_WorldTravelVolume12',                  content='Map.map_oubliette_02.1'),
    "Stockade 02 To Stockade 01"                           : LocationData(key='Oubliette_02_GAMEPLAY.BP_WorldTravelVolume11',                  content='Map.map_oubliette_01.2'),
    "Stockade 02 To Stockade 05"                           : LocationData(key='Oubliette_02_GAMEPLAY.BP_WorldTravelVolume12',                  content='Map.map_oubliette_05.2'),
    "Stockade 02 To Stockade 04"                           : LocationData(key='Oubliette_02_GAMEPLAY.BP_WorldTravelVolume13',                  content='Map.map_oubliette_04.2'),
    "Stockade 03 To Stockade 04"                           : LocationData(key='Oubliette_03_GAMEPLAY.BP_WorldTravelVolume13',                  content='Map.map_oubliette_04.3'),
    "Stockade 03 To Stockade 10"                           : LocationData(key='Oubliette_03_GAMEPLAY.BP_WorldTravelVolume14',                  content='Map.map_oubliette_10.3'),
    "Stockade 03 To Stockade 05"                           : LocationData(key='Oubliette_03_GAMEPLAY.BP_WorldTravelVolume15',                  content='Map.map_oubliette_05.3'),
    "Stockade 04 To Stockade 03"                           : LocationData(key='Oubliette_04_GAMEPLAY.BP_WorldTravelVolume14',                  content='Map.map_oubliette_03.4'),
    "Stockade 04 To Stockade 02"                           : LocationData(key='Oubliette_04_GAMEPLAY.BP_WorldTravelVolume16',                  content='Map.map_oubliette_02.4'),
    "Stockade 05_1 To Stockade 05"                         : LocationData(key='Oubliette_05_1_GAMEPLAY.BP_Interactable_WorldTravel13',         content='Map.map_oubliette_05.5_1'),
    "Stockade 05_2 To Stockade 05 Left"                    : LocationData(key='Oubliette_05_2_GAMEPLAY.BP_Interactable_WorldTravel12',         content='Map.map_oubliette_05.5_2'),
    "Stockade 05_2 To Stockade 05 Right"                   : LocationData(key='Oubliette_05_2_GAMEPLAY.BP_WorldTravelVolume18',                content='Map.map_oubliette_05.5_2B'),
    "Stockade 05_3 To Stockade 05"                         : LocationData(key='Oubliette_05_3_GAMEPLAY.BP_Interactable_WorldTravel16',         content='Map.map_oubliette_05.5_3'),
    "Stockade 05 To Stockade 05_2 Left"                    : LocationData(key='Oubliette_05_GAMEPLAY.BP_Interactable_WorldTravel10',           content='Map.map_oubliette_05_2.5'),
    "Stockade 05 To Stockade 05_3"                         : LocationData(key='Oubliette_05_GAMEPLAY.BP_Interactable_WorldTravel11',           content='Map.map_oubliette_05_3.5'),
    "Stockade 05 To Stockade 07_2"                         : LocationData(key='Oubliette_05_GAMEPLAY.BP_Interactable_WorldTravel12',           content='Map.map_oubliette_07_2.5'),
    "Stockade 05 To Stockade 05_1"                         : LocationData(key='Oubliette_05_GAMEPLAY.BP_Interactable_WorldTravel13',           content='Map.map_oubliette_05_1.5'),
    "Stockade 05 To Stockade 07_1"                         : LocationData(key='Oubliette_05_GAMEPLAY.BP_Interactable_WorldTravel14',           content='Map.map_oubliette_07_1.5'),
    "Stockade 05 To Stockade 02"                           : LocationData(key='Oubliette_05_GAMEPLAY.BP_WorldTravelVolume15',                  content='Map.map_oubliette_02.5'),
    "Stockade 05 To Stockade 06"                           : LocationData(key='Oubliette_05_GAMEPLAY.BP_WorldTravelVolume16',                  content='Map.map_oubliette_06.5'),
    "Stockade 05 To Stockade 03"                           : LocationData(key='Oubliette_05_GAMEPLAY.BP_WorldTravelVolume17',                  content='Map.map_oubliette_03.5'),
    "Stockade 05 To Stockade 05_2 Right"                   : LocationData(key='Oubliette_05_GAMEPLAY.BP_WorldTravelVolume18',                  content='Map.map_oubliette_05_2.5B'),
    "Stockade 06_1 To Stockade 07"                         : LocationData(key='Oubliette_06_1_GAMEPLAY.BP_Interactable_WorldTravel14',         content='Map.map_oubliette_07.6_1'),
    "Stockade 06_2 To Stockade 07"                         : LocationData(key='Oubliette_06_2_GAMEPLAY.BP_Interactable_WorldTravel13',         content='Map.map_oubliette_07.6_2'),
    "Stockade 06_3 To Stockade 07"                         : LocationData(key='Oubliette_06_3_GAMEPLAY.BP_WorldTravelVolume16',                content='Map.map_oubliette_07.6_3'),
    "Stockade 06_4 To Stockade 07"                         : LocationData(key='Oubliette_06_4_GAMEPLAY.BP_Interactable_WorldTravel16',         content='Map.map_oubliette_07.6_4'),
    "Stockade 06 To Stockade 05"                           : LocationData(key='Oubliette_06_GAMEPLAY.BP_WorldTravelVolume20',                  content='Map.map_oubliette_05.6'),
    "Stockade 06 To Stockade 07"                           : LocationData(key='Oubliette_06_GAMEPLAY.BP_WorldTravelVolume21',                  content='Map.map_oubliette_07.6'),
    "Stockade 06 To Stockade 10"                           : LocationData(key='Oubliette_06_GAMEPLAY.BP_WorldTravelVolume22',                  content='Map.map_oubliette_10.6'),
    "Stockade 07_1 To Stockade 05"                         : LocationData(key='Oubliette_07_1_GAMEPLAY.BP_Interactable_WorldTravel14',         content='Map.map_oubliette_05.7_1'),
    "Stockade 07_2 To Stockade 05"                         : LocationData(key='Oubliette_07_2_GAMEPLAY.BP_Interactable_WorldTravel14',         content='Map.map_oubliette_05.7_2'),
    "Stockade 07 To Stockade 06_1"                         : LocationData(key='Oubliette_07_GAMEPLAY.BP_Interactable_WorldTravel17',           content='Map.map_oubliette_06_1.7'),
    "Stockade 07 To Stockade 06_3"                         : LocationData(key='Oubliette_07_GAMEPLAY.BP_Interactable_WorldTravel18',           content='Map.map_oubliette_06_3.7'),
    "Stockade 07 To Stockade 06_2"                         : LocationData(key='Oubliette_07_GAMEPLAY.BP_Interactable_WorldTravel19',           content='Map.map_oubliette_06_2.7'),
    "Stockade 07 To Stockade 06_4"                         : LocationData(key='Oubliette_07_GAMEPLAY.BP_Interactable_WorldTravel20',           content='Map.map_oubliette_06_4.7'),
    "Stockade 07 To Stockade 06"                           : LocationData(key='Oubliette_07_GAMEPLAY.BP_WorldTravelVolume19',                  content='Map.map_oubliette_06.7'),
    "Stockade 07 To Stockade 13"                           : LocationData(key='Oubliette_07_GAMEPLAY.BP_WorldTravelVolume20',                  content='Map.map_oubliette_13.7'),
    "Stockade 07 To Stockade 09"                           : LocationData(key='Oubliette_07_GAMEPLAY.BP_WorldTravelVolume21',                  content='Map.map_oubliette_09.7'),
    "Stockade 08 To Stockade 11"                           : LocationData(key='Oubliette_08_GAMEPLAY.BP_WorldTravelVolume15',                  content='Map.map_oubliette_11.8'),
    "Stockade 08 To Stockade 09"                           : LocationData(key='Oubliette_08_GAMEPLAY.BP_WorldTravelVolume16',                  content='Map.map_oubliette_09.8'),
    "Stockade 08 To Stockade 13"                           : LocationData(key='Oubliette_08_GAMEPLAY.BP_WorldTravelVolume17',                  content='Map.map_oubliette_13.8'),
    "Stockade 09 To Stockade 08"                           : LocationData(key='Oubliette_09_GAMEPLAY.BP_WorldTravelVolume17',                  content='Map.map_oubliette_08.9'),
    "Stockade 09 To Stockade 10"                           : LocationData(key='Oubliette_09_GAMEPLAY.BP_WorldTravelVolume18',                  content='Map.map_oubliette_10.9'),
    "Stockade 09 To Stockade 07"                           : LocationData(key='Oubliette_09_GAMEPLAY.BP_WorldTravelVolume19',                  content='Map.map_oubliette_07.9'),
    "Stockade 10 To Stockade 09"                           : LocationData(key='Oubliette_10_GAMEPLAY.BP_WorldTravelVolume17',                  content='Map.map_oubliette_09.10'),
    "Stockade 10 To Stockade 03"                           : LocationData(key='Oubliette_10_GAMEPLAY.BP_WorldTravelVolume18',                  content='Map.map_oubliette_03.10'),
    "Stockade 10 To Stockade 17"                           : LocationData(key='Oubliette_10_GAMEPLAY.BP_WorldTravelVolume19',                  content='Map.map_oubliette_17.10'),
    "Stockade 10 To Stockade 06"                           : LocationData(key='Oubliette_10_GAMEPLAY.BP_WorldTravelVolume20',                  content='Map.map_oubliette_06.10'),
    "Stockade 11 To Stockade 12"                           : LocationData(key='Oubliette_11_GAMEPLAY.BP_WorldTravelVolume19',                  content='Map.map_oubliette_12.11'),
    "Stockade 11 To Stockade 08"                           : LocationData(key='Oubliette_11_GAMEPLAY.BP_WorldTravelVolume20',                  content='Map.map_oubliette_08.11'),
    "Stockade 11 To Stockade 14"                           : LocationData(key='Oubliette_11_GAMEPLAY.BP_WorldTravelVolume21',                  content='Map.map_oubliette_14.11'),
    "Stockade 11 To Stockade 13"                           : LocationData(key='Oubliette_11_GAMEPLAY.BP_WorldTravelVolume22',                  content='Map.map_oubliette_13.11'),
    "Stockade 11 To Stockade 13_2"                         : LocationData(key='Oubliette_11_GAMEPLAY.BP_WorldTravelVolume23',                  content='Map.map_oubliette_13_2.11'),
    "Stockade 11 To Stockade 13_1"                         : LocationData(key='Oubliette_11_GAMEPLAY.BP_WorldTravelVolume24',                  content='Map.map_oubliette_13_1.11'),
    "Stockade 12 To Stockade 11"                           : LocationData(key='Oubliette_12_GAMEPLAY.BP_WorldTravelVolume22_0',                content='Map.map_oubliette_11.12'),
    "Stockade 13_1 To Stockade 11"                         : LocationData(key='Oubliette_13_1_GAMEPLAY.BP_WorldTravelVolume24',                content='Map.map_oubliette_11.13_1'),
    "Stockade 13_2 To Stockade 11"                         : LocationData(key='Oubliette_13_2_GAMEPLAY.BP_WorldTravelVolume23',                content='Map.map_oubliette_11.13_2'),
    "Stockade 13 To Stockade 08"                           : LocationData(key='Oubliette_13_GAMEPLAY.BP_WorldTravelVolume21',                  content='Map.map_oubliette_08.13'),
    "Stockade 13 To Stockade 11"                           : LocationData(key='Oubliette_13_GAMEPLAY.BP_WorldTravelVolume22',                  content='Map.map_oubliette_11.13'),
    "Stockade 13 To Stockade 07"                           : LocationData(key='Oubliette_13_GAMEPLAY.BP_WorldTravelVolume23',                  content='Map.map_oubliette_07.13'),
    "Stockade 14 To Stockade 11"                           : LocationData(key='Oubliette_14_GAMEPLAY.BP_WorldTravelVolume13',                  content='Map.map_oubliette_11.14'),
    "Stockade 14 To Stockade 15"                           : LocationData(key='Oubliette_14_GAMEPLAY.BP_WorldTravelVolume14',                  content='Map.map_oubliette_15.14'),
    "Stockade 15 To Stockade 14"                           : LocationData(key='Oubliette_15_GAMEPLAY.BP_WorldTravelVolume15',                  content='Map.map_oubliette_14.15'),
    "Stockade 15 To Stockade 16"                           : LocationData(key='Oubliette_15_GAMEPLAY.BP_WorldTravelVolume16',                  content='Map.map_oubliette_16.15'),
    "Stockade 16 To Stockade 15"                           : LocationData(key='Oubliette_16_GAMEPLAY.BP_WorldTravelVolume17',                  content='Map.map_oubliette_15.16'),
    "Stockade 16 To Hinterlands 01"                        : LocationData(key='Oubliette_16_GAMEPLAY.BP_WorldTravelVolume18',                  content='Map.map_outside_01.O16'),
    "Stockade 17 To Stockade 10"                           : LocationData(key='Oubliette_17_GAMEPLAY.BP_WorldTravelVolume19',                  content='Map.map_oubliette_10.17'),
    "Stockade 17 To Verboten Domain 06"                    : LocationData(key='Oubliette_17_GAMEPLAY.BP_WorldTravelVolume20',                  content='Map.map_swamp_06.O17'),
    "Hinterlands 01 To Hinterlands 02"                     : LocationData(key='Outside_01_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_outside_02.1'),
    "Hinterlands 01 To Hinterlands 03"                     : LocationData(key='Outside_01_GAMEPLAY.BP_WorldTravelVolume3_8',                   content='Map.map_outside_03.1'),
    "Hinterlands 01 To Stockade 16"                        : LocationData(key='Outside_01_GAMEPLAY.BP_WorldTravelVolume4',                     content='Map.map_oubliette_16.O1'),
    "Hinterlands 02 To Hinterlands 01"                     : LocationData(key='Outside_02_GAMEPLAY.BP_WorldTravelVolume3_8',                   content='Map.map_outside_01.2'),
    "Hinterlands 03 To Hinterlands 01"                     : LocationData(key='Outside_03_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_outside_01.3'),
    "Hinterlands 03 To Twin Spires 09"                     : LocationData(key='Outside_03_GAMEPLAY.BP_WorldTravelVolume3_8',                   content='Map.map_fort_09.O3'),
    "Verboten Domain 04 To Verboten Domain 05"             : LocationData(key='Swamp_04_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_05.4'),
    "Verboten Domain 04 To Verboten Domain 3R"             : LocationData(key='Swamp_04_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_03.4'),
    "Verboten Domain 05 To Verboten Domain 04"             : LocationData(key='Swamp_05_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_04.5'),
    "Verboten Domain 05 To Verboten Domain 06"             : LocationData(key='Swamp_05_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_06.5'),
    "Verboten Domain 05 To Verboten Domain 07"             : LocationData(key='Swamp_05_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_07.5'),
    "Verboten Domain 05 To Verboten Domain 09"             : LocationData(key='Swamp_05_GAMEPLAY.BP_WorldTravelVolume4',                       content='Map.map_swamp_09.5'),
    "Verboten Domain 06 To Verboten Domain 05"             : LocationData(key='Swamp_06_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_05.6'),
    "Verboten Domain 06 To Stockade 17"                    : LocationData(key='Swamp_06_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_oubliette_17.S6'),
    "Verboten Domain 07 To Verboten Domain 16"             : LocationData(key='Swamp_07_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_16.7'),
    "Verboten Domain 07 To Verboten Domain 05"             : LocationData(key='Swamp_07_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_05.7'),
    "Verboten Domain 07 To Verboten Domain 08"             : LocationData(key='Swamp_07_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_08.7'),
    "Verboten Domain 07 To Verboten Domain 3B"             : LocationData(key='Swamp_07_GAMEPLAY.BP_WorldTravelVolume4',                       content='Map.map_swamp_03.7'),
    "Verboten Domain 08 To Verboten Domain 07"             : LocationData(key='Swamp_08_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_07.8'),
    "Verboten Domain 08 To Verboten Domain 15"             : LocationData(key='Swamp_08_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_15.8'),
    "Verboten Domain 08 To Verboten Domain 14"             : LocationData(key='Swamp_08_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_14.8'),
    "Verboten Domain 09 To Verboten Domain 13 Left"        : LocationData(key='Swamp_09_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_13.9'),
    "Verboten Domain 09 To Verboten Domain 03"             : LocationData(key='Swamp_09_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_03.9'),
    "Verboten Domain 09 To Verboten Domain 13 Right"       : LocationData(key='Swamp_09_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_13.9B'),
    "Verboten Domain 09 To Verboten Domain 05"             : LocationData(key='Swamp_09_GAMEPLAY.BP_WorldTravelVolume4',                       content='Map.map_swamp_05.9'),
    "Verboten Domain 01 To Verboten Domain 03"             : LocationData(key='Swamp_1_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_swamp_03.1'),
    "Verboten Domain 01 To Verboten Domain 02"             : LocationData(key='Swamp_1_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_swamp_02.1'),
    "Verboten Domain 10 To Verboten Domain 13"             : LocationData(key='Swamp_10_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_13.10'),
    "Verboten Domain 11 To Verboten Domain 15"             : LocationData(key='Swamp_11_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_15.11'),
    "Verboten Domain 11 To Verboten Domain 14"             : LocationData(key='Swamp_11_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_14.11'),
    "Verboten Domain 12 To Verboten Domain 15"             : LocationData(key='Swamp_12_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_15.12'),
    "Verboten Domain 12 To The Abyss 04"                   : LocationData(key='Swamp_12_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_abyss_04.2'),
    "Verboten Domain 13 To Verboten Domain 10"             : LocationData(key='Swamp_13_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_10.13'),
    "Verboten Domain 13 To Verboten Domain 09 Left"        : LocationData(key='Swamp_13_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_09.13'),
    "Verboten Domain 13 To Verboten Domain 09 Right"       : LocationData(key='Swamp_13_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_09.13B'),
    "Verboten Domain 13 To Verboten Domain 14"             : LocationData(key='Swamp_13_GAMEPLAY.BP_WorldTravelVolume4',                       content='Map.map_swamp_14.13'),
    "Verboten Domain 14 To Verboten Domain 13"             : LocationData(key='Swamp_14_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_13.14'),
    "Verboten Domain 14 To Verboten Domain 11"             : LocationData(key='Swamp_14_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_11.14'),
    "Verboten Domain 14 To Verboten Domain 08"             : LocationData(key='Swamp_14_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_08.14'),
    "Verboten Domain 15 To Verboten Domain 11"             : LocationData(key='Swamp_15_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_11.15'),
    "Verboten Domain 15 To Verboten Domain 08"             : LocationData(key='Swamp_15_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_08.15'),
    "Verboten Domain 15 To Verboten Domain 12"             : LocationData(key='Swamp_15_GAMEPLAY.BP_WorldTravelVolume4',                       content='Map.map_swamp_12.15'),
    "Verboten Domain 16 To Verboten Domain 07"             : LocationData(key='Swamp_16_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_07.16'),
    "Verboten Domain 16 To Verboten Domain 17"             : LocationData(key='Swamp_16_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_17.16'),
    "Verboten Domain 17 To Verboten Domain 18"             : LocationData(key='Swamp_17_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_swamp_18.17'),
    "Verboten Domain 17 To Verboten Domain 16"             : LocationData(key='Swamp_17_GAMEPLAY.BP_WorldTravelVolume2',                       content='Map.map_swamp_16.17'),
    "Verboten Domain 18 To The Abyss 01"                   : LocationData(key='Swamp_18_GAMEPLAY.BP_WorldTravelVolume_2',                      content='Map.map_abyss_01.S18'),
    "Verboten Domain 18 To Verboten Domain 17"             : LocationData(key='Swamp_18_GAMEPLAY.BP_WorldTravelVolume3',                       content='Map.map_swamp_17.18'),
    "Verboten Domain 02 To Verboten Domain 01"             : LocationData(key='Swamp_2_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_swamp_01.2'),
    "Verboten Domain 02 To Witch's Thicket 09"             : LocationData(key='Swamp_2_GAMEPLAY.BP_WorldTravelVolume2',                        content='Map.map_forest_09.S2'),
    "Verboten Domain 03 To Verboten Domain 09"             : LocationData(key='Swamp_3_GAMEPLAY.BP_WorldTravelVolume_2',                       content='Map.map_swamp_09.3'),
    "Verboten Domain 03 To Verboten Domain 01"             : LocationData(key='Swamp_3_GAMEPLAY.BP_WorldTravelVolume2',                        content='Map.map_swamp_01.3'),
    "Verboten Domain 03 To Verboten Domain 04"             : LocationData(key='Swamp_3_GAMEPLAY.BP_WorldTravelVolume3',                        content='Map.map_swamp_04.3'),
    "Verboten Domain 03 To Verboten Domain 07"             : LocationData(key='Swamp_3_GAMEPLAY.BP_WorldTravelVolume4',                        content='Map.map_swamp_07.3'),
    "Cliffside Hamlet 01 To Cliffside Hamlet 02"           : LocationData(key='Village_01_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_02.0'),
    "Cliffside Hamlet 01 To White Parish 08"               : LocationData(key='Village_01_GAMEPLAY.BP_WorldTravelVolume2',                     content='Map.map_church_08.1'),
    "Cliffside Hamlet 02 To Cliffside Hamlet 03"           : LocationData(key='Village_02_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_03.0'),
    "Cliffside Hamlet 02 To Cliffside Hamlet 01"           : LocationData(key='Village_02_GAMEPLAY.BP_WorldTravelVolume2',                     content='Map.map_village_01.1'),
    "Cliffside Hamlet 02 To Cliffside Hamlet 13"           : LocationData(key='Village_02_GAMEPLAY.BP_WorldTravelVolume3',                     content='Map.map_village_13.2'),
    "Cliffside Hamlet 03 To Cliffside Hamlet 02"           : LocationData(key='Village_03_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_02.1'),
    "Cliffside Hamlet 03 To Cliffside Hamlet 05"           : LocationData(key='Village_03_GAMEPLAY.BP_WorldTravelVolume2',                     content='Map.map_village_05.3'),
    "Cliffside Hamlet 03 To Cliffside Hamlet 13"           : LocationData(key='Village_03_GAMEPLAY.BP_WorldTravelVolume3',                     content='Map.map_village_13.0'),
    "Cliffside Hamlet 04_1 To Cliffside Hamlet 04"         : LocationData(key='Village_04_1_GAMEPLAY.BP_WorldTravelVolume_2',                  content='Map.map_village_04.2'),
    "Cliffside Hamlet 04 To Cliffside Hamlet 04"           : LocationData(key='Village_04_GAMEPLAY.BP_Interactable_WorldTravel_2',             content='Map.map_village_04_1.0'),
    "Cliffside Hamlet 04 To Cliffside Hamlet 05"           : LocationData(key='Village_04_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_05.4'),
    "Cliffside Hamlet 05 To Cliffside Hamlet 04"           : LocationData(key='Village_05_GAMEPLAY.BP_Interactable_WorldTravel_2',             content='Map.map_village_04.5'),
    "Cliffside Hamlet 05 To Cliffside Hamlet 03"           : LocationData(key='Village_05_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_03.5'),
    "Cliffside Hamlet 05 To Cliffside Hamlet 06"           : LocationData(key='Village_05_GAMEPLAY.BP_WorldTravelVolume2',                     content='Map.map_village_06.0'),
    "Cliffside Hamlet 06 To Cliffside Hamlet 05"           : LocationData(key='Village_06_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_05.1'),
    "Cliffside Hamlet 06 To Cliffside Hamlet 07"           : LocationData(key='Village_06_GAMEPLAY.BP_WorldTravelVolume2',                     content='Map.map_village_07.0'),
    "Cliffside Hamlet 06 To Cliffside Hamlet 12"           : LocationData(key='Village_06_GAMEPLAY.BP_WorldTravelVolume3',                     content='Map.map_village_12.6'),
    "Cliffside Hamlet 06 To Cliffside Hamlet 08"           : LocationData(key='Village_06_GAMEPLAY.BP_WorldTravelVolume4',                     content='Map.map_village_08.6'),
    "Cliffside Hamlet 07 To Cliffside Hamlet 06"           : LocationData(key='Village_07_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_06.1'),
    "Cliffside Hamlet 07 To Cliffside Hamlet 09"           : LocationData(key='Village_07_GAMEPLAY.BP_WorldTravelVolume2_5',                   content='Map.map_village_09.7'),
    "Cliffside Hamlet 07 To Cliffside Hamlet 14"           : LocationData(key='Village_07_GAMEPLAY.BP_WorldTravelVolume3_8',                   content='Map.map_village_14.7'),
    "Cliffside Hamlet 08 To Cliffside Hamlet 06"           : LocationData(key='Village_08_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_06.8'),
    "Cliffside Hamlet 08 To Cliffside Hamlet 09"           : LocationData(key='Village_08_GAMEPLAY.BP_WorldTravelVolume2_5',                   content='Map.map_village_09.8'),
    "Cliffside Hamlet 09 To Cliffside Hamlet 10"           : LocationData(key='Village_09_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_10.0'),
    "Cliffside Hamlet 09 To Cliffside Hamlet 15"           : LocationData(key='Village_09_GAMEPLAY.BP_WorldTravelVolume2',                     content='Map.map_village_15.0'),
    "Cliffside Hamlet 09 To Cliffside Hamlet 07"           : LocationData(key='Village_09_GAMEPLAY.BP_WorldTravelVolume3',                     content='Map.map_village_07.9'),
    "Cliffside Hamlet 09 To Cliffside Hamlet 08"           : LocationData(key='Village_09_GAMEPLAY.BP_WorldTravelVolume4',                     content='Map.map_village_08.9'),
    "Cliffside Hamlet 10 To Cliffside Hamlet 09"           : LocationData(key='Village_10_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_09.1'),
    "Cliffside Hamlet 10 To Cliffside Hamlet 11"           : LocationData(key='Village_10_GAMEPLAY.BP_WorldTravelVolume2',                     content='Map.map_village_11.0'),
    "Cliffside Hamlet 11_1 To Cliffside Hamlet 11"         : LocationData(key='Village_11_1_GAMEPLAY.BP_WorldTravelVolume_2',                  content='Map.map_village_11.2'),
    "Cliffside Hamlet 11 To Cliffside Hamlet 11_1"         : LocationData(key='Village_11_GAMEPLAY.BP_Interactable_WorldTravel_2',             content='Map.map_village_11_1.0'),
    "Cliffside Hamlet 11 To Cliffside Hamlet 10"           : LocationData(key='Village_11_GAMEPLAY.BP_WorldTravelVolume_4',                    content='Map.map_village_10.1'),
    "Cliffside Hamlet 11 To Ruined Castle 01"              : LocationData(key='Village_11_GAMEPLAY.BP_WorldTravelVolume2',                     content='Map.map_castle_01.0'),
    "Cliffside Hamlet 12 To Cliffside Hamlet 06"           : LocationData(key='Village_12_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_06.12'),
    "Cliffside Hamlet 12 To Cliffside Hamlet 13"           : LocationData(key='Village_12_GAMEPLAY.BP_WorldTravelVolume2',                     content='Map.map_village_13.12'),
    "Cliffside Hamlet 12 To Catacombs 01"                  : LocationData(key='Village_12_GAMEPLAY.BP_WorldTravelVolume3',                     content='Map.map_cave_01.V12'),
    "Cliffside Hamlet 12 To Cliffside Hamlet 16"           : LocationData(key='Village_12_GAMEPLAY.BP_WorldTravelVolume4',                     content='Map.map_village_16.12'),
    "Cliffside Hamlet 13 To Cliffside Hamlet 03"           : LocationData(key='Village_13_GAMEPLAY.BP_WorldTravelVolume4',                     content='Map.map_village_03.2'),
    "Cliffside Hamlet 13 To Cliffside Hamlet 02"           : LocationData(key='Village_13_GAMEPLAY.BP_WorldTravelVolume5',                     content='Map.map_village_02.13'),
    "Cliffside Hamlet 13 To Cliffside Hamlet 12"           : LocationData(key='Village_13_GAMEPLAY.BP_WorldTravelVolume6',                     content='Map.map_village_12.13'),
    "Cliffside Hamlet 14 To Cliffside Hamlet 07"           : LocationData(key='Village_14_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_07.14'),
    "Cliffside Hamlet 15 To Cliffside Hamlet 09"           : LocationData(key='Village_15_GAMEPLAY.BP_WorldTravelVolume_2',                    content='Map.map_village_09.2'),
    "Cliffside Hamlet 15 To Twin Spires 01"                : LocationData(key='Village_15_GAMEPLAY.BP_WorldTravelVolume2_5',                   content='Map.map_fort_01.V15'),
    "Cliffside Hamlet 16 To Cliffside Hamlet 12"           : LocationData(key='Village_16_GAMEPLAY.BP_WorldTravelVolume3',                     content='Map.map_village_12.16'),
                                                           
    'Ending_A'                                             : LocationData(content='Ending_A'),
    'Ending_B'                                             : LocationData(content='Ending_B'),
    'Ending_C'                                             : LocationData(content='Ending_C'),

}
